/*
 * GLX Hardware Device Driver for S3 Savage3D and probably Savage/MX and
 * Savage/IX
 * Copyright (C) 2000 Dominik Behr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on S3 Virge driver by Jim Duchek <jimduchek@ou.edu>
 *
 * Dominik Behr <behr@promail.pl>
 * thanks to Raja Koduri and Tim Roberts   
 */

#ifndef __s3savglx_h
#define __s3savglx_h

#include "types.h"              /* MESA types */
#ifndef MESA31
#error "The S3 Savage driver requires Mesa 3.1 or higher"
#endif
#include "xsmesaP.h"    /* GLX header */

#include "hwtypes.h"
#include "hwlog.h"
#include "s3savpriv.h"
#include "s3savhw.h"
#include "mm.h"

/*******************************************************************************
 settings
*******************************************************************************/

/* 
 this one is strange
 there are some #ifdef REV_B in metal source and it seems
 they are sending last vertex two times through BCI
 */
#define DUPLICATE_LAST_VERTEX

/*
 if this one is defined shadow status in memory is used instead
 of reading status regs through MMIO
 reading MMIO status reg can hang bus
*/
#define USE_SHADOW_STATUS

/*
 if hardware culling on
 software culling is better
 and allows two sided lighting
*/
#define noHARDCULL

/*
 enable direct
*/
#define DIRECT_ENABLE

/*
 insert shadow status update command after every primitive,
 fill, blit and renderstate update
*/
#define FORCE_SHADOW_UPDATE

/*
 insert bci command wait for 3d idle after every triangle
 its no use
*/
#define noWAIT_AFTER_EVERY_TRIANGLE

/*
 DEBUG option
 if this is definied after every pseudo bci flush
 doflush waits till bci buffer becomes empty
*/
#define noFLUSHWAITEMPTY

// if pseudo bci size exceeds this value it flushes
#define BCI_FLUSH_SIZE 4096
//#define BCI_FLUSH_SIZE 256

#define COB_TRESHOLDS 0x68001800

// DEBUG option
#define noDEBUG
// DEBUG option - fprintf functions called
#define noTRACE
// DEBUG option - insert events on status update, triangle, clears, fills
#define noEVENT_TRACE

#define ALIGN16(a) (((a) + 0xF) & 0xFFFFFFF0)
#define ALIGN32(a) (((a) + 0x1F) & 0xFFFFFFE0)
#define ALIGN64(a) (((a) + 0x3F) & 0xFFFFFFC0)

#define S3SAV_SHMKEY 'S3SV'

int
s3savHookServerSymbols(void *handle);



/******************************************************************************

******************************************************************************/
typedef struct S3SAVSURFACEtag {
 PMemBlock pMemBlock;

 unsigned char *p; // mapped pointer

 unsigned int nWidth; // width in pixels
 unsigned int nHeight; // height in pixels
 unsigned int nStride; /* distance in bytes to next line
                          rounded to tile len
                        */
 unsigned int nBPP; // bytes per pixel/z buffer cell

} S3SAVSURFACE, *PS3SAVSURFACE;

/******************************************************************************

******************************************************************************/
typedef struct S3SAVBUFFERtag {
 struct S3SAVBUFFERtag *pNext;
 S3SAVSURFACE stBackBuffer;
 S3SAVSURFACE stZBuffer; 
} S3SAVBUFFER, *PS3SAVBUFFER;

//#define S3SAV_FLAG_
#define S3SAV_FLAG_PALADDR   0x0001
#define S3SAV_FLAG_TEXADDR   0x0002
#define S3SAV_FLAG_TEXDESC   0x0004
#define S3SAV_FLAG_TEXCTRL   0x0008
#define S3SAV_FLAG_FOGCTRL   0x0010
#define S3SAV_FLAG_DRAWCTRL  0x0020
#define S3SAV_FLAG_ZBCTRL    0x0040
#define S3SAV_FLAG_ZBADDR    0x0080
#define S3SAV_FLAG_DESTCTRL  0x0100
#define S3SAV_FLAG_SCSTART   0x0200
#define S3SAV_FLAG_SCEND     0x0400
/* when followin flag is set it means that some global register has been
   changed and we must wait for 3d engine to be idle before setting all
   the flags */
#define S3SAV_FLAG_GLOBAL    0x80000000
/******************************************************************************

******************************************************************************/
typedef struct S3SAVCONTEXTtag {

 PS3SAVBUFFER pDrawBuffer;
 GLcontext *gl_ctx; // mesa context with all states and other goodies 
 float fViewportHeight;
} S3SAVCONTEXT, *PS3SAVCONTEXT;


/******************************************************************************

******************************************************************************/
typedef struct S3SAVTEXTUREtag {
 struct S3SAVTEXTUREtag *pNext;
 /* pointer to mesa texture object */
 struct gl_texture_object *pMT;
 /* block of memory on the card,
    if texture is not resident it is 0 */
 PMemBlock pMemBlock; 
 PS3SAVCONTEXT pCtx;
 hwUI32 uFormat;
 hwUI32 nLevels;
 hwUI32 nTextureID;
} S3SAVTEXTURE, *PS3SAVTEXTURE;

/******************************************************************************

******************************************************************************/
#define S3SAV_PSEUDOBCI_SIZE 0x4000
typedef struct S3SAVSHMtag {
 volatile hwUI32 aBCI[S3SAV_PSEUDOBCI_SIZE];
 volatile hwUI32 nBCI;
 volatile hwUI32 aShadowBuf[2048];
 volatile hwUI32 uFBPhysAddr;
 volatile hwUI32 uMMIOPhysAddr;
 volatile hwUI32 uShadowPhysAddr;
 volatile hwUI32 uVideoMemorySize;
 volatile hwUI32 uPixmapCacheOffset;
 volatile hwUI32 uPixmapCacheSize;
 volatile hwUI32 uWidth;
 volatile hwUI32 uHeight;
 volatile hwUI32 uBPP;
 volatile hwUI32 uNumClients;
 volatile hwUI32 uBCIBytes;
 volatile hwUI32 uBCIFlushes;
 volatile hwUI32 uFrames;
 volatile hwUI32 uEvent;
} S3SAVSHM, *PS3SAVSHM;

/******************************************************************************

******************************************************************************/
typedef struct S3SAVDATAtag {
 /* private variables of s3 savage xserver (2d part) */
 S3VPRIV *pS3VPriv;
 int nSHMID;
 volatile S3SAVSHM *pSHM;

 /* 3d registers of S3 Savage */
 volatile S3SAVREGS *pRegs;

 volatile hwUI8 *pMMIOMem;
 volatile hwUI32 *pBCI;
 volatile float *pfBCI;
 hwUI32 uBCIPtr; 
 volatile hwUI32 *pStatus0;
 volatile hwUI32 *pStatus1;
 volatile hwUI32 *pStatus2;

 hwUI32 uVideoMemoryPtr; // pointer to 8 MB of video memory
 hwUI32 uVideoMemorySize; // size of video memory in KB
 S3SAVSURFACE stFrontBuffer;
 hwUI8 *aTiledApertures[6];
 
 // shadows of 3d registers
 unsigned int uChangedFlags;
 S3SAVTEXPALADDR stPALADDR;
 S3SAVTEXADDR stTEXADDR;
 S3SAVTEXDESC stTEXDESC;
 S3SAVTEXCTRL stTEXCTRL;
 S3SAVFOGCTRL stFOGCTRL;
 S3SAVDRAWCTRL stDRAWCTRL;
 S3SAVZBCTRL stZBCTRL;
 S3SAVZBADDR stZBADDR;
 S3SAVDESTCTRL stDESTCTRL;
 S3SAVSCSTART stSCSTART;
 S3SAVSCEND stSCEND;
 
 /* heap of memory on card */
 memHeap_t *pCardHeap;

 hwUI32 nTextureID;
 PS3SAVTEXTURE pTextures; // texture list
 /* this is LRU list used for texture management
 */
 PS3SAVCONTEXT pContexts; // context list
 PS3SAVCONTEXT pContext; // current context

 PS3SAVBUFFER pDrawBuffer; // current draw buffer
  
} S3SAVDATA, *PS3SAVDATA;

#define S3SAVDWORD(a) (*(hwUI32 *)&(a))
#define DW(a) (*(hwUI32 *)&(a))

extern S3SAVDATA stS3Sav;

typedef void (*PFNS3SAVDMAFLUSH)(void);
extern PFNS3SAVDMAFLUSH s3savDMAFlush;


#define BCIWR(a) stS3Sav.pBCI[stS3Sav.uBCIPtr++] = (a)
#define BCIWF(a) stS3Sav.pfBCI[stS3Sav.uBCIPtr++] = (a)
#define BCIRST if (stS3Sav.uBCIPtr >= BCI_FLUSH_SIZE) s3savDMAFlush()

#define BCISETREG(reg, val) BCIWR(BCI_CMD_SETREG | 0x00010000 | reg); BCIWR(val)

#define S3SAVRD(addr) *(volatile hwUI32 *)(stS3Sav.pMMIOMem + addr)
#define S3SAVWR(addr,val) *(volatile hwUI32 *)(stS3Sav.pMMIOMem + addr) = val

#if 0
#include <stdio.h>
static hwUI32
S3SAVRD(hwUI32 addr)
{
 hwUI32 nRes;
 volatile hwUI32 *p;
 p = (volatile hwUI32 *)(stS3Sav.pMMIOMem + addr);
 nRes = *p;
 fprintf(stderr, "RD[%08X+%08X=%08X] = %08X\n",
         stS3Sav.pMMIOMem,
	 addr,
	 p,
	 nRes);
 return nRes;
}
#endif

#define S3SAV_MAXFIFO (0x8000 + 0xC0)
typedef void (*PFNS3SAVFIFOWAIT)(unsigned int nEntrFree);
extern PFNS3SAVFIFOWAIT s3savFIFOWait;

hwUI32
s3savInsertEvent(void);

void
s3savWaitEvent(hwUI32 uEvent);

void
s3savUpdateShadow(void);

static void
s3savWait3DIdle(void)
{
 while ((*stS3Sav.pStatus0 & 0x1FFFF) != 0
        || (*stS3Sav.pStatus0 & S3SAV_3DI) == 0)
       {
        volatile int i;
        for (i = 0; i < 1000; i++)
	    ;
       }
}

static hwUI32
s3savGetTextureID(void)
{
 return ((*stS3Sav.pStatus1) >> 16);
}

PS3SAVCONTEXT
s3savCreateContext(GLcontext *ctx);

int
s3savDestroyContext(PS3SAVCONTEXT ctx);

int 
s3savMakeCurrent(PS3SAVCONTEXT ctx, 
                 PS3SAVBUFFER buf);

void 
s3savGLXCreateDepthBuffer(GLcontext* ctx);

void 
s3savGLXDestroyImage(GLXImage* image);

GLXImage* 
s3savGLXCreateImage(WindowPtr pwindow, 
                    GLvisual *visual,
		    int width, 
                    int height, 
                    GLXImage *old_image );

GLboolean 
s3savGLXMakeCurrent(XSMesaContext c);

GLboolean 
s3savGLXBindBuffer(XSMesaContext c, 
                   XSMesaBuffer b);

XSMesaContext 
s3savGLXCreateContext(XSMesaVisual v,
                      XSMesaContext share_list);

void 
s3savGLXDestroyContext(XSMesaContext c);

GLbitfield 
s3savClear(GLcontext *ctx, 
           GLbitfield mask, 
           GLboolean all,
           GLint x, 
           GLint y, 
           GLint width, 
           GLint height);


void 
s3savGLXSwapBuffers(XSMesaBuffer b);

void
s3savDoSwap(PS3SAVSURFACE pBuf,
            DrawablePtr pDraw);

void 
s3savDirectClientSwapBuffers(XSMesaBuffer b);

int 
s3savGLXVendorPrivate(ClientPtr client,
		      XSMesaContext ctx, 
		      xGLXVendorPrivateReq *stuff);

int 
s3savGLXAllowDirect(ClientPtr client);

void 
s3savPerformanceBoxes(int is_direct);

void 
s3savBackToFront(DrawablePtr drawable, 
                 PS3SAVBUFFER buf );

void 
s3sav_setup_DD_pointers(GLcontext *ctx);

void 
s3savDDExtensionsInit(GLcontext *ctx);

void 
s3savDDInitStatePointers(GLcontext *ctx);

void 
s3savDDUpdateState(GLcontext *ctx);

// s3savtri.c
void
s3savLine(GLcontext *pCtx, 
          GLuint e0, 
          GLuint e1, 
          GLuint pv);
	  
/*
 NT - no texture
 TX - texture
 NF - nofog
 FG - fog
*/	  
void
s3savLine_NT_NF(GLcontext *pCtx, 
                GLuint e0, 
                GLuint e1, 
                GLuint pv);
	     
void
s3savLine_TX_NF(GLcontext *pCtx, 
                GLuint e0, 
                GLuint e1, 
                GLuint pv);	     	  

void
s3savLine_NT_FG(GLcontext *pCtx, 
                GLuint e0, 
                GLuint e1, 
                GLuint pv);
	     
void
s3savLine_TX_FG(GLcontext *pCtx, 
                GLuint e0, 
                GLuint e1, 
                GLuint pv);	     	  

void 
s3savTriangle(GLcontext *pCtx, 
              GLuint e0, 
              GLuint e1, 
              GLuint e2, 
              GLuint pv);
	      
void 
s3savTriangle_NT_NF(GLcontext *pCtx, 
                    GLuint e0, 
                    GLuint e1, 
                    GLuint e2, 
                    GLuint pv);
		 
void 
s3savTriangle_TX_NF(GLcontext *pCtx, 
                    GLuint e0, 
                    GLuint e1, 
                    GLuint e2, 
                    GLuint pv);
		 
void 
s3savTriangle_NT_FG(GLcontext *pCtx, 
                    GLuint e0, 
                    GLuint e1, 
                    GLuint e2, 
                    GLuint pv);
		 
void 
s3savTriangle_TX_FG(GLcontext *pCtx, 
                    GLuint e0, 
                    GLuint e1, 
                    GLuint e2, 
                    GLuint pv);		 		 	      

// s3savtex.c

void 
s3savCreateTexObj(PS3SAVCONTEXT ctx, 
                  struct gl_texture_object *tObj);

GLboolean 
s3savIsTextureResident(GLcontext *ctx, 
                       struct gl_texture_object *tObj);

void 
s3savDeleteTexture(GLcontext *ctx, 
                   struct gl_texture_object *tObj);

void 
s3savTexSubImage(GLcontext *ctx, 
                 GLenum target,
		 struct gl_texture_object *tObj, 
                 GLint level,
		 GLint xoffset, 
                 GLint yoffset,
		 GLsizei width, 
                 GLsizei height,
		 GLint internalFormat,
		 const struct gl_texture_image *image);

void 
s3savTexImage(GLcontext *ctx, 
              GLenum target,
              struct gl_texture_object *tObj, 
              GLint level,
	      GLint internalFormat,
	      const struct gl_texture_image *image);
	      
void 
s3savUpdateTexturePalette(GLcontext *ctx, 
                          struct gl_texture_object *tObj);

void 
s3savLoadTexturePalette(hwUI8 *pal);

PMemBlock
s3savMakeRoom(unsigned int nSize,
              unsigned int nAlign);

#define APERTURE_BACKBUFFER 5
#define APERTURE_ZBUFFER 1

hwUI32
s3savTILE(unsigned int uOffset,   
          unsigned int uStride, /* stride in bytes */
	  unsigned int uBPP); /* bytes per pixel */              
void
s3savSetTile(unsigned int uTiledAperture,
             unsigned int uOffset,
	     unsigned int uStride,
	     unsigned int uBPP);	  
#endif /* __s3savglx_h */

