/* $Id: i810direct.c,v 1.5 2000/01/28 05:00:44 keithw Exp $ */
/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <stdlib.h>
#include <signal.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mm.h"
#include "i810buf.h"
#include "i810dd.h"
#include "i810lib.h"
#include "i810log.h"
#include "i810direct.h"
#include "i810dma.h"
#include "i810glx.h"
#include "i810symbols.h"

#include "glx_clients.h"
#include "glx_init.h"
#include "xf86_OSproc.h"

extern int i810ServerDmaFlush( int wait, int *retry_timeout );


static int i810ClientDMAFlush( int wait, int *unused );
static void i810ClientSwapBuffers( XSMesaBuffer b ) ;
static void i810ClientReleaseHardware( void );

/* 
 */
#define GLX_START                X_GLXDirectPrivateStart

/* Although not strictly necessary, I've used different numbers
 * to the mga driver.  
 */
#define X_GLXI810SwapBuffers   (GLX_START+5)
#define X_GLXI810DmaFlush      (GLX_START+6)
#define X_GLXI810FBGeom        (GLX_START+7)


/* The X server will kick us off a flushdma or swapbuffer
 * request after this many usec:
 */
#define RETRY_INTERVAL 500

/* Nice thing about direct rendering is that the server *is* the
 * client, in the sense that they should both be the same executable
 * image 'glx.so'.  This means that most of the protocol structs can
 * be private to this file.
 */
typedef struct {   
   char init_fn[20];		/* The first element is always the
				 * name of the function for the client
				 * to call to initialize glx.so.  */

   int screenNum;		/* not respected - assumed to be zero */

   int vgaInfoVirtualX;
   int vgaInfoVirtualY;
   int vgaInfoDepth;
   int vgaInfoDisplayWidth;
   int vgaInfoVideoRam;
   int vgaInfoChipID;
   unsigned long vgaInfoPhysBase;

   char vgaInfoChipset[80];
   int vgaBytesPerPixel;
   int vgaLinearSize;

   int i810chipset;
   int i810dcache_start;
   int i810dcache_end;

   int i810glx_dma;
   int i810glx_agpsize;
   int i810glx_cmdsize;

   int i810ActiveDmaBuffer;
   int glx_first_visual;

} i810DirectHWInfo;


extern int __glXNumClients();


static int (*send_vendor_private)( int opcode,
				   char *vp, 
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz ) = 0;


ClientPtr i810GLXClientInit( i810DirectHWInfo *hw, 
			     void *display,
			     int screen,
			     int (*send_func)(),
			     void (**release_hook)( void ))
{   
   if (!i810InstallLocalSymbols())
      return 0;

   send_vendor_private = send_func;

   i810DoDmaFlush = i810ClientDMAFlush;
   i810GLXSwapBuffers = i810ClientSwapBuffers;

   memset(&GLXSYM(vga256InfoRec), 0, sizeof(GLXSYM(vga256InfoRec)));   
   GLXSYM(vgaLinearSize) = hw->vgaLinearSize;
   GLXSYM(vga256InfoRec).virtualX = hw->vgaInfoVirtualX;
   GLXSYM(vga256InfoRec).virtualY = hw->vgaInfoVirtualY;
   GLXSYM(vga256InfoRec).depth = hw->vgaInfoDepth;
   GLXSYM(vga256InfoRec).displayWidth = hw->vgaInfoDisplayWidth;
   GLXSYM(vga256InfoRec).videoRam = hw->vgaInfoVideoRam;
   GLXSYM(vga256InfoRec).chipID = hw->vgaInfoChipID;
   GLXSYM(vga256InfoRec).physBase = hw->vgaInfoPhysBase;
   GLXSYM(vga256InfoRec).chipset = hw->vgaInfoChipset;
   GLXSYM(vgaBytesPerPixel) = hw->vgaBytesPerPixel;
   
   i810glx.dmaDriver = hw->i810glx_dma;
   i810glx.agpSize = hw->i810glx_agpsize;
   i810glx.cmdSize = hw->i810glx_cmdsize;

   GLXSYM(I810DcacheMem).Start = hw->i810dcache_start;
   GLXSYM(I810DcacheMem).End = hw->i810dcache_end;
   GLXSYM(I810DcacheMem).Size = GLXSYM(I810DcacheMem).End - GLXSYM(I810DcacheMem).Start;   
   GLXSYM(I810Chipset) = hw->i810chipset;

   i810ActiveDmaBuffer = hw->i810ActiveDmaBuffer;

   GLXSYM(vgaLinearBase) = xf86MapVidMem(hw->screenNum, 
					   LINEAR_REGION,
					   (pointer) GLXSYM(vga256InfoRec).physBase,
					   GLXSYM(vgaLinearSize));

   if (GLXSYM(vgaLinearBase) == (pointer) -1) {
      i810Error("failed to map vga linear region");
      i810ClientReleaseHardware();
      return 0;
   }

   *release_hook = i810ClientReleaseHardware;

   if (!glxInitDirectClient( display, screen, hw->glx_first_visual,
			     XF_SVGA, I_810 ))
      return 0;

   return direct_client;
}

static void i810ClientReleaseHardware( void )
{
   if (GLXSYM(vgaLinearBase) != (pointer) -1) {
      xf86UnMapVidMem(0,
		      LINEAR_REGION,
		      (pointer) GLXSYM(vga256InfoRec).physBase,
		      GLXSYM(vgaLinearSize));
      GLXSYM(vgaLinearBase) = 0;
   }
}


static int i810GLXGoDirect( ClientPtr client )
{
   xGLXGetStringReply reply;
   i810DirectHWInfo *hw;
   int l;

   /* Allow only one direct client, which must be the only active
    * client, and must be on the local server.  
    */
   if (direct_client || !__glx_is_server || __glXNumClients() != 1 || 
       !LocalClient(client)) 
      return BadAccess;

   if (i810glx.dmaDriver < 2) {
      i810Error("Direct clients only allowed with real dma");
      return BadMatch;
   }

   i810DmaFlush();

   direct_client = client;
   hw = (i810DirectHWInfo *)malloc( sizeof(*hw) );

   /* Name of function to call to revive the struct.  Client will
    * receive buf in reply to its go_direct request.  Client will map
    * glx.so, and use buf as a string argument to dlsym(), to retreive
    * a function to call, namely i810GLXClientInit.  It will then call
    * that function passing hw as its argument.
    *
    * hw should contain all the information necessary to get glx.so
    * talking to hardware from the client.
    */
   strcpy(hw->init_fn, "i810GLXClientInit");   

   hw->screenNum = 0;		/* oh really? */
   hw->vgaInfoVirtualX = GLXSYM(vga256InfoRec).virtualX;
   hw->vgaInfoVirtualY = GLXSYM(vga256InfoRec).virtualY;
   hw->vgaInfoDepth = GLXSYM(vga256InfoRec).depth;
   hw->vgaInfoDisplayWidth = GLXSYM(vga256InfoRec).displayWidth;
   hw->vgaInfoVideoRam = GLXSYM(vga256InfoRec).videoRam;
   hw->vgaInfoChipID = GLXSYM(vga256InfoRec).chipID;
   hw->vgaInfoPhysBase = GLXSYM(vga256InfoRec).physBase;

   strncpy(hw->vgaInfoChipset, GLXSYM(vga256InfoRec).chipset, 80);
   hw->vgaInfoChipset[79] = 0;

   hw->vgaBytesPerPixel = GLXSYM(vgaBytesPerPixel);
   hw->vgaLinearSize = GLXSYM(vgaLinearSize);

   hw->i810dcache_start = GLXSYM(I810DcacheMem).Start;
   hw->i810dcache_end = GLXSYM(I810DcacheMem).End;
   hw->i810chipset = GLXSYM(I810Chipset);

   hw->i810glx_dma = i810glx.dmaDriver;
   hw->i810glx_agpsize = i810glx.agpSize;
   hw->i810glx_cmdsize = i810glx.cmdSize;

   hw->i810ActiveDmaBuffer = i810ActiveDmaBuffer;
   hw->glx_first_visual = __glx_first_visual;

   reply.type=X_Reply;
   reply.sequenceNumber = client->sequence;
   reply.length=0;
   reply.n=l=(sizeof(*hw)+3)/4;

   WriteToClient(client, sizeof(xGLXGetStringReply), (char*)&reply);
   WriteToClient(client, sizeof(int)*l, (char*)hw);

   return client->noClientException;
}



static int i810GLXReleaseDirect( ClientPtr client )
{
   if (__glx_is_server && direct_client && direct_client == client) {
      direct_client = 0;
      return Success;
   }

   return BadAccess;
}




/* =======================================================================
 * Dma flush
 */
typedef struct  {
   CARD32 dma_head;
   CARD32 dma_buffer_id;	/* validation only */
   CARD32 dma_sync;
} DmaFlushReq;


typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD16 dma_buffer_id;	/* new active id */   
   CARD16 retry;
} DmaFlushReply;



static int i810GLXDirectDMAFlush( ClientPtr client, 
				 xGLXVendorPrivateReq *stuff )
{
   DmaFlushReq *vp = (DmaFlushReq *)(stuff + 1);
   xReply reply;
   DmaFlushReply *rep = (DmaFlushReply *)&reply;	
   int wait;

   if (client != direct_client) 
      return BadAccess;

   if ( i810ActiveDmaBuffer != vp->dma_buffer_id ) 
   {
      i810Error("someone's been playing with dma on the server\n");
      return BadImplementation;
   }

   i810glx.dma_buffer->head = vp->dma_head;
   i810glx.dma_buffer->space = i810glx.dma_buffer->mem.Size - vp->dma_head;

   wait = RETRY_INTERVAL;
   rep->retry  = i810ServerDmaFlush( vp->dma_sync, &wait );
   rep->type   = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->dma_buffer_id = i810ActiveDmaBuffer;
   
   WriteToClient( client, sizeof(xReply), (char *) &reply );
	
   return client->noClientException;
}


/*
 * i810ClientDMAFlush
 * When direct rendering, this is called instead of starting
 * the dma from user space.  This only happens for overflows,
 * explicit flush/finish, and software rendering.  The swapbuffers
 * interface handles flushing in normal cases.
 *
 * KW: Implemented a maximum time to wait for dma completion.  The
 *     client must now poll if the timeout occurs before completion.
 *     this allows the server to process outstanding events (esp. 
 *     mouse events). 
 */
static int i810ClientDMAFlush( int wait, int *unused ) 
{
   DmaFlushReq vp;
   DmaFlushReply *rep;
   xReply reply;

   vp.dma_buffer_id = i810ActiveDmaBuffer;
   vp.dma_head = i810glx.dma_buffer->head;
   vp.dma_sync = wait;

   do {
      if (!send_vendor_private( X_GLXI810DmaFlush,
				(char *)&vp, sizeof(vp), 
				&reply, 0, 0 ))
	 return 0;

      rep = (DmaFlushReply *)&reply;

   } while (rep->retry);

   i810ActiveDmaBuffer = rep->dma_buffer_id;
   i810DmaResetBuffer();
   return 0;
}


/* =======================================================================
 * Geometry request 
 *   - could just use XGetGeom, but this is simpler, really.
 */
typedef struct  {
   CARD32 drawable;
} FBGeomReq;


typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD16 width;
   CARD16 height;
} FBGeomReply;


static int i810GLXDirectGetGeometry( ClientPtr client, 
				    xGLXVendorPrivateReq *stuff )
{
   xReply reply;
   DrawablePtr frontbuf;
   FBGeomReq *vp = (FBGeomReq *)(stuff + 1);
   FBGeomReply *rep = (FBGeomReply *)&reply;	

   if (client != direct_client) 
      return BadAccess;

   frontbuf = LookupIDByClass( vp->drawable, RC_DRAWABLE );   
   if (!frontbuf) 
      return GLXBadDrawable;

   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->width = frontbuf->width;
   rep->height = frontbuf->height;
   
   WriteToClient( client, sizeof(xReply), (char *) &reply );
	
   return client->noClientException;
}


void i810ClientGetGeometry( DrawablePtr d ) 
{
   FBGeomReq vp;
   xReply reply;

   vp.drawable = d->id;    
   
   if (send_vendor_private( X_GLXI810FBGeom,
			    (char *)&vp, sizeof(vp), 
			    &reply, 0, 0 ))
   {
      FBGeomReply *rep = (FBGeomReply *)&reply;
      d->width = rep->width;
      d->height = rep->height;
   }
}




/* ===========================================================================
 * Buffer swap
 *   - Combines dma flush, buffer swap and get geometry.
 */

/* Need to issue the swap commands from the server as the client
 * doesn't have the clip rects.  This isn't so bad as we would want to
 * do a dma flush immediately afterwards anyhow.
 */
typedef struct {
   CARD16 back_width;
   CARD16 back_height;
   CARD16 back_pitch;
   CARD16 back_format;
   CARD32 back_offset;
   CARD32 front_drawable;
   CARD32 dma_head;
   CARD32 dma_buffer_id;	/* validation only */
   CARD32 flag;
} SwapBufferReq;

typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD16 dma_buffer_id;	/* new active id */   
   CARD16 retry;
   CARD16 width;
   CARD16 height;
} SwapBufferReply;

#define FLG_TEX_SWAP 0x1

static int i810GLXDirectSwapBuffers( ClientPtr client, 
				    xGLXVendorPrivateReq *stuff )
{
   SwapBufferReq *vp = (SwapBufferReq *)(stuff+1);
   xReply reply;
   SwapBufferReply *rep = (SwapBufferReply *)&reply;	
   struct i810_dest_buffer *old = i810DB;
   struct i810_dest_buffer backbuf;
   TMemBlock tmp;
   DrawablePtr frontbuf;
   int wait;

   if (client != direct_client) 
      return BadAccess;

   if ( i810ActiveDmaBuffer != vp->dma_buffer_id ) 
   {
      fprintf(stderr, "somebody's been playing with dma on the server %d %ld\n",
	      i810ActiveDmaBuffer, vp->dma_buffer_id);

      return BadImplementation;
   }

   i810glx.dma_buffer->head = vp->dma_head;
   i810glx.dma_buffer->space = i810glx.dma_buffer->mem.Size - vp->dma_head;

   frontbuf = LookupIDByClass( vp->front_drawable, RC_DRAWABLE );   

   if (!frontbuf) {
      return GLXBadDrawable;
   }

   backbuf.Width = vp->back_width;
   backbuf.Height = vp->back_height;
   backbuf.Pitch = vp->back_pitch;
   backbuf.Format = vp->back_format;
   backbuf.MemBlock = &tmp;
   backbuf.MemBlock->ofs = vp->back_offset;

   if (vp->flag & FLG_TEX_SWAP) i810glx.c_textureSwaps++;

   i810DB=&backbuf;
   i810PerformanceBoxes( 1 );
   i810DB=old;

   i810BackToFront(frontbuf, &backbuf);   

   wait = RETRY_INTERVAL;
   rep->retry = i810ServerDmaFlush( 0, &wait );
   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->dma_buffer_id = i810ActiveDmaBuffer;
   rep->width = frontbuf->width;
   rep->height = frontbuf->height;

   WriteToClient( client, sizeof(xReply), (char *) &reply );

   return client->noClientException;
}

static void i810ClientSwapBuffers( XSMesaBuffer b ) 
{
   SwapBufferReq vp;
   xReply reply;
   struct i810_dest_buffer *buf;
   SwapBufferReply *rep;

   if (!b ||
       !b->db_state || 
       !b->backimage ||
       !(buf = (struct i810_dest_buffer *)b->backimage->devPriv))
   {
      fprintf(stderr, "client swap buffers: wtf???\n");
      return;
   }
   
   if (i810Ctx && i810Ctx->gl_ctx) {
      FLUSH_VB(i810Ctx->gl_ctx, "i810 client swap buffers");
   }

/*     i810WarpFinishSerie(); */
/*     i810glx.swapBuffersCount++; */

   i810glx.dma_buffer->texture_age = ++i810glx.current_texture_age;

   if (i810Ctx) {
      if (i810Ctx->CurrentTex0Obj)
	 i810Ctx->CurrentTex0Obj->age = ++i810glx.current_texture_age; 

      if (i810Ctx->CurrentTex1Obj)
	 i810Ctx->CurrentTex1Obj->age = ++i810glx.current_texture_age; 
   }


   vp.front_drawable = b->frontbuffer->id;    
   vp.back_width = buf->Width;
   vp.back_height =  buf->Height;
   vp.back_pitch = buf->Pitch;
   vp.back_format = buf->Format;
   vp.back_offset = buf->MemBlock->ofs;

   vp.dma_buffer_id = i810ActiveDmaBuffer;
   vp.dma_head = i810glx.dma_buffer->head;
   vp.flag = 0;

   if (i810glx.c_textureSwaps) {
      vp.flag |= FLG_TEX_SWAP;
      i810glx.c_textureSwaps = 0;
   }

   do {
      if (!send_vendor_private( X_GLXI810SwapBuffers,
				(char *)&vp, sizeof(vp), &reply, 0, 0 ))
	 FatalError("clientSwapBuffers failed");

      rep = (SwapBufferReply *)&reply;

   } while (rep->retry);

   b->frontbuffer->width = rep->width;
   b->frontbuffer->height = rep->height;

   i810ActiveDmaBuffer = rep->dma_buffer_id;
   i810DmaResetBuffer();
}







/* ===========================================================================
 */

int i810GLXVendorPrivate( ClientPtr client,
			 XSMesaContext ctx, 
			 xGLXVendorPrivateReq *stuff )
{
   if (!__glx_is_server) 
      return GLXUnsupportedPrivateRequest;

   switch (stuff->opcode) {
   case X_GLXDirectGoDirect:
      return i810GLXGoDirect( client );
   case X_GLXDirectRelease:
      return i810GLXReleaseDirect( client );
   case X_GLXI810SwapBuffers:
      return i810GLXDirectSwapBuffers( client, stuff );
   case X_GLXI810DmaFlush:
      return i810GLXDirectDMAFlush( client, stuff );
   case X_GLXI810FBGeom:
      return i810GLXDirectGetGeometry( client, stuff );
   }

   i810Error("not-handled case");
   return GLXUnsupportedPrivateRequest;
}


int i810GLXAllowDirect( ClientPtr client )
{
#if (!I810_USE_BATCH)
      i810Error("Don't allow direct clients unless compiled for true dma");
      return 0;
#endif

   return 1;
}

