/*
    Copyright (C) 2001-2002  bjk <bjk@arbornet.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <grp.h>
#include "common.h"
#include "ui.h"

static int escapes(const char *str)
{
    static int c;

    if (str[0] != '\\')
	return str[0];

    switch (*++str) {
	case 't': c = '\t'; break;
	case 'n': c = '\n'; break;
	case '\\': c = '\\'; break;
	case 'v': c = '\v'; break;
	case 'b': c = '\b'; break;
	case 'f': c = '\f'; break;
	case 'r': c = '\r'; break;
	case '\'': c = '\''; break;
	default: c = 0; break;
    }

    return c;
}

static void usage(char *me)
{
    extern unsigned int delimchar;

    fprintf(stderr, "Usage: %s [-vhV] [-j tf] [-D c] [options] "
	    "[-f <filename> ... | user ...]\n", me);
    fprintf(stderr, "  Password/Group file information [-P (-%s)]:\n",
	    MOPT_ORDER_PASSWD);
    fprintf(stderr, "\t-l\tlogin name\t\t");
    fprintf(stderr, "\t-p\tencrypted password\n");
    fprintf(stderr, "\t-u\tuser id (uid)\t\t");
    fprintf(stderr, "\t-g\tgroup id (gid)\n");
    fprintf(stderr, "\t-c\tpassword change time\t");
    fprintf(stderr, "\t-k\tpassword expire time\n");
    fprintf(stderr, "\t-d\thome directory\t\t");
    fprintf(stderr, "\t-m\thome directory mode\n");
    fprintf(stderr, "\t-s\tlogin shell\n");
    fprintf(stderr, "\t-i opts\tgecos information "
	    "([n]ame,[o]ffice,[w]phone,[h]phone),[a]ll)\n\n");

    fprintf(stderr, "  Mail information [-M (-%s)]:\n", MOPT_ORDER_MAIL);
    fprintf(stderr, "\t-o\tforward address\t\t");
    fprintf(stderr, "-a\tmail aliases\n");
    fprintf(stderr, "\t-r\tlast read\t\t");
    fprintf(stderr, "-w\tnew mail\n");
    fprintf(stderr, "\t-z\tfolder size\n\n");

    fprintf(stderr, "  Login information [-L (-%s)]:\n", MOPT_ORDER_LOGIN);
    fprintf(stderr, "\t-y\ttty\t\t\t");
    fprintf(stderr, "-e\tmessage status\n");
    fprintf(stderr, "\t-t\tlogin time stamp\t");
    fprintf(stderr, "-b\tduration in minutes\n");
    fprintf(stderr, "\t-x\thostname\t\t");
#ifdef __linux__
#ifdef HAVE_PROCFS
    fprintf(stderr, "-q\tminutes idle\n");
    fprintf(stderr, "\t-C\tparent process id (ppid)\n");
#else
    fprintf(stderr, "-q\tminutes idle\n");
#endif
#else
    fprintf(stderr, "-q\tminutes idle\n");
    fprintf(stderr, "\t-C\tparent process id (ppid)\n");
#endif
    fprintf(stderr, "\t-n opts lastlog information"
	   " ([t]ty,[h]ostname,t[i]me,[a]ll)\n\n");

    fprintf(stderr, "  -A\tOutput all options (-PML).\n");
    fprintf(stderr, "  -D c\tSeparate output with specified character "
	    "('%c').\n", delimchar);
    fprintf(stderr, "  -j tf\tstrftime(3) time format ('%s').\n",
	    DEF_TIMEFORMAT);
    fprintf(stderr, "  -f\tUsers are treated as owners of the following files."
	    "\n");
    fprintf(stderr, "  -v\tVerbose output; displays gids with group names, "
	    "and filenames.\n");
    fprintf(stderr, "  -h\tThis help text.\n");
    fprintf(stderr, "  -V\tVersion information.\n\n");
    fprintf(stderr, "Output key: %s=unknown/error, %s=none, %s=yes/on, "
	    "%s=no/off\n", UNKNOWN, NONE, ON, OFF);

    exit(EXIT_FAILURE);
}

static int get_lastlog_opts(const char *args)
{
    int i = 0;

    if (strchr(args, 'a') != NULL) {
	args = "thi";
	get_lastlog_opts(args);
    }

    while (*args) {
	switch (*args++) {
	    case 't':
	    case 'h':
	    case 'i':
		lastlog_opts[i++] = *args;
		break;
	    default:
		return 1;
	}
    }

    return 0;
}

static int get_gecos_opts(const char *args)
{
    int i = 0;

    if (strchr(args, 'a') != NULL) {
	args = "nowh";
	get_gecos_opts(args);
    }

    while (*args) {
	switch (*args++) {
	    case 'n':
	    case 'o':
	    case 'w':
	    case 'h':
		gecos_opts[i++] = *args;
		break;
	    default:
		return 1;
	}
    }

    return 0;
}

static int command_line_opts(int myargc, char *myargv[], const char *opts)
{
    int opt;
    static int options;

    while ((opt = Getopt(myargc, myargv, opts)) != -1) {
	switch (opt) {
	    case 'P': if (majoropt[MOPT_PASSWD])
			  continue;

		      majoropt[MOPT_PASSWD] = 1; 
		      return MOPT_PASSWD;
		      break;
	    case 'M': if (majoropt[MOPT_MAIL])
			  continue;

		      majoropt[MOPT_MAIL] = 1; 
		      return MOPT_MAIL;
		      break;
	    case 'L': if (majoropt[MOPT_LOGIN])
			  continue;

		      majoropt[MOPT_LOGIN] = 1; 
		      return MOPT_LOGIN;
		      break;
	    case 'A': if (majoropt[MOPT_ALL]) 
			  continue;

		      majoropt[MOPT_PASSWD] = 1;
	    	      majoropt[MOPT_MAIL] = 1;
	    	      majoropt[MOPT_LOGIN] = 1;
	    	      majoropt[MOPT_ALL] = 1;
		      return MOPT_ALL;
		      break;
	    case 'D': if (optarg[0] != '\\' && strlen(optarg) > 1)
			  usage(myargv[0]);

		      if ((delimchar = escapes(optarg)) == 0)
			  usage(myargv[0]);
		      
		      break;
	    case 'j': strncpy(tf, optarg, sizeof(tf));
		      break;
	    case 'h': usage(myargv[0]); 
		      break;
	    case 'V': fprintf(stdout, "%s\n%s\n", VERSION, COPYRIGHT);
		      exit(EXIT_SUCCESS);
		      break;
	    case 'f': usefile = 1;
		      break;
	    case 'v': verbose = 1; 
		      break;
	    case 'i': if (majoropt[MOPT_PASSWD])
			  gecosopts = "a";
		      else
			  gecosopts = optarg;
	    case 'l':
	    case 's':
	    case 'u':
	    case 'd':
	    case 'm':
	    case 'p':
	    case 'c':
	    case 'k':
	    case 'g': passwd_info = 1;
		      break;
	    case 'z':
	    case 'o':
	    case 'r':
	    case 'a':
	    case 'w': mail_info = 1;
		      break;
	    case 'y':
	    case 'e':
	    case 'b':
	    case 'x':
	    case 'n': if (majoropt[MOPT_LOGIN])
			  lastopts = "a";
		      else
			  lastopts = optarg;
	    case 'q':
	    case 'C':
	    case 't': login_info = 1;
		      break;
	    case '?':
	    default: usage(myargv[0]); 
		     break;
	}

	optspec[options++] = opt;
    }

    return opt;
}

static int junction(INFO *info, const char *arg)
{
    char filename[FILENAME_MAX];
    struct stat mystat;
    
    setpwent();

#ifdef HAVE_GETSPNAM
    if (amroot)
	setspent();
#endif

    filename[0] = 0;

    if (usefile) {
	strncpy(filename, arg, sizeof(filename));

        if ((lstat(filename, &mystat)) == -1) {
	    perror(filename);
	    return 1;
	}

        if ((info->passwd = getpwuid(mystat.st_uid)) == NULL) {
	    fprintf(stderr, "%s: no such uid %i\n", filename, mystat.st_uid);
	    return 1;
        }

#ifdef HAVE_GETSPNAM
	if (amroot) {
	    if ((info->spwd = getspnam(info->passwd->pw_name)) == NULL) {
		fprintf(stderr, "getspnam(): unknown error\n");
	        return 1;
	    }
	} 
#endif
    }
    else {
        if ((info->passwd = getpwnam(arg)) == NULL) {
	    fprintf(stderr, "%s: no such user\n", arg);
	    return 1;
	} 

#ifdef HAVE_GETSPNAM
	if (amroot) {
	    if ((info->spwd = getspnam(arg)) == NULL) {
		fprintf(stderr, "getspnam(): unknown error\n");
	        return 1;
	    }
	} 
#endif
    }

    /* fill the info structure */
    if (passwd_info) 
	passwdinfo(info);

    if (mail_info)
	mailinfo(info, arg);

    if (login_info)
	logininfo(info, arg);

    outputinfo(info, arg);
    return 0;
}

static void cleanup(INFO *info)
{
    GROUP *gcur, *gnext;

    if (info->login)
	free(info->login);

    if (info->mail)
	free(info->mail);

    if (group_info) {
	for (gcur = info->group; gcur; gcur = gnext) {
	    gnext = gcur->next;
	    free(gcur);
	}
    }

    return;
}

int main(int argc, char *argv[])
{
    char progname[FILENAME_MAX];
    extern int optreset;
    int i;
    INFO *info;

    delimchar = DEFDELIM;
    strncpy(tf, DEF_TIMEFORMAT, sizeof(tf));
    strncpy(progname, argv[0], sizeof(progname));

    opterr = optind = 1;

    while ((i = command_line_opts(argc, argv, CMDLINE_OPTS)) != -1) {
	char *moptargv[2], *mopts = NULL, tmp[MAXOPTIONS];
	int oldoptind;

	tmp[0] = 0;

	switch (i) {
	    case MOPT_PASSWD:
		mopts = MOPT_ORDER_PASSWD;
		snprintf(tmp, sizeof(tmp), "-%s", MOPT_ORDER_PASSWD);
		moptargv[1] = tmp;
		break;
	    case MOPT_MAIL:
		mopts = MOPT_ORDER_MAIL;
		snprintf(tmp, sizeof(tmp), "-%s", MOPT_ORDER_MAIL);
		moptargv[1] = tmp;
		break;
	    case MOPT_LOGIN:
		mopts = MOPT_ORDER_LOGIN;
		snprintf(tmp, sizeof(tmp), "-%s", MOPT_ORDER_LOGIN);
		moptargv[1] = tmp;
		break;
	    case MOPT_ALL:
		snprintf(tmp, sizeof(tmp), "%s%s%s", MOPT_ORDER_PASSWD,
			MOPT_ORDER_MAIL, MOPT_ORDER_LOGIN);
		mopts = tmp;
		tmp[0] = 0;
		snprintf(tmp, sizeof(tmp), "-%s%s%s", MOPT_ORDER_PASSWD,
			MOPT_ORDER_MAIL, MOPT_ORDER_LOGIN);
		moptargv[1] = tmp;
		break;
	    default:
		break;
	}

	oldoptind = optind;
	optind = 1;
	optreset = 1;
	moptargv[0] = progname;

	if ((i = command_line_opts(ARRAYCNT(moptargv), moptargv, mopts)) != -1)
	    usage(progname);

	optind = oldoptind;
	optreset = 1;
    }

    if (!mail_info && !passwd_info && !login_info)
	usage(progname);

    if (argc == optind)
	use_stdin = 1;

#ifdef HAVE_SETPASSENT
    if (!setpassent(1))
	fprintf(stderr, "cannot keep passwd file open (will be slower)\n");
#endif

#ifdef HAVE_GETSPNAM
    if (getuid() == 0)
	amroot = 1;
#endif

    if ((info = (INFO *) malloc(sizeof(INFO))) == NULL) {
        perror("malloc");
        exit(errno);
    }

    bzero(info, sizeof(INFO));

    if (gecosopts) {
	if ((info->gecos = (GECOS *) malloc(sizeof(GECOS))) == NULL) {
	    perror("malloc()");
	    exit(errno);
	}

        if (get_gecos_opts(gecosopts) != 0)
            usage(progname);
    }
	
    if (lastopts) {
	if ((info->lastinfo = (LASTINFO *) malloc(sizeof(LASTINFO))) == NULL) {
	    perror("malloc()");
	    exit(errno);
	}

        if (get_lastlog_opts(lastopts) != 0)
            usage(progname);
    }
     
    while (1) {
	char *arg = argv[optind];
	char buf[LINE_MAX], *t;
	
	if (!use_stdin && optind == argc)
	    break;

	if (arg && arg[0] == '-' && arg[1] == 0)
	    use_stdin = 1;

	if (use_stdin) {
	    while ((t = fgets(buf, sizeof(buf), stdin)) != NULL) {
		group_info = 0;
		arg = stripstr(buf, "\n");
		
		if (junction(info, arg) != 0) {
		    free(arg);
		    continue;
		}
		
		cleanup(info);
		free(arg);
	    }

	    break;
	}

	group_info = 0;
	optind++;

	if (junction(info, arg) != 0) 
	    continue;

	cleanup(info);
    }

    if (info->gecos)
	free(info->gecos);

    if (info->lastinfo)
	free(info->lastinfo);

    free(info);

#ifdef HAVE_PROCFS
#ifndef USE_KVM
    if (procdir)
	closedir(procdir);
#endif
#endif

    if (utmpfd)
	close(utmpfd);

    if (lastfp)
	fclose(lastfp);

    if (aliasfp)
	fclose(aliasfp);

    if (passwd_info)
	endpwent();

#ifdef HAVE_GETSPNAM
    if (amroot) {
        if (passwd_info)
	    endspent();
    }
#endif

    if (group_info)
	endgrent();

    exit(EXIT_SUCCESS);
}
