/* ==================================================== ======== ======= *
 *
 *  umserver.hpp : Ubit Message Server (Dual Mouse/bi-manual interaction, etc.)
 *
 *  NOTE: sur une machine SVR4 (Solaris, etc), ajouter a la compilation : 
 *        -lsocket (pour les sockets) et -lnsl (pour htons...) 
 *
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)umserver.hpp	ubit:03.05.05"
#ifndef _umserver_hpp_
#define	_umserver_hpp_
#include <vector>
#include <list>
#include <iostream>
#include <X11/Xlib.h>
#include <ubit/udefs.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Mouse button mapping

struct UMSbuttonMapping {

  UMSbuttonMapping(u_int source_button, 
		   u_int source_modifiers,
		   u_int flow_button,
		   u_int flow_modifiers);
  /**< specifies the events generated by the buttons of the mouses
   * rule:
   *  source_button, source_modifiers ==> flow_button, flow_modifiers
   * see: UMServer::setNativeButtonMapping (for the native mouses)
   * and UMSeventSource::setButtonMapping (for alternate mouses)
   */

  // input from the device
  u_int source_button;
  u_int source_button_mask;
  u_int source_modifiers;     // button_mask not included
  // output to the event flow
  u_int flow_button;
  u_int flow_button_mask;
  u_int flow_modifiers;       // button_mask not included
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// other UMSserver in the topology

struct UMSneighbor {
  // attempt to re-connect (if connection failed) after this delay
  static const Time CNX_RETRY_DELAY = 6*1000;  // 1 mn

  const char* host;
  int  port;
  Time cnx_time;
  class UMSclient* client;

  UMSneighbor(const char* host, int port);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// settings of the calibration window (used for absolute positioning
// devices such as the MIMIO)

struct UMScalibration {
  static const int POINT_COUNT = 4;
  static const int CROSS_XYPOS = 50;
  static const int CROSS_SIZE  = 600;

  int curpoint;          // completed when == POINT_COUNT-1

  /// coords of the calibration window (= its pos. on the screen)
  int x, y, width, height;

  /// positions of calpoints in pixels on the screen
  int xp[POINT_COUNT], yp[POINT_COUNT];

  /// position of calpoints in source coords
  double xcm[POINT_COUNT], ycm[POINT_COUNT];

  char* message;

  UMScalibration();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// position on the screen

struct UMSpos {
  UMSpos();
  UMSpos(int rx, int ry, int wx, int wy, Window win,
         bool _ptr_in_uwin, int winsock);
  
  int rx, ry;
  int wx, wy;
  Window win;
  bool ptr_in_uwin; // att ce champ n'est mis a jour que par locatePoint
  // cad quand on interagit avec un mouse flow non natif
  int winsock;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// window

struct UMSwin {
  ~UMSwin();
  UMSwin(Window win, Window client_win, int command_argc, char** command_argv,
	 char* ubit_props, int winsock);
  ///< att: aucune duplication des arguments!.

  Window win, client_win;
  int command_argc;
  char** command_argv;
  char* ubit_props;
  int winsock;

private:
  UMSwin(const UMSwin&);
  UMSwin& operator=(const UMSwin&);
};

typedef std::list<UMSwin*> UMSwinList;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UMServer {
public:
  // SHORT_MESSAGE_SIZE == sizeof(XClientMessageEvent.data.b)
  static const int SHORT_MESSAGE_SIZE = 20;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  UMServer(int ums_port, const char* display_name);
  /**< constructor.
   * Args:
   * - ums_port: the port used by the UMS; uses UMS_DEFAULT_PORT if 0
   * - display_name == NULL means "local host"
   */

  ~UMServer();

  bool isInit() const {return is_init;}
  /**< checks if the UMS is initialized.
   * NB: returns true if the UMS port and the X display could be opened.
   */

  void start();
  /**< starts the main loop of the UMS server.
   * NB: the UMS must be initialized (see: isInit())
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Topology

  bool addNeighbor(const char* position, const char* host, int port = 0);
  bool addNeighbor(int l, int c, const char* host, int port = 0);
  UMSneighbor* getNeighbor(int l, int c) const {return neighbors[l][c];}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -     // Emulation of mouse buttons 2 & 3, etc.

  UMSbuttonMapping* getNativeButtonMapping(u_int source_button,
					   u_int source_modifiers);

  void setNativeButtonMapping(UMSbuttonMapping*);
  void setNativeButtonMapping(u_int source_button, 
			      u_int source_modifiers,
			      u_int flow_button,
			      u_int flow_modifiers);
  /**< 
   * changes the events generated by the buttons of the native mouses
   * rule:
   * (source_button, source_modifiers ==> flow_button, flow_modifiers)
   *
   * useful for machines that do not have 3 buttons such as the IPAQ
   * button mapping can also be done for alternate event sources
   * (see UMSeventSource)
   */

  void setActionKey(u_int keycode, u_int modifier, 
		    void (*press_action)(class UMServer&, XEvent&),
		    void (*release_action)(class UMServer&, XEvent&));
  /**< this key will perform some useful action when pressed and/or released.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Calibration

  bool isCalibrationCompleted() const;
  const UMScalibration& getCalibration() const; 

  void startCalibration(const char* title, bool full_screen);
  void configureCalibrationWin(int x, int y, int w, int h);
  void drawCalibrationPoint();
  void setCalibrationPoint(double x, double y);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Event Flows

  bool addEventFlow(class UMSeventFlow*);
  /**< adds an event flow.
   * NB: - several event flows can be created
   *     - their IDs MUST be != 0 (and have different values)
   *     - the native event flow (that corresponds to the standard X mouse)
   *       is created implicitely by the UMServer. Its ID is always 0.
   * Return value:
   *     - true if this flow could be added (ie. ID is != 0 and not flow
   *       was registered with the same ID)
   */

  bool removeEventFlow(class UMSeventFlow*);
  /**< removes an event flow.
   * NB: - this fct does not delete the flow.
   *     - the native event flow (whose ID = 0) can't be removed.
   * Return value:
   *     - true if this flow could be removed (= ID is != 0 and this flow
   *       was previously registered
   */

  class UMSeventFlow* getEventFlow(int id) const;
  class UMSmouseFlow* getNativeMouseFlow() const;
  class UMSmouseFlow* getOrCreateMouseFlow(int id);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Event Sources

  bool addEventSource(class UMSeventSource*);
  /**< adds an event source.
   * NB: - several event sources can be created
   * Return value:
   *     - true if this source could be added (ie. if this source
   *       was not previously registered)
   */

  bool removeEventSource(class UMSeventSource*);
  /**< removes an event source.
   * NB: - this fct does not delete the source.
   * Return value:
   *     - true if this source could be removed (= if this source
   *       was previously registered
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // X Display

  Display* getDisplay() const {return xdisplay;}
  Screen*  getScreen()  const {return xscreen;}
  Window   getRoot()    const {return xrootwin;}
  long getScreenWidth() const {return screen_width;}
  long getScreenHeight()const {return screen_height;}
  int  getScreenDepth() const {return screen_depth;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Windows

  bool locateWin(Window win, UMSpos& pos);
  /** trouver position de win dans root.
   * input:  win
   * output: pos.rx, pos.ry  (pos.win = win et le reste a 0)
   */

  bool locatePoint(Window win, int rx, int ry, UMSpos& pos);
  /** trouver subwindow de win contenant le point (rx,ry).
   * input:  win et point (rx,ry) dans root
   * output: pos.win = subwindow trouvee ou win et coords = pos.{rx,ry,wx,wy}
   */

  bool locatePoint(int rx, int ry, UMSpos& pos);
  /** trouver window de root contenant le point (rx,ry).
   * input:  point (rx,ry) dans root
   * output: pos.win = window trouvee et coords = pos.{rx,ry,wx,wy}
   */

  bool locatePointer(const class UMSmouseFlow&, UMSpos& pos);
  /**< trouver position et window du pointeur pour ce mouse flow.
   * Notes: 
   * - le mouse flow d'ID 0 controle le pointeur X.
   * - inititalise egalement pos.winsock si Ubit Win (sinon -1)
   */

  bool locateXpointer(UMSpos&);
  ///< trouver position et window du pointeur du flow standard de X.

  bool isPointer(Window);
  ///< is this window an UMS mouse pointer ?.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Windows

  void retrieveWindows();
  void processEvents(XEvent&);
  void showWindows();
  void showWindowName(Window);

  UMSwinList::iterator searchWinPos(UMSwinList& wl, Window win);
  const UMSwin* searchWin(UMSwinList& wl, Window win);
  const UMSwin* searchWinName(UMSwinList& wl, const char* pattern);
  const UMSwin* searchWinCommand(UMSwinList& wl, const char* pattern);

  bool addWin(UMSwinList& wl, Window win, bool init_wm_props, int sock);
  bool removeWin(UMSwinList& wl, Window win);
  void moveWin(UMSwinList& wl, 
	       UMSwinList::iterator& from, const UMSwinList::iterator& to);

  bool addCnx(int sock, const char* app_name);
  void removeCnx(int sock);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Events

  void sendKeyEvent(UMSmouseFlow&, UMSpos&, int keycode, bool press);
  void sendButtonEvent(UMSmouseFlow&, UMSpos&, u_id btn_maskid, bool press);
  void sendMotionEvent(UMSmouseFlow&, UMSpos&);
  void sendCrossingEvent(UMSmouseFlow&, UMSpos&, bool enter);

  void sendMessage(UMSeventFlow&, UMSpos&, const char* message);
  void sendShortMessage(UMSeventFlow&, UMSpos&, const char message[SHORT_MESSAGE_SIZE]);
  void sendLongMessage(UMSeventFlow&, UMSpos&, const char* message, int message_len);
      
  static Time getTime();   ///< in millisec.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  // Implementation

private:
  static int quietXErrorHandler(Display*, XErrorEvent*);
  bool initX(const char* display_name);

  int  openServerSocket(int server_port);
  void readServerSocket();
  int  readComSocket(int sock_com, bool called_by_serv);

  bool parseRequest(class UMSrequest&, class RequestAction&);
  int  processRequest(int sock_com, class UMSrequest&);
  bool processMouseRequest(UMSrequest&);
  bool processEventRequest(UMSrequest&);
  bool processMessageRequest(UMSrequest&);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  static int xerror;
  struct sockaddr_in& sock_server;
  bool is_init;
  int  ums_socket;
  int  xconnection;
  long screen_width;
  long screen_height;
  int  screen_depth;
  Display* xdisplay;
  Screen*  xscreen;
  Window   xrootwin;
  Window   calwin;
  GC       calgc;
  UMScalibration cal;
  bool calibrating;
  Atom WM_DELETE_WINDOW, UBIT_MESSAGE, UBIT_WINDOW;

  /// list of event sources (= the devices that are connected to the eflows)
  std::vector<class UMSeventSource*> sources;

  /// list of event flows (eflows[0] corresponds to the standard X server)
  std::vector<class UMSeventFlow*> eflows;

  /// list of connections (= connected applications)
  std::vector<class UMScnx*> cnxs;

  /// list of windows that are direct children of the root window.
  UMSwinList root_children;

  /// list of connected windows.
  UMSwinList cnx_wins;

  /// neigbhors in teh topoly of UMS servers
  UMSneighbor* neighbors[3][3];

  /// predefined keys that perform some action
  std::vector<class UMSactionKey> action_keys;

  // mapping of the buttons of the native mouses.
  std::vector<class UMSbuttonMapping*> button_map;
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif
