/* ==================================================== ======== ======= *
 *
 *  uuview.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuview.cpp	ubit:03.06.04"
#include <iostream>
#include <stdio.h>  //debug
#include <config.h>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucond.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ucolor.hpp>
#include <ustr.hpp>
#include <ucontext.hpp>
#include <uborder.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uview.hpp>
#include <uviewImpl.hpp>
#include <uwin.hpp>
#include <uappli.hpp>
#include <ugraph.hpp>
#include <uima.hpp>
#include <ufont.hpp>
#include <utable.hpp>
#include <uedit.hpp>
#include <upane.hpp>
#include <unatgraph.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UViewStyle::UViewStyle(UView* (*make_view)(UBoxLink*, UView*, UWinGraph*),
		       u_modes bm) : UMode(bm) {
  makeView = make_view;
}

void UViewStyle::update() {
//  fromLinks.updateParents(true, true, false);  A REVOIR!!
}

void UViewStyle::addingTo(ULink *selflink, UGroup *parent) {
  UBrick::addingTo(selflink, parent);

 if (parent->isDef(0, UMode::HAS_RENDERER)) {
   UError::error("warning@UViewStyle::addingTo", 
		 "multiple UViewStyle bricks in parent",parent->cname());
  }

  parent->setCmodes(UMode::HAS_RENDERER, true);
}

void UViewStyle::removingFrom(ULink *prevlink, UGroup *parent) {
  parent->setCmodes(UMode::HAS_RENDERER, false);
  UBrick::removingFrom(prevlink, parent);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UViewExt::UViewExt() {
  old_region = null;
  kept_width = kept_height = -1;
}

UViewExt::~UViewExt() {
  if (old_region) delete old_region;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UViewStyle UView::style(&UView::makeView, UMode::UCONST);

// "static" constructor used by UViewStyle to make a new view
UView* UView::makeView(UBoxLink*bl, UView* parview, UWinGraph* wgraph) {
  return new UView(bl, parview, wgraph);
}

UView::UView(UBoxLink *box_link, UView* par_view, UWinGraph *wgraph) {
  x = 0, y = 0;
  width = height = 0;
  chwidth = chheight = 0;
  vflex_count = hflex_count = 0; 
  boxlink = box_link;
  parview = par_view;
  wingraph= wgraph;
  vmodes  = 0;
  pext    = null;
  border_margins   = null;
  nextview= null;
  edit_shift = 0;  // !!
}

UView::~UView() {
  if (pext) delete pext;
  if (border_margins) delete border_margins;
  nextview = null;
  // notifies that this view has been deleted
  //wingraph->getAppli()->deleteNotify(this);
  wingraph->getDisp()->deleteNotify(this);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UView::setVmodes(int m, bool state) {
  if (state) vmodes = vmodes | m; 
  else vmodes = vmodes & ~m;
}

void UView::setXwin(int x_in_win) {x = x_in_win;}
void UView::setYwin(int y_in_win) {y = y_in_win;}

int UView::getX() const {
  if (parview) return x - parview->getXwin();
  else return 0;
}
int UView::getY() const {
  if (parview) return y - parview->getYwin();
  else return 0;
}

bool UView::where(u_pos &xloc, u_pos &yloc) const {
  if (parview) {
    xloc = x - parview->getXwin();
    yloc = y - parview->getYwin();
    return true;
  }
  else {
    //xloc = -1; yloc = -1;
    xloc = 0; yloc = 0;
    return false;
  }
}

/* ==================================================== ======== ======= */

bool UView::whereInWin(u_pos &xloc, u_pos &yloc) const {
  xloc = x; yloc = y;
  return true;
}

/* ==================================================== ======== ======= */

// returns false if undefined
bool UView::whereOnScreen(u_pos &xloc, u_pos &yloc) const {
  UWin *win = getHardwin();  //returns hardwin

  if (win && win->whereOnScreen(xloc, yloc, getDisp())) {
    //dans le cas des hardwin et softwin il n'y a rien a ajouter
    //(et dans le cas des softwins on ajouterait 2 fois x et y)

    //if (!getBox() || !dynamic_cast<UWin*>(getBox())) {
    if (!getBox() || !getBox()->winCast()) {
      xloc += x;
      yloc += y;
    }
    return true;
  }
  else {
    xloc = 0; yloc = 0;    //xloc = -1; yloc = -1;
    return false;
  }
}

u_pos UView::getXscreen() const {
  u_pos xx, yy;
  whereOnScreen(xx,yy);
  return xx;
}

u_pos UView::getYscreen() const {
  u_pos xx, yy;
  whereOnScreen(xx,yy);
  return yy;
} 

/* ==================================================== ======== ======= */

bool UView::whereOnAppli(u_pos &xloc, u_pos &yloc) const {
  UAppli* a = getAppli();
  UFrame* mf = null;
  if (!a || !(mf = a->getMainFrame())) {
    xloc = 0; yloc = 0;      //xloc = -1; yloc = -1;
    return false;
  }

  u_pos xx, yy, xx0, yy0;

  if (whereOnScreen(xx,yy) && mf->whereOnScreen(xx0,yy0,getDisp())) {
    xloc = xx - xx0;
    yloc = yy - yy0;
    return true;
  }
  else {
    xloc = 0; yloc = 0;      //xloc = -1; yloc = -1;
    return false;
  }
}

u_pos UView::getXappli() const {
  u_pos xx, yy;
  whereOnAppli(xx,yy);
  return xx;
}

u_pos UView::getYappli() const {
  u_pos xx, yy;
  whereOnAppli(xx,yy);
  return yy;
}

/* ==================================================== ======== ======= */

bool UView::convertPos(UView* to, u_pos& to_x, u_pos& to_y,
		       UView* from, u_pos from_x, u_pos from_y) {

  // marche que si both view in same win
  if (to->wingraph == from->wingraph) {
    //cerr << "same wingraph" << endl;
    to_x = from_x + from->x - to->x;
    to_y = from_y + from->y - to->y;
    return true;
  }
  else {
    //cerr << "mult wingraph" << endl;
    u_pos scr_from_x, scr_from_y, scr_to_x, scr_to_y;

    // on devrait plutot comparer les Screens !!!
    if (to->wingraph->getDisp()	== from->wingraph->getDisp()
	&& from->whereOnScreen(scr_from_x, scr_from_y)
	&& to->whereOnScreen(scr_to_x, scr_to_y)
	) {
	
      to_x = from_x + scr_from_x - scr_to_x;
      to_y = from_y + scr_from_y - scr_to_y;
      return true;
    }
    
    to_x = to_y = -1;
    return false;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// !!PAS DE UPDATE!!!
void UView::setWidth(u_dim w)  {width = w;}
void UView::setHeight(u_dim h) {height = h;}
void UView::resize(u_dim w, u_dim h) {width = w; height = h;}

UDisp*  UView::getDisp() const 
{return wingraph ? wingraph->getDisp() : null;}

UAppli* UView::getAppli() const 
{return wingraph ? wingraph->getAppli() : null;}

bool UView::isRealized() const 
{return wingraph ? wingraph->isWinRealized() : null;}

UNatWin* UView::getNatWin() const 
{return wingraph ? wingraph->getNatWin() : null;}

UWin* UView::getHardwin() const 
{return wingraph ? wingraph->getHardwin() : null;}

UView* UView::getHardwinView() const {
  UWin *win = getHardwin();
  return win ? win->getWinView(getDisp()) : null; 
}

// !ATT: returns null if parent is a Window
UBox* UView::getBoxParent() const {
  if (boxlink && boxlink->getParent()) 
    // normalement boxCast() est toujours verifie par construction
    return boxlink->getParent()->boxCast();
  else return null;
}

UBox* UView::getBox() const {
  if (boxlink) return boxlink->getBoxChild();
  else {
    UError::error("internal@UView::getBox", "null boxlink");
    return null;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMultiList::addSoftwinList(UGroup *grp) {
  //UWin* hardwin = dynamic_cast<UWin*>(grp);
  UWin* hardwin = grp->winCast();
  ULink* link = null;
  if (hardwin && (link = hardwin->getSubSoftwinLink())) {
    addList(link);
    //cerr << "ADDsoft hardwin: " << hardwin  << " " << hardwin->cname()<<endl;
  }
}

// adds an Elem (and its followers) as a sublist of the MultiList

void UMultiList::addList(ULink *l) {
  if (!l) return;  // nop (nb: test important: sauter listes nulles)
  else if (count < MAXCOUNT) list_tab[count++] = l;
  else UError::error("internal@MultiList::addList",
		     UError::Too_many_elements, count);
}


// parses the AttrList and the ElemList, put properties and add
// elements (and groups) as a sublist
// also takes into account the softwin list when applicable

UMultiList::UMultiList(UContext& curp, UGroup* grp, bool rescale) {
  for (count = 0; count < MAXCOUNT; count++) list_tab[count] = null;
  count = 0;
  UProp *prop;

  // put ALL the properties of the AttrList
  // does not take into account other elements
  for (ULink *ch = grp->getAttrLinks(); ch != null; ch = ch->getNext()) {
    if ((prop = ch->getChild()->propCast())) {
      if (ch->verifies(&curp, grp))  prop->putProp(&curp, grp);
    }
  }

  for (ULink *ch = grp->getChildLinks(); ch != null; ch = ch->getNext()) {
    // adds the first elem (or group) and it followers as a sublist
    if (ch->getChild()->elemCast() || ch->getChild()->groupCast()) {
      addList(ch);
      break;  		// passer a la suite
    }
    // put the properties of the ElemList that are BEFORE the first elem
    else if ((prop = ch->getChild()->propCast())) {
      if (ch->verifies(&curp, grp))  prop->putProp(&curp, grp);
    }
  }

  if (grp->isCmode(UMode::HARDWIN)) addSoftwinList(grp);
  is_softwin_list = grp->isCmode(UMode::SOFTWIN_LIST);

  if (rescale && curp.lscale != 0) curp.rescale();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UViewLayoutImpl::UViewLayoutImpl(UView *v) {
  view = v;
  view->hflex_count = view->vflex_count = 0; 
  visibleElemCount = 0;
  children_w = 0;
  children_h = 0;
  orient     = 0; // orient initialise plus tard en fct de curp
  mustLayoutAgain = false;
}

/* ==================================================== ======== ======= */

void UViewLayoutImpl::computeWidth(const UContext& curp,
                                   const UMargins& margins,
				   class UViewLayout& vl,
                                   bool minmax_defined)
{
  u_dim border_width = margins.left + margins.right 
    + curp.local.padding.left + curp.local.padding.right;
                   
  // auto width of this view (= children + margins/borders)
  u_dim auto_width = children_w + border_width;
  u_dim spec_width = curp.local.width;

  // 12jul03: tenir compte de vl.strategy pour les uflowbox emboites
  enum {RESIZE, KEEP_SIZE, FIXED_SIZE} mode;
  if (curp.local.width < UWidth::KEEP_SIZE) mode = RESIZE;
  else if (curp.local.width == UWidth::KEEP_SIZE) mode = KEEP_SIZE;
  else mode = FIXED_SIZE;

  if (vl.strategy == UViewLayout::NESTED) {
    // cas des nested flowboxes: il faut les retailler (sauf si taille specifiee)
    if (mode == KEEP_SIZE) mode = RESIZE;
  }
  // cas (vl.strategy == UViewLayout::IMPOSE_WSIZE) {
  // deja traite par UView::doLayout qui fait: curp.local.width = width = vl.spec_w;
  
  // size is computed the 1st time then does not change (except when zoomed)
  if (mode == KEEP_SIZE) {
    // INITIALIZING is set at the 1st update, INITIALIZED at the 2nd update
    // (a combination of layout+update+layout is necessary for correct init)

    if (!view->pext || view->width<=0 || !view->isVmode(UView::INITIALIZED)){
      if (!view->pext) view->pext = new UViewExt();
      // kept_width must not depend on the scale at init. time
      //  => inverted rescale on auto_width
      view->pext->kept_width =     // UFontFamily::getXYScale(curp.lscale));
        (u_dim)((float)auto_width / curp.xyscale);
    }
    // set width to the initial (kept) value 
    spec_width = view->pext->kept_width;
  }

  // rescale according to curp.lscale if not null (and spec_width specified)
  if (curp.lscale != 0) {
    if (spec_width > 0)         // UFontFamily::getXYScale(curp.lscale));
      spec_width = (u_dim)((float)spec_width * curp.xyscale);
  }

  // uwidth(int) with int >= 0 : size fixed by argument
  // uwidth(UWidth::keepSize) : keep initial size

  if (mode == RESIZE) {
    view->setVmodes(UView::FIXED_WIDTH, false);
    view->width = auto_width;
  }
  else {			// keepSize ou fixedSize
    view->setVmodes(UView::FIXED_WIDTH, true);
    // NB: the fixedSize can be changed by client or zoom => always reset width
    view->width = spec_width;
  }

  view->chwidth = children_w;   // width of children
  vl.spec_w = spec_width;	// nb: <= keepSize if undefined
  if (minmax_defined) {
    vl.cmin_w += border_width;
    vl.cmax_w += border_width;
  }
  else { 
    vl.cmin_w = vl.cmax_w = view->width; 
  }
}

/* ==================================================== ======== ======= */

void UViewLayoutImpl::computeHeight(const UContext& curp,
                                    const UMargins& margins,
                                    class UViewLayout& vl,
                                    bool minmax_defined)
{
  u_dim border_height = margins.top + margins.bottom
  + curp.local.padding.top + curp.local.padding.bottom;

  u_dim auto_height = children_h + border_height;

  u_dim spec_height = curp.local.height;
  if (spec_height == UHeight::KEEP_SIZE) {
    if (!view->pext || view->height<=0 || !view->isVmode(UView::INITIALIZED)){
      if (!view->pext) view->pext = new UViewExt();
      view->pext->kept_height =        // UFontFamily::getXYScale(curp.lscale));
        (u_dim)((float)auto_height / curp.xyscale);
    }
    spec_height = view->pext->kept_height;
  }

  if (curp.lscale != 0 && spec_height > 0)   // UFontFamily::getXYScale(curp.lscale));
    spec_height = (u_dim)((float)spec_height * curp.xyscale);

  if (spec_height < UHeight::KEEP_SIZE) { // autoResize
    view->setVmodes(UView::FIXED_HEIGHT, false);
    view->height = auto_height;
  }
  else {			// keepSize ou fixedSize
    view->setVmodes(UView::FIXED_HEIGHT, true);
    view->height = spec_height;
  }

  view->chheight = children_h;
  vl.spec_h = spec_height;	// nb: <= keepSize if undefined
  if (minmax_defined) {
    vl.cmin_h += border_height;
    vl.cmax_h += border_height;
  }
  else {
    vl.cmin_h = vl.cmax_h = view->height;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UView::doLayout(UContext &parp, UViewLayout &vl) {
  UViewLayoutImpl vd(this); 
  UBox *box = getBox();
  UContext curp(box, this, parp);
  
  if (vl.strategy == UViewLayout::IMPOSE_WIDTH) {
    //imposer la taille donnee par parent
    curp.local.width = width = vl.spec_w;
  }

  doLayout2(vd, box, curp, vl);

  // IMPORTANT:  Height depend de Width pour les UFlowView
  // ceci impose de refaire une seconde fois le layout du parent
  // (sauf dans le cas ou Width est fixe a priori auquel cas Height
  // (peut directement etre determine des la premier passe)
  return vd.mustLayoutAgain;
};

/* ==================================================== ======== ======= */

static
void hintElemV(UViewLayoutImpl& vd, const UContext& curp,
	       const UViewLayout& chvl, bool is_box);
static
void hintElemH(UViewLayoutImpl& vd, const UContext& curp,
	       const UViewLayout& chvl, bool is_box);
static
void hintElemViewport(UViewLayoutImpl& vd, const UContext& curp,
		      const UViewLayout& chvl, bool is_box);
static
void hintElemBorder(UViewLayoutImpl& vd, const UContext& curp,
		    const UViewLayout& chvl, bool is_box);

/* ==================================================== ======== ======= */

void UView::doLayout2(UViewLayoutImpl &vd, UGroup *grp,
                      UContext &curp, UViewLayout &vl)
{
  UMultiList mlist(curp, grp, true); // rescale
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  bool is_pane  = dynamic_cast<UPane*>(grp);   // !TEST
  bool is_border = grp->isCmode(UMode::BORDER_LIST);   // !TEST

  UGroup *chgrp = null;
  UBrick *b = null;
  UView *chboxview = null;
  ULink *ch = null;

  // interdiction de tenir compte de l'orient dans les UGroups
  // if (grp->boxCast()) vd.orient = curp.orient;
  
  // les UGroup sont normalement en UOrient::inherit
  vd.orient = grp->isBmode(UMode::IS_VERTICAL) ? 
    UOrient::VERTICAL : UOrient::HORIZONTAL;

  // if this group is not null (which generally is the case) the object
  // it contains are added to children for normal display
  // (can for instance be used for adding list-item markers, checkboxes...

  if (curp.local.content) {
    UGroup *content = curp.local.content;
    curp.local.content = null;	// avoid infinite recursion
    doLayout2(vd, content, curp, vl);    // pas de curp, meme vd
  }

  for (ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, grp)) {
      b = ch->getChild();
      chboxview = null;
      UViewLayout chvl; // !att: reinit by constr.
      chgrp = null;

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UProps

      if (b->propCast()) {
	b->propCast()->putProp(&curp, grp);
      }

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UElems

      else if (b->elemCast()) {
	b->elemCast()->getSize(&curp, &chvl.cmax_w, &chvl.cmax_h);

	if (is_border) 
	  hintElemBorder(vd, curp, chvl, false);
	else if (is_pane) 
	  hintElemViewport(vd, curp, chvl, false);
	else if (vd.orient == UOrient::VERTICAL)
	  hintElemV(vd, curp, chvl, false);
	else if (vd.orient == UOrient::HORIZONTAL)
	  hintElemH(vd, curp, chvl, false);
      }

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UGroups, UBoxes, UWins

      else if ((chgrp = b->groupCast()) && chgrp->isShowable()) {
	
	if (chgrp->boxCast()) {  //QUE BoxCast
	  
	  if ((chgrp->isDef(0,UMode::BOX)  // UBox ou UIncrust
	       // this==hardwin && child==softwin :
	       || (mlist.isSoftwinList() && chgrp->isDef(0,UMode::SOFTWIN))
	       )
	      && (chboxview = ((UBoxLink*)ch)->getViewInside(vd.view))
	      ) {
	    
	    if (chgrp->isDef(0, UMode::FLOATING)) {
	      //ne pas les compter dans taille globale
	      vd.mustLayoutAgain |= chboxview->doLayout(curp, chvl);
	    }
	    
	    else {  // cas normal
	      vd.mustLayoutAgain |= chboxview->doLayout(curp, chvl);

	      if (is_border) 
		hintElemBorder(vd, curp, chvl, true);
	      else if (is_pane)
		hintElemViewport(vd, curp, chvl, true);
	      else if (vd.orient == UOrient::VERTICAL)
		hintElemV(vd, curp, chvl, true);
	      else 
		hintElemH(vd, curp, chvl, true);

	    } // end cas normal
	  } // endif(isDef(Mode::BOX) ...)
	} // endif(boxCast)
	
	else {  // just an UGroup
	  UContext chcurp(chgrp, vd.view, curp);
	  doLayout2(vd, chgrp, chcurp, vl);   //own curp, same vd
	}
      }
    }
  
  // la suite ne concerne pas les UGroup
  if (grp->boxCast()) {

    // Border and Box size
    UMargins margins(0, 0);

    if (curp.local.border) {
       curp.local.border->getSize(curp, margins);

      if ((chgrp = curp.local.border->getSubGroup())) {

	// !! A GENERALISER AILLEURS !!

	if (!vd.view->getBorderMargins()) {
	  vd.view->setBorderMargins(new UMargins(0,0,0,0));
	  chgrp->initView(chgrp->makeLink(), vd.view);
	}
	UContext chcurp(chgrp, vd.view, curp);
	doLayout2(vd, chgrp, chcurp, vl);   //own curp, same vd
	margins.incr(*vd.view->getBorderMargins());
      }
    }

    vd.computeWidth(curp, margins, vl);
    vd.computeHeight(curp, margins, vl);
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */

void hintElemV(UViewLayoutImpl& vd, const UContext& curp,
	       const UViewLayout& chvl, bool is_box) {
  if (is_box) {
    // number of vertically "flex"ible child objects
    if (curp.local.valign == UValign::FLEX) vd.view->vflex_count++;
    // fait avant: vd.mustLayoutAgain |= chboxview->doLayout(curp, chvl);
  }

  // add spacing if something before
  if (vd.visibleElemCount > 0) vd.children_h += curp.local.vspacing;

  // NB: only takes into account Ubox or Uelem (not Ugroup nor floating)
  vd.visibleElemCount++;

  // add element size then take max
  vd.children_h += chvl.cmax_h;
  if (chvl.cmax_w > vd.children_w) vd.children_w = chvl.cmax_w;
}

/* ==================================================== ======== ======= */

void hintElemH(UViewLayoutImpl& vd, const UContext& curp,
	       const UViewLayout& chvl, bool is_box) {
  if (is_box) {
    // number of horizontally "flex"ible child objects
    if (curp.local.halign == UHalign::FLEX) vd.view->hflex_count++;
    // fait avant: vd.mustLayoutAgain |= chboxview->doLayout(curp, chvl);
  }

  // add spacing if something before
  if (vd.visibleElemCount > 0) vd.children_w += curp.local.hspacing;

  // NB: only takes into account Ubox or Uelem (not Ugroup nor floating)
  vd.visibleElemCount++;
   
  // add element size then take max
  vd.children_w += chvl.cmax_w;
  if (chvl.cmax_h > vd.children_h) vd.children_h = chvl.cmax_h;
}

/* ==================================================== ======== ======= */

void hintElemViewport(UViewLayoutImpl& vd, const UContext& curp,
                      const UViewLayout& chvl, bool is_box) {
  if (!is_box) {
    UError::error("UView::doLayout",
                  "a UPane can only contain UBox(es) in its central area");
    return;
  }

  // fait avant: vd.mustLayoutAgain |= chboxview->doLayout(curp, chvl);

  // old
  //vd.children_w = chvl.cmax_w;
  //vd.children_h = chvl.cmax_h;

  // new: 9jn03: on prend le max de tous les children
  if (vd.children_w < chvl.cmax_w) vd.children_w = chvl.cmax_w;
  if (vd.children_h < chvl.cmax_h) vd.children_h = chvl.cmax_h;
  
  //cerr << "-- hint viewport "<< vd.children_w << " " << chvl.cmax_w
  //  << " / " <<vd.children_h << " " << chvl.cmax_h << endl;
}

/* ==================================================== ======== ======= */

void hintElemBorder(UViewLayoutImpl& vd, const UContext& curp,
		    const UViewLayout& chvl, bool is_box) {

  if (!vd.view->getBorderMargins()) {
    UError::error("internal@hintElemBorder","null view");
    return;
  }
  UMargins& frame = *vd.view->getBorderMargins();

  //NB: top et bottom s'adaptent a la zone centrale en largeur
  //ils ne controlent donc que la hauteur de leurs zones respectives
  switch (curp.local.valign) {
  case UValign::TOP:
    if (chvl.cmax_h > frame.top)  frame.top = chvl.cmax_h;
    break;
  case UValign::BOTTOM:
    if (chvl.cmax_h > frame.bottom)  frame.bottom = chvl.cmax_h;
    break;
  case UValign::CENTER:
  case UValign::FLEX:
    if (chvl.cmax_h > vd.children_h)  vd.children_h = chvl.cmax_h;
    break;
  }
  
  //NB: left et right s'adaptent a la zone centrale en hauteur
  //ils ne controlent donc que la largeur de leurs zones respectives
  switch (curp.local.halign) {
  case UHalign::LEFT:
    if (chvl.cmax_w > frame.left)  frame.left = chvl.cmax_w;
    break;
  case UHalign::RIGHT:
    if (chvl.cmax_w > frame.right)  frame.right = chvl.cmax_w;
    break;
  case UHalign::CENTER:
  case UHalign::FLEX:
    if (chvl.cmax_w > vd.children_w)  vd.children_w = chvl.cmax_w;
    break;
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
// UPDATE
/* ==================================================== ======== ======= */

inline void drawWallPaperLine(UWinGraph& g, UContext &curp,
			      int y, int hali, 
			      const UIma *wallp, u_dim ima_w,
			      const URegion &r, const URegion &clip) {
  URegion wr = r;
  wr.y = y;
  
  if (hali == UHalign::CENTER) {
    wr.x = (r.width - ima_w) / 2 + r.x;
    //g.drawIma(wallp, x, y);
    wallp->paint(g, &curp, wr);
  }
  else if (hali == UHalign::FLEX) {
    wr.x = (r.width - ima_w) / 2 + r.x;           // A REVOIR!!!
    //g.drawIma(wallp, x, y);
    wallp->paint(g, &curp, wr);
  }
  else {
    for (wr.x = r.x; wr.x < r.x + r.width; wr.x += ima_w)
      if (wr.x + ima_w >= clip.x && wr.x < clip.x + clip.width)
	//g.drawIma(wallp, x, y); 
	wallp->paint(g, &curp, wr);
  }
}

static void drawWallPaper(UWinGraph& g, UContext &curp, 
                          const UIma *wallp, 
			  u_dim ima_w, u_dim ima_h,
			  const URegion &r, const URegion &clip) {

  //NB: les if ne servent qu'a eviter des appels de fct inutiles:
  // de toute facon l'intersection sera faite via le clip
  // (qui a ete prealablement affecte a 'g')

  const UHalign *halign = curp.local.background->getHalign(); 
  const UValign *valign = curp.local.background->getValign();

  int y;
  int vali = (valign ? valign->get() : UValign::top.get());
  int hali = (halign ? halign->get() : UHalign::left.get());
      
  if (vali == UValign::CENTER) {
    y = (r.height - ima_h) / 2 + r.y;
    drawWallPaperLine(g, curp, y, hali, wallp, ima_w, r, clip);
  }

  else if (vali == UValign::FLEX) {
    y = (r.height - ima_h) / 2 + r.y;        // A REVOIR!!!
    drawWallPaperLine(g, curp, y, hali, wallp, ima_w, r, clip);
  }

  else {
    for (y = r.y; y < r.y + r.height; y += ima_h)
      if (y + ima_h >= clip.y && y < clip.y + clip.height) {
	drawWallPaperLine(g, curp, y, hali, wallp, ima_w, r, clip);
      }
  }
}

/* ==================================================== ======== ======= */

void UViewUpdateImpl::updateBackground(UWinGraph& g, UContext &curp, 
				       const URegion &r, const URegion& clip){
  // wallpaper a afficher (s'il existe)
  const UIma *wallp = null;
  u_dim ima_w = 0, ima_h = 0;

  // pixmap a afficher en wallpaper (si on le trouve!)
  if (curp.local.background) {

    if (curp.local.background->getColor()) {
      curp.bgcolor = curp.local.background->getColor();
    }

    // PAS exclusif (pas de else!)  
    if (curp.local.background->getIma()) {
      wallp = curp.local.background->getIma();

      //NB: getSize charge l'image si pas deja fait
      wallp->getSize(&curp, &ima_w, &ima_h);
      
      // n'afficher que si on a vraiment reussi a charger l'image
      // !att: ne pas afficher UnknownPixmap en background en cas d'echec!
      
      if (!wallp->isRealized() || ima_w <= 0 || ima_h <= 0)
	wallp = null;  // inutilisable

      // problematique pour locateSource()
      // else			// sinon passer en mode transparent
      //   curp.bgcolor = &UBgcolor::none;
    }
  }

  if (curp.local.alpha == 0.
      // pas de wallpaper (ou pas trouve) et pas de bgcolor
      || (!wallp && curp.bgcolor->equals(UBgcolor::none))
      ) {
      //completement transparent: on n'affiche rien
    // view->setVmodes(UView::TRANSP_VIEW, true);
  }

  else {
    bool blend_paint = false;

    if (curp.local.alpha == 1.)   //opaque
      //view->setVmodes(UView::TRANSP_VIEW, false)
      ;
    else {                   //translucent => melange
      //view->setVmodes(UView::TRANSP_VIEW, true);
      if (can_paint /* && !upmode.elem */) {
	blend_paint = true;
      }
    }

    if (wallp) {     // il y a un wallpaper et on reussi a le charger
 
      // cas des shaped pixmaps: reafficher ce qu'il y a en dessous
      // puis le pixmap par dessus
      // if (wallp->isTransparent()) 
      //	view->setVmodes(UView::TRANSP_VIEW, true);

      if (can_paint) {
	//nb: reafficher clip et non r qui va deborder!!!
	if (blend_paint) g.beginBlend(clip, curp.local.alpha);

	drawWallPaper(g, curp, wallp, ima_w, ima_h, r, clip);  //old: chclip

	// terminer mode BLEND (=> merging des 2 plans)
	if (blend_paint) g.endBlend();
      }
    }

    else {        // uniform bgcolor   
      if (can_paint) {
#ifdef WITH_GL
	g.beginBlend(clip, curp.local.alpha);
	g.setColor(curp.bgcolor);
	g.fillRect(clip.x, clip.y, clip.width, clip.height);
	g.endBlend();
#else
	// NB: blendColor ne marchera pas correctement dans les 
	// Incrust (a cause du beginSubwin() prealable et ces
	// 2 operations ne se combinent pas correctment
	g.setColor(curp.bgcolor);
	if (blend_paint) g.blendColor(clip, curp.local.alpha);
	else      // reafficher zone minimale (= celle du clip)
	  g.fillRect(clip.x, clip.y, clip.width, clip.height);
#endif
      }
    }
  }
}

/* ==================================================== ======== ======= */

void UViewUpdateImpl::setMargins(UWinGraph& g, const UContext &curp,
                                 const URegion &r,
                                 bool add_frame_and_padding)
{
  margins.set(0, 0, 0, 0);  // margin doit il etre conserve dans vd ?????
   
  if (curp.local.border) {
    // add border decoration 
    curp.local.border->getSize(curp, margins);

    // add border frame elems
    if (add_frame_and_padding
        && view->border_margins && !curp.local.border->isOverlaid())
      margins.incr(*view->border_margins);
  }
 
  // add margin between border and content
  if (add_frame_and_padding) margins.incr(curp.local.padding);

  // !vd.margin ne sert plus apres ds la nvlle version => le virer de vd
  x = r.x + margins.left;
  y = r.y + margins.top;
  width  = r.width - margins.left - margins.right;
  height = r.height - margins.top - margins.bottom;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UViewUpdateImpl::callbacks(UGroup *grp, UView* win_view) {

  if (grp->isDef(UMode::VIEW_CHANGE_CB,0)) {
    if (!view->pext) view->pext = new UViewExt();

    if (!view->pext->old_region) {
      view->pext->old_region = new URegion(*view); // init: on n'appelle rien
    }
    else {
      URegion* oldr = view->pext->old_region;

      if (oldr->x != view->x || oldr->y != view->y) {
	UEvent e(UEvent::viewMove, null, win_view, null);
	e.setSource(view);
	e.getSourceProps().redrawStatus = true;
	//e.getSourceProps().redrawClip.set(chclip);
	e.getSourceProps().redrawClip = chclip;

	// mettre fire() en dernier sinon boucle infinie
	oldr->x = view->x;
	oldr->y = view->y;
	grp->fire(e, UOn::viewMove);
      }

      if (oldr->width != view->width || oldr->height != view->height) {
	UEvent e(UEvent::viewResize,  null, win_view, null);
	e.setSource(view);
	e.getSourceProps().redrawStatus = true;
	//e.getSourceProps().redrawClip.set(chclip);
	e.getSourceProps().redrawClip = chclip;
	oldr->width  = view->width;
	oldr->height = view->height;
	// fire() doit etre APRES modif width/height sinon boucle infinie
	// si le callback change lui meme la taille (car rappel de cette
	// meme fonction et old jamais modifie...)
	grp->fire(e, UOn::viewResize);
      }
    }
  }

  // tester d'abord si INIT ?
  UEvent e(UEvent::viewPaint,  null, win_view, null);
  e.setSource(view);
  e.getSourceProps().redrawStatus = true;
  //e.getSourceProps().redrawClip.set(chclip);
  e.getSourceProps().redrawClip = chclip;
  grp->fire(e, UOn::viewPaint);
}

/* ==================================================== ======== ======= */
//!NB: ne doit etre appele QUE pour UMode::FLOATING

void UViewUpdateImpl::setFloating(UContext& curp, UGroup* grp, URegion& r) {
  // rajouter d'eventuelles coords relatives ajoutees par UPos 
  // (= cas UMode::FLOATING) mais PAS en mode search !!!
  // !!!NB: devrait tenir compte du scale!!!
    
  //cerr << "setfloating1 "<< grp << " " << grp->cname() << endl;

  if (upmode.mode < UViewUpdate::LOCATE_ELEM_POS) {
    //ne rien faire dans les autres cas sinon les coords peuvent 
    //etre ajoutees plusieurs fois
    view->x = (r.x += curp.local.xpos);
    curp.local.xpos = 0; //eviter pbm si recursion avec content
    view->y = (r.y += curp.local.ypos);
    curp.local.ypos = 0;

    //cerr << "setfloating2 "<< view->x << " " << view->y << endl;

    //les coords ont ete remises a jour
    view->setVmodes(UView::OBS_COORDS, false);
  }
  //NB: y'a peut etre des cas ou les coords sont pas a jour ?    
}

/* ==================================================== ======== ======= */

UViewUpdateImpl::UViewUpdateImpl(UView *v, const URegion &r, 
				 UViewUpdate &_upmode)
  : upmode(_upmode) {
  view = v;
  hflex_space = vflex_space = 0;
  can_paint = false;
  edit = null;
  tabview = dynamic_cast<UTableView*>(v);   // si c'est une table:
  if (tabview) {tabview->ccur = tabview->lcur = 0;}

  // necessaire pour viewport qui calcule le max des tailles
  chr.width = chr.height = 0;
  
  if (!view->isVmode(UView::INITIALIZED)) {   // !!new
    if (!view->isVmode(UView::INITIALIZING))
      view->setVmodes(UView::INITIALIZING, true);
    else {
      view->setVmodes(UView::INITIALIZING, false);
      view->setVmodes(UView::INITIALIZED, true);
    }
  }

  //if (vup.getMode() == UViewUpdate::PAINT_CHANGED) {
  //  //att aux floatings: pas encore deplaces ?
  //  if (v->x == r.x && v->y == r.y
  //	&& v->width == r.width && v->height == r.height)
  //    return;  //nothing changed
  //  else can_paint = true;
  // }
  // if (v->width != r.width || v->height != r.height) size_changed = true;
  //  else size_changed = false;

   // ATT: on peut pas init les coords. ici a cause des floatings 
   // dont les coords. dependent de curp.

  switch (upmode.mode) {
  case UViewUpdate::PAINT_ALL:
    can_paint = true;
    // ca ne sert a rien de gerer les damaged et de detecter l'etat
    // "after painting all damaged" puisque de toute facon on repaint
    // tous les layers
    break;

  case UViewUpdate::PAINT_DAMAGED:
    if (view->isVmode(UView::DAMAGED)) {
      can_paint = true;
      // on incremente le nombre de vues damaged traversees
      upmode.damaged_level++;
    }

    else if (upmode.damaged_level > 0)
      // faut afficher si on est dans un descendant d'un damaged
      can_paint = true;

   else if (upmode.after_damaged) 
     // ou bien si on est "after painting all damaged"
     // seuls les Floatings doivent etre reaffiches dans ce cas 
     // can_paint = true;  (code ah hoc dans update2() pour Floatings)
    break;

  case UViewUpdate::UPDATE_DATA:
    if (view->isVmode(UView::DAMAGED)) {
      // on incremente le nombre de vues damaged traversees
      upmode.damaged_level++;
    }
    break;

  default:
    // NB: can_paint reste a false pour les ELEM_OPS
    break;
  }
}

/* ==================================================== ======== ======= */

UViewUpdateImpl::~UViewUpdateImpl() {
  switch (upmode.mode) {
  case UViewUpdate::PAINT_ALL:
    view->setVmodes(UView::DAMAGED, false);
    // ca ne sert a rien de gerer les damaged (voir constructeur)
    break;

  case UViewUpdate::PAINT_DAMAGED:
  case UViewUpdate::UPDATE_DATA:

    if (view->isVmode(UView::DAMAGED)) {
      view->setVmodes(UView::DAMAGED, false);
      // on decremente le nombre de vues damaged traversees
      upmode.damaged_level--;

      if (upmode.damaged_level == 0) {
	// on passe dans l'etat "after painting all damaged": ca signifie
	// qu'il faudra repaindre toutes les vues (en partculier les floatings)
	// a afficher dans la region de clipping
	upmode.after_damaged = true; 
      }
      else if (upmode.damaged_level < 0)
	UError::error("internal@UViewUpdateImpl::~UViewUpdateImpl", 
		      "unbalanced damaged view");
    }
    break;

  default:
    break;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UView::doUpdate(UContext& parp, URegion r, URegion clip, 
		     UViewUpdate& upmode) {
  UBox *box = getBox();
  
  // test pas valable pour les FLOATING car leurs coords dependent de curp
  // et sont donc changees apres creation et parsing de ce dernier

  if (!box->isDef(0, UMode::FLOATING)) {
    // toujours initialiser les coords meme si objet cache par autre chose,
    // sinon des objets theoriquement non visibles vont se retrouver dans 
    // une zone visible car leurs coords x,y vaudront 0,0 faute d'avoir
    // ete initialisees (en part. s'il y a du clipping avec des scrollpane)
    this->set(r);

    // elem is not visible because of scrolling, etc... 
    // l'intersection de clip et de r est affectee dans clip
    if (clip.setInter(r) == 0) return;
  }

  UViewUpdateImpl vd(this, r, upmode);
  UContext curp(box, this, parp);
  doUpdate2(vd, box, curp, r, clip, upmode);
  //NB: finalisation par destructeur de UViewUpdateImpl
}

/* ==================================================== ======== ======= */

static void initLayoutH(UViewUpdateImpl&, const UContext&, const URegion& r);
static void layoutElemH(UViewUpdateImpl&, const UContext&, 
			UGroup *grp, ULink* link, 
			u_dim ww, u_dim hh, UView* view);
static void nextElemH(UViewUpdateImpl&, const UContext&);

static void initLayoutV(UViewUpdateImpl&, const UContext&, const URegion& r);
static void layoutElemV(UViewUpdateImpl&, const UContext&, 
			UGroup *grp, ULink* link,
			u_dim ww, u_dim hh, UView* view);
static void nextElemV(UViewUpdateImpl&, const UContext&);

static void initLayoutViewport(UViewUpdateImpl&, const UContext&, const URegion& r);
static void layoutElemViewport(UViewUpdateImpl&, const UContext&, 
			       UGroup *grp, ULink*,
			       u_dim ww, u_dim hh, UView* view);
static void nextElemViewport(UViewUpdateImpl&, const UContext&);

static void layoutElemBorder(UViewUpdateImpl& vd, const UContext& curp,
			     UGroup *grp, ULink* link,
			     u_dim ww, u_dim hh, UView* view);
static void nextElemBorder(UViewUpdateImpl&, const UContext&);

/* ==================================================== ======== ======= */
// - mode SearchElem: juste recuperer l'elem et sa position sans redessiner
//   !ATT il faut IMPERATIVEMENT elem_props != null dans le mode SearchElem !
// - clip est passe en valeur, pas r
// - noter que coords de r et views sont != dans le cas des UWIn INBOX

void UView::doUpdate2(UViewUpdateImpl& vd, UGroup* grp,
		      UContext& curp, URegion& r,
                      URegion& clip, UViewUpdate& vup)
{
  UMultiList mlist(curp, grp, true);  // rescale
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  if (grp->isDef(0, UMode::FLOATING)) {
    vd.setFloating(curp, grp, r);
    if (clip.setInter(r) == 0) return;

    // on est dans l'etat "after painting all damaged": ca signifie qu'il
    // faut traiter a part les floatings car ils peuvent recouvrir les
    // damaged precedemment affiches
    // NB: ne rien faire de special si damaged_level > 0 (on est deja 
    //     a l'interieur d'un damaged, donc on repaint de tt facon)

    // !! voir uuflow.cc !!

    if (vup.damaged_level == 0 && vup.after_damaged) {
      switch (vup.mode) {
      case UViewUpdate::PAINT_DAMAGED:
	vd.can_paint = true;
	vup.above_damaged_count++;
	break;

      case UViewUpdate::UPDATE_DATA:
	vup.above_damaged_count++;
	break;

      default:	// rien a faire dans les autres cas
	break;
      }
    }
  }

  if (UAppli::getDefaults().max_app_width > 0 && grp->winCast()) {
    u_dim xx = getXappli();
    if (xx + width > UAppli::getDefaults().max_app_width)
      width = UAppli::getDefaults().max_app_width - xx;
  }

  if (UAppli::getDefaults().max_app_height > 0 && grp->winCast()) {
    u_dim yy = getXappli();
    if (yy + height > UAppli::getDefaults().max_app_height)
      height = UAppli::getDefaults().max_app_height - yy;
  }

  bool is_pane = false;   // !TEST
  bool is_border = grp->isCmode(UMode::BORDER_LIST);   // !TEST
  bool g_end = false;
  UWinGraph &g = curp.winview->wg();
  UGroup* chgrp = null;

  if (grp->boxCast()) {  // que pour les Boxes 
    vd.edit = curp.local.edit;
    is_pane = dynamic_cast<UPane*>(grp);   // !TEST

    // !att: setClip peut provoquer un seg fault si appele avec des donnees
    // non initialisee (c'est le cas pour les operations LOCATE sans PAINT)
    // MAIS (correct:15May03: g doit etre init. pour les INCRUST dans
    // tous les cas d'affichage meme si can_paint = false afin que le
    // contenu ne soit pas decale lors de l'affichage des children

    if (vup.mode < UViewUpdate::UPDATE_DATA 
	&& grp->isCmode(UMode::INCRUST)){
      // sub-hardwin incrustee X -> decaler les coords
      g.beginSubwin(clip, r.x, r.y); 
      g_end = true;      // NB: ne pas oublier de faire g.end()
    }
    else if (vd.can_paint) {
      if (grp->isDef(0, UMode::DOUBLE_BUFFER)){
	g.beginDoublebuf(clip); 
	g_end = true;      // NB: ne pas oublier de faire g.end()
      }
      else g.setClip(clip);
    }

    // interdiction de tenir compte de l'orient dans les UGroups
    // vd.orient = curp.orient;
    vd.orient = grp->isBmode(UMode::IS_VERTICAL) ? 
      UOrient::VERTICAL : UOrient::HORIZONTAL;

    vd.updateBackground(g, curp, r, clip);

    // margin normal qui prend en compte le border frame
    // initializes vd.x, vd.y, vd.width, vd.height
    vd.setMargins(g, curp, r, true);

    // clipping limits (NB: les bords sont exclus !)
    vd.chclip.set(vd.x, vd.y, vd.width, vd.height);
  
    // pas la peine de chercher a afficher les enfants
    // s'ils sont hors zone de clip
    if (vd.chclip.setInter(clip) == 0) {
      //if (g_end) g.end();
      //return;
      goto END;
    }

    if (is_pane) 
      initLayoutViewport(vd, curp, r);
    else if (vd.orient == UOrient::HORIZONTAL) 
      initLayoutH(vd, curp, r);
    else 
      initLayoutV(vd, curp, r);

  } // endif (grp->boxCast())


  // if this group is not null (which generally is the case) the object
  // it contains are added to children for normal display
  // (can for instance be used for adding list-item markers, checkboxes...

  if (curp.local.content) {
    UGroup *content = curp.local.content; // same curp, same vd
    curp.local.content = null;	// avoid infinite recursion
    doUpdate2(vd, content, curp, r, clip, vup);
  }

  for (ULink* ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, grp)) {
 
      UBrick* b = ch->getChild();
      UView* chboxview = null;
      UElem* it = null; 
      chgrp = null; // !!reinit necessaire
      u_dim ww, hh;

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UProps

      if (b->propCast())
	b->propCast()->putProp(&curp, grp);

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UElems

      else if ((it = b->elemCast())) {
	it->getSize(&curp, &ww, &hh);

	if (is_border)
	  layoutElemBorder(vd, curp, grp, ch, ww, hh, null);
	else if (is_pane)
	  layoutElemViewport(vd, curp, grp, ch, ww, hh, null);
	else if (vd.orient == UOrient::HORIZONTAL) 
	  layoutElemH(vd, curp, grp, ch, ww, hh, null);
	else 
	  layoutElemV(vd, curp, grp, ch, ww, hh, null);

	if (vd.chr.width > 0 && vd.chr.height > 0) {
	    
	  //NB: ITEM_OPS => vd.can_paint == false mais pas l'inverse !
	  if (vd.can_paint) {
	    if ((vd.orient == UOrient::HORIZONTAL
		 && vd.chr.x + vd.chr.width > vd.chclip.x
		 && vd.chr.x < vd.chclip.x + vd.chclip.width
		 )
		||
		(vd.orient == UOrient::VERTICAL
		 && vd.chr.y + vd.chr.height > vd.chclip.y
		 && vd.chr.y < vd.chclip.y + vd.chclip.height)
		) {
	      g.setClip(vd.chclip);
	      it->paint(g, &curp, vd.chr);

	      if (vd.edit && vd.edit->getCaretStr() == (const UElem*)it) {
		// nb: danger: ne pas ecraser le curseur en reaffichant
		// l'elem suivant par dessus
		vd.edit->paint(g, &curp, vd.chr);
	      }
	    }
	  } //endif(can_paint)
 
	  // cas UPDATE_DATA: calculer position du cursor pour shifter
	  // correctement les children au PAINT suivant
	  else if (vup.mode == UViewUpdate::UPDATE_DATA) {
	    if (vd.edit && vd.edit->getCaretStr() == (const UElem*)it) {
	      int xpos = vd.edit->getXpos(g, &curp, vd.chr);

              //cerr << "EDIT XPos = " << xpos << " limit " << vd.x + vd.width << endl;
              if (xpos > vd.x + vd.width -1) {
                edit_shift += (vd.x + vd.width) - (xpos + 1);
              }
              else if (xpos < vd.x) {
                edit_shift += vd.x - xpos + 10;
                if (edit_shift > 0) edit_shift = 0;
              }
              else if (edit_shift < 0) {
                int caret_pos = -1;
                if (!vd.edit->getCaretStr(caret_pos)
                    || (caret_pos == 0 && !vd.edit->getPreviousStr(grp))
                    )
                  edit_shift = 0;
              }
            }
          }

	  else if (vup.mode >= UViewUpdate::LOCATE_ELEM_POS) {
	    // do not draw, just find
	    // !att: il faut que e->elem_props soit =!null
	    // nb: s'arrete de cherche qunad true est renvoye'
	    if (vup.mode == UViewUpdate::LOCATE_ELEM_POS) {
	      if (vd.orient == UOrient::HORIZONTAL) {
		if (vd.view->locateElemH(curp, ch, g, vd.chr, vup))
		  //return;
		  goto END;
	      }
	      else {
		if (vd.view->locateElemV(curp, ch, g, vd.chr, vup))
		  //return;
		  goto END;
	      }
	    }
	    else if (vup.mode == UViewUpdate::LOCATE_ELEM_PTR) {
	      if (vd.view->locateElemRef(curp, ch, g, vd.chr, vup))
		//return;
		goto END;
	    }
	  } //endif(ITEM_OPS)

	    // increment 
	  if (is_border)
	    nextElemBorder(vd, curp);
	  else if (is_pane) 
	    nextElemViewport(vd, curp);
	  else if (vd.orient == UOrient::HORIZONTAL)
	    nextElemH(vd, curp);
	  else 
	    nextElemV(vd, curp);
	  
	  } // endif (vd.chr.width > 0 && vd.chr.height > 0)	 
      } // endif elemCast()

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UGroups

      else if ((chgrp = b->groupCast()) && chgrp->isDef(0, UMode::GROUP)) {
	if (chgrp->isShowable()) {
	  UContext chcurp(chgrp, vd.view, curp);   // own curp, same vd

	  if (vd.tabview  && dynamic_cast<UTrow*>(chgrp)) {  // tables
	    vd.height = 
	      std::max(vd.tabview->lines[vd.tabview->lcur].max_dim,
		       vd.tabview->lines[vd.tabview->lcur].spec_dim);
  	    vd.tabview->ccur = 0;
	    doUpdate2(vd, chgrp, chcurp, r, clip, vup);
	    vd.chr.x = vd.x;
	    vd.y += vd.height + curp.local.vspacing;
	    (vd.tabview->lcur)++;
	  }
      	  else {
	    doUpdate2(vd, chgrp, chcurp, r, clip, vup);
	  }
	}
      }

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // UBoxes

      else if (chgrp && chgrp->isShowable()
	       && (chgrp->isDef(0, UMode::BOX)
		   // this==hardwin && child==softwin :
		   || (mlist.isSoftwinList() && chgrp->isDef(0,UMode::SOFTWIN))
		   )
	       && (chboxview = ((UBoxLink*)ch)->getViewInsideSize(vd.view, &ww, &hh))
	       ) {

	// 1::cas des Floatings
	if (chgrp && chgrp->isDef(0, UMode::FLOATING)) {
	  // fl_chr pour ne pas abimer vd.chr qui est cumulatif
	  // soit en x soit en y !!!
	  URegion fl_chr;
	  fl_chr.width  = ww;
	  fl_chr.height = hh;
	  // !att: on rajoute la MARGE: les coords sont locales 
	  // % a l'INTERIEUR du CADRE et non % a l'origine de la Box.
	  fl_chr.x = vd.x;
	  fl_chr.y = vd.y;
	  // coords relatives rajoutees ensuite via setCoords
	  chboxview->doUpdate(curp, fl_chr, vd.chclip, vup);
	}

	// 2::tous les autres uboxes
	else {

	  if (vd.tabview) {  // cas des tables
	    // tenir compte des rowspan des lignes precedentes
	    if (vd.tabview->cols[vd.tabview->ccur].rowspan > 1) {
	      vd.chr.x += 
		std::max(vd.tabview->cols[vd.tabview->ccur].spec_dim,
			 vd.tabview->cols[vd.tabview->ccur].max_dim);
	      (vd.tabview->cols[vd.tabview->ccur].rowspan)--;
	      vd.tabview->ccur += vd.tabview->cols[vd.tabview->ccur].colspan;
	    }

	    // si UTcell tenir compte de colspan, rowspan
	    UTcell *cell = dynamic_cast<UTcell*>(chgrp);
	    if (cell) {
	      vd.tabview->cols[vd.tabview->ccur].colspan = cell->getColspan();
	      vd.tabview->cols[vd.tabview->ccur].rowspan = cell->getRowspan();
	      vd.tabview->ccur += cell->getColspan();
	    }
	    else vd.tabview->ccur++;
	  }

	  if (is_border)
	    layoutElemBorder(vd, curp, grp, ch, ww, hh, chboxview);
	  else if (is_pane)
	    layoutElemViewport(vd, curp, grp, ch, ww, hh, chboxview);
	  else if (vd.orient == UOrient::HORIZONTAL)
	    layoutElemH(vd, curp, grp, ch, ww, hh, chboxview);
	  else 
	    layoutElemV(vd, curp, grp, ch, ww, hh, chboxview);

	  if (vd.chr.width > 0 && vd.chr.height > 0) {
	    chboxview->doUpdate(curp, vd.chr, vd.chclip, vup);
	    // increment

            if (is_border)
	      nextElemBorder(vd, curp);
	    else if (is_pane) 
	      nextElemViewport(vd, curp);
	    else if (vd.orient == UOrient::HORIZONTAL) 
	      nextElemH(vd, curp);
	    else 
	      nextElemV(vd, curp);
	  } 
	}
      }
    }

  // deplace: ne porte pas sur le border (ce qui peut etre un PBM ?)

  // call paint and resize callback: 
  // NB: desormais appele en FIN de traitement pour permettre utilisation
  // cohererente de UGraph ds traitements clients)
  if (vd.can_paint 
      && grp->isDef(UMode::VIEW_PAINT_CB|UMode::VIEW_CHANGE_CB,0)) 
    vd.callbacks(grp, curp.winview);

 END:
  if (curp.local.border) {
    
    // paint border decoration
    if (vd.can_paint) {
      g.setClip(clip);  // (necessaire apres clip children)
      curp.local.border->paint(g, curp, r);
    }

    // !! A GENERALISER
    
    // paint border frame elems
    // en fait faudrait dessiner les bords APRES (surtout si superposes!)
    if (vd.view->border_margins && (chgrp = curp.local.border->getSubGroup())) {
      // margin special qui ne prend pas en compte le border frame
      // ni le padding interne
      
      vd.setMargins(g, curp, r, false);
      
      // NB: att au CLIP: doit etre celui de la view tt entiere !
      vd.chclip.set(vd.x, vd.y, vd.width, vd.height);
      if (vd.chclip.setInter(clip) != 0) {
	UContext chcurp(chgrp, vd.view, curp);
	doUpdate2(vd, chgrp, chcurp, r, clip, vup);
      }
    }
  }

  if (g_end) g.end();  // ne pas oublier!!
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */

static void initLayoutH(UViewUpdateImpl& vd, const UContext& curp, 
			const URegion& r) {
  vd.hflex_space = 0;

  // !!TOUTES CES OPTIONS DEVRAIENT ETRE RE-TESTEES POUR **CHAQUE** ELEMENT!!

  // there are flexible objects => use all space
  if (vd.view->hflex_count > 0)  {
    vd.chr.x = vd.x;
    // calcul espace restant
    vd.hflex_space = (vd.width - vd.view->chwidth) / vd.view->hflex_count;
    vd.view->chwidth = vd.width;
  }
  else switch (curp.local.halign) {
  case UHalign::LEFT:
    //vd.chr.x = vd.x;
    // NEW: 12sep03: edit_shift scrolle le texte edite
    vd.chr.x = vd.x + vd.view->edit_shift;
    break;

  case UHalign::CENTER:
    // ajouter espace restant a vd.x
    vd.chr.x = vd.x + (vd.width-vd.view->chwidth)/2 + vd.view->edit_shift;

    // NEW:12sep03
    // si trop grand alors mettre en UHalign::LEFT pour que le debut
    // apparaisse et ne soit pas clipe (en part quand il y a du texte)
    if (vd.chr.x < vd.x && vd.view->edit_shift == 0) vd.chr.x = vd.x;
    break;

  case UHalign::RIGHT:     
    // ajouter espace restant a vd.x
    // reste centre (et clipe) si trop grand (peut deborder!)

    // !!COMPLETEMENT FAUX : voir VCENTER
    vd.chr.x = vd.x + (vd.width-vd.view->chwidth) + vd.view->edit_shift;
    break;

  default:
    // cas d'erreur: comme LEFT
    // se produit en particulier qd il y a une deconnade entre doLayout
    // et doUpdate et que flexCount==0 avec UView::HFLEX
    // (ca vient du fait que toutes les options qui precedent devraient
    // etre retestees pour chaque element)
    vd.chr.x = vd.x;
    break;
  }
}

/* ==================================================== ======== ======= */

static void initLayoutV(UViewUpdateImpl& vd, const UContext& curp, 
			const URegion& r) {
  vd.vflex_space = 0;

  // !!TOUTES CES OPTIONS DEVRAIENT ETRE RE-TESTEES POUR **CHAQUE** ELEMENT!!

  // there are flexible objects => use all space
  if (vd.view->vflex_count > 0)  {
    vd.chr.y = vd.y;
    vd.vflex_space = (vd.height - vd.view->chheight) / vd.view->vflex_count;
    vd.view->chheight = vd.height;
  }
  else switch (curp.local.valign) {
  case UValign::TOP:
    vd.chr.y = vd.y;
    break;
	
  case UValign::CENTER:
    // !!COMPLETEMENT FAUX : en fait c'est la hauteur des enfants
    // concernes qu'il faut enlever. c'est parfois approxime par 
    // favoriteHeight mais pas du tout dans le cas ou on definit uheight()
    //obs: vd.chr.y = vd.y + (vd.height-vd.view->favoriteHeight)/2;

    // ajouter espace restant a vd.y
    vd.chr.y = vd.y + (vd.height-vd.view->chheight)/2;

    // NEW:12sep03
    // si trop grand alors mettre en UHalign::TOP pour que le haut
    // apparaisse et ne soit pas clipe (en part quand il y a du texte)
    if (vd.chr.y < vd.y) vd.chr.y = vd.y;
    break;

  case UValign::BOTTOM:
    // ajouter espace restant a vd.y
    // reste centre (et clipe) si trop grand (peut deborder!)

    // !!COMPLETEMENT FAUX : voir VCENTER
    //obs: vd.chr.y = vd.y + (vd.height-vd.view->favoriteHeight);
    vd.chr.y = vd.y + (vd.height-vd.view->chheight);
    break;

  default:
    // cas d'erreur: comme TOP
    // VOIR note pour HORIZONTAL
    vd.chr.y = vd.y;
    break;
  }
}

/* ==================================================== ======== ======= */

static void initLayoutViewport(UViewUpdateImpl& vd, const UContext& curp, 
			       const URegion& r){
  UPaneView* paneview = dynamic_cast<UPaneView*>(vd.view);
  if (!paneview) {
    UError::error("internal@initLayoutViewport","null PaneView");
    return;
  }
  paneview->margins = vd.margins;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */

static void layoutElemH(UViewUpdateImpl& vd, const UContext& curp,
			UGroup *grp, ULink* link,
			u_dim ww, u_dim hh, UView*chview) {
  switch (curp.local.valign) {
  case UValign::TOP:
    vd.chr.height = hh;
    vd.chr.y = vd.y;
    break;

  case UValign::FLEX:
  case UValign::CENTER:
    // !NOTE: pour les Elems, VFLEX est identique a VCENTER

    // Cas: BOX + FLEX
    if (chview && curp.local.valign == UValign::FLEX) {
      // - tous les enfants sont justifies verticalement
      // sur celui qui a la plus grande hauteur
      // - selon les cas (voir ligne precedente) ils sont ou non
      // egalement adaptes par rapport a laur parent      
      //NB: ! NE PAS changer chw sinon le scroll va deconner
      vd.chr.height = vd.height;
      vd.chr.y = vd.y;
    }

    // Cas CENTER ou Elem{FLEXouCENTER})
    else {
      vd.chr.height = hh;
      // ajouter espace restant a vd.y
      // reste centre (et clipe) si trop grand (peut deborder!)
      vd.chr.y = vd.y + (vd.height-hh)/2;
    }
    break;

  case UValign::BOTTOM:
    vd.chr.height = hh;
    // ajouter espace restant a vd.y
    // reste centre (et clipe) si trop grand (peut deborder!)
    vd.chr.y = vd.y + (vd.height-hh);
    break;
  }

  // flexible horizontal object  ==>  add flexible width space
  // !!==> sauf pour les Elems qui sont jamais flexibles
  if (chview && curp.local.halign == UHalign::FLEX) {
    vd.chr.width = ww + vd.hflex_space;
  }
  else vd.chr.width = ww;
}

/* ==================================================== ======== ======= */

static void layoutElemV(UViewUpdateImpl& vd, const UContext& curp,
			UGroup *grp, ULink* link,
			u_dim ww, u_dim hh, UView*chview) {
  switch (curp.local.halign) {
  case UHalign::LEFT:
    vd.chr.width = ww;
    vd.chr.x = vd.x;
    break;
    
  case UHalign::FLEX:  
  case UHalign::CENTER:
    // !NOTE: pour les Elems, HFLEX est identique a HCENTER
    
    // Cas: BOX + FLEX
    if (chview && curp.local.halign == UHalign::FLEX) {
      // - tous les enfants sont justifies horizontalement
      // sur celui qui a la plus grande largeur
      // - selon les cas (voir ligne precedente) ils sont ou non
      // egalement adaptes par rapport a laur parent
      //NB: ! NE PAS changer chw sinon le scroll va deconner
      vd.chr.width = vd.width;
      vd.chr.x = vd.x;
    }
    
    // Cas CENTER ou Elem{FLEXouCENTER})
    else {
      vd.chr.width = ww; 
      // ajouter espace restant a vd.x
      // reste centre (et clipe) si trop grand (peut deborder!)
      vd.chr.x = vd.x + (vd.width-ww)/2;
    }
    break;
    
  case UHalign::RIGHT:
    vd.chr.width = ww;
    // ajouter espace restant a vd.x
    // reste centre (et clipe) si trop grand (peut deborder!)
    vd.chr.x = vd.x + (vd.width-ww);
    break; 
  }
  
  // flexible vertical object  ==>  add flexible height space
  // !!==> sauf pour les Elems qui sont jamais flexibles
  if (chview && curp.local.valign == UValign::FLEX) {
    //vd.chr.height = U_MIN(vd.flex_space, hh + vd.flex_space);
    vd.chr.height = hh + vd.vflex_space;
  }
  else vd.chr.height = hh;
}

/* ==================================================== ======== ======= */

static void layoutElemViewport(UViewUpdateImpl& vd, const UContext& curp,
			       UGroup *grp, ULink* link,
			       u_dim ww, u_dim hh, UView* chview) {
  if (!chview) {
    UError::error("warning@UView::doUpdate",
                  "a UPane can only contain UBox(es) in its central area");
    return;
  }

  UPaneView* paneview = dynamic_cast<UPaneView*>(vd.view);
  UPane* pane = dynamic_cast<UPane*>(grp);
  if (!paneview || !pane) {
    UError::error("internal@layoutElemViewport", 
		  "null UPane or null PaneView");
    return;
  }
  vd.chr.x = vd.x - paneview->xscroll;
  vd.chr.y = vd.y - paneview->yscroll;

  // new: 9jn03: on prend le max de tous les children
  /* deconne: ne diminue jamais
  u_dim chw = ww > vd.width ? ww : vd.width;
  vd.chr.width  = chw > vd.chr.width ? chw : vd.chr.width;
  u_dim chh = hh > vd.height ? hh : vd.height;
  vd.chr.height = chh > vd.chr.height ? chh : vd.chr.height;
  */

  // si scrollbar, child prend sa taille naturelle et ce scrollbar
  // permettra de faire defiler child. sinon child doit s'adapter
  // a la taille du pane comme si c'etait une box. de plus dans ce
  // denier cas, ca permet de superposer plusieurs children qui
  // seront superpose et de meme taille comme dans un cardbox
  
  if (pane->getHScrollbar()) {
    u_dim chw = ww > vd.width ? ww : vd.width;
    vd.chr.width  = chw > vd.chr.width ? chw : vd.chr.width;
  }
  else {
    // deconne dans le cas ou il y a un scroll: child a la meme
    // taille que pane, mais comme il est decale il n'atteint
    // pas le bord droit du pane
    // vd.chr.width = vd.width;
    vd.chr.width = vd.width + paneview->xscroll;
  }

  if (pane->getVScrollbar()) {
    u_dim chh = hh > vd.height ? hh : vd.height;
    vd.chr.height = chh > vd.chr.height ? chh : vd.chr.height;
  }
  else {
    // meme remarque
    //vd.chr.height = vd.height;
    vd.chr.height = vd.height + paneview->yscroll;
  }
}

/* ==================================================== ======== ======= */

static void layoutElemBorder(UViewUpdateImpl& vd, const UContext& curp,
			     UGroup *grp, ULink* link,
			     u_dim ww, u_dim hh, UView*) {
  if (!vd.view->getBorderMargins()) {
    UError::error("internal@layoutElemBorder","null frame view");
    return;
  }
  UMargins& frame = *vd.view->getBorderMargins();

  switch (curp.local.valign) {
  case UValign::TOP:
    //vd.chr.y = r.y + margin.top;
    vd.chr.y = vd.y;
    vd.chr.height = frame.top;
    break;
	    
  case UValign::BOTTOM:
    //vd.chr.y = r.y + r.height - frame.bottom - margin.bottom;
    vd.chr.y = vd.y + vd.height - frame.bottom;
    vd.chr.height = frame.bottom;
    break;
	    
  case UValign::CENTER: // ne s'adapte PAS a la taille du Pane
  case UValign::FLEX: // Middle adaptatif: s'adapte a la taille du Pane
    //vd.chr.y = r.y + frame.top + margin.top;
    vd.chr.y = vd.y + frame.top;
	    
#if EXX
    // !NOTE: pour les Elems, VFLEX est identique a MIDDLE
    if (!it && curp.local.valign == VFLEX)
      vd.chr.height = r.height - frame.top - frame.bottom - margin.top - margin.bottom;
    else vd.chr.height = chview->h;
#endif

    //dans TOUS les cas (flex ou center/middle, Elem ou Box) :
    // => TOUJOURS justifie
    //vd.chr.height = r.height - frame.top - frame.bottom 
    //  - margin.top - margin.bottom;
    vd.chr.height = vd.height - frame.top - frame.bottom;
    break;
  }
	  
  switch(curp.local.halign) {
  case UHalign::LEFT:
    //vd.chr.x = r.x + margin.left;
    vd.chr.x = vd.x;
    vd.chr.width = frame.left;
    break;
    
  case UHalign::RIGHT:
    //vd.chr.x = r.x + r.width - frame.right - margin.right;
    vd.chr.x = vd.x + vd.width - frame.right;
    vd.chr.width = frame.right;
    break;
	    
  case UHalign::CENTER: // ne s'adapte PAS a la taille du Pane
  case UHalign::FLEX: // Center adaptatif: s'adapte a la taille du Pane
    //vd.chr.x = r.x + frame.left + margin.left;
    vd.chr.x = vd.x + frame.left;

#if EXX
    // !NOTE: pour les Elems, FLEX est identique a CENTER
    if (!it && curp.local.halign == HFLEX)
      chr.width = r.width - frame.left - frame.right - margin.left - margin.right;	 
    else chr.width = chview->w;
#endif

    //dans TOUS les cas (flex ou center/middle, Elem ou Box) :
    // => TOUJOURS justifie
    //vd.chr.width = r.width - frame.left - frame.right
    //  - margin.left - margin.right;	 
    vd.chr.width = vd.width - frame.left - frame.right;
    break;
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */

static void nextElemH(UViewUpdateImpl& vd, const UContext& curp) {
  // increment vd.chr.x in all cases
  vd.chr.x += vd.chr.width + curp.local.hspacing;
}

/* ==================================================== ======== ======= */

static void nextElemV(UViewUpdateImpl& vd, const UContext& curp) {
  // increment vd.chr.y in all cases
  vd.chr.y += vd.chr.height + curp.local.vspacing;
}

/* ==================================================== ======== ======= */

static void nextElemViewport(UViewUpdateImpl& vd, const UContext& curp) {
  //NB: les children s'affiche les uns sur les autres (ce qui a un sens
  // s'ils sont transparents ou si on veut faire un cardbox
}

/* ==================================================== ======== ======= */

static void nextElemBorder(UViewUpdateImpl& vd, const UContext& curp) {
  // nop
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NOTE: Cette fct accumule les props heritees via le graphe d'instanciation
// dans le ucontext: le contect complete' est renvoyee bia e->props
// ATTENTION: 
// -1 e->props ne prend pas en compte les props de la trouvee UBox
//       (ie. l'heritage s'arrete au PARENT de cette UBox)
// -2- cette fct n'a de sens et ne doit etre utilisee QUE SI l'objet 
//     est VISIBLE


UView* UView::contains(u_pos xmouse, u_pos ymouse, URegion &clip) { //mofifies clip
  // Notes:
  // 1)- cas View rectangulaire
  // 2)- clip est l'intersection de this et du clip initial
  //     ==> necessaire pour calculer le bon redrawClip qd les objets 
  //         sont scrolles ou clippes 

  /* obsolete
     UBox *box = getBox();
     if (box->isDef(0, UMode::SUB_HARDWIN)) {
    // cas part. des subwindow (UIncrust) ne pas prendre en compte view->x,y
    // car les UEvent sont relatifs a cette subwindow
    if (clip.setInter(0,0,width,height) == 0) return null;
  }
  else {
    if (clip.setInter(*this) == 0) return null;
  }
  */

  if (clip.setInter(*this) == 0) return null;
  if (xmouse < clip.x 
      || ymouse < clip.y 
      || xmouse >= (clip.x + clip.width) 
      || ymouse >= (clip.y + clip.height))
    return null;

  return this;
}

UView* UView::contains(UView *searchv, URegion &clip) { //modifies clip
  // Notes:
  // 1)- cas View rectangulaire
  // 2)- clip est l'intersection de this et du clip initial
  //     ==> necessaire pour calculer le bon redrawClip qd les objets 
  //         sont scrolles ou clippes 

  /* obsolete
     UBox *box = getBox();
  if (box->isDef(0, UMode::SUB_HARDWIN)) {
    // cas part. des subwindow (UIncrust) ne pas prendre en compte view->x,y
    // car les UEvent sont relatifs a cette subwindow
    if (clip.setInter(0,0,width,height) == 0) return null;//suprime
  }
  else {  
    if (clip.setInter(*this) == 0) return null;
  }
  */
    if (clip.setInter(*this) == 0) return null;
  if (this != searchv)  return null;
  else return this;
}

/* ==================================================== ======== ======= */
/****

//layers (depth)
//ugroups et formes qq
//cas du serachview
// FAUT FAIRE UNE RECHERCHE EN LARGEUR D'ABORD !!!

static UView* viewLocateSource(UView *view, UEvent *e, UContext *parp, 
			       UGroup *grp, const URegion clip);
// this function sets:
// -- e->{source, sourceView, sourceProps, redrawnClip}

UView* UView::locateSource(UEvent *e, UContext *parp, URegion clip) {
  UBox *box = getBox();
  if (! box->isShowable() 
      || box->isDef(UMode::IGNORE_EVENTS,0)
      || box == e->through 
      )
    return null;

  if (!contains(e->xmouse, e->ymouse, clip)) return null;

  // mouse coords must be defined in order to determine which objects
  // are traversed by events
  // ==> ceci implique que seules les UBox peuvent flagger les events
  //     (car les UGroup n'ont pas de coords)
  if (box->isDef(UMode::FLAG_EVENTS,0)) {
    e->sourceView = this;
    box->fire(e, UOn::traverse);
  }

  UView *subview = viewLocateSource(this, e, parp, box, clip);

  if (subview) return subview;
  //sinon c'est bien dans cette view et aucune subview que se trouve mouse
  else {
    e->setSource(this);
    e->redrawStatus = true;
    e->redrawClip = clip;

    // e->firstTranspView = the first view that is transparent in the tree
    // !att: curp.transpView ET NON parp.transpView
    if (isDef(TRANSP_VIEW)) {
      if (!parp || !parp->firstTranspView) e->firstTranspView = this;
      else e->firstTranspView = parp->firstTranspView; 
     }
    // si cette vue est opaque il est alors inutile de tenir compte
    // des vues transparentes precedentes ==> ecraser firstTranspView
    else e->firstTranspView = null;

    // !att: en final il faut tenir compte de fixedSizes pour la vue courante
    // et non pour son parent ==> recopie de firstLayoutView dans parp
    // lui-meme recopie dans un champ de l'event 'e'
    // de plus: si FLOATING, inutile de recalculer le layout des parents

    if (parp && (box->isDef(0,UMode::FLOATING)
		 || isAllDef(FIXED_WIDTH|FIXED_HEIGHT)))
      parp->firstLayoutView = this;

    // !ATT: recopie le PARENT context!!
    if (parp) e->dumpParentContext(*parp);
    return this;
  }
}

// ==================================================== ======== =======

static UView* viewLocateSource(UView *view, UEvent *e, UContext *parp, 
			       UGroup *grp, const URegion clip) {
  UContext curp(grp, view, *parp);
  //mlist.parseProps(grp, curp);
  //mlist.addList(grp->getChildLinks());
  //if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  UMultiList mlist(curp, grp);
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  // on continue gentiment de scanner la liste jusqu'au bout car
  // il peut y avoir une autre 'found_view' superposee
  // et c'est cette derniere qu'on veut recuperer

  UView *subview = null;
  for (ULink* ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, grp)) {
      UBrick *b = ch->getChild();
      UProp  *prop;
      UGroup *chgrp;

      if ((prop = b->propCast()))
	prop->putProp(&curp,  grp);
      
      else if ((chgrp = b->groupCast())
	       && chgrp->isShowable() 
	       && (!chgrp->isDef(UMode::IGNORE_EVENTS,0))
	       && chgrp != e->through 
	       ) {
	UView  *chview;

	if (chgrp->isDef(0,UMode::GROUP)
	    && (chview = viewLocateSource(view, e, &curp, chgrp, clip))){
	  subview = chview;
	}
	else if ((chgrp->isDef(0,UMode::BOX)      //elimine les UWin
		  // this==hardwin && child==softwin :
		  || (mlist.isSoftwinList() && chgrp->isDef(0,UMode::SOFTWIN))
		  )
		 // ch est un UBoxLink par construction
		 && (chview = ((UBoxLink*)ch)->getView(view))){
	  URegion chclip(clip);
	  if (chview->contains(e->xmouse, e->ymouse, chclip))
	    subview = chview;
	}
      }
    } //endfor

  if (subview) {
    UView* subsubview = subview->locateSource(e, &curp, clip);
    if (subsubview) subview = subsubview;
  }
  return subview;
}

// ==================================================== ======== =======

static UView* viewLocateSource(UEvent *e, UView *view, UContext *parp, 
			       UGroup *grp, const URegion clip, UView *sv);

UView* UView::locateSource(UEvent *e, UContext *parp, URegion clip, UView *sv) {
  UBox *box = getBox();
  if (!contains(sv, clip)) {
    UView *subview = viewLocateSource(e, this, parp, box, clip, sv);
    //cf cas des group pas initailises
    if (subview) return subview;
    else return null;
  }
  //sinon c'est bien cette view que l'on cherche
  else {
    e->setSource(this);
    e->redrawStatus = true;
    e->redrawClip = clip;

    // e->firstTranspView = the first view that is transparent in the tree
    // !att: curp.transpView ET NON parp.transpView
    if (isDef(TRANSP_VIEW)) {
      if (!parp || !parp->firstTranspView) e->firstTranspView = this;
      else e->firstTranspView = parp->firstTranspView; 
     }
    // si cette vue est opaque il est alors inutile de tenir compte
    // des vues transparentes precedentes ==> ecraser firstTranspView
    else e->firstTranspView = null;

    // !att: en final il faut tenir compte de fixedSizes pour la vue courante
    // et non pour son parent ==> recopie de firstLayoutView dans parp
    // lui-meme recopie dans un champ de l'event 'e'
    // de plus: si FLOATING, inutile de recalculer le layout des parents

    if (parp && (box->isDef(0,UMode::FLOATING) 
		 || isAllDef(FIXED_WIDTH|FIXED_HEIGHT)))
      parp->firstLayoutView = this;

    // !ATT: recopie le PARENT context!!
    if (parp) e->dumpParentContext(*parp);
    return this;
  }
}

// ==================================================== ======== ======= 

static UView* viewLocateSource(UEvent *e, UView *view, UContext *parp, 
			       UGroup *grp, const URegion clip, UView *sv) {
  UContext curp(grp, view, *parp);
  //mlist.parseProps(grp, curp);
  //mlist.addList(grp->getChildLinks());

  UMultiList mlist(curp, grp);
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  UView *subview = null;
  for (ULink* ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, grp)) {
      UBrick *b = ch->getChild();
      UProp  *prop;
      UGroup *chgrp;

      if ((prop = b->propCast()))
	prop->putProp(&curp,  grp);
      
      else if ((chgrp = b->groupCast())) {
	UView *chview = null;

	if (chgrp->isDef(0,UMode::GROUP) && chgrp->isShowable()) {
	  if (view == sv) return view;
	  else if ((subview = viewLocateSource(e, view, &curp, chgrp, clip, sv)))
	    return subview;
	}
	else if (chgrp->isDef(0,UMode::BOX)  //elimine les UWin
		 // ch est un UBoxLink par construction
		 && (chview = ((UBoxLink*)ch)->getView(view))){
	  if ((subview = chview->locateSource(e, &curp, clip, sv))) 
	    return subview;
	}
      }
    } //endfor

  return null;
}
****/

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */

// NB: clip recopie a cause du Border (on veut recuperer le clip le l'objet
// entier meme si on a clique a l'interieur du border)

UView* UView::locateSource2(UGroup* grp, const UContext& parp, URegion clip,
			    UEvent& e, const UView* searchedView,
			    USourceProps& vl)
{
  UContext curp(grp, this, parp);
  UMultiList mlist(curp, grp, false); //no rescale
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  UGroup *chgrp = null;
  UView  *chview = null, *found_view = null;

  if (grp->boxCast()) {   // pas de sens pour UGroups et border

    vl.set(this, grp, curp);

    if (getBorderMargins() && curp.local.border
        && (chgrp = curp.local.border->getSubGroup())
	) {
      
      found_view = locateSource2(chgrp, curp, clip, e, searchedView, vl);
      
      // soit l'objet est dans le border et on le retourne
      // soit il est (eventuellement) dans la zone interieure et dans ce
      // cas il faut modifier le clip :
      // a) pour limiter la zone de recherche et d'affichage (le redrawClip)
      //    pour des raison d'optimisation evidentes
      // b) pour que xle redrawClip corresponde exactement a la zone interne
      //    ce qui est essentiel pour le scrolling dans les viewports
      //    car on optimise l'affichage par des copyArea et il ne faut 
      //    surtout pas copier des pixels en dehors de la zone interne 
      
      if (found_view) return found_view;
      //cf TRANSPARENT_SCROLL_MODE
      else {
	UMargins margins(0, 0, 0, 0);      // voir setMargins
	
	curp.local.border->getSize(curp, margins);
	margins.incr(*getBorderMargins());

	clip.set(clip.x + margins.left,
		 clip.y + margins.top,
		 clip.width - margins.left - margins.right,
		 clip.height - margins.top - margins.bottom);
      }
      //endcf
    }
  }


  UView *last_found_view = null;
  for (ULink* ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, grp)) {
      UBrick *b = ch->getChild();
      UProp  *prop;
      found_view = null;
      chgrp = null;

      if ((prop = b->propCast()))
	prop->putProp(&curp,  grp);
      
      else if ((chgrp = b->groupCast())
	       && chgrp->isDef(0,UMode::GROUP)
	       && chgrp->isShowable() 
	       && (!chgrp->isDef(UMode::IGNORE_EVENTS,0))
	       // && ( (curp.local.alpha < 1.)
	       //    ? v->setVmodes(UView::TRANSP_VIEW, true), true : true )
	       && (found_view = locateSource2(chgrp, curp, clip, e, 
					      searchedView, vl)) 
	       ) {
	last_found_view = found_view;
      }

      else if (chgrp
	       && (chgrp->isDef(0,UMode::BOX) //elimine les UWin
		   // this==hardwin && child==softwin :
		   || (mlist.isSoftwinList() && chgrp->isDef(0,UMode::SOFTWIN))
		   )
	       // ch est donc un UBoxLink par construction
	       && (chview = ((UBoxLink*)ch)->getViewInside(this))
	       && (found_view = chview->locateSource(curp, clip, e, 
						     searchedView, vl))
	       ) {
	// on continue gentiment de scanner la liste jusqu'au bout car
	// il peut y avoir une autre 'found_view' superposee
	// et c'est cette derniere qu'on veut recuperer
	last_found_view = found_view;
      }

    } //endfor

  return last_found_view;
}

/* ==================================================== ======== ======= */
// NB: UViewLocate recopie: layoutView et transpView sont propres
// a un niveau de profondeur d'appel recursif de locateSource

UView* UView::locateSource(const UContext& parp, URegion clip, UEvent& e, 
			   const UView* searchedView, USourceProps vl) {
  UBox *box = getBox();
  if (!box) {
    UError::error("internal@UView::locateSource","View points to null box");
    return null;
  }

  if (! box->isShowable() 
      || box->isBmode(UMode::IGNORE_EVENTS) // B mode !!
      //|| box == e.through 
      )
    return null;

  if (e.resendEvent && e.aux == box) return null;

  // NB: setInter() renvoie false si l'arg est une region de w ou h nulle
  // ca peut poser pbm si la vue est de taille (0,0) : on 'arrivera pas
  // a la localiser par cette fct
  if (clip.setInter(*this) == 0) return null;

  if (searchedView) {
    if (searchedView == this) {

      UContext curp(box, this, parp);
      UMultiList mlist(curp, box, false);   // necessaire pour parser curp
      mlist.first();		// sert juste a eviter un warning a la compil
      vl.set(this, box, curp);
      goto TROUVE;
    }
    // sinon chercher dans la liste des enfants via la boucle for() ci-apres
  }

  else {
    // Notes:
    // 1)- cas View rectangulaire
    // 2)- clip est l'intersection de this et du clip initial
    //     ==> necessaire pour calculer le bon redrawClip qd les objets 
    //         sont scrolles ou clippes 
    
    if (e.xmouse < clip.x 
	|| e.ymouse < clip.y 
	|| e.xmouse >= (clip.x + clip.width) 
	|| e.ymouse >= (clip.y + clip.height))
      return null;

    // NB: event flagging is meaningless in 'searchedView' mode:
    // mouse coords must be defined in order to determine which objects
    // are traversed by events
    // ==> ceci implique que seules les UBox peuvent flagger les events
    //     (car les UGroup n'ont pas de coords)
    if (box->isDef(UMode::PRE_CHILD_EVENT_CB,0)) {
      e.setSource(this);
      e.preChildEventNotify = true;
      box->fire(e, UOn::preChildEvent);
    }
  }

  { // la { est impose par syntaxe du goto
    UView *found_view = locateSource2(box, parp, clip, e, searchedView, vl);
    if (found_view) return found_view;
  }

  // sinon on n'a pas trouve de vue incluse et il faudra donc retourner
  // la vue courante. Pour ce faire on continue en sequence jusqu'au
  // tag 'TROUVE'

  // si uc.mode searchedView et qu'on n'est PAS alle' directement au
  // tag 'TROUVE' c'est qu'on a pas trouve le bon objet
  if (searchedView) return null;

 TROUVE:
  // si l'objet n'a pas d'enfant ou si la souris n'est contenue dans aucun 
  // de ses enfants alors retourner l'objet en personne. (apres avoir
  // copie' toutes les infos desirees)

  // copier la source trouvee et les parametres dans ViewLocate
  e.setSource(this);

  vl.redrawStatus = true;
  vl.redrawClip = clip;

  // recopier le PARENT context si ce champ est non nul
  if (vl.parentContext) *(vl.parentContext) = parp;
  e.sp = vl;
  return this;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static void saveElemProps(UContext &props, 
			 ULink *link, UWinGraph &g, 
			 const URegion &r, UViewUpdate &upmode,
			 bool exact_match) {
  int strpos = -1;
  UElem *elem = link->getChild()->elemCast();
  UStr *str;
  if (elem && (str = elem->strCast())) {
    // search the strpos
    strpos = g.getCharPos(props.fontdesc, str->chars(), str->length(),
			  upmode.e->getXwin() - r.x);
  }
  upmode.elem_props->set(props, elem, link, r, strpos, exact_match);
}

// Horizontal Search. 
// retrieves elem and info from x,y position
// !!ATT: 
// - suppose que elem_props!= null !
// - return==true signifie: ne pas chercher davantage car:
//                trouve' OU pas trouve' mais plus la peine de chercher
// - return==false signifie: continuer a chercher car:
//                pas trouve' mais on peut encore trouver 

bool UView::locateElemH(UContext &props, ULink *link, UWinGraph &g, 
			const URegion &r, UViewUpdate &upmode) {

  if (r.x > upmode.e->xmouse) return true;  // plus rien a chercher (not found)

  if (r.y <= upmode.e->ymouse && upmode.e->ymouse < r.y + r.height) {
    if (upmode.e->xmouse < r.x + r.width) {
      // elem exactly found
      saveElemProps(props, link, g, r, upmode, true);
      return true;  // trouve' => finir la recherche
    }
    // elem approximatively found
    else saveElemProps(props, link, g, r, upmode, false);
  }

  return false;	// continuer a chercher (dans les 2cas)
}

bool UView::locateElemV(UContext &props, ULink *link, UWinGraph &g, 
			const URegion &r, UViewUpdate &upmode) {

  if (r.y > upmode.e->ymouse) return true;  // plus rien a chercher (not found)

  if (r.x <= upmode.e->xmouse && upmode.e->xmouse < r.x + r.width) {
    if (upmode.e->ymouse < r.y + r.height) {
      saveElemProps(props, link, g, r, upmode, true);
      return true;
    }
    else saveElemProps(props, link, g, r, upmode, false);
    
  }
  return false;	// continuer a chercher (dans les 2cas)
}

bool UView::locateElemRef(UContext &props, ULink *link, UWinGraph &g, 
			  const URegion &r, UViewUpdate &upmode) {
  UElem *elem;

  // retrieves info from Elem link or Elem ptr
  if ((upmode.elem_props->elemLink == link 
       || upmode.elem_props->elem == link->getChild()
       )
      && ((elem = link->getChild()->elemCast()))
      ) {
    upmode.elem_props->set(props, elem, link, r, upmode.elem_props->strpos, true);
    return true;    //trouve, inutile de continuer a chercher
  }

  //tous les cas pas trouve (PAS de else!)
  return false;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UView::updateWinData(const URegion &region) {
  UView *winview = wg().getHardwin()->getWinView(getDisp());
  if (!winview) return -1;

  UViewUpdate upmode(UViewUpdate::UPDATE_DATA);
  UWinContext winctx(winview);

  setVmodes(UView::DAMAGED, true);
  winview->doUpdate(winctx, *winview, region, upmode);
  //inutile: setVmodes(UView::DAMAGED, false);

  //return vup.overlapp;
  return upmode.above_damaged_count;
}

/* ==================================================== ======== ======= */
// update view paint in window coordinates:
// ATT: CHANGEMENT:  updateAllLayers = false par DEFAUT !!!

void UView::updateWinPaint(const URegion &region, UView *opaqueView,
			   bool doublebuff, bool all_layers) {
  UWinGraph &g = wg();
  UView *winview = g.getHardwin()->getWinView(getDisp());
  if (!winview) return;

  UWinContext winctx(winview);
  UViewUpdate upmode(UViewUpdate::PAINT_ALL);

#ifdef WITH_GL   // all layers  // [MODE OGL_DOUBLEBUF]
  if (g.getDisp()->getConf().double_buffering) {
    // tout reafficher a cause du glXSwapBuffers appele par end()
    g.begin(*winview);
    winview->doUpdate(winctx, *winview, *winview, upmode);
    g.end();
    return;  //ATT: on sort de la fct!
  }
#endif  // ATT: continue en sequence si !double_buffering

  /* NOTE:
   * le mode PAINT_DAMAGED est incompatible avec le DoubleBuffering
   * puisque justement on ne reaffiche pas tout (ce qui generera
   * un fond non initialise donc blanc dans certains cas)
   * ==> imposer PAINT_ALL
   * (une autre solution coherente consisterait d'abord a recopier
   * le fond existant puis a ne generer que les parties a modifier
   * grace a PAINT_DAMAGED)
   */
  if (doublebuff) {
    g.beginDoublebuf(region);
    winview->doUpdate(winctx, *winview, region, upmode);
    g.end();
  }
  //else if (mode == 2) {      //update all CHANGED layers ???
  //  vup.setMode(UViewUpdate::PAINT_CHANGED);
  //  g.begin2(region);
  //  winview->doUpdate(winctx, winview, region, vup);
  //  g.end2();
  //}
  else if (all_layers) {             //update all layers
    g.begin(region);
    winview->doUpdate(winctx, *winview, region, upmode);
    g.end();
  }
  else {
    // s'il y a un transp egalement afficher a partir du PARENT du transp
    //if (firstTranspView && firstTranspView->getParentView()) 
    //  firstTranspView->getParentView()->setVmodes(UView::DAMAGED, true);

    if (!opaqueView) opaqueView = winview;  // the hardwin!
    opaqueView->setVmodes(UView::DAMAGED, true);

    setVmodes(UView::DAMAGED, true);
    upmode.setMode(UViewUpdate::PAINT_DAMAGED); // !!

    g.begin(region);
    winview->doUpdate(winctx, *winview, region, upmode);
    g.end();
  }
}

/* ==================================================== ======== ======= */
// update view paint in view coordinates:

void UView::updatePaint(URegion *region, bool doublebuff) {
  if (!region) region = this;
  else {
    region->x += x;
    region->y += y;
  }
  updateWinPaint(*region, null, doublebuff, true);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
