/* ==================================================== ======== ======= *
 *
 *  umsclient.hpp : UMS [Ubit Mouse Server] client
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _umsclient_hpp_
#define	_umsclient_hpp_
//pragma ident	"@(#)umsclient.hpp	ubit:03.06.04"
#include <ubit/udefs.hpp>
#include <ubit/ubrick.hpp>
#include <ubit/ustr.hpp>


class UMSclient {
public:
  enum Stat {
    HostNotFound  = -2,
    CannotOpen    = -1,
    NotOpened     =  0,
    Opened        =  1
  };

  UMSclient(UAppli&);
  UMSclient(const UStr& client_name) ;
  ~UMSclient();

  int open(const UStr& umsd_hostname, int umsd_port = 0);
  /**<
   * opens connection to the UMS (Ubit Multiple Mouse/Message Server).
   * This makes it possible to control the mouse pointer(s) and to send
   * events and messages to X applications on the remote display where
   * the UMS is running.
   *  Args:
   *  - ums_host: the host where the UMS is running; local host if null
   *  - ums_port: the port used by the UMS; uses UMS_DEFAULT_PORT if 0
   *  - returns connection status (see: enum UMSclient::Stat).
   *  Note: 
   *  - the UMS must already be running (ie. the 'umsd' program)
   */

  void close();

  int getStatus() const;
  /**< returns connection status (see: enum UMSclient::Stat).
   *  value is > 0 if connected to an UMS server
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  bool sendMessage(const char* target, const UStr& message);
  bool sendMessage(const char* target, const char* message);
  /**<
   * sends a message to target.
   *  event_flow == 0 is the standard X mouse.
   *  event_flow > 0 are alternate mouse event flows (must be managed
   *  by the UMS)
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  bool sendMouseClick(const char* target, u_pos x, u_pos y, u_id mouse_button);
  bool sendMousePress(const char* target, u_pos x, u_pos y, u_id mouse_button);
  bool sendMouseRelease(const char* target, u_pos x, u_pos y, u_id mouse_button);
  /**<
   * sends pseudo events to target.
   * args:
   *  - x and y are relative to the target
   * - 'mouse_button' is one of UEvent::MButton1, MButton2...
   *   (only one mouse button at a time: should not be ORed !)
   * notes:
   * - press and release events must be balanced.
   * - these fct. does NOT changes the state of the mouse event flow
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  bool moveMouse(int mouse_event_flow, u_pos x, u_pos y, bool absolute_coords);
  /**<
   *  moves the mouse controlled by this event flow.
   * args:
   *  - event_flow == 0 is the standard X mouse.
   *  - event_flow > 0 are alternate  mouse event flows (must be managed
   *    by the UMS)
   *  - x and y are screen coordinates if 'absolute_coords' is true
   *    and relative to the previous location of the mouse otherwise. 
   */

  bool pressMouse(int mouse_event_flow, u_id mouse_button);
  bool releaseMouse(int mouse_event_flow, u_id mouse_button);
  /**<
   * presses or releases the mouse controlled by this event flow.
   * args:
   * - mouse_button is one of UEvent::MButton1, MButton2 etc.
   *   (only one mouse button at a time: should not be ORed !)
   * notes:
   * - press and release events must be balanced.
   * - these fct. changes the state of the mouse event flow (excepts for
   *   flow 0, the standard X mouse because there is no mean to do it).
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  bool winOpened(u_id win_id);
  bool winClosed(u_id win_id);
  bool cnxOpened(const UStr& cnx_name);
  bool cnxClosed();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

private:
  class UAppli* appli;
  struct sockaddr_in* rhost;
  uptr<UInput> input;
  UStr client_name;
  UStr remote_host;
  int  remote_port;
  int  com_sock;
  int  stat;

  void constructs(UAppli*, const UStr& client_name);
  int  openCom(const UStr& cnx_name, bool keep_cnx_request);
  bool sendRequest(class UMSrequest& request);
  void receive();
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
