/* ==================================================== ======== ======= *
 *
 *  udir.hpp
 *  Ubit Project  [Elc][2003]
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _udir_hpp_
#define	_udir_hpp_
#include <sys/stat.h>

class UDir {
public:

  struct Entry {
    Entry(const class UDir& dir, const char* dname);

    bool isDir()  const;
    bool isFile() const;
    bool isLink() const;
    bool isExec() const;

    UStr* getName() const {return name;}
    ///< this function returns the name of this entry (this object is automatically freed, don't 'delete' it).

    UIma* getSmallIcon() const;
    UIma* getLargeIcon() const;
    ///< this function returns a shared Icon (this object is automatically freed, don't 'delete' it).

    unsigned long getSize() const {return size;}
    mode_t getMode() const {return mode;}
    time_t getTime() const {return mtime;}

    UStr* createSizeStr() const;
    UStr* createModeStr() const;
    UStr* createTimeStr() const;
    ///< these functions create a new UStr containing the corresponding info (use 'delete' to free memory)

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    uptr<UStr> name;
    unsigned long size;
    mode_t mode;
    time_t mtime;
  };
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UDir(const std::string pathname);
  UDir(const UStr& pathname);
  UDir(const UStr& pathname, const UStr& prefix, 
       const UStr& filter, bool get_hidden_files);
  /**< opens this directory and gets file entries.
   * Args:
   * - prefix: only get files thats start witrh this prefix
   * - filter: only get files with this filter.
   *   filter syntax: "C++ Files (*.cpp; *.hpp)" or "*.cc; *.hh" or "*.*" 
   */
				
  virtual ~UDir();

  const UStr& getPath() const {return path;}
  /**< returns the full pathname of the directory
   * note:
   *	UDir normalizes the pathname: getPath() may return a value that 
   * differs from the argument given to the constructor
   */

  std::vector<UDir::Entry*> getFileEntries() const {return entries;}
  UDir::Entry* getFileEntry(int k) const {return entries[k];}

  /**< returns the (filtered) file entries of the directory.
   * Notes:
   * - depending on how the UDir constructor was called, some file entries
   *   may have been filtered (and dismissed)
   * - all entries are destroyed then the UDir is deleted. However, data
   *   obtained by calling Entry::getName(), Entry::get*Icon() etc. remain 
   *   valid if pointed by an uptr<>.
   */
  
  unsigned int getFileCount() const {return entries.size();}
  ///< returns the entry count.
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //	Impl.
  
protected:
  friend class UDir::Entry;
		
  UStr path;
  struct stat& statbuf;
  bool want_attributes;
  std::vector<UDir::Entry*> entries;
  std::vector<UStr*> filters;
		
  virtual void constructs(const UStr& pathname, const UStr& prefix, 
                          const UStr& filter, bool get_hidden_files);
public:
  static void parseDir(UStr& pathname);
  static void parseFilter(std::vector<UStr*>& filter_list,
			  const UStr& filter_str);
  static bool compareEntries(const UDir::Entry*, const UDir::Entry*);
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif
