/* ==================================================== ======== ======= *
 *
 *  uuscroll.cc
 *  Ubit Project [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuscroll.cc	ubit:b1.11.7"
#include <ubrick.hh>
#include <ucolor.hh>
#include <usymbol.hh>
#include <uborder.hh>
#include <uevent.hh>
#include <ucall.hh>
#include <ubox.hh>
#include <uscroll.hh>
#include <ustyle.hh>
#include <ugraph.hh>
#include <uview.hh>
#include <uviewImpl.hh>
#include <ugadgets.hh>
#include <upix.hh>
#include <ucall.hh>
#include <uobs.hh>


/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// UScrollbar and inner classes
/* ==================================================== ======== ======= */
// Slider

const UStyle* UScrollbar::Slider::getStyle(const UBox *parent) {
  // !!!the parent should always be a Scrollbar
  UScrollbar *scrollbar = (UScrollbar*)parent;
  if (!scrollbar) return null;
  else return scrollbar->getScrollbarStyle(null)->slider;
}

UScrollbar::Slider::Slider(UArgs a) : UButton(a) {
  // ne pas allumer le slider qunad on entre dedans et quand on 
  // appuie dessus car ca provoque des effets desagerables
  /*
  setCmodes(UMode::ENTER_HIGHLIGHT 
	    | UMode::ENTER_HIGHBORDER
	    | UMode::CAN_ARM
	    //| UMode::CAN_CLOSE_MENU,
	    ,false);
  */
  setCmodes(UMode::HAS_AUTOCLOSE_BHV, true);
  setCmodes(UMode::AUTOCLOSE_BHV, false);
}

/* ==================================================== ======== ======= */
// LessScroller

const UStyle* UScrollbar::LessScroller::getStyle(const UBox *parent) {
  // !!!the parent should always be a Scrollbar
  UScrollbar *scrollbar = (UScrollbar*)parent;
  if (!scrollbar) return null;
  else return scrollbar->getScrollbarStyle(null)->lessScroller;
}

UScrollbar::LessScroller::LessScroller(UArgs a) : UButton(a) {
  //setCmodes(UMode::CAN_CLOSE_MENU, false);
}

/* ==================================================== ======== ======= */
// MoreScroller

const UStyle* UScrollbar::MoreScroller::getStyle(const UBox *parent) {
  // !!!the parent should always be a Scrollbar
  UScrollbar *scrollbar = (UScrollbar*)parent;
  if (!scrollbar) return null;
  else return scrollbar->getScrollbarStyle(null)->moreScroller;
}

UScrollbar::MoreScroller::MoreScroller(UArgs a) : UButton(a) {
  //setCmodes(UMode::CAN_CLOSE_MENU, false);
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// Scrollbar Container

const UClass UScrollbar::uclass("UScrollbar");

const int UScrollbar::never           = 0;
const int UScrollbar::always          = 1;
const UOrient &UScrollbar::horizontal = UOrient::horizontal;
const UOrient &UScrollbar::vertical   = UOrient::vertical;

UScrollbarStyle *UScrollbar::horizontalStyle;
UScrollbarStyle *UScrollbar::verticalStyle;

UScrollbarStyle::UScrollbarStyle(const UScrollbarStyle *model) {
  scrollbar    = new UStyle(null);
  slider       = new UStyle(null);
  lessScroller = new UStyle(null);
  moreScroller = new UStyle(null);
  if (model) set(model);
}

void UScrollbarStyle::set(const UScrollbarStyle *model) {
  *scrollbar    = *(model->scrollbar);
  *slider       = *(model->slider);
  *lessScroller = *(model->lessScroller);
  *moreScroller = *(model->moreScroller);
}

const UStyle* UScrollbar::getStyle(const UBox *parent) {
  return getScrollbarStyle(parent)->scrollbar;
}


static const char *boule_xpm[] = {
/* columns rows colors chars-per-pixel */
"10 10 8 1",
". c None s None",
"X c None s None",
"O c gray55",
"+ c #acacac",
"@ c #bcbcbc",
"# c gainsboro",
"$ c #ececec",
"% c white",
/* pixels */
"..X###@...",
".@#$%%%#O.",
"X#$%%$$#@.",
"#$%%$$##@O",
"#%%$$##@@O",
"#%$$##@@+O",
"@%$##@@++O",
"X###@@++O.",
".O@@@++O..",
"..XOOOO...",
};
static UPix boule(boule_xpm);


const UScrollbarStyle* UScrollbar::getScrollbarStyle(const UBox *parent) {

  static const UColor **bgcolors = null;
  if (!bgcolors) {
    bgcolors = UStyle::makeColors(&UBgcolor::darkgrey, &UBgcolor::lightgrey); 
  }

  // VERTICAL
  if (orient == vertical.get()) {

    if (!verticalStyle) {
      verticalStyle = new UScrollbarStyle(null);
      UStyle *s;

      // nouveau style du scrollbar
      s = verticalStyle->scrollbar;
      s->viewStyle      = &UScrollbarView::style;
      s->local.orient   = UOrient::vertical.get();
      s->local.halign   = UHalign::flex.get();
      s->local.valign   = UValign::top.get();
      s->local.hspacing = 0;
      s->local.vspacing = 1; //!!
      ///s->local.border   = &UBorder::shadowIn; ancienne version
      ///s->local.border   = &UBorder::etchedOut; classique, vers l'arriere
      s->local.border   = &UBorder::shadowOut; //similaire etchedIn en moins gros

      s->local.padding.set(0,0);
      s->bgcolors       = bgcolors;

      const UStyle *button_style = slider->UButton::getStyle(null);

      // nouveau style du slider
      s = verticalStyle->slider;
      // recopier le Button Style(!!att: les tableaux ne sont pas dupliques)
      s->set(button_style);
      s->local.orient = UOrient::vertical.get();
      s->local.halign = UHalign::center.get();
      s->local.valign = UValign::center.get();
      s->local.vspacing = 0;
      s->local.hspacing = 0;
      s->local.padding.set(2,2); //!!
      s->local.content = &ugroup(boule+boule+boule);
      s->local.border  = &UBorder::shadowIn;
      //alt: s->local.padding.set(0,0);
      //alt: s->bgcolors = UStyle::makeColors(&UBgcolor::none, &UBgcolor::lightgrey); 
      //alt: s->local.border = null;

      // nouveau style du lessScroller
	s = verticalStyle->lessScroller;
      s->set(button_style);
      s->local.orient = UOrient::vertical.get();
      s->local.halign = UHalign::center.get();
      s->local.valign = UValign::center.get();
      s->local.padding.set(1,0);  //petite marge verticale
      s->local.content = &ugroup(USymbol::up);
      s->local.border = &UBorder::shadowIn;
      //alt:s->local.border = &UBorder::etchedIn;

      // nouveau style du moreScroller
      s = verticalStyle->moreScroller;
      s->set(button_style);
      s->local.orient = UOrient::vertical.get();
      s->local.halign = UHalign::center.get();
      s->local.valign = UValign::center.get();
      s->local.padding.set(0,0);
      s->local.content = &ugroup(USymbol::down);
      s->local.border = &UBorder::shadowIn;
      //alt: s->local.border = &UBorder::etchedIn;
    }

    return verticalStyle;
  }

  // HORIZONTAL
  else {

    if (!horizontalStyle) {
      horizontalStyle = new UScrollbarStyle(null);
      UStyle *s;

      // nouveau style du scrollbar
      s = horizontalStyle->scrollbar;
      s->viewStyle      = &UScrollbarView::style;
      s->local.orient   = UOrient::horizontal.get();
      s->local.halign   = UHalign::left.get();
      s->local.valign   = UValign::flex.get();
      s->local.hspacing = 1; //!!
      s->local.vspacing = 0;
      s->local.padding.set(0,0);
      s->bgcolors       = bgcolors;
      s->local.border   = &UBorder::shadowOut; //similaire etchedIn en moins gros

      const UStyle *button_style = slider->UButton::getStyle(null);

      // nouveau style du slider
      s = horizontalStyle->slider;
      // recopier le Button Style(!!att: les tableaux ne sont pas dupliques)
      s->set(button_style);
      s->local.orient = UOrient::horizontal.get();
      s->local.halign = UHalign::flex.get();
      s->local.valign = UValign::flex.get();
      //s->local.border = null;
      s->local.vspacing = 0;
      s->local.hspacing = 0;
      s->local.padding.set(0,0);
      s->local.padding.set(2,2);
      s->local.content = &ugroup(boule+boule+boule);
	/*&ugroup(
	 UOn::mpress / ugroup(USymbol::left+ USymbol::left+ USymbol::left)
	 + UOn::mdrag / ugroup(USymbol::left+ USymbol::left+ USymbol::left)
	 + UOn::idle / ugroup(boule+boule+boule)
	 );
	*/
      s->local.border = &UBorder::shadowIn;

      // nouveau style du lessScroller
      s = horizontalStyle->lessScroller;
      s->set(button_style);
      s->local.orient = UOrient::horizontal.get();
      s->local.halign = UHalign::center.get();
      s->local.valign = UValign::center.get();
      s->local.padding.set(0,0);
      s->local.content = &ugroup(USymbol::left);
      s->local.border = &UBorder::shadowIn;

      // nouveau style du moreScroller
      s = horizontalStyle->moreScroller;
      s->set(button_style);
      s->local.orient = UOrient::horizontal.get();
      s->local.halign = UHalign::center.get();
      s->local.valign = UValign::center.get();
      s->local.padding.set(0,0);
      s->local.content = &ugroup(USymbol::right);
      s->local.border = &UBorder::shadowIn;
    }
    return horizontalStyle;
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

UScrollbar& uscrollbar(const UOrient &o, UArgs a) {
  return *(new UScrollbar(o, a));
}
UScrollbar& uhscrollbar(UArgs a) {
  return *(new UScrollbar(UScrollbar::horizontal, a));
}
UScrollbar& uvscrollbar(UArgs a) {
  return *(new UScrollbar(UScrollbar::vertical, a));
}

/* ==================================================== ======== ======= */

UButton* UScrollbar::makeSlider(UArgs a) {
  UButton* bt = new UScrollbar::Slider(pos); //autodel?
  bt->addlist(a);
  return  bt;
}

UButton* UScrollbar::makeLessScroller(UArgs a) {
  return new UScrollbar::LessScroller(a); //autodel?
}

UButton* UScrollbar::makeMoreScroller(UArgs a) {
  return new UScrollbar::MoreScroller(a); //autodel?
}

//UFlag UScrollbar::lessFlag;
//UFlag UScrollbar::moreFlag;

UScrollbar::UScrollbar(const UOrient &_orient, UArgs a) : UBox(a) {
  //setCmodes(UMode::CAN_CLOSE_MENU, false); 
  val    = 0.0;
  incr   = 10;
  orient = _orient.get();
  pos.autodel(false);
  activeView = null;
  pane   = null;

  slider = makeSlider();
  slider->addlist(UOn::mdrag      / ucall(this, &UScrollbar::dragSlider)
		  + UOn::mpress   / ucall(this, &UScrollbar::pressSlider)
		  + UOn::mrelease / ucall(this, &UScrollbar::releaseSlider));

  lessScroller = makeLessScroller();
  lessScroller->add(UOn::mpress / ucall(this, &UScrollbar::pressScroller, -1));

  moreScroller = makeMoreScroller();
  moreScroller->add(UOn::mpress / ucall(this, &UScrollbar::pressScroller, +1));

  addlist(UOn::mpress / ucall(this, &UScrollbar::pressBackground)
	  + lessScroller + moreScroller + slider);
}

UScrollbar::~UScrollbar() {
  // Note: les components de UScrollbar (slider, less, more...) sont detruits
  // automatiquement via le DAG d'instance
  // Par contre il faut gerer les dependance avec le UPane control
  // par le scrollbar (NB: ne pas utiliser setHScrollbar, setVScrollbar
  // car dependances et actions croisees)
  if (pane) {
    if (this == pane->vscrollbar) pane->vscrollbar = null;
    if (this == pane->hscrollbar) pane->hscrollbar = null;
    pane = null;
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

void UScrollbar::setPane(class UPane *p) {pane = p;}

void UScrollbar::setValue(float newval) {

  if (newval < 0 ) newval = 0;
  else if (newval > 100) newval = 100.0;

  // test inverse' avec ce qui precede pour bloquer les extremes 
  // (= ne pas reafficher indefiniment qunad la souris sort du scrollbar)

  //if (newval == value) return;   //!Faux: marche pas quand ajout
  val = newval;

  //if a pane is attached to this scrollbar then scroll it
  if (pane) {
    if (orient == vertical.get()) {
      // -1 means: do nothing in this direction
      pane->scroll(-1, val, false);
      if (pane->getYScroll() <= 0.0) val = 0.0;
    }
    else {
      pane->scroll(val, -1, false);
      if (pane->getXScroll() <= 0.0) val = 0.0;
    }
  }

  // pour toutes les Views du Scrollbar

  for (ULinkLink *ll = parents.first(); ll; ll = ll->next()) {
    //NB: au moins UBoxLink* par construction
    //UBoxLink *boxlink = ll->link()->boxLinkCast();
    UBoxLink *boxlink = dynamic_cast<UBoxLink*>(ll->link());

    for (int kv = 0; kv < boxlink->getViewCount(); kv++) {
      UScrollbarView *sbar_view = (UScrollbarView*) boxlink->getViews()[kv];

      // !!NB toujours verifier views[k] != null 
      // (certaines vues peuvent avoir ete supprimees)
      if (sbar_view /* && (winview = sbar_view->getWinView())*/) {
	//NB: UPDATE fait par cette fct (via un MOVE, d'ou efficacite)
	sbar_view->setScroll(val);
      }
    }
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UScrollbar::pressSlider(UEvent*e) {
  if (e->getSource() == slider) {
    //activeView: the scrollbar view that where interaction currently takes place
    activeView = (UScrollbarView*)(e->getView()->getParentView());

    if (orient == vertical.get()) 
      activeView->setLastMouse(e->getYwin());
    else
      activeView->setLastMouse(e->getXwin());
  }
}

void UScrollbar::releaseSlider(UEvent*e) {
  //effet: le slider se synchronise avec la pos ou on a lache' la souris
  // meme si on est sorti de la fenetre (car dans ce cas, a moisn qu'il
  // n'y ait un xgrab, le scrollbar ne suit plus la souris)

  //dragSlider(e); plutot penible car bouge le slider qunad on clique
  //dessus sans modifier sa position (a cause de la manier dont sont
  //calculees les positions)
  activeView = null;
}


float foo(UEvent *e, UScrollbarView *activeView, short orient) {
  int pos;
  if (orient == UScrollbar::vertical.get())
    pos = e->getYwin() - activeView->getYwin();
  else
    pos = e->getXwin() - activeView->getXwin() ;
  
  return 
    (float) (pos - activeView->getScrollMin())
    / (activeView->getScrollMax() - activeView->getScrollMin()) 
    * 100.0;
}

void UScrollbar::dragSlider(UEvent *e) {
  if (!activeView) return;
  // activeView->getAY() == vertical absolute origin of the scrollbar view
  /*
  int pos;
  if (orient == vertical.get())
    pos = e->getYwin() - activeView->getYwin();
  else
    pos = e->getXwin() - activeView->getXwin() ;

  setValue((double) (pos - activeView->getScrollMin())
	   / (activeView->getScrollMax() - activeView->getScrollMin()) 
	   * 100.0);
  */
  setValue(foo(e, activeView, orient));
}

// the background of the Scrollbar was pressed
void UScrollbar::pressBackground(UEvent*e) {
  if (e->getButtonNumber() == 1) {
    float pressed_val = foo(e, (UScrollbarView*)e->getView(), orient);
    if (pressed_val > val) 
      setValue(val + incr);
    else 
      setValue(val - incr);
  }
  else if (e->getButtonNumber() == 2) {
    setValue(foo(e, (UScrollbarView*)e->getView(), orient));
  }
}

// the more or less Scroller was pressed
void UScrollbar::pressScroller(UEvent*, int dir) {
  setValue(val + incr * dir);
}


/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UScrollbarView
/* ==================================================== ======== ======= */

UViewStyle UScrollbarView::style(&UScrollbarView::makeView, UMode::UCONST);

UScrollbarView::UScrollbarView(UBoxLink *box_link, UView* par_view, 
			       UWinGraph *wgraph)
  : UView(box_link, par_view, wgraph) {
  lastmouse = scrollmin = scrollmax = 0;
}

// "static" constructor used by UViewStyle to make a new view
UView* UScrollbarView::makeView(UBoxLink*bl, UView* parview, 
				    UWinGraph *wgraph) {
  return new UScrollbarView(bl, parview, wgraph);
}

u_bool UScrollbarView::doLayout(UContext &parp, UViewLayout &vl) {
  return UView::doLayout(parp, vl); 
}

void UScrollbarView::doUpdate(UContext &parp, URegion r, URegion clip, 
			      UViewUpdate &vup) {
  // l'intersection de clip et de r est affectee dans clip
  if (clip.setInter(r) == 0) return;

  // il faut appeler DEUX fois doUpdate:   !!!!!!
  // -- la premiere pour calculer correctement la taille du UScrollbarView
  //    (car do layout ne le fait pas correctemt dans tous les cas
  //    comme le laisse suggerer son nom intelligemmenmt choisi)
  // -- la seconde vraiment pour tout afficher correctement 
  //    a la bonne position (c'est a dire APRES avoir bouge le slider)

  UView::doUpdate(parp, r, clip, vup);

  UScrollbar *scr = (UScrollbar*)getBox();
  // returns the corresponding views of the 'less', 'more', 'slider' buttons
  //UView *lessview = scr->lessScroller->getViewContainedInParent(this, 0);
  UView *moreview = scr->moreScroller->getViewContainedInParent(this, 0);
  UView *sliderview = scr->slider->getViewContainedInParent(this, 0);

  if (scr->orient == UScrollbar::vertical.get()) {
    // decaler de la hauteur des 2 boutons (s'ils sont en haut)
    //scrollmin = lessview->getHeight() + moreview->getHeight() + 2;
    scrollmin = moreview->getY() + moreview->getHeight() + 1;

    // ici y'a un pbm : on ne connait pas la place prise par le bord
    // ==> on risque de deborder d'ou le -2 !!!
    scrollmax = getHeight() - sliderview->getHeight() - 2;

    // repositionner le slider
    // ATT: set(,,false) ==> no update; surtout ne pas appeler move car 
    // ca rappellerait cette meme fonction => boucle infinie

    // en x : centrer / largeur du scrollbar
    scr->pos.set((getWidth() - sliderview->getWidth()) / 2, 
		 int(scrollmin+scr->getValue() * (scrollmax-scrollmin) / 100.0),
		 false);
  }

  else {  // horizontal
    // decaler de la largeur des 2 boutons (s'ils sont a gauche)
    //    scrollmin = lessview->getWidth() + moreview->getWidth() + 2;
    scrollmin = moreview->getX() + moreview->getWidth() + 1;
    scrollmax = getWidth() - sliderview->getWidth() - 2; 
    
    // repositionner et centrer le slider
    scr->pos.set(int(scrollmin+scr->getValue() * (scrollmax-scrollmin) / 100.0),
		 (getHeight() - sliderview->getHeight()) / 2,
		 false); 
  }

  // voir remarque ci-dessus sur le fait qu'il faut appeler DEUX FOIS
  // la fonction doUpdate
  UView::doUpdate(parp, r, clip, vup);
}

/* ==================================================== ======== ======= */
// Attention: cette fonction fait un update implicite!
// (-> ne pas l'appler dans doLayout ou duUpdate!) 

void UScrollbarView::setScroll(float value) {

  UScrollbar *scr = (UScrollbar*)getBox();

  float val = scrollmin + value * (scrollmax - scrollmin) / 100.0;
  if (val >= 0.0/* && val <= 100.0*/) {
    if (scr->orient == UScrollbar::vertical.get())
      scr->pos.setY((int) val);   //NB: fait update
    else
      scr->pos.setX((int) val);   //NB: fait update
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UPane
/* ==================================================== ======== ======= */

const UClass UPane::uclass("UPane");
const UClass UScrollPane::uclass("UScrollPane");

UStyle *UPane::style = null;

const UStyle* UPane::getStyle(const UBox *parent) {
  if (!style) {
    style = new UStyle(null);
    style->viewStyle      = &UPaneView::style;
    style->local.orient   = UOrient::horizontal.get();
    style->local.halign   = UHalign::flex.get();
    style->local.valign   = UValign::flex.get();
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    //style->local.padding.set(1,1);
    // will take initial size and won't change
    style->local.height   = UHeight::keepSize; // !!
    style->local.width    = UWidth::keepSize;  // !!
  }
  return style; 
}

/* ==================================================== ======== ======= */

UPane& upane(UArgs l) {return *(new UPane(l));}

UScrollPane& uscrollPane(int vs_mode, int hs_mode, UArgs l) {
  return *(new UScrollPane(vs_mode, hs_mode, l));
}
UScrollPane& uscrollPane(UArgs l) {
  return *(new UScrollPane(UScrollbar::always, UScrollbar::always, l));
}

UScrollPane& uscrollpane(int vs_mode, int hs_mode, UArgs l) {
  return *(new UScrollPane(vs_mode, hs_mode, l));
}
UScrollPane& uscrollpane(UArgs l) {
  return *(new UScrollPane(UScrollbar::always, UScrollbar::always, l));
}

/* ==================================================== ======== ======= */

UPane::UPane(UArgs a): UBox(a) {
  hscrollbar = null;
  vscrollbar = null;
  viewportLink = null;
}

UPane::~UPane() {
  //!!Att: scrollbar pas forcement enfnats du Pane !
  // (seront automatiquement detruit via le DAG inutile de s'en charger ici)
  if (vscrollbar) vscrollbar->setPane(null); vscrollbar = null;
  if (hscrollbar) hscrollbar->setPane(null); hscrollbar = null;
}

UScrollPane::UScrollPane(int vs_mode, int hs_mode, UArgs a): UPane(a) {
  if (vs_mode == UScrollbar::always) {
    add(UValign::flex);
    add(UHalign::right);
    add(vscrollbar = new UScrollbar(UScrollbar::vertical));
      // pour que les add() suivants mettent les enfants en zone centrale
    add(UHalign::flex);
    vscrollbar->setPane(this);
  }

  if (hs_mode == UScrollbar::always) {
    add(UValign::bottom); 
    add(UHalign::flex);
    add(hscrollbar = new UScrollbar(UScrollbar::horizontal));
    // pour que les add() suivants mettent les enfants en zone centrale
    add(UValign::flex);
    hscrollbar->setPane(this);
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

UScrollbar* UPane::getHScrollbar() {return hscrollbar;}
UScrollbar* UPane::getVScrollbar() {return vscrollbar;}

UBox* UPane::getViewport() {
  //return (viewportLink ? viewportLink->boxCast() : null);
  return (viewportLink ? dynamic_cast<UBox*>(viewportLink->brick()) : null);
}

UView *UPane::getViewportView(UView *pane_view) {
  if (! viewportLink) return null;
  else return viewportLink->getView(pane_view);
}

void UPane::_setViewportLink(UBoxLink*l) {
  viewportLink = l;
}

// Set the Vertical and Horizontal Scrollbars
// Notes:
// - if 'add_to_pane' is true (default case), the scrollbar is added to
//   the Pane child list and it will appear inside the Pane.
// - otherwise, the scrollbar is NOT added to the Pane and MUST be added
//   as a child of another box. This makes it possible to CONTROL a Pane
//   by a scrollbar that is NOT included in this Pane.

void UPane::setVScrollbar(UScrollbar *sc, u_bool add_to_pane) {
  // enleve le scrollbar du Pane (s'il y est: sinon ne fait rien)
  // mais NE LE DETRUIT PAS (car on ne sait pas qui a cree ce scrollbar:
  // il n'a pas forcement ete cree par le UPane ou le Scrollpane)
  if (vscrollbar) {
    vscrollbar->setPane(null);
    remove(vscrollbar, false);
  }
  vscrollbar = sc;
  if (sc) {
    sc->setPane(this);
    if (add_to_pane) {
      add(UValign::flex);
      add(UHalign::right);
      add(sc);
      // pour que les add() suivants mettent les enfants en zone centrale
      add(UHalign::flex);
    }
  }
}


void UPane::setHScrollbar(UScrollbar *sc, u_bool add_to_pane) {
  // enleve le scrollbar du Pane (s'il y est: sinon ne fait rien)
  // mais NE LE DETRUIT PAS!
  if (hscrollbar) {
    hscrollbar->setPane(null);
    remove(hscrollbar, false);
  }
  hscrollbar = sc;
  if (sc) {
    sc->setPane(this);
    if (add_to_pane) {
      add(UValign::bottom);
      add(UHalign::flex);
      add(sc);
      // pour que les add() suivants mettent les enfants en zone centrale
      add(UValign::flex);
    }
  }
}

/* ==================================================== ======== ======= */

void UPane::scroll(float new_xscroll, float new_yscroll, 
		   u_bool update_scrollbars) {
  int xoffset_max = 0, yoffset_max = 0;
  xscroll = new_xscroll;
  yscroll = new_yscroll;

  if (update_scrollbars) {
    if (hscrollbar) hscrollbar->setValue(xscroll);
    if (vscrollbar) vscrollbar->setValue(yscroll);
  }

  // pour toutes les Views de 'pane' 
  for (ULinkLink *ll = getParentList().first(); ll; ll = ll->next()) {

    //NB: au moins UBoxLink* par construction
    UBoxLink *boxlink = dynamic_cast<UBoxLink*>(ll->link());
    UView *winview = null;

    for (int kv = 0; kv < boxlink->getViewCount(); kv++) {
      //UPaneView par construction
      UPaneView *pane_view  = (UPaneView*)boxlink->getViews()[kv];

      // !!NB toujours verifier views[k] != null 
      // (certaines vues peuvent avoir ete supprimees)

      if (pane_view && (winview = pane_view->getHardwinView())) {

	UView *viewport_view = getViewportView(pane_view);
	//if (!viewport_view) warning("setValue","No child to scroll");
	if (viewport_view) {

	  UEvent e(UEvent::viewPaint, winview, null);
	  e.locateSource(viewport_view);
	  int deltax = 0, deltay = 0;
	  int xoffset = 0, yoffset = 0;
	  
	  if (yscroll >= 0) {               // vertical scroll
	    int viewport_h = pane_view->getHeight()
	      - pane_view->margin.top - pane_view->margin.bottom
	      - pane_view->frame.top - pane_view->frame.bottom;
	    
	    yoffset = int(yscroll * (viewport_view->getFavoriteHeight()
				     - viewport_h) / 100.0);
	    if (yoffset >= 0) {
	      deltay = yoffset - pane_view->getYScroll() ;

	      //NB: setYScroll ne fait pas de update
	      pane_view->setYScroll(yoffset);

	      viewport_view->y -= deltay;
	      viewport_view->height += deltay;
	    }
	  }
	  
	  if (xscroll >= 0) {		// horizontal scroll
	    int viewport_w = pane_view->getWidth()
	      - pane_view->margin.left - pane_view->margin.right
	      - pane_view->frame.left - pane_view->frame.right;
	    
	    xoffset = int(xscroll * (viewport_view->getFavoriteWidth()
				     - viewport_w) / 100.0);
	    if (xoffset >= 0) {
	      deltax = xoffset - pane_view->getXScroll() ;

	      //NB: setXScroll ne fait pas de update
	      pane_view->setXScroll(xoffset);

	      viewport_view->x -= deltax;
	      viewport_view->width += deltax;
	    }
	  }

	  //printf("yscroll %f - yoffset %d \n", yscroll, yoffset);
	  //printf("xscroll %f - xoffset %d \n", xscroll, xoffset);

	  if (xoffset >= 0 && yoffset >= 0) {
	    xoffset_max = U_MAX(xoffset_max, xoffset);
	    yoffset_max = U_MAX(yoffset_max, yoffset);
	    
	    // fait une copie d'ecran et ne genere que la partie manquante
	    UUpdate upd(UUpdate::SCROLL);
	    upd.setOffset(deltax, deltay);

	    viewport_view->getBox()->updateView(e, viewport_view, upd);
	  }
	}
      }
    }
  }

  if (xoffset_max == 0) xscroll = 0;
  if (yoffset_max == 0) yscroll = 0;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UViewStyle UPaneView::style(&UPaneView::makeView, UMode::UCONST);

UPaneView::UPaneView(UBoxLink *box_link, UView* par_view, UWinGraph *wgraph) 
  : UView(box_link, par_view, wgraph)
{
  xscroll = yscroll = 0;
}

// "static" constructor used by UViewStyle to make a new view

UView* UPaneView::makeView(UBoxLink*bl, UView* parview, 
			     UWinGraph* wgraph) {
  return new UPaneView(bl, parview, wgraph);
}

UPane *UPaneView::getPane(){  // same as getBox() but with correct type
  //return boxlink ? (UPane*)boxlink->boxCast() : null;
  return (boxlink ? dynamic_cast<UPane*>(boxlink->brick()) : null);
}


/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// keepSize : ne pas changer la taille (w,h) de la view
// utilise qunad on fait un update partiel A L"INTERIEUR d'une autre vue
// pour eviter de deborder ou de deconner sur le reste
// !!PAR CONTRE: keepSize doit etre false sur les objets inclus ou
// qunad on refait un update global

u_bool UPaneView::doLayout(UContext &parp, UViewLayout &vl) {
  UBox *box = getBox();
  UContext curp(box, this, parp);
  u_bool mustLayoutAgain = false;

  //ULink *ch = doPrelude(box, curp);
  UMultiList mlist(curp, box);

  frame.set(0, 0, 0, 0);
  int children_w = 0, children_h = 0;
  box->flexCount = 0;

  //for ( ; ch != null; ch = ch->next())
  for (ULink *ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, box)) {

      UBrick *b = ch->brick();
      UView *chboxview = null;
      UItem *it = null;
      UGroup *chgrp = null; //!att reinit!
      UViewLayout chvl; //!att: reinit by constr.

      if (b->propCast())  
	b->propCast()->putProp(&curp, box);

      /* a completer...
      else if ((chgrp = b->groupNotBoxCast())) {
	if (chgrp->isShowable())
	  doUpdateG(vd, b->groupCast(), curp, g, r, clip, vup); //meme vd !
      }
      */

      // UItems + UBoxes, UWins
      else if ((it = b->itemCast())
	       ||
	       ((chgrp = b->boxCast()) 
		&& chgrp->isShowable()
		&& chgrp->isDef(0, UMode::BOX)
		&& (chboxview = ((UBoxLink*)ch)->getView(this)))
	       ) {

	if ((chgrp && chgrp->isDef(0, UMode::FLOATING)) 
	    || ch->floatingCast()) {
	  mustLayoutAgain |= chboxview->doLayout(curp, chvl);
	}

	else {
	  if (it) it->getSize(&curp, &(chvl.cmax_w), &(chvl.cmax_h));
	  else {
	    mustLayoutAgain |= chboxview->doLayout(curp, chvl);
	  }

	  //NB: top et bottom s'adaptent a la zone centrale en largeur
	  //ils ne controlent donc que la hauteur de leurs zones respectives
	  switch (curp.local.valign) {
	  case TOP:
	    if (chvl.cmax_h > frame.top)  frame.top = chvl.cmax_h;
	    break;
	  case BOTTOM:
	    if (chvl.cmax_h > frame.bottom)  frame.bottom = chvl.cmax_h;
	    break;
	  case VCENTER:
	  case VFLEX:
	    if (chvl.cmax_h > children_h)  children_h = chvl.cmax_h;
	    break;
	  }

	  //NB: left et right s'adaptent a la zone centrale en hauteur
	  //ils ne controlent donc que la largeur de leurs zones respectives
	  switch (curp.local.halign) {
	  case LEFT:
	    if (chvl.cmax_w > frame.left)  frame.left = chvl.cmax_w;
	    break;
	  case RIGHT:
	    if (chvl.cmax_w > frame.right)  frame.right = chvl.cmax_w;
	    break;
	  case HCENTER:
	  case HFLEX:
	    if (chvl.cmax_w > children_w)  children_w = chvl.cmax_w;
	    break;
	  }
	}
      }
  } //endfor
  
  // Border and Box size
  //!!ATT: margin est membre de l'objet (cxontrairement aux autres *view)
  margin.set(0, 0);
  if (curp.local.border) 
    curp.local.border->doLayout(this, curp, margin);

  //!att a 'favorite_*' : il y a des cas ou cette valeur reste inchangee
  //(si width ou height ext fixe : 'favorite_*' est initalisee la 1ere
  //(fois puis reste inchangee ensuite sauf en cas de 'flex'ibles)

  // voir note an haut

  vl.spec_w = curp.local.width;  // < keepSize si indefini

  // -- uwidth(int) with int >= 0   : size fixed by argument and won't change
  // -- uwidth(UWidth::keepSize) : will take initial size and won't change

  if (curp.local.width >= UWidth::keepSize) {

    //vl.wmode = UViewLayout::FIXED_BY_CHILD;

    if (!curp.boxIsHFlex || width <=0) {     // w <=0 : initialization
      // favorite_w initialisee, ne changera pas sauf si 'flex'ible
      //BUG: w = favorite_w = curp.local.width;
      children_w += margin.left + margin.right
	+ frame.left + frame.right
	+ curp.local.padding.left + curp.local.padding.right;

      // favorite doit etre le MAX de la taille desiree et de la taille
      // requise par les enfants (sinon le scroll ne marchera pas dans
      // le 1e cas et le 'flex' ne marchera pas dans le second)
      favoriteWidth = U_MAX(curp.local.width, children_w);
    }
    if (width <=0) {
      //fixedSizes = FIXED_WIDTH;
      setVmodes(UView::FIXED_WIDTH, true);
      if (curp.local.width == UWidth::keepSize)
	width = favoriteWidth; //init to favorite
      else 
	width = curp.local.width; //init to spec.
    }
  }
  else {
    favoriteWidth = children_w + margin.left + margin.right
      + frame.left + frame.right
      + curp.local.padding.left + curp.local.padding.right;
    //obs: if (!vl.keepSize) width = favoriteWidth;
    width = favoriteWidth;
  }
    

  vl.spec_h = curp.local.height;  // < keepSize si indefini

  // see note above
  if (curp.local.height >= UHeight::keepSize) {

    //vl.hmode = UViewLayout::FIXED_BY_CHILD;

    if (!curp.boxIsVFlex || height <=0) {    // w <=0 : initialization
      // favorite_h initialisee, ne changera pas sauf si 'flex'ible
      //BUG::h = favorite_h = curp.local.height;
      // voir note pour favorite_w ci-dessus.
      children_h += margin.top + margin.bottom + frame.top + frame.bottom
	+ curp.local.padding.top + curp.local.padding.bottom ;
      favoriteHeight = U_MAX(curp.local.height,  children_h);
    }
    if (height <=0) {
      //if (fixedSizes == FIXED_WIDTH) fixedSizes = FIXED_SIZE;
      //else fixedSizes = FIXED_HEIGHT;
      setVmodes(UView::FIXED_HEIGHT, true);

      if (curp.local.height == UHeight::keepSize)
	height = favoriteHeight; //init to favorite
      else
	height = curp.local.height; // init to spec.
    }
  }
  else {
    favoriteHeight = children_h + margin.top + margin.bottom
      + curp.local.padding.top + curp.local.padding.bottom ;
    //obs: if (!vl.keepSize) height = favoriteHeight;
    height = favoriteHeight;
  }
  
  // IMPORTANT:
  // Height depend de Width pour les UFlowView
  // ceci impose de refaire une seconde fois le layout du parent
  // (sauf dans le cas ou Width est fixe a priori auquel cas Height
  // (peut directement etre determine des la premier passe)
  //
  // must layout again
  ///shown = false;
  vl.cmin_w = vl.cmax_w = width;
  vl.cmin_h = vl.cmax_h = height;
  return mustLayoutAgain;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

static void scrollDoUpdate(UViewUpdateImpl&, UPaneView*, UContext &curp, 
			   UPane*, UMultiList &mlist, UWinGraph &g, 
			   const URegion &r, URegion &clip, UViewUpdate &vup);

//NB: mode SearchItem: juste recuperer l'item et sa position sans redessiner
//!ATT il faut IMPERATIVEMENT itemData != null dans le mode SearchItem !
//NB: clip est passe en valeur, pas r

//Special PaneView: xscroll, yscroll qui ne portent que sur la zone centrale
// ie. le "viewport")

void UPaneView::doUpdate(UContext &parp, URegion r, URegion clip, 
			 UViewUpdate &vup) {
  // UPane* par construction
  UPane* pane = getPane();

  // item is not visible because of scrolling, etc... 
  // l'intersection de clip et de r est affectee dans clip
  // (test pas valable pour les FLOATING car coords changees ensuite)

  if (!pane->isDef(0, UMode::FLOATING)) { 
    if (clip.setInter(r) == 0) return;
  }

  UViewUpdateImpl vd(this, r, vup, pane->flexCount);
  UContext curp(pane, this, parp);

  //ULink *lastprelude = doPrelude(box, curp);
  UMultiList mlist(curp, pane);
  //fait aussi:
  //a) if (grp->isDef(0,UMode::HARDWIN)) mlist.addSoftwinList(grp);
  //b) u_bool is_softwin_list = grp->isDef(0,UMode::SOFTWIN_LIST);

  if (pane->isDef(0, UMode::FLOATING)) {
    vd.setFloating(curp, pane, r);
    if (clip.setInter(r) == 0) return;
  }

  {
    UWinGraph &g = curp.winview->wg();
    int gmode = 0;

    // setClip peut provoquer un seg fault si appele avec des donnees non
    // initialisee (et c'est le cas pour les operations LOCATE sans PAINT)
    if (vup.mode < UViewUpdate::LOCATE_ITEM_POS) {
     //sub-hardwin incrustee X -> decaler les coords
      if (pane->isDef(0, UMode::SUB_HARDWIN))
	gmode = g.beginShifted(clip,-r.x,-r.y); 
      //pas de blending (mais le background est peut etre none)
      else if (curp.local.alpha == 1.)
	gmode = g.begin(clip);
      //translucent --> NB: serait OK pour les groups ???
      else {
	gmode = g.beginBlended(clip, r);
	//!!!penser locate;
      }
      //NB: ne pas oublier de TOUJOURS faire g.end() !
    }

    vd.chclip.set(clip);
    // (nb: vd.margin init. by updateBackgroundAndBorder)
    vd.backgroundAndBorder(g, curp, r);
  
    //!! a la differnce des autres Renderers, UScrollView conserve
    //ses propres marges en argument (il faut donc recopier vd.margin
    // calcule par updateBackgroundAndBorder dans this->margin
    this->margin = vd.margin;
    
    // clipping limits
    vd.chclip.set(r.x + margin.left, r.y + margin.top,
		  r.width - margin.right, r.height - margin.bottom);
    
    // pas la peine de chercher a afficher les enfants
    // s'ils sont hors zone de clip // generalement inutile et dangereux
    if (vd.chclip.setInter(clip) != 0)
      scrollDoUpdate(vd, this, curp, pane, mlist, g, r, clip, vup);

   g.endAll(gmode, r);
  }

  //NB: finalisation par destructeur de UViewUpdateImpl vd
}

/* ==================================================== ======== ======= */

static void scrollDoUpdate(UViewUpdateImpl &vd, UPaneView* paneview,
			   UContext &curp, UPane* pane, 
			   UMultiList &mlist, UWinGraph &g, 
			   const URegion &r, URegion &clip, UViewUpdate &vup) {
  UMargin& margin = paneview->margin;
  UMargin& frame  = paneview->frame;
  u_dim& xscroll  = paneview->xscroll;
  u_dim& yscroll  = paneview->yscroll;

  //for ( ; ch != null; ch = ch->next())
  for (ULink* ch = mlist.first(); ch != null; ch = mlist.next(ch))
    if (ch->verifies(&curp, pane)) {
	
      UBrick *b = ch->brick();
      UView *chboxview = null;
      UItem *it = null; 
      UGroup *chgrp = null; //!att reinit!
      
      if (b->propCast())  
	b->propCast()->putProp(&curp, pane);
      
      /* a completer
	 else if ((chgrp = b->groupNotBoxCast())) {
	 if (chgrp->isShowable())
	 doUpdateG(vd, b->groupCast(), curp, g, r, clip, vup); //meme vd !
	 }
      */
	
      // UItems + UBoxes, UWins
      else if ((it = b->itemCast())
	       ||
	       ((chgrp = b->boxCast()) 
		&& chgrp->isShowable()
		&& chgrp->isDef(0, UMode::BOX)
		&& (chboxview = ((UBoxLink*)ch)->getView(vd.view)))
	       ) {
	
	// 1::cas des Floating
	
	if ((chgrp && chgrp->isDef(0, UMode::FLOATING))
	    || ch->floatingCast()) {
	  
	  URegion fl_chr; //chr for floatings
	  chboxview->getSize(fl_chr.width, fl_chr.height);
	  
	  if (ch->floatingCast()) {
	    // transformer les coords relatives en coords absolues
	    // puis afficher l'objet a cet endroit precis
	    fl_chr.x = r.x + margin.left + ch->floatingCast()->getX();
	    fl_chr.y = r.y + margin.top + ch->floatingCast()->getY();
	  }
	  else {
	    //!att: on rajoute la MARGE: les coords sont locales 
	    // % a l'INTERIEUR du CADRE et non % a l'origine de la Box.
	    fl_chr.x = r.x + margin.left;
	    fl_chr.y = r.y + margin.top;
	    // coords relatives rajoutees ensuite dans doUpdate
	  }
	  
	  chboxview->doUpdate(curp, fl_chr, vd.chclip, vup);
	}
	
	// 2- Cas particulier de la zone centrale:
	// celle-ci est la seule a pouvoir etre scrollee
	
	else if ((curp.local.valign==UView::VFLEX || curp.local.valign==UView::VCENTER)
		 &&
		 (curp.local.halign==UView::HFLEX || curp.local.halign==UView::HCENTER)
		 ){
	  
	  //prend en compte le dernier affiche (note devrait etre dans Layout)
	  pane->_setViewportLink(dynamic_cast<UBoxLink*>(ch));
	  
	  vd.chr.x = r.x + frame.left + margin.left
	    - xscroll;
	  vd.chr.y = r.y + frame.top  + margin.top
	    - yscroll;
#if EXXX
	  // !NOTE: pour les Items, VFLEX est identique a MIDDLE
	  if (!it && curp.local.valign == VFLEX)
	    chr.height = r.height - frame.top - frame.bottom
	      - margin.top - margin.bottom;
	  else chr.height = chview->h;
	  
	  // !NOTE: pour les Items, HFLEX est identique a CENTER
	  if (!it && curp.local.halign == HFLEX)
	    chr.width = r.width - frame.left - frame.right
	      - margin.left - margin.right;	 
	  else chr.width = chview->w; 
#endif
	  //dans TOUS les cas (flex ou center/middle, Item ou Box) :
	  // => TOUJOURS justifie
	  vd.chr.height =
	    r.height - frame.top - frame.bottom 
	    - margin.top - margin.bottom;
	  vd.chr.width =
	    r.width - frame.left - frame.right 
	    - margin.left - margin.right;
	  
	  // scrolling: il faut egalement agrandir la zone apres l'avoir
	  // decalee vers la gauche ou vres le haut
	  vd.chr.width  += xscroll;
	  vd.chr.height += yscroll;
	  
	  //CP de la zone centrale : il faut redefinir le clip
	  //(car seule cette zone est scrollee)
	  URegion chclip2(r.x + margin.left + frame.left,
			  r.y + margin.top + frame.top,
			  r.width - margin.right - frame.right, 
			  r.height - margin.bottom - frame.bottom);
	  
	  if (chclip2.setInter(clip) == 0) continue;
	  
	  if (vd.chr.width > 0 && vd.chr.height > 0) {
	    
	    //pas un item => Box
	    if (!it) chboxview->doUpdate(curp, vd.chr, chclip2, vup);
	    else {
	      
	      //NB: ITEM_OPS => can_paint == false mais pas l'inverse !
	      if (vd.can_paint) {
		g.setClip(chclip2);		 //chclip2!!
		it->paint(g, &curp, vd.chr);
	      }
	      
	      else if (vup.mode >= UViewUpdate::LOCATE_ITEM_POS) {
		// do not draw, just find
		// !att: il faut que e->itemData soit =!null
		// s'arrete de cherche qunad true est renvoye'
		
		if (vup.mode == UViewUpdate::LOCATE_ITEM_POS) {
		  if (vd.view->locateItemVPos(curp, ch, g, vd.chr, vup))
		    return;
		}
		else if (vup.mode == UViewUpdate::LOCATE_ITEM_PTR) {
		  if (vd.view->locateItemPtr(curp, ch, g, vd.chr, vup))
		    return;
		}
	      }
	    }
	    
	  } //endif (chr.width > 0 && chr.height > 0)
	}
	
	
	// 3- cas du frame entourant la zone centrale
	else {
	  
	  switch (curp.local.valign) {
	  case UView::TOP:
	    vd.chr.y = r.y + margin.top;
	    vd.chr.height = frame.top;
	    break;
	    
	  case UView::BOTTOM:
	    vd.chr.y = r.y + r.height - frame.bottom - margin.bottom;
	    vd.chr.height = frame.bottom;
	    break;
	    
	  case UView::VCENTER: // ne s'adapte PAS a la taille du Pane
	  case UView::VFLEX: // Middle adaptatif: s'adapte a la taille du Pane
	    vd.chr.y = r.y + frame.top + margin.top;
	    
#if EXX
	    // !NOTE: pour les Items, VFLEX est identique a MIDDLE
	    if (!it && curp.local.valign == VFLEX)
	      vd.chr.height = r.height - frame.top - frame.bottom - margin.top - margin.bottom;
	    else vd.chr.height = chview->h;
#endif
	    //dans TOUS les cas (flex ou center/middle, Item ou Box) :
	    // => TOUJOURS justifie
	    vd.chr.height = r.height - frame.top - frame.bottom 
	      - margin.top - margin.bottom;
	    break;
	  }
	  
	  switch(curp.local.halign) {
	  case UView::LEFT:
	    vd.chr.x = r.x + margin.left;
	    vd.chr.width = frame.left;
	    break;
	    
	  case UView::RIGHT:
	    vd.chr.x = r.x + r.width - frame.right - margin.right;
	    vd.chr.width = frame.right;
	    break;
	    
	  case UView::HCENTER: // ne s'adapte PAS a la taille du Pane
	  case UView::HFLEX: // Center adaptatif: s'adapte a la taille du Pane
	    vd.chr.x = r.x + frame.left + margin.left;
#if EXX
	    // !NOTE: pour les Items, FLEX est identique a CENTER
	    if (!it && curp.local.halign == HFLEX)
	      chr.width = r.width - frame.left - frame.right - margin.left - margin.right;	 
	    else chr.width = chview->w;
#endif
	    //dans TOUS les cas (flex ou center/middle, Item ou Box) :
	    // => TOUJOURS justifie
	    vd.chr.width = r.width - frame.left - frame.right
	      - margin.left - margin.right;	 
	    break;
	  }//endswitch()
	  
	  
	  if (vd.chr.width > 0 && vd.chr.height > 0) {
	    
	    //pas un item => Box
	    if (!it) chboxview->doUpdate(curp, vd.chr, vd.chclip, vup);
	    else {
	      
	      //NB: ITEM_OPS => can_paint == false mais pas l'inverse !
	      if (vd.can_paint) {
		g.setClip(vd.chclip);
		it->paint(g, &curp, vd.chr);
	      }
	      
	      else if (vup.mode >= UViewUpdate::LOCATE_ITEM_POS) {
		// do not draw, just find
		// !att: il faut que e->itemData soit =!null
		// s'arrete de cherche qunad true est renvoye'
		
		if (vup.mode == UViewUpdate::LOCATE_ITEM_POS) {
		  if (vd.view->locateItemVPos(curp, ch, g, vd.chr, vup))
		    return;
		}
		else if (vup.mode == UViewUpdate::LOCATE_ITEM_PTR) {
		  if (vd.view->locateItemPtr(curp, ch, g, vd.chr, vup))
		    return;
		}
	      } //endif(ITEM_OPS)
	    }
	    
	  }// endif (chr.width > 0 && chr.height > 0)
	}//endelse (in frame)
      }
    }
  
  // call callback (desormais appele en fin de traitement)
  if (vd.can_paint 
      && pane->isDef(UMode::VIEW_PAINT_CB|UMode::VIEW_CHANGE_CB,0)) 
    vd.callbacks(pane, curp.winview);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// different de UView::locateSource car il faut tenir compte du scroll
// sur la partie centrale

UView* UPaneView::locateSource(UEvent *e, UContext *parp, URegion clip,
				  UView *searchedView) {
  UBox *box = getBox();
  if (!box) {
    uerror("UPaneView::locateSource",
	   "Internal error: View points to null box");
    return null;
  }

  //OLD: if (!shown) return null; // View pas visible
  if (! box->isShowable() 
      //|| box->isDef(0, UMode::IGNORE_EVENTS)
      || box->isDef(UMode::IGNORE_EVENTS,0)
      || box == e->through 
      || clip.setInter(*this) == 0  // intersects clip !
      )
    return null;

  if (searchedView) {
    if (searchedView == this) goto TROUVE;
    //sinon on va chercher dans la liste des enfants via la boucle
    //'for ()' ci-apres
  }
  else {
    // Notes:
    // 1)- cas View rectangulaire
    // 2)- clip est l'intersection de this et du clip initial
    //     ==> necessaire pour calculer le bon redrawClip qd les objets 
 
    if (e->getXwin() < clip.x 
	|| e->getYwin() < clip.y 
	|| e->getXwin() >= (clip.x + clip.width) 
	|| e->getYwin() >= (clip.y + clip.height))
      return null;
 
    // NB: event flagging is meaningless in 'searchedView' mode:
    // mouse coords must be defined in order to determine which objects
    // are traversed by events
    // ==> ceci implique que seules les UBox peuvent flagger les events
    //     (car les UGroup n'ont pas de coords)
  
    //if (box->isDef(0, UMode::FLAG_EVENTS)) {
    if (box->isDef(UMode::FLAG_EVENTS,0)) {
      //e->sourceView = this;
      e->setViewSource(this);
      box->fire(*e, UOn::traverse);
    }
  }

  { // { impose par syntaxe du goto
    //if (!gstate.isEnabled()) e->props.inherit.enabled = false; !!!!
    UContext curp(box, this, *parp);

    // chercher d'abord si objets actifs dans border (laisse beton!)
    //if (curp.local.border
    //    && (src_view = curp.local.border->locateSource(e, this, &curp)))
    //  return src_view;
    
    UView *last_found_view = null;

    // on continue gentiment de scanner TOUTE la liste jusqu'au bout car
    // il peut y avoir une autre 'found_view' superposee
    // et c'est cette derniere qu'on veut recuperer

    for (ULink *ch = getBox()->getChildLinks(); ch; ch = ch->next()) {
      UBrick *b = ch->brick();
      UView *chview = null, *found_view = null;

      if (b->propCast())  
	b->propCast()->putProp(&curp, box);

      else if (b->boxCast() //&& b->boxCast()->isDef(0,UMode::INBOX)
	       && b->boxCast()->isDef(0,UMode::BOX)
	       // ch est donc un UBoxLink par construction
	       && (chview = ((UBoxLink*)ch)->getView(this))) {

	if ((curp.local.valign==VFLEX || curp.local.valign==VCENTER)
	    &&
	    (curp.local.halign==HFLEX || curp.local.halign==HCENTER)
	    ){

	  // CP de la zone centrale : il faut redefinir le clip
	  // (car seule cette zone est scrollee)

	  if (ch == getPane()->viewportLink) {
	    // cas zone central ET object actif

	    URegion clip2(x + margin.left + frame.left,
			  y + margin.top + frame.top,
			  width - margin.right - frame.right, 
			  height - margin.bottom - frame.bottom);
	    if (clip2.setInter(clip) > 0) {
	      found_view = chview->locateSource(e, &curp, clip2, searchedView);
	      if (found_view) last_found_view = found_view;
	    }
	  }
	  // else que dalle: objet central mais cache
	}

	// objet hors zone centrale => traitement standard
	else {
	  found_view = chview->locateSource(e, &curp, clip, searchedView);
	  if (found_view) last_found_view = found_view;
	}
      }

    } //endfor

    // sinon on n'a pas trouve de vue incluse et il faudra donc retourner
    // la vue courante. Pour ce faire on continue en sequence jusqu'au
    // tag 'TROUVE'
    if (last_found_view) return last_found_view;
  }


  // si mode searchedView et qu'on n'est PAS alle' directement au
  // tag 'TROUVE' c'est qu'on a pas trouve le bon objet
  //
  if (searchedView) return null;


 TROUVE:
  // si l'objet n'a pas d'enfant ou si la souris n'est contenue dans aucun 
  // de ses enfants alors retourner l'objet en personne. (apres avoir
  // copie' toutes les infos desirees)
 
  e->setViewSource(this);
  e->redrawStatus = 1;
  e->redrawClip = clip;

  // e->transpView = the first view that is transparent in the tree
  // !att: curp.transpView ET NON parp.transpView

  //EX: if (box->isTransparent()) { propr maintenant relative a UView
  if (isDef(TRANSPARENT)) {
   if (!parp || !parp->firstTranspView) e->firstTranspView = this;
    else e->firstTranspView = parp->firstTranspView;
  }
  else e->firstTranspView = null;

  //!att: recopie du CONTENU du context du PARENT
  if (parp) e->dumpParentContext(*parp); 
  return this;
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:00] ======= */

