/* ==================================================== ======== ======= *
 *
 *  uupix.cc
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uupix.cc	ubit:b1.11.1"
#include <udefs.hh>
#include <ubrick.hh>
#include <uprop.hh>
#include <ucontext.hh>
#include <ugraph.hh>
#include <uctrl.hh>
#include <uview.hh>
#include <ucolor.hh>
#include <ustr.hh>
#include <upix.hh>
#include <ugraph.hh>
#include <unat.hh>
#include <uevent.hh>
#include <uappli.hh>
#include <ubox.hh>

static UPix *UnknownIma = &UPix::question;  //!!!

#define TMP_PIX_FILE "/usr/tmp/ubit_tmp_pix.xpm"

const UClass UIma::uclass("UIma");
const UClass UPix::uclass("UPix");

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UIma& uima(const UStr &filename) {
  return *(new UIma(filename));
}
UIma& uima(const UStr *filename) {
  return *(new UIma(filename));
}
UIma& uima(const char *filename) {
  return *(new UIma(filename));
}
UIma& uima(const char **xpm_data) {
  return *(new UIma(xpm_data));
}

UPix& upix(const UStr &filename) {
  return *(new UPix(filename));
}
UPix& upix(const UStr *filename) {
  return *(new UPix(filename));
}
UPix& upix(const char *filename) {
  return *(new UPix(filename));
}
UPix& upix(const char **xpm_data) {
  return *(new UPix(xpm_data));
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UIma::UIma(const char *file_name, u_modes m) : UItem(m) {
  name = null;
  natimas = null;
  set(file_name, false);
}
UIma::UIma(const UStr *file_name, u_modes m) : UItem(m) {
  name = null;
  natimas = null;
  set(file_name, false);
}
UIma::UIma(const UStr &file_name, u_modes m) : UItem(m) {
  name    = null;
  natimas = null;
  set(file_name, false);
}
UIma::UIma(const char **xpm_data, u_modes m) : UItem(m) {
  name    = null;
  natimas = null;
  set(xpm_data, false);
}

UIma::~UIma() {
  if (name) free(name);
  cleanCache();
}

void UIma::cleanCache() {
  UNatIma *pim = natimas, *nextim = null;
  while (pim) {
    nextim = pim->next;  //store nextim *before* deleting pim!
    delete pim;
    pim = nextim;
  }
  natimas = null;
  // NB: on ne detruit pas data (data n'est pas copie et 
  // pointe donc sur l'original)
}

/* ==================================================== ======== ======= */

void UIma::set(const UStr *fname, u_bool upd) {
  set(fname->chars(), upd);
}
void UIma::set(const UStr &fname, u_bool upd) {
  set(fname.chars(), upd);
}

void UIma::set(const char *fname, u_bool upd) {
  cleanCache();
  natimas   = null;
  if (name) free(name);
  name      = fname ? strdup(fname) : null;
  data      = null;
  bgcolor   = null;
  load_stat = 0;
  read_from_file = true;
  read_from_data = false;
  changed(upd);
}

void UIma::set(const char **xpm_data, u_bool upd) {
  cleanCache();
  natimas   = null;
  if (name) free(name);
  name      = null;
  data      = xpm_data;  //!!att: NOT copied!
  bgcolor   = null;
  load_stat = 0;
  read_from_file = false;
  read_from_data = true;
  changed(upd);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UPix::UPix(const char* file_name, u_modes m) : UIma(file_name, m) {
  //is_pix  = true;
  natpix  = null;
}
UPix::UPix(const UStr* file_name, u_modes m) : UIma(file_name, m) {
  //is_pix  = true;
  natpix  = null;
}
UPix::UPix(const UStr& file_name, u_modes m)  : UIma(file_name, m) {
  //is_pix  = true;
  natpix  = null;
}
UPix::UPix(const char **xpm_data, u_modes m) : UIma(xpm_data, m) {
  //is_pix  = true;
  natpix = null;
}

UPix::UPix(const char **xpm_data, const UBgcolor &bc, u_modes m) 
  : UIma(xpm_data, m) {
  //is_pix  = true;
  bgcolor = &bc;
  natpix = null;
}

UPix::~UPix() {
  //!att: appel necessaire car ~UIma appelara le cleanCache de UIma
  // (l'appel des fct virtuelles ne l'est dans dans les destructeurs!)
  cleanCache();
}

void UPix::cleanCache() {
  if (natpix) delete natpix;
  natpix = null;
  UIma::cleanCache();
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UIma::update() {
  parents.updateParents(UUpdate::layout);
}

u_bool UIma::isRealized() const {
  return (natimas ? natimas->isRealized() : false);
}

u_dim UIma::getHeight() const {
  return (natimas ? natimas->getHeight() : -1);
}

u_dim UIma::getWidth() const {
  return (natimas ? natimas->getWidth() : -1);
}

int UIma::getStatus() {return load_stat;}

u_bool UIma::isShaped() const {
  return (natimas ? natimas->isShaped() : false);
}

  // the following variants of the set() methods makes it possible
  // to specify the background color when useful (ie. when the
  // image data is shaped)
  // Special cases:
  // -- UBgcolor::none    --> transparent background (the default
  //                          when using other set() methods)
  // -- UBgcolor::inherit --> the background color is inherited from the
  //                          FIRST parent (beware if several parents!)

//etc....

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UIma::getSize(UContext *props, u_dim *w, u_dim *h) const {
  UWinGraph &g = props->winview->wg();

  /*obs:  (pas coherent avec partage des parents
    const UColor *bg;
    // inherit from the parent color if UBgcolor::inherit
    if (this->bgcolor && this->bgcolor->equals(UBgcolor::inherit))
    bg = props->bgcolor;
    // otherwise take the (possibly transparent) specified color
    else bg = this->bgcolor;
  */

  if (load_stat==0 && !natimas) realize(g.getDisp(), false);
  //realize(g.getDisp(), props->color, bg, false);

  // !!ATTENTION: l'image peut etre partage par plusieurs objets,
  //   d'ou la necessite d'initialiser TOUS les links correctement
  //   meme si l'image en elle-meme n'est charge qu'une seule fois

  if (!natimas) {
    if (this != UnknownIma)  //empecher recusriosn infinies!
    UnknownIma->getSize(props, w, h);
  }
  else {
    int fsize;
    UNatIma *pim = findImaInCache(g.getNatDisp(), natimas,
				  props->lscale, fsize);    
    if (!pim) pim = addImaInCache(g.getNatDisp(), natimas, 
				  props->lscale, fsize);
    if (!pim) {
      *w = *h = 0;
      return;
    }
    else {
      *w = pim->getWidth();
      *h = pim->getHeight();
    }
  }
}

void UPix::getSize(UContext *props, u_dim *w, u_dim *h) const {

  if (natpix && natpix->lscale == props->lscale) {
      *w = natpix->getWidth();
      *h = natpix->getHeight();
  }
  else UIma::getSize(props, w, h);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UIma::paint(UWinGraph &g, UContext *props, const URegion &r) const {
  if (!natimas) {
    if (this != UnknownIma)  //empecher recusrions infinies!
      UnknownIma->paint(g, props, r);
  }
  else {
    int fsize;
    UNatIma *pim = findImaInCache(g.getNatDisp(), natimas, 
				  props->lscale, fsize);
    //etre coherent avec getSize: ne rien faire sinon
    if (pim) g.drawIma(pim, r.x, r.y);      
  }
}

void UPix::paint(UWinGraph &g, UContext *props, const URegion &r) const {
  if (!natimas) {
    if (this != UnknownIma)  //empecher recusrions infinies!
      UnknownIma->paint(g, props, r);
  }
  //!ATT: on ne dessine l'image telle quelle QUE SI elle a vraiment
  //exactement la meme taille (sinon retailler)
  else if (natpix && natpix->lscale == props->lscale) {
    g.drawIma(natpix, r.x, r.y);
  }
  else {
    int fsize;
    UNatIma *pim = findImaInCache(g.getNatDisp(), natimas, 
				  props->lscale, fsize);
    //etre coherent avec getSize: ne rien faire sinon
    if (pim) {
      // deletes the previous natpix
      if (natpix) delete natpix; 
      natpix = null; 

      // creates the new natpix
      UWinGraph &g = props->winview->wg();
      natpix = new UNatPix(g.getNatDisp(), pim);

      if (natpix) {  	// cas normal: draw the natpix
	natpix->lscale = props->lscale;
	g.drawIma(natpix, r.x, r.y);
      }
      //pbm eventuel de creation du natpix ==> draw ima
      else g.drawIma(pim, r.x, r.y);

    }
  }
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

UNatIma* UIma::findImaInCache(UNatDisp *nd, 
			      UNatIma *natimas, //source image
			      int lscale,       //logical scale
			      int &fsize) {     // font size

  // convertit lscale en fsize, taille de font correspondante
  // et verifie que l'on ne depasse pas les limites MIN et MAX
  // Si c'est le cas --> modifier lsize de telle sorte qu'on cherche
  // une image dans le cache aynat la taille limite (au lieu de
  // creer une nouvelle image qui aurait en fait de nouveau
  // la meme taille limite)

  fsize = lscale + UFont::MEDIUM_LSIZE;
  if (fsize > UFont::MAX_LSIZE) {
    //printf("MAX scaling factor reached\n");
    fsize = UFont::MAX_LSIZE;
    //set lscale to MAX!
    lscale = fsize - UFont::MEDIUM_LSIZE;
  }
  else if (fsize < UFont::MIN_LSIZE) {
    //printf("MIN scaling factor reached\n");
    fsize = UFont::MIN_LSIZE;
    //set lscale to MIN!
    lscale = fsize - UFont::MEDIUM_LSIZE;
  }

  //NB: lscale=0 => nominal size / first natima always at nominal size
  //printf("search at scale %d\n", lscale);

  // search in the cache if this image was already created with
  // the same size
  for (UNatIma *pim = natimas; pim != null; pim = pim->next) 
    if (pim->lscale == lscale) return pim;   //found in cache

  return null; //not found
}

/* ==================================================== ======== ======= */

UNatIma* UIma::addImaInCache(UNatDisp *nd, 
			     UNatIma *natimas, //source image
			     int lscale,       //logical scale
			     int fsize) {      //font size
  if (!natimas) return null;

  // on detruit systematiquement celui qui suit natimas 
  // (natimas pointant sur l'original)
  if (natimas->next) {
    delete natimas->next;
    natimas->next = null;
  }

  //derives xscale and yscale from lscale automatically from font sizes  
  float xyscale = UFontFamily::getDefaultFontScale(fsize);

  UNatIma *pim = natimas->clone(nd, xyscale, xyscale);
  if (pim) {
    pim->lscale   = lscale;
    pim->next     = natimas->next;
    natimas->next = pim;
  }

  return pim; 
}
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

int UIma::realize(class UDisp *disp, u_bool force_reload) const {

  if (force_reload || (load_stat==0 && !natimas)) {  
    // ne pas essayer de recharger une seconde fois quelque soit
    // le resultat de cet essai (pour ne pas boucler sur le load)
    //must_load = false;

    load_stat = InvalidFile;
    char *fpath = null;
    char *filename_or_data = null;
    UNatIma::Reader reader = null;

    if (read_from_data) {
      if (data) {
	filename_or_data = (char*)data;
	reader = UXpmDataReader;
	fpath = strdup("[XMP compiled data]");
      }
      else {
	//warning("realize", "Null XPM compiled data");
	load_stat = InvalidData;
      }
    }
    
    else if (read_from_file) {
      UAppli *appli = disp->getAppli();
      fpath = appli->makeImaPath(name);

      if (!fpath || !*fpath) {
	//warning("realize", "Null or empty filename"); //file not found a terme
	load_stat = InvalidFile;
      }
      else {
	filename_or_data = fpath;
	const char *ext = null;  //extension

	if (name && name[0]) ext = u_strext(name);  //trouver le format
	if (!ext) ext = "";

	if (strcasecmp(ext, "gif")==0)    // GIF Image assumed
	  reader = UGifFileReader;
	else if (strcasecmp(ext, "xpm")==0)  // XPM Pixmap assumed	
	  reader = UXpmFileReader;
#if KKK
	else if (strcasecmp(ext, "jpg")==0)     // JPG Image assumed
	  reader = UJpgFileReader;
	else                                // X Bitmap assumed
	  reader = UXbmFileReader;
#endif
	else {
	  load_stat = InvalidFormat;
	  //warning("realize", "Could not open image '%s': invalid name or unsupported format", fpath);
	}
      }
    }

    if (reader) {
      //const UColor *_fgcol, const UColor *_bgcol, 

      UNatDisp *nd = disp->getNatDisp();
      /*
	Pixel bgpixel = 0;
	if (!_bgcol || _bgcol->equals(UBgcolor::none)) transp_bg = true;
	else {
	  transp_bg = false;   // will use the specified bgcolor
	  bgpixel = nd->getXPixel(_bgcol);
        }
      */
      // null and none colors ==> the background will be transparent
      u_bool transp_bg; 
      Pixel bgpixel = 1;
      if (!bgcolor || bgcolor->equals(UBgcolor::none)) transp_bg = true;
      else {
	transp_bg = false;   // will use the specified bgcolor
	bgpixel = nd->getXPixel(bgcolor);
      }

      //Pixel fgpixel = nd->getXPixel(_fgcol);
      Pixel fgpixel = 0;

      UX_Image ima = null, shapeima = null;  //!!!devrait pas etre la!!!
      u_dim  width = 0, height = 0;

      load_stat = (*reader)(nd, filename_or_data,
			    fgpixel, bgpixel, transp_bg,
			    ima, shapeima, width, height);

      if (load_stat >= 0) natimas = new UNatIma(nd, ima, shapeima);
    }
    if (fpath) free(fpath);
  }
  //return natimas ? true : false;
  return load_stat;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

// a wrapping for using some of the great FVWM pixmaps in the Ubit toolkit
// -- these icons were originally designed by Ludvig A. Norin
// -- other icons are in file uufvwmpix.cc
// -- please refer to this file for details

const char *fvwm_mini_folder_xpm[] = {
"16 16 6 1",
"       c None s None",
".      c #808080",
"X      c #c0c0c0",
"o      c #ffff00",
"O      c black",
"#      c white",
"                ",
"  .....         ",
" .XoXoX.        ",
".XoXoXoX......  ",
".############.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
".#XoXoXoXoXoX.O ",
".#oXoXoXoXoXo.O ",
"..............O ",
" OOOOOOOOOOOOOO ",
"                ",
"                "};

const char *fvwm_mini_ofolder_xpm[] = {
/* width height num_colors chars_per_pixel */
"    16    16       6            1",
/* colors */
"  c None s None",
". c black",
"# c #808080",
"g c white",
"h c #c0c0c0",
"j c yellow",
/* pixels */
"                ",
"   #####        ",
"  #ggggg#       ",
" #ghjhjhg###### ",
" #gjhjhjhggggg#.",
" #ghjhjhjhjhjh#.",
"#############j#.",
"#gggggggggg#.h#.",
"#ghjhjhjhjhj.##.",
" #ghjhjhjhjh#.#.",
" #gjhjhjhjhjh.#.",
"  #gjhjhjhjhj#..",
"  ############..",
"   .............",
"                ",
"                ",
};


const char * fvwm_mini_doc_xpm[] = {
"16 16 4 1",
"       c None s None",
".      c black",
"X      c white",
"o      c #808080",
"                ",
"   .......      ",
"   .XXXXX..     ",
"   .XoooX.X.    ",
"   .XXXXX....   ",
"   .XooooXoo.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   .XooooooX.o  ",
"   .XXXXXXXX.o  ",
"   ..........o  ",
"    oooooooooo  ",
"                "};

const char *fvwm_mini_diskette_xpm[] = {
/* width height ncolors cpp [x_hot y_hot] */
"16 16 7 1 0 0",
/* colors */
"       c #000080",
".      s iconColor1    m black c black",
"X      s none  m none  c none",
"o      s iconColor6    m white c yellow",
"O      s iconGray1     m white c #e0e0e0",
"+      s iconColor2    m white c white",
"@      c #C0C0C0",
/* pixels */
"  ...........  X",
"  ooooooooooo  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"  OOOOOOOOOOO  .",
"  +++++++++++  .",
"               .",
"    @@@@@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"    @  @@@@    .",
"X   @@@@@@@    .",
"XX.............."};


const char *fvwm_mini_cross_xpm[] = {
"16 16 4 1",
"       c None s None",
".      c red",
"X      c #808080",
"o      c black",
"                ",
"                ",
"           .    ",
"   ..X    ...   ",
"    ..X  .....  ",
"     .......oo  ",
"      .....o    ",
"      ....o     ",
"     ......     ",
"     ..o ...    ",
"    ..o   ...   ",
"    .o     ..X  ",
"   .o       .o  ",
"   o         o  ",
"                ",
"                "};


//::::::::::::::
//mini_rball.xpm
//::::::::::::::

const char *fvwm_mini_rball_xpm[] = {
"10 10 4 1",
"       c None s None",
".      c red",
"X      c white",
"o      c gray50",
"          ",
"          ",
"   ...    ",
"  .XX..   ",
"  .XX..o  ",
"  .....o  ",
"   ...oo  ",
"    ooo   ",
"          ",
"          "};

// *** = predefined Pixmaps with WHITE background

UPix UPix::Wfolder(fvwm_mini_folder_xpm, UBgcolor::white, UMode::UCONST);
UPix UPix::Wofolder(fvwm_mini_ofolder_xpm, UBgcolor::white, UMode::UCONST);
UPix UPix::Wdoc(fvwm_mini_doc_xpm, UBgcolor::white, UMode::UCONST);

// *** = predefined Pixmaps with TRANSPARENT background

UPix UPix::folder(fvwm_mini_folder_xpm, UMode::UCONST);
UPix UPix::ofolder(fvwm_mini_ofolder_xpm, UMode::UCONST);
UPix UPix::doc(fvwm_mini_doc_xpm, UMode::UCONST);
UPix UPix::rball(fvwm_mini_rball_xpm, UMode::UCONST);
UPix UPix::diskette(fvwm_mini_diskette_xpm, UMode::UCONST);
UPix UPix::cross(fvwm_mini_cross_xpm, UMode::UCONST);

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
