/* ==================================================== ======== ======= *
 *
 *  ufont.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ufont_hh
#define	_ufont_hh
//pragma ident	"@(#)ufont.hh	ubit:b1.11.0"

#include <uprop.hh>

//shortcuts (see notes below):
class UFont& ufont(const UFont&);


/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UFont : public UProp {
  friend class UDisp;
  friend class UNatDisp;
  friend class UFontDesc;
protected:
  const class UFontFamily *family;
  short lsize;			 // logical font size of this UFont
  short onStyles, offStyles;	 // activated and disactivated styles
public:
  static const int FONTMAP_SIZE;  // internals
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  // minimum, medium and maximum LOGICAL size (not a point size!)
  static const int MIN_LSIZE;    // = 1
  static const int MEDIUM_LSIZE; // = 7
  static const int MAX_LSIZE;	 // = 15

  struct FontSize {
    int logsize;	// logical size
    int ptsize;		// point size
    const char *natspec;// native point spec
  };

  enum FontStyle {
    BOLD	= 1<<0,	// DON'T CHANGE VALUE!
    ITALIC	= 1<<1,	// DON'T CHANGE VALUE!
    UNDERLINE	= 1<<2,
    FILL        = 1<<3
  };

  // IMPORTANT NOTE:
  // ** UFont Mofifiers can be combined (when applicable)
  // Example:
  //    UFont::bold + UFont::italic + "abcd"  --> displayed in bold italic

  // Style Modifiers:
  // -- 'normal' means not bold, not italic, not underlined
  static UFont normal, bold, italic, underline;

  // -- prefix '_' means NOT: example:  '_bold' means 'not bold'
  static UFont _bold, _italic, _underline;

  // Size Modifiers:
  // -- medium <==> logical lsize == MEDIUM_LSIZE
  static UFont xx_small, x_small, small, medium, large, x_large, xx_large;

  // FontFamily Modifiers:
  static UFont helvetica, courier, times, fixed, symbol;

  // -- 'standard' is the default FontFamily with medium size
  // -- 'inherit' means that the font is inherited from parents
  static UFont standard, inherit;

  // Creates a new UFont:
  UFont(const class UFontFamily& family, 
	// integer bitmask that specifes the font styles (styles are activated 
	// if value is positive and inactivated if value is negative)
	int style = 0,
	// LOGICAL size (not the point size) must be:
	// -- in interval [MIN_LSIZE, MAX_LSIZE]
	// -- or 0 in which case the actual size is inherited from parents
	int lsize = 0,
	u_modes = 0);

  // Creates a new UFont that is a copy of a pre-existing UFont
  UFont(const UFont& = UFont::standard, u_modes = 0);

  void set(const UFont&, u_bool update = true);
  void set(const UFont*, u_bool update = true);

  // merge only sets NON default values (and combines styles)
  void merge(const UFont&);
  void merge(const UFont*);

  u_bool equals(const UFont&) const;
  u_bool equals(const UFont*) const;

  u_bool isBold() const;
  u_bool isItalic() const;

  const class UFontFamily* getFamily() const;
  void setFamily(const class UFontFamily&, u_bool update);
  void setFamily(const UFont &f, u_bool update);


  // integer bitmask that specifes the font styles (styles are activated 
  // if value is positive and inactivated if value is negative)
  // 0 means 'no specific style'
  // note that this function resets existing modes
  void setStyle(int styles, u_bool update);
  void setStyle(const UFont &f, u_bool update);


  // Lsize = LOGICAL size: from UFont::MIN_LSIZE to UFont::MAX_LSIZE
  // Ptsize = POINT size
  int getLsize() const;
  int getPtsize() const;

  void setSize(const UFont &f, u_bool update);
  void setLsize(int logical_size, u_bool update);
  void setPtsize(int point_size, u_bool update);


  // update parents' graphics
  virtual void update();

  // allocates the physical resources of the UFont. This operation is
  // automatically performed the first time the font is used. Thus, this 
  // function should only be called for allocating resources in advance.
  // (note that the appli. can be given as an argument)
  u_bool realize(class UDisp*);

  //==== Ubit Intrinsics ====
  void set(const UFontDesc*, u_bool update);
  virtual void putProp(class UContext*, class UCtrl*);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UFontFamily represents a collection of fonts for all possible sizes
// and styles (however, actual fonts are loaded dynamically when necessary)

class UFontFamily {
  friend class UDisp;
  friend class UNatDisp;
  friend class UFont;
public:
  // Predefined font families:
  // ** 'standard' is the default font family. 'standard' must always
  //    be physically available and it is preloaded for all styles.
  // ** 'any' represents any font family. It is used in UFont descriptions
  //    when changing a style or a size without changing the font family.
  //
  static UFontFamily standard, any, helvetica, times, courier, fixed, symbol;

  // Creates a new FontFamily from a native system font name
  // !Warnings:
  // -- the strings must NOT be freed nor changed by the client!
  // -- this API may change in future releases

  UFontFamily(const char *family_name,
	      const char *native_family_name_specs,
	      const char *normal_style_specs,
	      const char *bold_style_specs, 
	      const char *italic_style_specs,
	      // this vector must be MAX_LSIZE long (or null -> take default)
	      const UFont::FontSize *font_sizes = null);

  // simplified API: assumes default options
  UFontFamily(const char *fname, const char *native_name);

  // thes functions can be used for converting point sizes to logical sizes
  // and vice versa
  int lsizeToPtsize(int logical_size) const;
  int ptsizeToLsize(int point_size) const;

  // returns an array of MAX_LSIZE
  static const UFont::FontSize* getDefaultFontSizes();

  // get default scale factor related to medium standard size
  static float getDefaultFontScale(int logical_size);

protected:
  static const UFont::FontSize *DEFAULT_FONT_SIZES;  // array of MAX_LSIZE
  static int familyCount;
  int  ix;
  const char *name;
  const char *natName, *normalStyle, *boldStyle, *italicStyle;
  const UFont::FontSize *fontSizes;   // array of MAX_LSIZE
  UFontFamily(const char *name, UFontFamily&);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
//==== Ubit Intrinsics ====
// Internal representation: do not use directly

struct UFontDesc {
  const class UFontFamily *family;
  short font_ix, lsize, styles;

  void set(const class UFont*);
  void set(const class UFont*, int delta_scale);
  // merge only sets NON default values (and combines styles)
  void merge(const class UFont*, int delta_scale);
  // note: decrements if the value is negative
  void incrScale(int delta_scale);
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */

