/* ==================================================== ======== ======= *
 *
 *  ucall.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ucall_hh
#define	_ucall_hh
//pragma ident	"@(#)ucall.hh	ubit:b1.11.1"

/* MANUAL: how to add callback functions and methods

 **** 1. You can add callbacks to UGroup/UBox subclasses as follows:
 *
 *       ubox( .... + UOn::xxx / ucall(...) + ...)
 * or:
 *       UBox *box = &ubox(...);
 *       box->add( UOn::xxx / ucall(....) );
 * where:
 * --    UOn::xxx is an Event Condition such as:
 *       UOn::action, UOn::select, UOn::mpress ... (see file uctrl.hh)


 **** 2. Callbacks can be object METHODS or ordinary FUNCTIONS:
 *       -- callback METHODS can have 0 to 3 arguments 
 *       -- callback FUNCTIONS can have 1 to 3 arguments 
 *       -- callback functions and methods can also have an optional
 *          UEvent* argument (that must be the first argument)
 *
 * example 1:  UOn::mpress / ucall(foo, arg1, arg2)
 *       could call a FUNCTION defined as:
 *             void foo(TYPE1 arg1, TYPE2 arg2)  {...}
 *       or:   void foo(UEvent *event, TYPE1 arg1, TYPE2 arg2)  {...}

 * example 2:  UOn::select / ucall(obj, &MyClass::foo, arg1)
 *       could call a METHOD defined as:
 *             void MyClass::foo(TYPE1 arg1)  {...}
 *       or:   void MyClass::foo(UEvent *, TYPE1 arg1)  {...}
 *       with 'obj' being an object POINTER of class MyClass


 **** 3. the ucloseWin() predefined callback closes the window 
 *       (Menu or Dialog) that contains this object
 *
 * example: clicking on the "Close' button will close the dialog       
 *             udialog( ubutton( "Close" + UOn::action / ucloseWin() )
 * note:
 *       ucloseWin() closes any containing UWin (= any UDialog or UMenu)
 *       while ucloseDialog() only closes UDialogs and ucloseMenu() UMenus


 **** 4. the uset(a,b) predefined callback performs:  a->set(b)
 * example:  
 *             ubutton( "Do a = b" + UOn::action / uset(a, b) )
 * note:
 *       'a' and 'b' must be pointers and 'a' must have a 'set()'
 *       method that can take 'b' as an argument


 **** 5. Remarks:
 * -- ordinary FUNCTIONS must have (at least) ONE argument
 *    while object METHODS may have NO argument
 *
 * -- be VERY careful with argument types as they must EXACTLY match 
 *    (no implicit type conversion apply in templates)
 *
 * -- templates work well with object POINTERS. However (depending
 *    on which compiler you use), you may have compilation problems 
 *    with C++ references.
 */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UCall: public UBrick {
protected:
  const UCond *cond;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  virtual class UCall*  callCast() {return this;}

  UCall();
  UCall(UCond*);
  UCall(UCond&);
  virtual ~UCall() {clean();}
  //virtual void update() {}

  //package_private: ====[internal implementation]======================

  //removingFrom => destructor!
  virtual void addingTo(ULink *selflink, UGroup *parent);
  virtual void removingFrom(ULink *selflink, UGroup *parent);

  virtual void call(UEvent*) = 0; //abstract
  const UCond* getCond() {return cond;}
  void setCond(const UCond &c) {cond = &c;}
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// ucloseWin()    will close the Window that encloses this object
// ucloseDialog() will only close UDialog objects
// ucloseMenu()   will only close UMenu objects

UCall& ucloseWin();
UCall& ucloseDialog();
UCall& ucloseMenu();

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// uset(a,b) will call 'a->set(b)'
// 'a' must have a set() method that can take 'b' as an argument

template <class CC, class VAL> 
class UCall_set : public UCall{
  CC  obj;
  VAL val;
public:
  UCall_set(CC _obj, VAL _val) 
    : UCall() {obj = _obj; val = _val;}
  void call(UEvent*) {obj->set(val);}
};

template <class CC, class VAL> 
UCall& uset(CC obj, VAL val) {
  return *(new UCall_set<CC,VAL> (obj, val));
}

/*
template <class CC, class VAL> 
class UCall_set : public UCall{
  CC  *obj;
  VAL val;
public:
  UCall_set(CC *_obj, VAL _val) : UCall() {
    obj = _obj; val = _val;
  }
  void call(UEvent*) {obj->set(val);}
};

template <class CC, class VAL> 
UCall& uset(CC *obj, VAL val) {
  return *(new UCall_set<CC,VAL> (obj, val));
}
*/
// ==================================================== [Elc:01] ======= 
// ==================================================== ======== =======
// Callback methods WITHOUT an argument:
// method:   void MyClass::foo()
// or:       void MyClass::foo(UEvent*)
// call:     UOn:xxx / ucall(object, &MyClass::foo)

template <class CC,class MM> 
class UCall_method0 : public UCall{
  CC obj;
  void (MM::*member)();
public:
  UCall_method0(CC object, void (MM::*m)()) 
    : UCall(){obj = object; member = m;}
  void call(UEvent *e) {(obj->*member)();}
};

template <class CC, class MM> 
class UCall_method0_ev : public UCall {
  CC obj;
  void (MM::*member)(UEvent*);
public:
  UCall_method0_ev(CC object, void (MM::*m)(UEvent*)) 
    : UCall() {obj = object; member = m;}
  void call(UEvent *e) {(obj->*member)(e);}
};

// ucall(object, &MyClass::foo)  for:  void MyClass::foo()

template <class CC, class MM> 
UCall& ucall(CC obj, void (MM::*method)()) {
  return *(new UCall_method0<CC,MM>(obj, method));
}

// ucall(object, &MyClass::foo)  for:  void MyClass::foo(UEvent*)

template <class CC, class MM> 
UCall& ucall(CC obj, void (MM::*method)(UEvent*)) {
  return *(new UCall_method0_ev<CC,MM>(obj, method));
}

// ==================================================== [Elc:01] =======
// ==================================================== ======== =======
// Callback methods with ONE argument:
// method:  void MyClass::foo(ARG1 arg1)
// or:      void MyClass::foo(UEvent*, ARG1 arg1)
// call:    UOn:xxx / ucall(object, &MyClass::foo, arg1)

template <class CC, class MM, class ARG1> 
class UCall_method1 : public UCall {
  CC obj;
  void (MM::*member)(ARG1);
  ARG1 arg1;
public:
  UCall_method1(CC object, void (MM::*m)(ARG1), ARG1 a1) 
    : UCall() {obj = object; member = m; arg1 = a1;}
  void call(UEvent*) {(obj->*member)(arg1);}
};

template <class CC, class MM, class ARG1> 
class UCall_method1_ev : public UCall {
  CC obj;
  void (MM::*member)(UEvent*, ARG1);
  ARG1 arg1;
public:
  UCall_method1_ev(CC object, void (MM::*m)(UEvent*,ARG1), ARG1 a1) 
    : UCall() {obj = object; member = m; arg1 = a1;}
  void call(UEvent *e) {(obj->*member)(e, arg1);}
};

// ucall(obj, &MyClass::foo, arg1)  for:  void MyClass::foo(ARG1)

template <class CC, class MM, class ARG1> 
UCall& ucall(CC obj, void (MM::*method)(ARG1), ARG1 a1) {
  return *(new UCall_method1<CC,MM,ARG1> (obj, method, a1));
}

// ucall(obj, &MyClass::foo, arg1)  for:  void MyClass::foo(UEvent*, ARG1)

template <class CC, class MM, class ARG1> 
UCall& ucall(CC obj, void (MM::*method)(UEvent*,ARG1), ARG1 a1) {
  return *(new UCall_method1_ev<CC,MM,ARG1> (obj, method, a1));
}

// ==================================================== [Elc:01] ======= 
// ==================================================== ======== ======= 
// Callback Methods with TWO arguments
// method:  void MyClass::foo(ARG1 arg1, ARG2 arg2)
// or:      void MyClass::foo(UEvent*, ARG1 arg1, ARG2 arg2)
// call:    UOn:xxx / ucall(object, &MyClass::foo, arg1, arg2)

template <class CC, class MM, class ARG1, class ARG2>
class UCall_method2 : public UCall {
  CC obj;
  void (MM::*member)(ARG1, ARG2);
  ARG1 arg1;
  ARG2 arg2;
public:
  UCall_method2(CC object, void (MM::*m)(ARG1, ARG2), ARG1 a1, ARG2 a2)
    : UCall() {obj = object; member = m; arg1 = a1; arg2 = a2;}
  void call(UEvent *e) {(obj->*member)(arg1, arg2);}
};

template <class CC, class MM, class ARG1, class ARG2> 
class UCall_method2_ev : public UCall {
  CC obj;
  void (MM::*member)(UEvent*, ARG1, ARG2);
  ARG1 arg1;
  ARG2 arg2;
public:
  UCall_method2_ev(CC object, void (MM::*m)(UEvent*,ARG1,ARG2), ARG1 a1, ARG2 a2)
    : UCall() {obj = object; member = m; arg1 = a1; arg2 = a2;}
  void call(UEvent *e) {(obj->*member)(e, arg1, arg2);}
};

// ucall(object, &MyClass::foo, arg1, arg2) for: void MyClass::foo(ARG1, ARG2)

template <class CC, class MM, class ARG1, class ARG2> 
UCall& ucall(CC obj, void (MM::*method)(ARG1, ARG2), ARG1 a1, ARG2 a2) {
  return *(new UCall_method2<CC,MM,ARG1,ARG2>(obj, method, a1, a2));
}

// ucall(object, &MyClass::foo, arg1, arg2) for: void MyClass::foo(UEvent*, ARG1, ARG2)

template <class CC, class MM, class ARG1, class ARG2> 
UCall& ucall(CC obj, void (MM::*method)(UEvent*,ARG1,ARG2), ARG1 a1, ARG2 a2){
  return *(new UCall_method2_ev<CC,MM,ARG1,ARG2>(obj, method, a1, a2));
}

// ==================================================== [Elc:01] =======
// ==================================================== ======== =======
// Callback methods with THREE arguments
// method:  void MyClass::foo(ARG1 arg1, ARG2 arg2, ARG3 arg3)
// or:      void MyClass::foo(UEvent*, ARG1 arg1, ARG2 arg2, ARG3 arg3)
// call:    UOn:cond / ucall(object, &MyClass::foo, arg1, arg2, arg3)

template <class CC, class MM, class ARG1, class ARG2, class ARG3>
class UCall_method3 : public UCall {
  CC obj;
  void (MM::*member)(ARG1, ARG2, ARG3);
  ARG1 arg1;
  ARG2 arg2;
  ARG3 arg3;
public:
  UCall_method3(CC object, void (MM::*m)(ARG1,ARG2,ARG3), 
		ARG1 a1, ARG2 a2, ARG3 a3)
 : UCall() {obj = object; member = m; arg1 = a1; arg2 = a2; arg3 = a3;}
  void call(UEvent *e) {(obj->*member)(arg1, arg2, arg3);}
};

template <class CC, class MM, class ARG1, class ARG2, class ARG3> 
class UCall_method3_ev : public UCall {
  CC obj;
  void (MM::*member)(UEvent*, ARG1, ARG2, ARG3);
  ARG1 arg1;
  ARG2 arg2;
  ARG3 arg3;
public:
  UCall_method3_ev(CC object, void (MM::*m)(UEvent*,ARG1,ARG2,ARG3), 
		  ARG1 a1, ARG2 a2, ARG3 a3)
    : UCall() {obj = object; member = m; arg1 = a1; arg2 = a2; arg3 = a3;}
  void call(UEvent *e) {(obj->*member)(e, arg1, arg2, arg3);}
};


// templates for: ucall(object, &MyClass::foo, arg1, arg2, arg3)

template <class CC, class MM, class ARG1, class ARG2, class ARG3> 
UCall& ucall(CC obj, void (MM::*method)(ARG1,ARG2,ARG3), 
	     ARG1 a1, ARG2 a2, ARG3 a3) {
  return *(new UCall_method3<CC,MM,ARG1,ARG2,ARG3>(obj,method,a1,a2,a3));
}

template <class CC, class MM, class ARG1, class ARG2, class ARG3> 
UCall& ucall(CC obj, void (MM::*method)(UEvent*,ARG1,ARG2,ARG3), 
	     ARG1 a1, ARG2 a2, ARG3 a3) {
  return *(new UCall_method3_ev<CC,MM,ARG1,ARG2,ARG3>(obj,method,a1,a2,a3));
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* Callback Function with ONE argument
 *       void foo(ARG1 a1)
 *  or:  void foo(UEvent*, ARG1 a1)
 *  --> UOn:cond / ucall(foo, a1)
 */

template <class ARG1> 
class UCall_fun1 : public UCall {
  void (*fun)(ARG1);
  ARG1 arg1;
public:
  UCall_fun1(void (*function)(ARG1), ARG1 a1) : UCall() {
    fun = function; arg1 = a1;
  }
  void call(UEvent *e) {(*fun)(arg1);}
};

template <class ARG1> 
class UCall_fun1_ev : public UCall {
  void (*fun)(UEvent*, ARG1);
  ARG1 arg1;
public:
  UCall_fun1_ev(void (*function)(UEvent*,ARG1), ARG1 a1) : UCall() {
    fun = function; arg1 = a1;
  }
  void call(UEvent *e) {(*fun)(e, arg1);}
};

// templates for: ucall(foo, arg1)

template <class ARG1> 
UCall& ucall(void (*function)(ARG1), ARG1 a1) {
  return *(new UCall_fun1<ARG1> (function, a1));
}

template <class ARG1> 
UCall& ucall(void (*function)(UEvent*,ARG1), ARG1 a1) {
  return *(new UCall_fun1_ev<ARG1> (function, a1));
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* Callback Function with TWO arguments
 *       void foo(ARG1 a1, ARG2 a2)
 *  or:  void foo(UEvent*, ARG1 a1, ARG2 a2)
 *  --> UOn:cond / ucall(foo, a1, a2)
 */

template <class ARG1, class ARG2> 
class UCall_fun2 : public UCall{
  void (*fun)(ARG1, ARG2);
  ARG1 arg1;
  ARG2 arg2;
public:
  UCall_fun2(void (*f)(ARG1,ARG2), ARG1 a1, ARG2 a2) : UCall(){
    fun = f; arg1 = a1; arg2 = a2;
  }
  void call(UEvent *e) {(*fun)(arg1, arg2);}
};

template <class ARG1, class ARG2> 
class UCall_fun2_ev : public UCall{
  void (*fun)(UEvent*, ARG1, ARG2);
  ARG1 arg1;
  ARG2 arg2;
public:
  UCall_fun2_ev(void (*f)(UEvent*,ARG1,ARG2), ARG1 a1, ARG2 a2) : UCall(){
    fun = f; arg1 = a1; arg2 = a2;
  }
  void call(UEvent *e) {(*fun)(e, arg1, arg2);}
};

// templates for: ucall(foo, arg1, arg2)

template <class ARG1, class ARG2> 
UCall& ucall(void (*function)(ARG1,ARG2), ARG1 a1, ARG2 a2) {
  return *(new UCall_fun2<ARG1,ARG2> (function, a1, a2));
}

template <class ARG1, class ARG2> 
UCall& ucall( void (*function)(UEvent*,ARG1,ARG2), ARG1 a1, ARG2 a2) {
  return *(new UCall_fun2_ev<ARG1,ARG2> (function, a1, a2));
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* Callback Function with THREE arguments
 *       void foo(ARG1 a1, ARG2 a2, ARG3 a3)
 *  or:  void foo(UEvent*, ARG1 a1, ARG2 a2, ARG3 a3)
 *  --> UOn:cond / ucall(foo, a1, a2, a3)
 */

template <class ARG1, class ARG2, class ARG3> 
class UCall_fun3 : public UCall{
  void (*fun)(ARG1, ARG2, ARG3);
  ARG1 arg1;
  ARG2 arg2;
  ARG3 arg3;
public:
  UCall_fun3(void (*f)(ARG1,ARG2,ARG3), 
	     ARG1 a1, ARG2 a2, ARG3 a3) : UCall(){
    fun = f; arg1 = a1; arg2 = a2; arg3 = a3;
  }
  void call(UEvent *e) {(*fun)(arg1, arg2, arg3);}
};

template <class ARG1, class ARG2, class ARG3> 
class UCall_fun3_ev : public UCall{
  void (*fun)(UEvent*, ARG1, ARG2, ARG3);
  ARG1 arg1;
  ARG2 arg2;
  ARG3 arg3;
public:
  UCall_fun3_ev(void (*f)(UEvent*,ARG1,ARG2,ARG3), 
		ARG1 a1, ARG2 a2, ARG3 a3) : UCall(){
    fun = f; arg1 = a1; arg2 = a2; arg3 = a3;
  }
  void call(UEvent *e) {(*fun)(e, arg1, arg2, arg3);}
};

// templates for: ucall(foo, arg1, arg2, arg3)

template <class ARG1, class ARG2, class ARG3> 
UCall& ucall(void (*function)(ARG1,ARG2,ARG3), 
	     ARG1 a1, ARG2 a2, ARG3 a3) {
  return *(new UCall_fun3<ARG1,ARG2,ARG3> (function, a1, a2, a3));
}

template <class ARG1, class ARG2, class ARG3> 
UCall& ucall( void (*function)(UEvent*,ARG1,ARG2,ARG3), 
	      ARG1 a1, ARG2 a2, ARG3 a3) {
  return *(new UCall_fun3_ev<ARG1,ARG2,ARG3> (function, a1, a2, a3));
}

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
