#ifndef VHDLKERNEL_CC
#define VHDLKERNEL_CC
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
// $Id: VHDLKernel.cc,v 1.180 1999/10/28 04:32:31 mal Exp $
//
//---------------------------------------------------------------------------

#include "VHDLKernel.hh"
#include "FileType.hh"
#include "SignalBase.hh"
#include "Block.hh"

VHDLKernel::VHDLKernel() : markedQ(this) {
  postponed = false;
}

VHDLKernel::~VHDLKernel() {
  outputGcollect(PINFINITY);
  name = NULL;
}

void
VHDLKernel::initialize() {
  ((VHDLKernel_state *) state->current)->initState(proc, this);
  // set GUARD signals to result of their expression
  if (getGuard() != NULL) {
    updateGuard(getGuard());
  }
  executeVHDL();
}

void
VHDLKernel::finalize() {
  ((VHDLKernel_state *) state->current)->cleanState();
}

void
VHDLKernel::executeProcess() {
  scalarSignalsUpdateArray.reset();
  compositeResolvedSignalsUpdateArray.reset();

  eventsToCancel.reset();
  eventsInCurrentCycle.reset();

  VHDLEvent* event = NULL;
  
  //Collect all the events to be processed in this
  //simulation time cycle
  //Define an expandable array to hold the events in the current
  //simulation cycle

  do {
    //get the elements from the inputQ 
    //add them to the expandable array of pointers
    event = (VHDLEvent*)getEvent();
    
    //Check if the event is a Cancellation Event 
    //that has been sent during the Marking process
    if(event->transactionType == CANCELTRANSACTION ) { //CANCELTRANSACTION
      eventsToCancel.addElement(event);
    }
    else {
      eventsInCurrentCycle.addElement(event);
    }

  } while ((!postponed && inputQ.peekEvent() != NULL && 
	    inputQ.peekEvent()->recvTime == getTimeNow() && 
	    inputQ.peekEvent()->dest == id) ||
	    (postponed && inputQ.peekEvent() != NULL &&
	     inputQ.peekEvent()->recvTime.time == getTimeNow().time &&
	     inputQ.peekEvent()->dest == id));
  // Only test time, not time & delta, for a postponed process.

  //CANCEL out the messages by looking at the events to be processed
  //in this cycle and the remove them
  int ii = 0;
  int jj = 0;

  for(ii=0; ii < eventsToCancel.getNumberOfElements(); ii++) {
    int matchFound = 0;
    SigEvent* toCancel = (SigEvent*) eventsToCancel.getElement(ii);
    for(jj=0; jj < eventsInCurrentCycle.getNumberOfElements(); jj++) {
      SigEvent* postedEvent = (SigEvent*) eventsInCurrentCycle.getElement(jj);
      if((postedEvent->sigId == toCancel->sigId) &&
	 (postedEvent->sigsrc == toCancel->sigsrc) &&
	 (postedEvent->rejTime == toCancel->rejTime) &&
	 (postedEvent->sender == toCancel->sender) &&
	 (memcmp(((postedEvent)+1), ((toCancel)+1), 
		 toCancel->size-sizeof(SigEvent)) == 0)) {
	matchFound = 1;
	break;
      }
      else {
	continue;
      }
    }
    if(matchFound == 1) {
      //jj has the index of the element to be removed
      eventsInCurrentCycle.removeElement(jj);
    }
    else {
      cerr << name << "Error: can't find Transaction to cancel after marking" << endl;
      cerr << "Proceeding with the simulatio" << endl;
      cerr << "Simulation may not be functionally correct" << endl;
    }
  }

  // process all events at the current time
  for(ii=0; ii < eventsInCurrentCycle.getNumberOfElements(); ii++) {
    event = (VHDLEvent*)eventsInCurrentCycle.getElement(ii); // updates lVT
    if (event != NULL) {
      switch (event->type) {
	SigEvent* sEvent;
      case WAIT:
	updateWait((WaitEvent*)event);
	break;
      case EXPLICIT:
      case G_BUS:
	sEvent = (SigEvent*)event;
	updateDriver(sEvent);
	break;
      case NULL_TRANSACTION:
	sEvent = (SigEvent*)event;
	disconnectDriver(sEvent);
	break;
      default:
	cerr << "Error!  Unknown VHDL event received!" << endl;
	cerr << "event->type is " << event->type << " (" << (int)(event->type)
	     << ")" << endl;
	abort();
	break;
      }

    }
    else {
      return;
    }
  }
#ifdef SEQUENTIAL
  outputGcollect(VTime(getLVT().time, 0));
#endif
    
  executeSignalUpdationPhase();
  executeVHDL();
  eventSigs.reset();
}


#ifndef SEQUENTIAL

void
VHDLKernel::outputGcollect(VTime gtime, Container<BasicEvent>*& outputQptr) {
  SimulationObj::outputGcollect(gtime, outputQptr);
#ifdef VHDLDBG
  cout << name << ": gcollecting up to " << gtime
       << ", markedQ.length before = " << markedQ.size() << endl;
#endif
  markedQ.gcollect(gtime);
#ifdef VHDLDBG
  cout << name << ": markedQ.length after = " << markedQ.size() << endl;
#endif
}

#else

void
VHDLKernel::outputGcollect(VTime gtime){
#ifdef VHDLDBG
  cout << name << ": gcollecting up to " << gtime
       << ", markedQ.length before = " << markedQ.size() << endl;
#endif
  markedQ.gcollect(gtime);
  
  
#ifdef VHDLDBG
  cout << name << ": markedQ.length after = " << markedQ.size() << endl;
#endif
  SimulationObj::outputGcollect(gtime);
}

#endif

#ifndef SEQUENTIAL
void
VHDLKernel::rollback(VTime rollbackTime) {

  // A postponed process needs all messages of the current main cycle


  if (postponed == true) {
    rollbackTime.delta = 1;
#ifdef VHDLDBG
    cout << name << ": rollback for postponed process to "
	 << rollbackTime << endl;
#endif
  }
  
#ifdef VHDLDBG
  cout << name << ": rollback--rolling back from " << getTimeNow() 
       << " to " << rollbackTime  << "--markedQ is:\n";
  markedQ.print(cout);
#endif
  SimulationObj::rollback(rollbackTime);
  VTime tempTime(getTimeNow());
  markedQ.rollbackTo(tempTime);
  
#ifdef VHDLDBG
  cout << name << ": rollbackTo finished--lVT is now " << getTimeNow() 
       << "--markedQ is:\n";
  markedQ.print(cout);
#endif
}

#endif

void
VHDLKernel::cancelSingleMessage(BasicEvent* toCancel) {

  SigEvent *cancelEvent = (SigEvent*)toCancel;
  int size = cancelEvent->size;

  SigEvent* negEvent = (SigEvent*) new char[size];
  new(negEvent) SigEvent();
  *negEvent = *((SigEvent*)cancelEvent);
  memcpy((char*)(negEvent + 1), (char*)(cancelEvent +1), size - sizeof(SigEvent));

  negEvent->size = size;
  negEvent->transactionType = CANCELTRANSACTION; //CANCELTRANSACTION is 1
  sendEventUnconditionally(negEvent);

}

#ifdef SEQUENTIAL

void
VHDLKernel::sendEventUnconditionally(BasicEvent* toSend) {
  sendEvent(toSend);
}

#endif

void
VHDLKernel::sendNegEvent(BasicEvent* toCancel) {
  SigEvent *sEvent, *cancelEvent;
  MarkedEvent *cancelMarkedElement;
// Since we change the currentPos, we have to restore it to a proper value
// at the end.
  Container<MarkedEvent> *savedPos = markedQ.getCurrent();
  
  sEvent = (SigEvent*) toCancel;
  cancelEvent = sEvent;
#ifdef VHDLDBG
  cout << "Sending out negative event:\n\t" << sEvent 
	<< " " << *sEvent << endl;
#endif
  
  // find corresponding event in the markedQ
  cancelEvent = markedQ.find(sEvent->recvTime, EQUAL);
  markedQ.setCurrentToFind();
  while ((cancelEvent != NULL) && 
	 (cancelEvent->recvTime == sEvent->recvTime)) {
    if ((cancelEvent->sendTime == sEvent->sendTime) &&
	(cancelEvent->sigId == sEvent->sigId) &&
	0 == memcmp((cancelEvent+1), (sEvent+1),
		    sEvent->size - sizeof(SigEvent))) {
      break;
    }
    // find event on our signal in set of events at the proper time
    cancelEvent = markedQ.seek(1, CURRENT)->event;
  }
  
  if ((cancelEvent == NULL) || 
      (cancelEvent->sigId !=sEvent->sigId) ||
      (cancelEvent->sendTime != sEvent->sendTime) ||
      (0 != memcmp((cancelEvent+1), (sEvent+1),
		   sEvent->size - sizeof(SigEvent)))) {
    cerr << "cancelEvent: " << cancelEvent->sendTime << " != " 
	 << "sEvent: " << sEvent->sendTime << endl;
    
    cerr << name << ": attempting to cancel a bogus event!\n";
    cerr << "I'm trying to cancel " << sEvent << " " << *sEvent << endl;
    cerr << "which I can't find in my markedQ:\n\n";
    markedQ.print(cerr);
    cerr.flush();
    abort();
  }

  //#ifndef SEQUENTIAL
  SignalBase* cancelSig = ((VHDLKernel_state *) state->current)->locateSig(cancelEvent->sigId);

#ifdef VHDLDBG
  // cancel message I sent it to myself
  cout << name << " cancelling " << cancelEvent << " "
       << *(SigEvent*)cancelEvent << endl;
#endif
  //cancelSingleMessage(cancelEvent);
  
  // Now cancel messages sent for this signal's fanout.
  for (register int i = 0; i < cancelSig->fanout; i++) {
    ASSERT(cancelEvent->sender == id);
    cancelEvent->dest =  cancelSig->fanDest[i].objid;
    ((SigEvent*)cancelEvent)->sigId = cancelSig->fanDest[i].sigid;
    cancelSingleMessage(cancelEvent);
  }
  //#endif
  // If we are deleting the savedPos, SortedList will take care of setting
  // the currentPos to the next element in the list.  If not, we have to
  // restore the currentPos to savedPos.
  if(savedPos == markedQ.getCurrent()) {
    cancelMarkedElement = markedQ.removeCurrent();
  } else {
    cancelMarkedElement = markedQ.removeCurrent();
    markedQ.setCurrent(savedPos);
  }
  
#ifdef SEQUENTIAL  
  cancelMarkedElement->~MarkedEvent();
  delete [] (char*)cancelMarkedElement;
#endif  
}


bool
VHDLKernel::resumeWait(int wId, const EnumerationType& cond) {
  bool resume = false;
  Wait *wait;

  if (wId != ((VHDLKernel_state *) state->current)->waitLabel) {
    cerr << name << " resumeWait: wId (" << wId 
	 << ") not the one I'm waiting for! (" << ((VHDLKernel_state *) state->current)->waitLabel 
	 << ")" << endl;
    exit(-14);
  }

  if (((VHDLKernel_state *) state->current)->waitExpired == true) {
    // timeout has been satisfied
    resume = true;
#ifdef VHDLDBG
    cout << name << ": wait timeout has expired-resuming\n";
#endif
  } else {
    if(wId == WAITING_IN_PROC) {
      wait = ((VHDLKernel_state *) state->current)->procWait;
    } else {
      wait = (((VHDLKernel_state *) state->current)->wait + wId);
    }
    if (wait->sensSize > 0) {
      // wait has a sensitivity list
      for (register int i = 0; i < eventSigs.num; i++) {
	if (wait->sensitiveTo(eventSigs.sig[i])) {
	  // this wait is sensitive to a signal that has had an event this
	  // cycle.  Only resume the wait if there was a hit on the
	  // sensList _and_ the condition clause evaluates to true.
	  resume = (cond == SAVANT_BOOLEAN_TRUE);
#ifdef VHDLDBG
	  cout << name << " had event on sens. sig " << eventSigs.sig[i]
		<< ": condition evals to " << (resume ? "true":"false") <<endl;
#endif
	  break;
	} // if
      }	// for
    } // if(wait->sensSize > 0)
  } // else 

  if (resume == true) {
    ((VHDLKernel_state *) state->current)->waitLabel = NOT_WAITING;
#ifdef VHDLDBG
    cout << name << " resuming wait " << wId << " at " << getTimeNow() <<endl;
  } else {
    cout << name << " not resuming wait " << wId << " at " 
	 << getTimeNow() << endl;
#endif
  }

  return resume;
}


void
VHDLKernel::executeWait(int wId, const PhysicalType& timeout) {
  VTime delay;
  
  // record which wait we're waiting on
  ((VHDLKernel_state *) state->current)->waitLabel = wId;
  
  // This is required in update Wait.
  ((VHDLKernel_state *) state->current)->waitTime = getTimeNow();

  delay = ((const PhysicalType &) timeout).getVTime();

#ifdef VHDLDBG
  cout << name << " executing wait "<< wId << " at time " 
       << getTimeNow() << ", will resume in: " << delay << endl;
#endif

  ((VHDLKernel_state *) state->current)->waitExpired = false;
  if (delay != PINFINITY) {
    // schedule resumption from this wait by sending event to self
    WaitEvent* waitEv = (WaitEvent*) new char[sizeof(WaitEvent)];
    new(waitEv) WaitEvent();
    waitEv->sender = id;
    waitEv->dest = id;
    waitEv->recvTime = getTimeNow() + delay;
    waitEv->size = sizeof(WaitEvent);
    waitEv->type = WAIT;
    waitEv->waitId = wId;
    sendEvent(waitEv);
  }
}


EnumerationType
VHDLKernel::savantendfile(VHDLKernelBase*, int fileidx) {
  return savantendfile_boolean(NULL, fileidx);
}

EnumerationType
VHDLKernel::savantendfile_boolean(VHDLKernelBase*, int fileidx) {
  if (inFileQ[fileidx].access().peek() == EOF) {
#ifdef VHDLDBG
    cout << name << ": EOF reached" << endl;
#endif
    //        return true;
    // return EnumerationType(ObjectBase::VARIABLE, UniversalBoolean(true));
    return SAVANT_BOOLEAN_TRUE;
  }
#ifdef VHDLDBG
  cout << name << ": EOF tested, but not reached" << endl;
#endif
  //  return false;
  // return EnumerationType(ObjectBase::VARIABLE, UniversalBoolean(false));
  return SAVANT_BOOLEAN_FALSE;
}

EnumerationType
VHDLKernel::savantendfile(VHDLKernelBase* base, FileType& file) {
  return savantendfile_boolean(base, file.get_fileHandle());
}

EnumerationType
VHDLKernel::savantendfile_boolean(VHDLKernelBase* base, FileType& file) {
  return savantendfile_boolean(base, file.get_fileHandle());
}

int 
VHDLKernel::savantreadline(VHDLKernelBase*, int fileidx, 
				  AccessVariable<char*>& line) {
  int linesize;
  
  if (line.val != NULL) {  // dispose of old data
    delete [] line.val;
  }

  line.val = getline(inFileQ[fileidx].access(), linesize);
  line.dataLength = linesize;
  line.curPos = 0;

#ifdef VHDLDBG
  cout << name << "(id = " << id << "): line read from file "
       << fileidx << "(VTime = " << getLVT() << "):\n";
  line.print(cout);
  cout << endl;
#endif
  
  return NORMAL_RETURN;
}

int 
VHDLKernel::savantwriteline(VHDLKernelBase*, int fileidx,
				   AccessVariable<char*>& line) {
  ostrstream outbuf;

  FileData *outputLine = new FileData;
  
  for (register int i = 0; i < line.dataLength; i++) {
    outbuf << line.val[i];
  }
  outbuf << "\n" << ends;
  
  outputLine->line = outbuf.str();        // VHDL data structure
  outputLine->length = line.dataLength + 1;
  outputLine->time = ((VHDLKernel_state *) state->current)->lVT;

#ifdef VHDLDBG
  cout << "Writing data to file " << fileidx << ":\n|" << *outputLine << "| "
       << "at VTime = " << getLVT() << endl;
#endif
  if(fileidx == FileQueue::standardOut){
    stdoutFileQ.insert(outputLine);
  }
  else {
    outFileQ[fileidx].insert(outputLine); // insert element into the queue
  }
  line.dataLength = 0;
  line.curPos = 0;
  delete [] line.val;
  line.val = NULL;
  return NORMAL_RETURN;
}

int 
VHDLKernel::savantwriteline(VHDLKernelBase *base, int fileidx,
				   SavantlineType &line) {
  ostrstream outbuf;
  char *outbuf_str;
  int returnValue;
  
  outbuf << line << ends;

  AccessVariable <char *> tempStr;

  outbuf_str = outbuf.str();
  tempStr.val = new char[strlen(outbuf_str) + 1];
  strcpy(tempStr.val, outbuf_str);
  tempStr.dataLength = strlen(tempStr.val);
  
  returnValue = savantwriteline(base, fileidx, tempStr);

  delete line.val;
  line.val = NULL;
  delete [] outbuf_str;
  
  return returnValue;
}

int
VHDLKernel::savantreadline(VHDLKernelBase*base, int fileidx, 
				  SavantlineType& line)
{
  int returnValue;
  AccessVariable<char*> tempStr;
  ostrstream outbuf;
  char *outbuf_str;
  
  if (line.val != NULL) {
    delete line.val;
    line.val = NULL;
  }
  
  if ((returnValue = savantreadline(base, fileidx, tempStr))
      == NORMAL_RETURN) {
    
    for(int length = 0; (length < tempStr.dataLength); length++) {
      outbuf << tempStr.val[length];
    }
    
    outbuf << ends;
    
    outbuf_str = outbuf.str();
    line.val = new SavantstringType(ObjectBase::VARIABLE, outbuf_str);
    delete [] outbuf_str;
  }
  
  return returnValue;
}

int
VHDLKernel::savantreadline(VHDLKernelBase*base, int fileidx, 
			   SavantlineType& line,
			   IntegerType &count)
{
  int returnValue, dataRead = 0;
  AccessVariable<char*> tempStr;
  ostrstream outbuf;
  char *outbuf_str;
  
  if (line.val != NULL) {
    delete line.val;
    line.val = NULL;
  }
  
  if ((returnValue = savantreadline(base, fileidx, tempStr))
      == NORMAL_RETURN) {
    
    for(int length = 0; (length < tempStr.dataLength); length++) {
      outbuf << tempStr.val[length];
      if (tempStr.val[length] == FILE_DELIMITER) {
	dataRead++;
      }
    }
    
    outbuf << ends;
    
    outbuf_str = outbuf.str();
    line.val = new SavantstringType(ObjectBase::VARIABLE, outbuf_str);
    delete [] outbuf_str;
  }
  
  count = IntegerType(ObjectBase::VARIABLE, UniversalInteger(dataRead));
  
  return returnValue;
}

int
VHDLKernel::savantreadline(VHDLKernelBase*base, const FileType &file,
			   SavantlineType& line)
{
  return savantreadline(base, file.get_fileHandle(), line);
}

int
VHDLKernel::savantwriteline(VHDLKernelBase*base, FileType &file,
			    SavantlineType& line)
{
  return savantwriteline(base, file.get_fileHandle(), line);
}

void
VHDLKernel::assignSignal(VHDLType& dest, int srcId, 
			 const VHDLType& src,
			 const PhysicalType& delay, 
			 const PhysicalType& rejTime, 
			 const ArrayInfo& dinfo,
			 const ArrayInfo& sinfo) {
  register int i;
  register int dest_i;
  register int src_i;
  int dest_lbound, dest_rbound;
  int src_lbound, src_rbound;
  int length;
  ArrayDirn_t src_dir, dest_dir;
  
  ArrayInfo* src_ainfo  = (ArrayInfo *) &sinfo;
  ArrayInfo* dest_ainfo = (ArrayInfo *) &dinfo;
  
  switch(dest.get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    assignSignal(*((SignalBase*)(*(ScalarType*)&dest).object), srcId, 
		 (*(ScalarType*)&src).object->readVal(),
		 ((PhysicalType &) delay).getVTime(), 
		 ((PhysicalType &) rejTime).getVTime(), 
		 defaultInfo, defaultInfo);
    break;
  case ARRAY_TYPE:
    if(sinfo == nullInfo) {
      src_ainfo = src.get_bounds();
    }
    if(dinfo == nullInfo) {
      dest_ainfo  = dest.get_bounds();
    }
    switch(src.get_kind()) {
    case INTEGER_TYPE:
    case REAL_TYPE:
    case ENUMERATION_TYPE:
    case PHYSICAL_TYPE:
      assignSignal(dest[((ArrayType &)dest).object->bounds.left()], srcId,
		   src, delay, rejTime, *dest_ainfo, *src_ainfo);
      break;
    case ARRAY_TYPE:
      assignSignal((*(ArrayType*)&dest).get_array(), srcId,
		   (*(ArrayType*)&src).get_array(), delay, 
		   rejTime, *dest_ainfo, *src_ainfo);
      break;
    default:
      abort();
    } 
    break;
  case RECORD_TYPE:
    for(i = 1; i <= (*(RecordType*)&dest).numberOfFields; i++) {
      assignSignal((*(RecordType*)&dest).get_field(i), srcId,
		   (*(RecordType*)&src).get_field(i),
		   delay, rejTime, dinfo, sinfo);
    }
    break;
  case VECTOR_BASE:
    if ((sinfo != nullInfo) && (sinfo != defaultInfo)) {
      src_lbound  = sinfo.left();
      src_rbound  = sinfo.right();
      src_dir     = sinfo.dirn();
      length      = sinfo.length();
    } else {
      src_lbound = src.left();
      src_rbound = src.right();
      src_dir    = src.dirn();
      length     = src.length();
    }
    if ((dinfo != nullInfo) && (dinfo != defaultInfo)) { 
      dest_lbound = dinfo.left();
      dest_rbound = dinfo.right();
      dest_dir    = dinfo.dirn();
    } else {
      dest_lbound = dest.left();
      dest_rbound = dest.right();
      dest_dir    = dest.dirn();
    }
    dest_i = dest_lbound;
    src_i  = src_lbound;
    for(i = length ; (i > 0); i--) {
      assignSignal(dest[dest_i], srcId, src[src_i], delay, rejTime, 
		   nullInfo, nullInfo);
      dest_i += dest_dir;
      src_i  += src_dir;
    }
    break;
  default:
    cerr << "Signal Assignment statement called for a non Signal Type "<< endl;
    break;
  };
}

void
VHDLKernel::assignSignal(SignalBase& dest, int srcId, 
			 const VHDLData& src,
				const VTime& delay, const VTime& rejTime, 
				const ArrayInfo& dinfo,
				const ArrayInfo& sinfo) {
  const int eventSize = sizeof(SigEvent);
  register int dataSize;
  SigEvent* newev;

#ifdef VHDLDBG
  cout << name << " assigning ";
  src.print(cout);
  cout << " to signal " << dest.name << ": sTime = " << getTimeNow()
       << ", rTime = " << getTimeNow() + delay << endl;
#endif
  if ((sinfo != nullInfo) && (dinfo != nullInfo) &&
      (sinfo.length() == dinfo.length())) {
    dataSize        = src.getSize();
    newev           = (SigEvent*) new char[eventSize + dataSize];
    new(newev) SigEvent();	// sets up virtual table
    newev->size     = eventSize + dataSize;
    newev->type     = dest.type;
    newev->sendTime = getTimeNow();
    newev->recvTime = newev->sendTime + delay;
    newev->sign     = POSITIVE;
    newev->dest     = id;
    newev->sigId    = dest.id;
    newev->sigsrc   = srcId;
    newev->rejTime  = rejTime;
    newev->sBound   = sinfo;
    newev->dBound   = dinfo;
    newev->sender   = id;
    memcpy((char*)(newev + 1), (char*)&src, dataSize);
    
    SigEvent *msg;
    for (register int i = 0; (i < dest.fanout); i++) {
      // make copies and send it to everyone in the fanout list
      msg = (SigEvent*)new char[newev->size];
      new(msg) SigEvent();
      memcpy(msg, newev, newev->size);
      msg->dest        = dest.fanDest[i].objid;
      msg->sigId       = dest.fanDest[i].sigid;
      msg->senderSigId = dest.id; //sender's signalId needed for TypeConversion
      sendEvent(msg);             // senderId is set here
      
#ifdef VHDLDBG
      cout << name << ": transmitting event:\n" << *msg << endl;
#endif
    }
    // put original copy in the markedQ
    markedQ.insert(newev);
  }
  else {
    cerr << "wugga wugga" << endl;
  }
}


// When this is called, it is assumed to be for a savantBit.
// Anything else will screw up--in particular, chars.
void
VHDLKernel::assignSignal(SignalBase& dest, int srcId, const char src,
			 const VTime& delay, const VTime& rejTime, 
			 const ArrayInfo& dinfo, 
			 const ArrayInfo& sinfo) {
  savantBit b(src);
  assignSignal(dest, srcId, b, delay, rejTime, dinfo, sinfo);
}

void
VHDLKernel::assignNullTransaction(VHDLType& dest, int srcId, 
				  const PhysicalType& delay, 
				  const PhysicalType& rejTime, 
				  const ArrayInfo&,
				  const ArrayInfo&) {
  register int i = 0;
  register int dest_i = 0;
  int dest_lbound = 0, dest_rbound = 0;
  ArrayDirn_t dest_dir;
  ArrayInfo* vec_bounds = NULL;
  
  switch(dest.get_kind()){
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    assignNullTransaction(*((SignalBase*)(*(ScalarType*)&dest).object), srcId, 
		 ((PhysicalType &) delay).getVTime(), 
		 ((PhysicalType &) rejTime).getVTime(), nullInfo, nullInfo);
    break;
  case ARRAY_TYPE:
    assignNullTransaction((*(ArrayType*)&dest).get_array(), srcId,
			  delay, rejTime, nullInfo, nullInfo);
    break;
  case RECORD_TYPE:
    for(i = 1; i <= (*(RecordType*)&dest).numberOfFields; i++) {
      assignNullTransaction((*(RecordType*)&dest).get_field(i), srcId,
			    delay, rejTime, nullInfo, nullInfo);
    }
    break;
  case VECTOR_BASE:
    vec_bounds  = ((VectorBase*)&dest)->get_bounds();
    dest_lbound = vec_bounds->left();
    dest_rbound = vec_bounds->right();
    dest_dir    = vec_bounds->dirn();
    dest_i      = dest_lbound;
    
    for(i = vec_bounds->length(); (i > 0); i--) {
      assignNullTransaction(dest[dest_i], srcId, delay, rejTime, nullInfo,
			    nullInfo);
      dest_i += dest_dir;
    }
    break;
  default:
    cerr << "Signal Assignment statement called for a non Signal Type "<< endl;
    break;
  };
}

void
VHDLKernel::assignNullTransaction(SignalBase& dest, int srcId, 
				  const VTime& delay, const VTime& rejTime, 
				  const ArrayInfo&,
				  const ArrayInfo&) {
  const int eventSize = sizeof(SigEvent);
  SigEvent* newev = NULL;
  
#ifdef VHDLDBG
  cout << name << " assigning NULL TRANSACTION ";
  cout << " to signal " << dest.name << ": sTime = " << getTimeNow()
       << ", rTime = " << getTimeNow() + delay << endl;
#endif
  newev           = (SigEvent*) new char[eventSize];
  new(newev) SigEvent();	// sets up virtual table
  newev->size     = eventSize;
  newev->type     = NULL_TRANSACTION;
  newev->sendTime = getTimeNow();
  newev->recvTime = newev->sendTime + delay;
  newev->sign     = POSITIVE;
  newev->dest     = id;
  newev->sigId    = dest.id;
  newev->sigsrc   = srcId;
  newev->rejTime  = rejTime;
  newev->sBound   = Others;
  newev->dBound   = Others;
  newev->sender   = id;
  
  SigEvent *msg = NULL;
  for (register int i = 0; (i < dest.fanout); i++) {
    // make copies and send it to everyone in the fanout list
    msg        = (SigEvent*)new char[newev->size];
    memcpy(msg, newev, newev->size);
    msg->dest  = dest.fanDest[i].objid;
    msg->sigId = dest.fanDest[i].sigid;
    sendEvent(msg); // senderId is set here
    
#ifdef VHDLDBG
    cout << name << ": transmitting event:\n" << *msg << endl;
#endif
  }
  // put original copy in the markedQ
  markedQ.insert(newev);
}

void
VHDLKernel::disconnectDriver(SigEvent* event) {
  int sigId          = event->sigId;
  int srcId          = event->sigsrc;
  SignalBase* signal = NULL;
  
  signal = ((VHDLKernel_state *) state->current)->locateSig(sigId); 
#ifdef VHDLDBG
  cout << name << " Applying a NULL TRANSACTION to signal " << signal->name << " driver " 
       << srcId;
  cout << " at " << getTimeNow() << endl;
#endif
  
  //Update the signals driver corresponding to the source ID.
  signal->disconnectDriver(srcId);
  if(signal->isCompositeResolvedSignal() == false) {
    //Add this element into the list of signals that are to be 
    //updated this simulation cycle
    scalarSignalsUpdateArray.addElement(signal);
  }
  else {
    compositeResolvedSignalsUpdateArray.addElement(signal->getParentCompositeType());
  }
}

//The following Function updates the driver of the signal
//corresponding  to its sig id. Also adds this signal to the
//list of signals that should be updated in the current simulation cycle

void
VHDLKernel::updateDriver(SigEvent* event) {

  int sigId = event->sigId;
  int srcId = event->sigsrc;
  int senderSigId = event->senderSigId;
  
  VHDLData* data = (VHDLData*)(event +1);
  ArrayInfo* dInfo = &event->dBound;
  ArrayInfo *sInfo = &event->sBound;
  SignalBase* signal;

  signal = ((VHDLKernel_state *) state->current)->locateSig(sigId); 
#ifdef VHDLDBG
  cout << name << " updating signal " << signal->name << " driver " 
       << srcId << " slice " << *dInfo << " with value ";
  data->print(cout);
  cout << " slice " << *sInfo << " at " << getTimeNow() << endl;
#endif

  //Update the signals driver corresponding to the source ID.
  signal->updateDriver(senderSigId, srcId, data, sInfo, dInfo);
  if(signal->isCompositeResolvedSignal() == false) {
    //    const VHDLData &tempVal = signal->readDriver(srcId)->getObject()->readVal();
    //const VHDLData *driverVal = &tempVal;
    //signal->updateDrvVal(driverVal, sInfo, dInfo);
    //Add this element into the list of signals that are to be 
    //updated this simulation cycle
    scalarSignalsUpdateArray.addElement(signal);
  }
  else {
    compositeResolvedSignalsUpdateArray.addElement(signal->getParentCompositeType());
  }
}

//The following functions updates all the signals whose driver got 
//changed in the current simulation cycle

void
VHDLKernel::executeSignalUpdationPhase() {
  register int i=0;

  //Call updateSignal() of all the unresolved scalar Signals,
  //Or on elements of composite unresolved signals or
  //on elements of composite resolved signals
  for(i=0; i < scalarSignalsUpdateArray.getNumberOfElements(); i++) {
    updateSignal((SignalBase*)scalarSignalsUpdateArray.getElement(i));
  }
  scalarSignalsUpdateArray.reset();

  //Call updateSignal() of all signals that are composite resolved signals
  for(i=0; i<compositeResolvedSignalsUpdateArray.getNumberOfElements(); i++) {
    updateSignal((VHDLType*)compositeResolvedSignalsUpdateArray.getElement(i));
  }
  compositeResolvedSignalsUpdateArray.reset();
  
    // Update the implicit guard signal if it exists
  if (getGuard() != NULL) {
    updateGuard(getGuard());
  }
}

//The following function calls the resolution function and 
//The type conversion functions if any for a signal and
//updates the values of its dependent signals and attributes

void
VHDLKernel::updateSignal(SignalBase* signal, bool initializingSimulation) {

  const VHDLData *oldEffVal = signal->readEffVal();
  const VHDLType *newEffVal = NULL;
  bool eventOccurred = false;

  newEffVal = signal->resolve(this);

  if ( newEffVal == NULL ){
    // This could happen in tha case of signals of register kind where
    // if all the dirvers are disconnected the register value doesn't change
    return;
  }
  if (*oldEffVal != newEffVal->getObject()->readVal()) {  
    // update sensitivity list
    if (signal->sensitive == true) {
      eventSigs.add(signal->id);
    }
    eventOccurred = true;
  }
  
#ifdef VHDLDBG
  cout << name << " updating signal " << signal->name << " with drvVal(";
  signal->readDrvVal()->print(cout);
  cout << ") and effVal(";
  signal->readEffVal()->print(cout);
  cout << ") and oldEffVal(";
  oldEffVal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif

  if (initializingSimulation == false) {
    // update the implicit attributes iff simulation is in progress.
    // other dont. 
    updateImplicitAttributes(signal, oldEffVal, eventOccurred);
  }
  
  //Update the effective Value of the signal if there is an event
  if(eventOccurred == true) {
    signal->updateEffVal(newEffVal->getObject()->readVal());
  }
}

//The following function calls the resolution function and 
//The type conversion functions if any for a Composite resolved signal and
//updates the values of its dependent signals and attributes

void
VHDLKernel::updateSignal(VHDLType* signal) {

  const VHDLType *newEffVal = NULL;
  newEffVal = signal->resolve(this);
  
#ifdef VHDLDBG
  cout << name << " updating signal " << ((SignalBase *) signal)->name
       << " with drvVal(";
  newEffVal->print(cout);
  cout << ") and effVal(";
  newEffVal->print(cout);
  cout << ") and oldEffVal(";
  signal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  
  updateImplicitAttributes(signal, newEffVal);
  signal->updateEffVal(newEffVal);
}

//The following function updates the implicit signal attributes
//of the currently updated composite resolved signal

void
VHDLKernel::updateImplicitAttributes(VHDLType* signal,
					    const VHDLType* newEffVal) {
  SignalBase* signalSubElement = NULL;
  VHDLData* newValue = NULL;
  VHDLData* oldEffectiveValue = NULL;
  bool eventOccurred = false;
  register int i=0;
  switch(signal->get_kind()){
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    ASSERT(((ScalarType*)signal)->getKind() == ObjectBase::SIGNAL);
    signalSubElement = (SignalBase*)((ScalarType*)signal)->getObject();
    oldEffectiveValue = signalSubElement->readEffVal();
    ASSERT(newEffVal->is_scalar_type() == true);
    newValue = (VHDLData*)&((ScalarType*)newEffVal)->getVHDLData();
    if(*oldEffectiveValue != *newValue) {
      eventOccurred = true;
    }
    updateImplicitAttributes(signalSubElement, oldEffectiveValue, eventOccurred);
    break;
  case ARRAY_TYPE:
    ASSERT(newEffVal->get_kind() == ARRAY_TYPE);
    updateImplicitAttributes(((ArrayType*)signal)->object, ((ArrayType*)newEffVal)->object);
    break;
  case RECORD_TYPE:
    ASSERT(newEffVal->get_kind() == RECORD_TYPE);
    for(i = 1; i <= (*(RecordType*)signal).numberOfFields; i++) {
      updateImplicitAttributes(&((RecordType*)signal)->get_field(i),&((RecordType*)newEffVal)->get_field(i));
    }
    break;
  case VECTOR_BASE:
    ASSERT(newEffVal->get_kind() == VECTOR_BASE);
    for(i = 0; i < ((VectorBase*)signal)->numElems; i++) {
      updateImplicitAttributes(&((VectorBase*)signal)->get_element(i),&((VectorBase*)newEffVal)->get_element(i));
    }
    break;
  default:
    cerr << "ERROR: updateImplicitAttributes(VHDLType*,const VHDLType*)" << endl
	 << " of VHDLType: " << (int) signal->get_kind() << " Called"<< endl;
    abort();
    break;
  };
}

//The following function updates the implicit signal attributes
//of the currently updated function

void
VHDLKernel::updateImplicitAttributes(SignalBase* signal,
				     const VHDLData* oldEffVal,
				     bool eventOccurred) {
  
  for(int i = 0; i < signal->numAttributes; i++) {
    switch(signal->attributeList[i].attrib) {
    case LAST_EVENT: 
      if(eventOccurred == true) {
	updateAttribute(&(signal->attributeList[i]));
      }
      break;
    case LAST_VALUE: 
      if(eventOccurred == true) {
	updateAttribute(&(signal->attributeList[i]), *oldEffVal);
      }
      break;
    case LAST_ACTIVE:
      updateAttribute(&(signal->attributeList[i]));
      break;
    case DRIVING :
      cerr << name 
	   << ": VHDLKernel::updateSignal : Not yet implemented" << endl;
      break;
    case TRANSACTION: 
      updateAttribute(&(signal->attributeList[i]));
      break;
    case EVENT: 
    case DRIVING_VALUE:
    case INVALID:
    case ACTIVE:
      break;
    case STABLE:
      if(((SignalBase*) signal->attributeList[i].value->getObject())->sensitive == true && eventOccurred == true) {
	eventSigs.add(((SignalBase*) signal->attributeList[i].value->getObject())->id);
      }
      break;
    case QUIET:
      if(((SignalBase*) signal->attributeList[i].value->getObject())->sensitive == true) {
	eventSigs.add(((SignalBase*) signal->attributeList[i].value->getObject())->id);
      }
      break;
    default:
      cerr << name 
	   << ": VHDLKernel::updateSignal : (invalid attribute)" << endl;
      break;
    } // switch
  } // for
  updateDefaultAttributes(signal, oldEffVal, eventOccurred);
}

void
VHDLKernel::updateDefaultAttributes(SignalBase* signal,
				    const VHDLData* oldEffVal,
				    bool eventOccurred) {
  if(eventOccurred == true) {
    
    signal->allAttributes.last_event = UniversalLongLongInteger(((VHDLKernel_state *) state->current)->lVT.time);
    signal->allAttributes.event = true;
  }

  signal->allAttributes.last_active = UniversalLongLongInteger(((VHDLKernel_state *) state->current)->lVT.time);

  switch(signal->allAttributes.sigtype) {
  case VHDLData::UNIVERSAL_BOOLEAN:
  case VHDLData::UNIVERSAL_INTEGER:
    signal->allAttributes.last_value.inttype = ((UniversalInteger*)oldEffVal)->val;
    break;
  case VHDLData::UNIVERSAL_REAL:
    signal->allAttributes.last_value.floattype = ((UniversalReal*)oldEffVal)->val;
    break;
  case VHDLData::UNIVERSAL_LONG_LONG_INTEGER:
    signal->allAttributes.last_value.physicaltype = ((UniversalLongLongInteger*)oldEffVal)->val;
    break;
  default:
    break;
  }
}

// Resolution functions are not called from updateSignal.  The driver
// which modifies the value of the signal has the responsibility to
// update the values of all the fanouts of the signals and send the
// corresponding messages to them.  This is done to avoid multiple
// resolution function computation.  If the destination were to
// resolve its signals, the same values will be computed by each of
// the fanouts.  
// This is why the check of resolved funciton has been removed from
// updateSignal.

void
VHDLKernel::updateSignal(int sigId, int srcId, VHDLData* data,
			 const ArrayInfo* dInfo, 
			 const ArrayInfo *sInfo){
  
  register int i;
  SignalBase* sig;
  bool eventOccurred = false;	// Set if an event occurs on this signal.

  sig = ((VHDLKernel_state *) state->current)->locateSig(sigId); 
#ifdef VHDLDBG
  cout << name << " updating signal " << sig->name << " driver " 
       << srcId << " slice " << *dInfo << " with value ";
  data->print(cout);
  cout << " slice " << *sInfo << " at " << getTimeNow() << endl;
#endif

  const VHDLData *oldEffVal = sig->readEffVal();
  const VHDLType *newEffVal;

  // Update the driver as specified by srcId, and resolve the signal.
  // The resolve is an overloaded function in ImplicitSignal and Signal.
  // If the Signal has to be resolved, the signal will resolve itself and
  // update the effective value of the signal.
  if(sig->getKind() == ObjectBase::IMPLICIT_SIGNAL){
    newEffVal = (VHDLType *)data;
  }
  else{
    sig->updateDriver(sigId, srcId, data, sInfo, dInfo);
    const VHDLData &tempVal = sig->readDriver(srcId)->getObject()->readVal();
    const VHDLData *driverVal = &tempVal;
    sig->updateDrvVal(driverVal, sInfo, dInfo);
    newEffVal = sig->resolve(this);
  }
  // check if there's been an event
  if (*oldEffVal != newEffVal->getObject()->readVal()) {  
    // update sensitivity list
    if (sig->sensitive == true) {
      eventSigs.add(sig->id);
    }
    eventOccurred = true;
  }

#ifdef VHDLDBG
  cout << name << " updating signal " << sig->name << " with drvVal(";
  sig->readDrvVal()->print(cout);
  cout << ") and effVal(";
  sig->readEffVal()->print(cout);
  cout << ") and oldEffVal(";
  oldEffVal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  for(i = 0; i < sig->numAttributes; i++) {
    switch(sig->attributeList[i].attrib) {
    case LAST_EVENT: 
      if(eventOccurred == true) {
	updateAttribute(&(sig->attributeList[i]));
      }
      break;
    case LAST_VALUE: 
      if(eventOccurred == true) {
	updateAttribute(&(sig->attributeList[i]), *oldEffVal);
      }
      break;
    case LAST_ACTIVE:
      updateAttribute(&(sig->attributeList[i]));
      break;
    case DRIVING :
      cerr << name 
	   << ": VHDLKernel::updateSignal : Not yet implemented" << endl;
      break;
    case TRANSACTION: 
      updateAttribute(&(sig->attributeList[i]));
      break;
    case EVENT: 
    case DRIVING_VALUE:
    case INVALID:
    case ACTIVE:
      break;
    case STABLE:
      if(((SignalBase*) sig->attributeList[i].value->getObject())->sensitive == true && eventOccurred == true) {
	eventSigs.add(((SignalBase*) sig->attributeList[i].value->getObject())->id);
      }
      break;
    case QUIET:
      if(((SignalBase*) sig->attributeList[i].value->getObject())->sensitive == true) {
	eventSigs.add(((SignalBase*) sig->attributeList[i].value->getObject())->id);
      }
      break;
    default:
      cerr << name 
	   << ": VHDLKernel::updateSignal : (invalid attribute)" << endl;
      break;
    } // switch
  } // for
  if(eventOccurred == true) {
    sig->updateEffVal(newEffVal->getObject()->readVal());
  }
}


void
VHDLKernel::updateGuard(VHDLType *guardSig) {
  SignalBase *signal = (SignalBase*)guardSig->getObject();
  const VHDLData *oldEffVal = signal->readEffVal();
  const VHDLType *newEffVal = NULL;
  bool eventOccurred = false;

  newEffVal = getGuardExpression();

  if ( newEffVal == NULL ){
    return;
  }
  if (*oldEffVal != newEffVal->getObject()->readVal()) {  
    // update sensitivity list
    if (signal->sensitive == true) {
      eventSigs.add(signal->id);
    }
    eventOccurred = true;
  }
  
#ifdef VHDLDBG
  cout << name << " updating signal " << signal->name << " with drvVal(";
  signal->readDrvVal()->print(cout);
  cout << ") and effVal(";
  signal->readEffVal()->print(cout);
  cout << ") and oldEffVal(";
  oldEffVal->print(cout);
  cout <<") at " << getTimeNow() << endl;
#endif
  
  //Update the effective Value of the signal if there is an event
  if(eventOccurred == true) {
    signal->updateEffVal(newEffVal->getObject()->readVal());
  }
  //  delete newEffVal;
}

void
VHDLKernel::updateWait(WaitEvent* event) {

  if (event->waitId != ((VHDLKernel_state *) state->current)->waitLabel) {
    // This case is NOT an error case.  It could happen that I was waiting
    // with the following VHDL statement:
    //    wait on sig until <cond> for <time> ns;
    // If the signal has an event on it, I will no longer wait here, but
    // the wait-resume event that I sent out earlier WILL arrive here.  I
    // should not abort on that.  I have to check if I was waiting on this
    // wait stmt.  If not, just return without doing anything.
    return;
  }
  // We need to check if we are still waiting for this wait event.  It
  // could be that we have resumed the wait statement for which this event
  // was sent, but we have reached the same statement again, and are
  // waiting on it.  Then, this wait event CAN NOT cause this execution of
  // the wait statement to resume, since we have to wait for the wait
  // event that we sent out when we executed this wait statment.  We keep
  // track of this by maintainig when this wait was executed.
  // ((VHDLKernel_state *) state->current)->waitTime is set in VHDLKernel::executeWait().
  if(((VHDLKernel_state *) state->current)->waitTime == event->sendTime) {
    ((VHDLKernel_state *) state->current)->waitExpired = true;
  }
#ifdef VHDLDBG
  cout << name << " updating wait " << event->waitId << ", waiting on "
	<< ((VHDLKernel_state *) state->current)->waitLabel << " from " << ((VHDLKernel_state *) state->current)->waitTime;
  if(((VHDLKernel_state *) state->current)->waitExpired == true) {
    cout << " wait expired." << endl;
  } else {
    cout << " wait has not expired (waitTime = " << ((VHDLKernel_state *) state->current)->waitTime
	  << " sendTime = " << event->sendTime << endl;
  }
#endif
}

// This method first tries to fit the file line into a static buffer.  If
// the line fits, this value is reused.  If the buffer is too small, the
// file is rewound to its original position, and the line is reread into a
// dynamically allocated buffer twice the size of the static one.  If the
// line still doesn't fit, the process is repeated, doubling the buffer
// size each time, until the line if successfully read in its entirety.

char*
VHDLKernel::getline(istream& file, int& linesize) {
  register unsigned int bytesread = 0;
  register unsigned int chunksize = 1024;
  char buf[1024];
  char* retstr;
  streampos linestart;

  linestart = file.tellg(); // save position of beginning of line
  file.get(buf, chunksize, '\n'); // get the line and the # of chars read
  bytesread = file.gcount();

  if (bytesread < (chunksize - 1)) {
    // line fit into original buffer
    file.seekg(1, ios::cur);
    linesize = bytesread;
    retstr = new char[bytesread + 1];
    memcpy(retstr, buf, bytesread);
    *(retstr + bytesread) = '\0';
    return retstr;
  }
  else {
    // we filled the static buffer--see if we're really done
    if (file.peek() == '\n') {
      // we've got the whole line, so skip that pesky newline!
      file.seekg(1, ios::cur);
      linesize = bytesread;
      retstr = new char[bytesread + 1];
      memcpy(retstr, buf, bytesread);
      *(retstr + bytesread) = '\0';
      return retstr;
    }
    else {
      // the line's longer than the initial value of chunksize, so dynamic
      // memory is used, repeatedly growing exponentially, until we do get
      // the line in one piece.  Trying static memory first, then using
      // dynamic is ugly, but it prevents having to use any dynamic memory
      // in all but the most massive input file lines.
      char* tmpptr = NULL;
      do {
	if (tmpptr != NULL) delete [] tmpptr;
	file.seekg(linestart); // rewind back to beginning of the line
	chunksize *= 2;  // get more chars this time around
	tmpptr = new char[chunksize];
	file.get(tmpptr, chunksize, '\n');
	bytesread = file.gcount();
      } while (bytesread == (chunksize - 1));

      file.seekg(1, ios::cur);
      linesize = bytesread;
      retstr = new char[bytesread + 1];
      memcpy(retstr, tmpptr, bytesread);
      *(retstr + bytesread) = '\0';
      delete [] tmpptr;
      return retstr;
    }
  }
}

bool 
VHDLKernel::eatwhite(AccessVariable<char*>& line) {
  if (line.curPos >= line.dataLength) {
    cerr << name << " VHDLKernel::eatwhite: line variable " << line.name
	 << " is empty!\n\n";
    line.print(cerr);
    cerr << endl;
    return false;
  }

  // skip any whitespace
  while ((line.val[line.curPos] == ' ') || 
	 (line.val[line.curPos] == '\t')) {
    line.curPos++;
    if (line.curPos == line.dataLength) {
      // line exhausted, no bit found
      cerr << name << " VHDLKernel::eatwhite Warning: EOL reached" << endl;
      return false;
    }
  }
  return true;
}

int
VHDLKernel::openFile(const char *fileName, const _savant_file_open_kind &mode, ios::open_mode opening_mode) {
  if (mode.val == READ_MODE) {
    inFileQ[numInFiles].Open(fileName);
    numInFiles++;
    return (numInFiles - 1);
  }
  else {
    outFileQ[numOutFiles].open(fileName, opening_mode);
    numOutFiles++;
    return (numOutFiles - 1);
  }
}

void
VHDLKernel::closeFile(const _savant_file_open_kind &mode, int fileHandle) {
  if (mode.val == READ_MODE) {
    inFileQ[fileHandle].close();
    numInFiles--;
  }
  else {
    outFileQ[fileHandle].close();
    numOutFiles--;
  }
}

#ifndef SEQUENTIAL
void
VHDLKernel::initState() {
  if (getCurrentState() != NULL) {
    // ((VHDLKernel_state *) getCurrentState())->cleanState();
    delete getCurrentState();
  }
  
  getStateManager()->createInitialState();
  getCurrentState()->inputPos     = inputQ.getHead(localId);
  getCurrentState()->outputPos    = outputQ.getTail();
  getCurrentState()->myLastOutput = false;
  
  ((VHDLKernel_state *) getCurrentState())->initState(proc, this);
  if (getGuard() != NULL) {
    updateGuard(getGuard());
  }
  
  getCurrentState()->inputPos     = inputQ.getHead(localId);
  getCurrentState()->outputPos    = outputQ.getTail();
  getCurrentState()->myLastOutput = false;
}
#endif

#ifdef SEQUENTIAL
int
compareFunction(const BasicEvent *e1, const BasicEvent *e2) {
  if (e1->sendTime > e2->sendTime) {
    return 1;
  }
  else if (e1->sendTime == e2->sendTime) {
    return 0;
  }
  
  return -1;
}
#endif

#endif

