#include "SourceInfo.hh"
#include "SourceData.hh"
#include "VHDLType.hh"

const int CHILD_NOT_ADDED = -99;

SourceInfo::SourceInfo(void) : SourceBase() {
  upConvertFnId = DEFAULT_ID;
  downConvertFnId = DEFAULT_ID;
  resolveFnId = DEFAULT_ID;

  numChildren = 0;
  child = NULL;

  internalTable = NULL;
  childTable = NULL;
  tablesize = 0;
  retval = NULL;
}

SourceInfo::~SourceInfo() {
  int child_no ;
  
  for ( child_no = 0 ; child_no < numChildren ; child_no++){
    delete child[child_no];
  }
  
  delete [] child;
  child = NULL;
  
  if ( childTable != NULL ){
    delete [] childTable ;
    childTable = NULL;
  }
  if ( internalTable != NULL ){
    delete [] internalTable ;
    internalTable = NULL;
  }
  
}

SourceInfo&
SourceInfo::operator = (const SourceInfo& source) {
  register int i;
  numChildren = source.getNumChildren();
  for(i = 0; i < numChildren; i++) {
    *(child[i]) = *(source.getChild(i)); // This recursively copies the
  }				         // grandchildren as well!!
  return (*this);
}

void 
SourceInfo::setUpTypeConversionFnId(TypeConversionFnId_t fnId) {
  upConvertFnId = fnId;
}

void 
SourceInfo::setDownTypeConversionFnId(TypeConversionFnId_t fnId) {
  downConvertFnId = fnId;
}

void 
SourceInfo::setResolutionFnId(ResolutionFnId_t fnId) {
  resolveFnId = fnId;
}

int 
SourceInfo::getNumChildren() const {
  return numChildren;
}

int
SourceInfo::getNumActiveChildren() const {
  int num_child = 0;
  for(int i=0; i < numChildren; i++) {
    if(child[i]->_is_driver_connected() == true) {
      num_child++;
    }
  }
  return num_child;
}

void 
SourceInfo::setNumChildren(int num) {
  numChildren = num;
}
  
bool
SourceInfo::_is_child_present(SourceId_t childId) {
  int i;
  if(parent != NULL) {
    return parent->_is_child_present(childId);
  } else {
    for(i = 0; i < int(tablesize); i++) {
      if(internalTable[i] == childId) {
	return true;
      }
    }
    return false;
  }
}

bool
SourceInfo::_is_driver_connected() {
  // Return true only if atleast one child can contribute a driver!!
  
  for(int i=0; i < numChildren; i++) {
    if(child[i]->_is_driver_connected() == true) {
      return true;
    }
  }
  return false;
}

int 
SourceInfo::addChild(SourceBase *newChild) {
  register int i = 0;
  SourceBase **newChildArray;

  if(newChild->getNumChildren() == 0) {
    if(newChild->get_kind() == SourceBase::SOURCE_INFO) {
      return CHILD_NOT_ADDED;
    }
  }
  int childPresent = 0;
  if(newChild->get_kind() == SourceBase::SOURCE_DATA) {
    if(_is_child_present(newChild->getSourceId())) {
      cerr << "Try to add another driver for the same process"
	   << " with id" << newChild->getSourceId() << endl;
      childPresent = 1;
    }
    else {
      childPresent = 0;
    }
  }
  else if(newChild->get_kind() == SourceBase::SOURCE_INFO) {
    SourceInfo* siPtr = (SourceInfo*)newChild;
    for(i=0; i < int(tablesize); i++) {
      if(siPtr->_is_child_present(internalTable[i])) {
	childPresent = 1;
	cerr << "Try to add another driver for the same process"
	     << " with SourceInfo containing id" << internalTable[i] << endl;
	break;
      }
    }
  } else if ( newChild->get_kind() == SourceBase::SOURCE_TYPE_CONVERT){
    // nothing to be checked for a node of SOURCE_TYPE_CONVERT
    // childPresent = 0; but that's done in initialization itself, so why worry
  }

  
  if(childPresent == 1) {
    return CHILD_NOT_ADDED;
  }
  else {
    newChildArray = (SourceBase**) new char[sizeof(SourceBase*)*(numChildren+1)];

    for(i = 0; i < numChildren; i++) {
      newChildArray[i] = child[i];
    }
    numChildren++;
    newChildArray[numChildren - 1] = newChild;
    newChild->setParent(this);

    // only if the newChild is a SOURCE_DATA or SOURCE_INFO then the
    // the drivers are collected at the root of the resolution tree.
    if ( newChild->get_kind() != SourceBase::SOURCE_TYPE_CONVERT){
      copyTable(newChild);
    }
    
    delete [] child;
    child = newChildArray;
    
    return numChildren;
  }
}

int 
SourceInfo::addChild(VHDLType *newDriver, SourceId_t childId) {
  SourceData *newChild = NULL;
  register int i = 0;

  if(_is_child_present(childId)) {
    return CHILD_NOT_ADDED;
  }

  SourceBase **newChildArray = (SourceBase**) new (SourceBase*)[numChildren+1];
  ASSERT(newDriver != NULL);

  for(i = 0; i < numChildren; i++) {
    newChildArray[i] = child[i];
  }
  newChild = new SourceData();
  newChild->setParent(this);
  newChild->addChild(newDriver, childId);
  numChildren++;
  newChildArray[numChildren - 1] = newChild;

  if (childId != ANONYMOUS_PROCESS_ID) {
    // Sure waste of time. Should strip this call off Mal.
    addToTable(newChild);
  }

  delete [] child;
  child = newChildArray;

  return numChildren;
}

SourceBase *
SourceInfo::getChild(const int childId) const {
  ASSERT(childId < numChildren);
  return child[childId];
}

SourceBase&
SourceInfo::operator[] (const int childId) {
  ASSERT(childId < numChildren);
  return *getDriver(childId);
}

// This method is used to build the fast access tables in the source data
// structure.  The tables are required only in the root node of the tree.
// If this is not the root node, it passes the argument to its parent.  If
// this is the root node, it allocates space for the new data, copies the
// old data, deletes the space allocated before, and finally, adds the
// data to the tables.
void
SourceInfo::addToTable(SourceData *leafNode) {
  if(parent != NULL) {		// This is not the root node.
    parent->addToTable(leafNode);
  } else {			// This is the root node.
    int *newInternalTable = new int[tablesize+1];
    SourceData **newChildTable = new SourceData*[tablesize+1];

    for(register int i = 0; i < int(tablesize); i++) {
      newInternalTable[i] = internalTable[i];
      newChildTable[i] = childTable[i];
    } // for
    delete [] internalTable;
    delete [] childTable;
    internalTable = newInternalTable;
    childTable = newChildTable;

    tablesize++;
    internalTable[tablesize-1] = leafNode->getSourceId();
    childTable[tablesize-1] = leafNode;
  } // else
}

void 
SourceInfo::copyTable(const SourceBase *newNode) {
  if(parent != NULL) {
    parent->copyTable(newNode);
  } else {
    for(register int i = 0; i < newNode->getTableSize(); i++) {
      addToTable(newNode->getDriveratIndex(i));
    } // for
  } // else
}

VHDLType *
SourceInfo::resolve(VHDLKernelBase* processPtr, SigType typ) {
  int activeChildren = getNumActiveChildren();
  VHDLType **valueArray = NULL;
  register int i = 0;
  register int j = 0;
  if(activeChildren == 0) {	// Sanity check.
    // This could happen for signals of kind bus or register
    // When the kind is a bus then pass a NULL array to the resolution
    // function to find the effcetive value whereas in case of a register
    // the value does not change 
    if (typ == G_BUS) {
      // Pass a NULL array to the resolution function
      ASSERT(resolveFnId != DEFAULT_ID);
      retval = ((savantResolutionFn[resolveFnId])(processPtr, 0, NULL));
      return retval;
    } else {
      return NULL;
    }
  }

  valueArray = (VHDLType **) new VHDLType*[activeChildren];
  TypeConversionFnId_t fnId = DEFAULT_ID;
  VHDLType *temp_vhdl_type_ptr = NULL;  
  // Two counters are used to point to the drivers.
  // "i" refers to all the drivers in the simulation
  // "j" refers to the set of all connected Drivers during the
  // current simulation period.
  for(j=0, i = 0; i < numChildren; i++) {
    if(child[i]->_is_driver_connected() == true) {
      //Call the resolution function if the driver is connected
      valueArray[j] = child[i]->resolve(processPtr);
      //Call the type conversion if the driver is connected
      // First we have to do the up-type-conversion. Then we will pass on the
      // resultant vector to the resolution function. The down-type
      // conversion will be taken care of by the fanout, since this is a
      // sequence of functions operating on a data.
      temp_vhdl_type_ptr = NULL;
      fnId = child[i]->getUpTypeConversionFnId();
      if(fnId != DEFAULT_ID) {
	temp_vhdl_type_ptr = valueArray[j];
	valueArray[j] = (savantTypeConversionFn[fnId])(processPtr, valueArray[j]);
	delete temp_vhdl_type_ptr;
      }
      j++;
    }
  }
  
  if (j != activeChildren) {
    // Insanity check!!!
    cerr << "Internal Error : Number of active children != number of"
	 << " drivers.\n";
    activeChildren = j;
  }
  
  //Call the Resolution function after calling the resolve of the 
  //Drivers
  if(activeChildren > 1) {
    ASSERT(resolveFnId != DEFAULT_ID);
    retval = ((savantResolutionFn[resolveFnId])(processPtr, activeChildren, valueArray));
    delete []valueArray;
    return retval;
  } else {
    retval = valueArray[0];
    delete [] valueArray;
    return retval;
  }
}

void
SourceInfo::print(ostream& os) const {
  static int indent = 0;
  register int i;

  for(i = 0; i < indent; i++) {
    os << " ";
  }

  indent += 4;			// Indent each child by 4 spaces.
  
  os << "numChildren(" << numChildren << ") "
     << "upFn(" << upConvertFnId << ") "
     << "downFn(" << downConvertFnId << ") "
     << "resolveFn(" << resolveFnId << ")" << endl;

  for(i = 0; i < numChildren; i++) {
    child[i]->print(os);
  }

  indent -= 4;
}

// SourceInfo::getDriver searches the tree for the SourceData node that
// points to the driver requested by the id and returns the node.  It
// aborts if the requested driver is not available in the tree.
SourceData*
SourceInfo::getDriver(const SourceId_t id) const {
  register int i;
  if(parent != NULL) {
    return parent->getDriver(id);
  } else {
    for(i = 0; i < int(tablesize); i++) {
      if(internalTable[i] == id) {
	break;
      }	// if
    } // for
    if(i == int(tablesize)) {
      //For the following VHDL code
      //  type two_array is array (1 to 2) of integer;
      //  signal v : two_array;
      //       proc1 : process
      // 	begin
      
      // 	v(1) <= 1 after 10 ns;
      //       wait;
      
      //       end process;
      //There should be only one driver added to the signal since it is a
      //unresolved signal. But the source of the signal for each sub-element
      //can be different. Currently the elaborator ensures that only one driver
      // is added to the any unresolved signal. But the source of the signal's
      //sub-element is not determined at elaboration time, and hence the
      //following fix
      if(i == 1) {
	SourceId_t srcId = id;
	internalTable[0] = srcId;
	childTable[0]->setSourceId(srcId);
	return childTable[0];
      }
      else {
	//	cerr << "Driver " << id << " not present in the source tree.  " 	     << "Aborting." << endl;
	//abort();
	return NULL;
      }
    } else {
      return childTable[i];
    }
  }
}

SourceData*
SourceInfo::getDriveratIndex(int index) const {
  if(childTable != NULL) {
    if(index >= int(tablesize)) {
      cerr << "Driver " << index << " not present in the source tree.  " 
	   << "Aborting." << endl;
      abort();
      return NULL;
    }
    else {
      return childTable[index];
    }
  }
  else {
    return NULL;
  }
}
