/* vim: set tabstop=4: */
/*
 * This file is part of TraceProto.
 * Copyright 2004-2005 Eric Hope and others; see the AUTHORS file for details.
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libnet.h>
#include <pcap.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <ctype.h>
#include <time.h>

#include "traceproto.h"
#include "tp_miscfunc.h"
#include "tp_output.h"
#include "config.h"
#include "tp_as.h"

#if defined ( HAVE_NCURSES_H )
#include <ncurses.h>
#elif defined ( HAVE_NCURSES_NCURSES_H )
#include <ncurses/ncurses.h>
#elif defined ( HAVE_CURSES_H )
#include <curses.h>
#endif

#if defined ( HAVE_LIBNCURSES )
#define	TP_USE_CURSES 1
#elif defined ( HAVE_LIBCURSES )
#define	TP_USE_CURSES 1
#endif

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>
#endif /* HAVE_LIBDMALLOC */

void report_none ( int output_item, __attribute__((__unused__)) struct in_addr * from, __attribute__((__unused__)) int packet_type )
{
	if ( output_item == TP_OUT_HEADER )
	{
		printf ( "%s to %s (%s), %d hops max\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			behavior.max_ttl );
		if ( behavior.default_if == NO )
			printf ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printf ( "trace time: %s\n", behavior.timestamp_str );
	}

	if ( output_item == TP_OUT_FOOTER )
		hop_audit ( );

	fflush ( stdout );

	return;
}

void report_minimum ( int output_item, __attribute__((__unused__)) struct in_addr * from, int packet_type )
{
	const char * minimal_packet_id[] = {
		" ",
		"*",
		"S",
		"A",
		"S",
		"S",
		"R",
		"R",
		"F",
		"U",
		"X",
		"P",
		"H",
		"N",
		"I",
		"E",
		"H",
		"?"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		if ( behavior.timestamp == YES )
			printf ( "%s\n", behavior.timestamp_str );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "%d  ", state.current_hop );
		break;
	case TP_OUT_HOP_INFO:
		printf ( "%s ", minimal_packet_id [ packet_type ] );
		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
		break;
	case TP_OUT_FOOTER:
		hop_audit ( );
		break;
	}

	fflush ( stdout );

	return;
}

void report_scriptable ( int output_item, struct in_addr * from, int packet_type )
{
	const char * script_packet_id[] = {
		"NULL",
		"NR",
		"TS",
		"TA",
		"TSA",
		"TSC",
		"TR",
		"TAR",
		"TF",
		"U",
		"ITX",
		"IPU",
		"IHU",
		"INU",
		"I",
		"ER",
		"PRO",
		"O",
	};
    switch ( output_item )
    {
    case TP_OUT_HEADER:
		if ( behavior.timestamp == YES )
			printf ( "%s\n", behavior.timestamp_str );
   	    break;
    case TP_OUT_HOP_NUMBER:
        break;
	case TP_OUT_HOP_INFO:
		if ( packet_type == TP_TYPE_NR )
		{
			printf ( "%d 0.0.0.0 %s 0\n",
				state.current_hop,
				script_packet_id [ packet_type ] );
		} else {
			printf ( "%d %s %s %3g\n",
				state.current_hop,
				inet_ntoa ( * from ),
				script_packet_id [ packet_type ],
				state.trip_time );
		}
		break;
	case TP_OUT_FOOTER:
		hop_audit ( );
		break;
	}
	fflush ( stdout );
			
	return;
}

/* as close as possible to the original traceroute
 * needs a bit of work to make it match exactly */
void report_classic ( int output_item, struct in_addr * from, int packet_type )
{
	static struct in_addr previous_in_addr;

	const char * classic_packet_id[] = {
		" ",
		"*",
		" ",
		" ",
		" ",
		" ",
		"!R",
		"!R",
		" ",
		" ",
		" ",
		"!P",
		"!H",
		"!N",
		" ",
		"*",
		"!H",
		"!?"
	};

	switch ( output_item  )
	{
	case TP_OUT_HEADER:
		printf ( "%s to %s (%s), %d hops max\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			behavior.max_ttl );
		if ( behavior.default_if == NO )
			printf ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printf ( "trace time: %s\n", behavior.timestamp_str );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "%d  ", state.current_hop );
		previous_in_addr.s_addr = ( unsigned long ) NULL;
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( previous_in_addr.s_addr != ( unsigned long ) NULL )
				printf ( "\n\t" );
			if ( behavior.as_discovery == YES )
			{
				find_as ( inet_ntoa ( * from ) );
				printf ( "%s (%s) [%s] ",
					libnet_addr2name4 ( from->s_addr,
						behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ),
					as_string );
			} else {
				printf ( "%s (%s)  ",
					libnet_addr2name4 ( from->s_addr,
						behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ) );
			}
		}
		if ( packet_type == TP_TYPE_NR )
			printf ( "*  " );
		else
			printf ( "%3g ms %s ",
				state.trip_time,
				classic_packet_id [ packet_type ] );

		if ( ( state.packets_this_hop + 1 ) >= behavior.packets_per_hop )
			printf ( "\n" );

		break;
	case TP_OUT_FOOTER:
		hop_audit ( );
		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}

/*
 * traceproto standard format, somewhat more readable than the original
 * but still fairly space efficient
 */
void report_std ( int output_item, struct in_addr * from, int packet_type )
{
	static struct in_addr previous_in_addr;

	const char * std_packet_id[] = {
		"null",
		"no response",
		"TCP Syn",
		"TCP Ack",
		"TCP Syn Ack",
		"TCP Syn Ecn",
		"TCP Reset",
		"TCP Ack Reset",
		"TCP Fin",
		"UDP",
		"ICMP Time Exceeded",
		"Port Unreachable",
		"Host Unreachable",
		"Network Unreachable",
		"ICMP",
		"Echo Reply",
		"ICMP Prohibited",
		"Unknown"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		printf ( "%s: trace to %s (%s), port %d\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			packet.dst_port );
		if ( behavior.default_if == NO )
			printf ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printf ( "trace time: %s\n", behavior.timestamp_str );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "ttl  %d:", state.current_hop );
		previous_in_addr.s_addr = ( unsigned long ) NULL;
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( previous_in_addr.s_addr != ( unsigned long ) NULL )
				printf ( "\n\t" );
			if ( behavior.as_discovery == YES )
			{
				find_as ( inet_ntoa ( * from ) );
				printf ( "  %s from %s (%s) [%s]\n",
					std_packet_id [ packet_type ],
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ),
					as_string );
			} else {
				printf ( "  %s from %s (%s)\n",
					std_packet_id [ packet_type ],
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ) );
			}
		}
		if ( packet_type == TP_TYPE_NR )
	/*		printf ( "\t%s", std_packet_id [ packet_type ] ); */
			printf ( "%s     ", std_packet_id [ packet_type ] );
		else
			printf ( "\t%#.5g ms", state.trip_time );

		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
		break;
	case TP_OUT_FOOTER:
		hop_audit ( );
		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}

void report_graphic ( int output_item, struct in_addr * from, int packet_type )
{
	int i;
	static struct in_addr previous_in_addr;

	const char * graphic_packet_id[] = {
		"null",
		"no response",
		"TCP Syn",
		"TCP Ack",
		"TCP Syn Ack",
		"TCP Syn Ecn",
		"TCP Reset",
		"TCP Ack Reset",
		"TCP Fin",
		"UDP",
		"ICMP Time Exceeded",
		"Port Unreachable",
		"Host Unreachable",
		"Network Unreachable",
		"ICMP",
		"Echo Reply",
		"ICMP Prohibited",
		"Unknown"
	};

	switch ( output_item )
	{
	case TP_OUT_HEADER:
		printf ( "%s: trace to %s, port %d\n",
			state.prog,
			behavior.target,
			packet.dst_port );
		if ( behavior.default_if == NO )
			printf ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printf ( "trace time: %s\n", behavior.timestamp_str );
		break;
	case TP_OUT_HOP_NUMBER:
		printf ( "ttl  %d:  ", state.current_hop );
		break;
	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( behavior.as_discovery == YES )
			{
				find_as ( inet_ntoa ( * from ) );
				printf ( "%s from %s (%s) [%s]\n",
					graphic_packet_id [ packet_type ],
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ),
					as_string );
			} else {
				printf ( "%s from %s (%s)\n",
					graphic_packet_id [ packet_type ],
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ) );
			}
		}
		if ( packet_type == TP_TYPE_NR )
			state.trip_time = 0;

		for ( i = 0; i < ( int ) state.trip_time; i++ )
			printf ( "#" );

		printf ( "  %#.5g ms\n", state.trip_time );

		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printf ( "\n" );
		break;
	case TP_OUT_FOOTER:
		hop_audit ( );
		break;
	}

	fflush ( stdout );
	if ( from != 0 )
		previous_in_addr = * from;

	return;
}

void report_curses ( int output_item, struct in_addr * from, __attribute__((__unused__)) int packet_type )
{
		static struct in_addr previous_in_addr;
		static int stats_start_line = 4;
		static int ttl_column = 0;
		/* static int host_column  = 9; */
		/* static int as_column; */
		static int stats_column = 43;
		static int stats_len = 36;
		static int other_x;
		static int low_point = 4;
		int low_point_test;

#ifdef TP_USE_CURSES

	switch ( output_item )
	{
	case TP_OUT_HEADER:

		/* start the curses stuff */
		initscr ( );
		intrflush ( stdscr, 1 );

		/* header info */
		move ( 0, 0 );
		printw ( "%s: trace to %s (%s), port %d\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			packet.dst_port );
		if ( behavior.default_if == NO )
			printw ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printw ( "trace time: %s\n", behavior.timestamp_str );
		refresh ( );

		stats_column = COLS - stats_len - 2;
		getyx ( stdscr, stats_start_line, other_x );
		stats_start_line += 1;

		/* print the stats header */
		move ( stats_start_line - 1, ttl_column );
		printw ( "ttl  : hostname" ); 
		move ( stats_start_line - 0, ttl_column );
		printw ( "---------------" );
		move ( stats_start_line - 1, stats_column );
		printw ( "  last   min    ave    max  rcvd lost" );
		move ( stats_start_line - 0, stats_column );
		printw ( " ------------------------------------\n" );
		refresh ( );

		break;

	case TP_OUT_HOP_NUMBER:
		move ( stats_start_line + state.current_hop, 0 );
		refresh ( );
		printw ( "ttl %d: ", state.current_hop );
		previous_in_addr.s_addr = ( unsigned long ) NULL;
		break;

	case TP_OUT_HOP_INFO:
		if ( from != NULL && previous_in_addr.s_addr != from->s_addr )
		{
			if ( previous_in_addr.s_addr != ( unsigned long ) NULL )
				printw ( "\n\t" );
/*
			if ( behavior.as_discovery == YES )
			{
				find_as ( inet_ntoa ( * from ) );
				printw ( "  %s from %s (%s) [%s]\n",
					curses_packet_id [ packet_type ],
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ),
					inet_ntoa ( * from ),
					as_string );
			} else {
*/
				printw ( "%s",
					libnet_addr2name4 ( from->s_addr, behavior.libnet_resolve_choice ) );
/*			} */
		}
		move ( stats_start_line + state.current_hop, stats_column );
		refresh ( );
		printw ( " %#.5g  %#.5g  %#.5g  %#.5g  %3d  %3d",
			state.trip_time,
			state.hop_record[ state.current_hop ].min_time,
			state.hop_record[ state.current_hop ].ave_time,
			state.hop_record[ state.current_hop ].max_time,
			state.hop_record[ state.current_hop ].num_packets,
			state.hop_record[ state.current_hop ].lost_packets );

		if ( ( state.packets_this_hop + 1 ) == behavior.packets_per_hop )
			printw ( "\n" );

		getyx ( stdscr, low_point_test, other_x );
		low_point = ( low_point_test > low_point ) ? low_point_test : low_point;

		break;
	case TP_OUT_FOOTER:

		move ( low_point, 0 );
		refresh ( );
		endwin ( );
		delwin( stdscr );

		printf ( "\n%s: trace to %s (%s), port %d\n",
			state.prog,
			behavior.target,
			behavior.target_reverse,
			packet.dst_port );
		if ( behavior.default_if == NO )
			printf ( "using interface %s\n", behavior.interface );
		if ( behavior.timestamp == YES )
			printf ( "trace time: %s\n", behavior.timestamp_str );

		hop_audit ( );
		break;
	}

	refresh ( );
/*	fflush ( stdout ); */

	if ( from != 0 )
		previous_in_addr = * from;

#else /* ! TP_USE_CURSES */

	/* If the curses libs aren't available,
	   print and error and degrade gracefully. */
	printf ( "curses interface unavailable\n" );
	behavior.output_style = TP_STD_OUTPUT;
	behavior.report = report_std;
	behavior.report( TP_OUT_HEADER, NULL, 0 );

#endif /* TP_USE_CURSES */

	return;
}




void usage( void )
{
	size_t i;
const char * usage_str[] = {
	"usage:",
	"\ttraceproto [options] destination",
	"\t-p protocol (default tcp)",
	"\t\tcurrently available: tcp, udp, and icmp",
	"\t-d minimum (or only) destination port (default 80)",
	"\t-D maximum destination port",
	"\t-s minimum source port (default 10240)",
	"\t-S maximum source port",
	"\t-m minimum ttl (default 1)",
	"\t-M maximum ttl (default 30)",
	"\t-w wait timeout (default 5 sec)",
	"\t-W wait timeout before sending a new packet (default 0 ms)",
	"\t-a hop accounting (default 2)",
	"\t\t 0 == no accounting",
	"\t\t 1 == total only",
	"\t\t 2 == full accounting",
	"\t-P payload byte count (default 12)",
	"\t-k comma separated list of hops to skip (default none)",
	"\t-c trace continuously without accounting",
	"\t-C continuous trace with accounting",
	"\t-I number of reoccuring traces",
	"\t-H packets per hop (default 3)",
	"\t-f set the don't fragment bit",
	"\t-F specify the network interface",
	"\t-A do AS number lookup",
	"\t-T print timestamp",
	"\t-i incr/decr source/destination ports",
	"\t\t s == decrement source port",
	"\t\t S == increment source port (default)",
	"\t\t d == decrement destination port",
	"\t\t D == increment destination port",
	"\t\t n == static source port",
	"\t\t N == static destination port (default)",
	"\t\t Note: nN may result in inaccurate responses",
	"\t-o output style",
	"\t\t s == standard output (default)",
	"\t\t g == graphical output",
	"\t\t c == classic output",
	"\t\t m == minimal output",
	"\t\t p == scriptable output",
	"\t\t n == no individual hop output, accounting only",
	"\t\t C == (n)curses output",
	"\t-t tcp flags to use",
	"\t\t S == SYN (default)",
	"\t\t A == ACK",
	"\t\t R == RST",
	"\t\t U == URG",
	"\t\t P == PUSH",
	"\t\t F == FIN",
	"\t\t E == ECE",
	"\t\t C == CWR",
	"\t-R start at max ttl and decrement",
	"\t-h this help message",
	"\t-v version info",
	"",
	"Note that you can set generally set illogical or contradictory",
	"options if you wish, combining -d 80 and -p icmp is permissible",
	"but silly.  The option that doesn't make sense is usually ignored.",
	"In the case of contradictory options, the last one seen is",
	"generally authoritative",
	"" };

	for ( i = 0; i < sizeof (  usage_str ) / sizeof ( char * ); i++ )
		printf ( "%s\n", usage_str [ i ] );
}
