// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "Vec.h"
#include "mx_low_level.h"

namespace Torch {

Vec::Vec(real * ptr_, int n_dim)
{
  ptr = ptr_;
  n = n_dim;
  ptr_is_allocated = false;
}

Vec::Vec(int n_dim)
{
  n = n_dim;
  ptr = (real *) xalloc(sizeof(real) * n);
  ptr_is_allocated = true;
}

void Vec::copy(Vec * vec, int start_i)
{
  if (vec == this)
    return;

  real *ptr_r = vec->ptr + start_i;
  real *ptr_w = ptr + start_i;
  for (int i = 0; i < n - start_i; i++)
    *ptr_w++ = *ptr_r++;
}

void Vec::zero()
{
  real *ptr_w = ptr;
  for (int i = 0; i < n; i++)
    *ptr_w++ = 0.;
}

real Vec::norm1(Vec * weights)
{
  real sum = 0.0;
  real *ptr_x = ptr;
  if (weights)
  {
    real *ptr_w = weights->ptr;
    for (int i = 0; i < n; i++)
      sum += *ptr_w++ * fabs(*ptr_x++);
  }
  else
  {
    for (int i = 0; i < n; i++)
      sum += fabs(*ptr_x++);
  }

  return sum;
}

real Vec::norm2(Vec * weights)
{
  real sum = 0.0;
  real *ptr_x = ptr;
  if (weights)
  {
    real *ptr_w = weights->ptr;
    for (int i = 0; i < n; i++)
    {
      real z = *ptr_x++;
      sum += *ptr_w++ * z * z;
    }
  }
  else
  {
    for (int i = 0; i < n; i++)
    {
      real z = *ptr_x++;
      sum += z * z;
    }
  }

  return sqrt(sum);
}

real Vec::normInf()
{
  real *ptr_x = ptr;
  real max_val = fabs(*ptr_x++);

  for (int i = 1; i < n; i++)
  {
    real z = fabs(*ptr_x);
    if (max_val < z)
      max_val = z;
    ptr_x++;
  }

  return max_val;
}

real Vec::iP(Vec * vec, int start_i)
{
  return (mxIp__(ptr + start_i, vec->ptr + start_i, n - start_i));
}

Vec *Vec::subVec(int dim1, int dim2)
{
  Vec *vec = new Vec(ptr + dim1, dim2 - dim1 + 1);

  return (vec);
}

Vec::~Vec()
{
  if (ptr_is_allocated)
    free(ptr);
}

}

