// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "SaturationMeasurer.h"

namespace Torch {

SaturationMeasurer::SaturationMeasurer(GradientMachine *machine_, DataSet *data_, FILE *file_) : Measurer(data_, file_)
{
  machine = machine_;
  measure = xalloc(sizeof(real)*4);

  real *r_measure = (real *)measure;
  sum_outputs = r_measure;
  sum_derivees = r_measure+1;
  sum_outputs_c = r_measure+2;
  sum_derivees_c = r_measure+3;

  reset();
}

SaturationMeasurer::SaturationMeasurer(GradientMachine *machine_, DataSet *data_, const char *filename) : Measurer(data_, filename)
{
  machine = machine_;
  measure = xalloc(sizeof(real)*4);

  real *r_measure = (real *)measure;
  sum_outputs = r_measure;
  sum_derivees = r_measure+1;
  sum_outputs_c = r_measure+2;
  sum_derivees_c = r_measure+3;

  reset();
}

void SaturationMeasurer::measureEx()
{
  List *outputs_ = machine->outputs;
  while(outputs_)
  {
    real *out = (real *)outputs_->ptr;
    for(int i = 0; i < outputs_->n; i++)
    {
      *sum_outputs_c += fabs(*out++);
      n_sum_out++;
    }
    outputs_ = outputs_->next;
  }

  List *der_params_ = machine->der_params;
  while(der_params_)
  {
    real *der = (real *)der_params_->ptr;
    for(int i = 0; i < der_params_->n; i++)
    {
      *sum_derivees_c += fabs(*der++);
      n_sum_der++;
    }
    der_params_ = der_params_->next;
  }
}

void SaturationMeasurer::measureIter()
{
  if(n_sum_out != 0)
    *sum_outputs = *sum_outputs_c / n_sum_out;
  if(n_sum_der != 0)
    *sum_derivees = *sum_derivees_c / n_sum_der;

  fprintf(file, "%g %g\n", *sum_outputs, *sum_derivees);
  fflush(file);
  reset();
}

void SaturationMeasurer::init()
{
  Measurer::init();
  reset();
}

void SaturationMeasurer::reset()
{
  *sum_outputs_c = 0;
  *sum_derivees_c = 0;
  n_sum_out = 0;
  n_sum_der = 0;
}

SaturationMeasurer::~SaturationMeasurer()
{
  free(measure);
}

}

