// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "LogSigmoid.h"

namespace Torch {

LogSigmoid::LogSigmoid(int n_units)
{
  n_outputs = n_units;
  buff = NULL;
}

int LogSigmoid::numberOfParams()
{
  return(0);
}

void LogSigmoid::allocateMemory()
{
  GradientMachine::allocateMemory();
  buff = (real *)xalloc(sizeof(real)*n_outputs);
}

void LogSigmoid::freeMemory()
{
  if(is_free)
    return;

  GradientMachine::freeMemory();
  free(buff);

  is_free = true;
}

void LogSigmoid::forward(List *inputs)
{
  real *ptr_out = (real *)outputs->ptr;
  real *ptr_buff = buff;
  while(inputs)
  {
    real *x = (real *)inputs->ptr;
    for(int j = 0; j < inputs->n; j++)
    {
      *ptr_buff = exp(- *x++);
      *ptr_out++ = -log(1.+ *ptr_buff++);
    }
    inputs = inputs->next;
  }
}

void LogSigmoid::backward(List *inputs, real *alpha)
{
  real *alpha_ptr = alpha;
  real *beta_ptr = beta;
  real *buff_ptr = buff;
  for(int i = 0; i < n_outputs; i++)
  {
    real z = *buff_ptr++;
    *beta_ptr++ = *alpha_ptr++ * z / (1. + z);
  }
}

LogSigmoid::~LogSigmoid()
{
  freeMemory();
}

}

