/* Tower Toppler - Nebulus
 * Copyright (C) 2000-2006  Andreas Röver
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef MENUSYS_H
#define MENUSYS_H

#include <string>
#include <vector>
#include <map>

#include "event.h"
#include "decl.h"

#include "SDL_keyboard.h"


class Menusystem; 

/* Menu option flags */
typedef enum {
  MOF_NONE     = 0x00,
  MOF_PASSKEYS = 0x01,  /* Do keys get passed onto this option? */
  MOF_LEFT     = 0x02,  /* Option string is left justified */
  MOF_RIGHT    = 0x04   /* Option string is right justified */
} menuoptflags;


/* menu option callback function. gets called with the
   menusystem as it's parameter, and should return a
   string describing this option. If the parameter
   is null, then this is called just to get the
   string back. */
typedef const char *FDECL((*menuopt_callback_proc), (Menusystem *));

/* menu background callback procedure. this should draw the
   background screen for the menu. */
typedef void FDECL((*menubg_callback_proc), (void));
    
bool men_input(char* origs, int max_len, int xpos, int ypos,
                               const char *allowed);

class Menuoption {

public:
  SDLKey get_quickkey;
  Menuoption(const char *nname,menuopt_callback_proc pr,
                       SDLKey nquickkey, menuoptflags nflags, int nstate);
  const std::string &get_name() {return name;}
  menuoptflags get_flags() {return flags;}
  bool is_empty() {return(name.empty()); }
  bool has_proc() {return callback;}
  const char *run_proc(Menusystem *m) {return (*callback)(m); }
  void set_name(const char *n) {name=n; }
private:
  std::string name;
  const menuopt_callback_proc callback;
  const SDLKey quickkey;
  const menuoptflags flags;
  int state;
};


class Menusystem {
public:
    Menusystem(const char *title, menuopt_callback_proc pr, 
               int molen, int ystart);

    static void set_bgproc(menubg_callback_proc proc);
    static void call_bgproc() {if(bgproc) (*bgproc)(); } 
    void add_option(const char *name,menuopt_callback_proc pr,
		    SDLKey quickkey = SDLK_UNKNOWN, 
		    menuoptflags flags = MOF_NONE, int state = 0);
    void add_spacer();
    void run(Menusystem *parent=0);
    void set_timeproc(long t, menuopt_callback_proc pr);
    Event *get_last_event() { return last_ev; }
    void clear_last_event() {if(last_ev) delete(last_ev); last_ev=0; }
    void set_last_event(Event *e) {if(last_ev) delete(last_ev); last_ev=e; }
    void capture() { captured=hilited; }
    void release() { captured=-1; }
    void set_exit() { exitmenu=true; } 
    void set_state(int s) { state=s; }
    int get_state() { return(state); } 
    void set_wraparound(bool w) { wraparound=w; }
private:
    void reset_hilited();
    void move_hilite_down();
    void move_hilite_up();

    Menusystem *parent;
    void draw_background();
    void draw(int dx,int dy);
    int hilited;
    int captured;
    int yhilitpos;
    int maxoptlen; // maximum width of an option (for bar)
    int curr_mtime;
    bool exitmenu;
    bool wraparound;
    int mtime;
    bool opt_steal_control;
    int ystart;
    std::string title;
    std::vector<Menuoption *> options;
    static menubg_callback_proc bgproc;
    menuopt_callback_proc callback;
    menuopt_callback_proc timeproc; // should this be static?
    Event *last_ev;
    std::map<SDLKey,int> quickkeys;
    int state; // a global state var someone can use
};



#endif

