namespace eval grep {
    variable grep

    # Define the grep array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set grep {
	list.reset ""
	list.clear ""
	entry.file ""
	entry.regexp ""
	dialog ""
	file ""
	regexp ""
	regexp_type ""
	context ""
	byte_offset ""
	count ""
	no_filename ""
	ignore_case ""
	file_match ""
	line_number ""
	quiet ""
	silent ""
	revert_match ""
	word_regexp ""
	line_regexp ""
	binary ""
	unix_byte_offset ""
	protect_regexp ""
    }
}

# grep::create --
#
#   Method to create the dialog box for the grep command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::create { } {
    global tkWorld
    variable grep

    # Put the focus on the grep dialog if it is already open.
    if [winfo exists $grep(dialog)] {
	switch -- [wm state $grep(dialog)] {
	    normal {
		raise $grep(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $grep(dialog)
	    }
	}
	focus $grep(dialog)
	return
    } else {
	set grep(dialog) [dialog::create .grep Grep]
    }

    # The first tab has file and regexp information, along with
    # the standard command line options.
    set tab1 [tabnotebook::page [dialog::interior $grep(dialog)] "Options"]

    # Use a frame to encapsulate the File and Regexp options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command grep::open
    set grep(entry.file) [entry $f1.entry_file \
	    -width 35 \
	    -textvariable grep::grep(file)]
    label $f1.regexp_label \
	    -text "Regexp" \
	    -anchor e \
	    -width 7
    set grep(entry.regexp) [entry $f1.entry_regexp \
	    -width 25 \
	    -textvariable grep::grep(regexp)]
    grid $f1.file_button $f1.entry_file \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.regexp_label $f1.entry_regexp \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.byte_offset \
	    -text "Byte Offset" \
	    -variable grep::grep(byte_offset) \
	    -onvalue "b" \
	    -offvalue ""
    checkbutton $f2.count \
	    -text "Count Matches" \
	    -variable grep::grep(count) \
	    -onvalue "c" \
	    -offvalue ""
    checkbutton $f2.no_filename \
	    -text "No Filename" \
	    -variable grep::grep(no_filename) \
	    -onvalue "h" \
	    -offvalue ""
    checkbutton $f2.ignore_case \
	    -text "Ignore Case" \
	    -variable grep::grep(ignore_case) \
	    -onvalue i \
	    -offvalue ""
    checkbutton $f2.line_number \
	    -text "Line Number" \
	    -variable grep::grep(line_number) \
	    -onvalue "n" \
	    -offvalue ""
    checkbutton $f2.quiet \
	    -text "Quite Mode" \
	    -variable grep::grep(quiet) \
	    -onvalue "q" \
	    -offvalue ""
    checkbutton $f2.silent \
	    -text "Silent Mode" \
	    -variable grep::grep(silent) \
	    -onvalue "s" \
	    -offvalue ""
    checkbutton $f2.revert_match \
	    -text "Revert Match" \
	    -variable grep::grep(revert_match) \
	    -onvalue "v" \
	    -offvalue ""
    checkbutton $f2.word_regexp \
	    -text "Match Whole Word" \
	    -variable grep::grep(word_regexp) \
	    -onvalue "w" \
	    -offvalue ""
    checkbutton $f2.line_regexp \
	    -text "Match Whole Line" \
	    -variable grep::grep(line_regexp) \
	    -onvalue "x" \
	    -offvalue ""
    checkbutton $f2.binary \
	    -text "Binary File" \
	    -variable grep::grep(binary) \
	    -onvalue "U" \
	    -offvalue ""
    checkbutton $f2.unix_byte_offset \
	    -text "Unix Byte Offset" \
	    -variable grep::grep(unix_byte_offset) \
	    -onvalue "e" \
	    -offvalue ""
    checkbutton $f2.protect_regexp \
	    -text "Protect Regexp" \
	    -variable grep::grep(protect_regexp) \
	    -onvalue "--" \
	    -offvalue ""

    grid $f2.byte_offset $f2.count $f2.no_filename \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.ignore_case $f2.line_number $f2.quiet \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.silent $f2.revert_match $f2.word_regexp \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.line_regexp $f2.binary $f2.unix_byte_offset \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.protect_regexp \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build the first tab.
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $grep(entry.file)

    # The second tab has to do with exclusive relationships on the
    # grep command and regular expression.
    set tab2 [tabnotebook::page [dialog::interior .grep] "Advanced"]

    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]

    # Radio box for command engine types.
    set rb1 [radiobox::create $f3.engine_type "Engine Type"]
    foreach {a b} {Basic B Extended E Fixed F Off ""} {
	radiobox::add $rb1 $a $b grep::grep(regexp_type)
    }

    # Radio box for how man lines to print out besides the
    # matching line.
    set rb2 [radiobox::create $f3.context "Context"]
    foreach {a b} {Both 5 Trailing A5 Leading B5 Off ""} {
	radiobox::add $rb2 $a $b grep::grep(context)
    }

    # Radiobox to print out matching/non-matching filenames.
    set rb3 [radiobox::create $f3.file "Print Files"]
    foreach {a b} {"Without Match" L "With Match" l Off ""} {
	radiobox::add $rb3 $a $b grep::grep(file_match)
    }

    # Pack up each radiobox.
    grid $rb1 $rb2 $rb3 \
	    -sticky n \
	    -padx 2 \
	    -pady 2

    # Move the radioboxes to the top of the second tab.
    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Define the lists for the reset and clear methods
    set grep(list.reset) "byte_offset count no_filename ignore_case \
	    line_number quiet silent revert_match word_regexp \
	    line_regexp binary unix_byte_offset protect_regexp \
	    regexp_type context file_match"
    set grep(list.clear) "file regexp"
}

# grep::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::ok { } {
    global tkWorld
    variable grep

    set flist ""
    foreach f $grep(file) {
	if [regexp $tkWorld(working_dir) $f] {
	    append flist "$f "
	} else {
	    append flist "$tkWorld(working_dir)/$f "
	}
    }

    # Build the command line argument based on the basic options
    # with no arguments.
    set cmd_arg ""
    foreach x $grep(list.reset) {
	if [string length $grep($x)] {
	    switch $x {
		context -
		protect_regexp {
		    # Skip these for special treatment later.
		}
		default {
		    append cmd_arg $grep($x)
		}
	    }
	}
    }
    if [string length $cmd_arg] {
	set cmd_arg "-$cmd_arg "
    }

    # The value of advanced options which use the radiobox object
    # and must be seperate from the basic options on the command line.
    foreach i "context protect_regexp" {
	if [string length $grep($i)] {
	    append cmd_arg "-$grep($i) "
	}
    }

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	set cmd_arg [string trimright $cmd_arg " "]
	$tkWorld(cmd_center) insert insert \
		[subst {grep $cmd_arg "$grep(regexp)" $flist}]
    } else {
	$tkWorld(cmd_center) insert insert \
		[subst {grep "$grep(regexp)" $flist}]
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the file entry.
    $grep(entry.file) configure -bg white -fg black
    $grep(entry.regexp) configure -bg white -fg black

    # Give a warning on null regular expressions.
    if ![string length $grep(regexp)] {
	$grep(entry.regexp) configure -bg yellow
	stderr::log 3001
    }
}

# grep::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::reset { } {
    variable grep

    # It is easier to list the elements to exclude rather than list
    # all of the values to reset.
    foreach x $grep(list.reset) {
	set grep($x) ""
    }
}

# grep::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::clear { } {
    variable grep

    # Reset the data structure elements and bg/fg.
    foreach x $grep(list.clear) {
	set grep($x) ""
	$grep(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $grep(entry.file)
}

# grep::help --
#
#   Method to invoke the Grep Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::help { } {
    global tkWorld

    help::create "help/grep.html" "Grep Command Help"
}

# grep::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::close { } {
    variable grep
    
    balloonhelp::cancel
    destroy $grep(dialog)
}

# grep::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc grep::open { } {
    variable grep

    # Insert the file list and move the cursor.
    $grep(entry.file) insert insert [file::select]
    $grep(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $grep(entry.file)
}
