/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	 Department of Computer Science,
 *       The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#if __GNUC__
#define UNUSED __attribute__ ((unused)) 
#else
#define UNUSED
#endif

static char rcsid[] UNUSED
 = "$Id: isemLoad.cpp 1.3 Sat, 13 Sep 1997 12:04:11 -0600 maccabe $";

#define IDENT_CHARS "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <tk.h>

#include "config.h"     /* Get configure-time defs (e.g., possible WORDS_BIGENDIAN) */

#include "a_out.h"
#include "sizedefs.h"
#include "RegBlock.h"
#include "IU.h"
#include "MMU.h"
#include "globals.h"

#include "Instruct.h"
#include "Isem.h"

//
//-----------------------------------------------------------------------------
// Symbol Tables
static Tcl_HashTable super_data_syms, super_text_syms;
static Tcl_HashTable user_data_syms, user_text_syms;
static char *super_strings = NULL, *user_strings = NULL;

static char tmp_buf[256];
static char xtmp_buf[256];
static char super_labl_buf[256], user_labl_buf[256];
static unsigned long super_laddr, user_laddr;

static inline char *get_label( Tcl_HashTable *tbl, unsigned long addr ) {
    Tcl_HashEntry *entry = Tcl_FindHashEntry( tbl, (char *)addr );
    if( entry != NULL ) {
	return (char *)Tcl_GetHashValue(entry);
    } else {
	return NULL;
    }
}

static inline void swap_ends(char& a, char&b) {
    char temp = a;
    a = b;
    b = temp;
}

#ifdef WORDS_BIGENDIAN
static inline void big_to_little_endian(void*, size_t, int) {
    return;
}
#else
static void big_to_little_endian(void* obj, size_t sz, int n) {
    char* p = (char*)obj;

    Assert( !(sz&3), "unable to swap bytes properly" );

    while (n--) {
        for (unsigned i = 0; i < sz; i+= sizeof(UInt32), p+=sizeof(UInt32)) {
            //UInt32* q = (UInt32*)p;

            swap_ends(p[0], p[3]);
            swap_ends(p[1], p[2]);
        }
    }
}
#endif

static void clear_hash_table( Tcl_HashTable *tbl ) {
    // clear all of the entries in a hash table

    Tcl_HashSearch srch;
    Tcl_HashEntry *cur;

    cur = Tcl_FirstHashEntry( tbl, &srch );
    while( cur != NULL ) {
	Tcl_DeleteHashEntry( cur );
	cur = Tcl_NextHashEntry( &srch );
    }
}

void CreateSymbolTable( Tcl_Interp *interp, FILE* infile, struct exec& header,
			char * mode ) 
{
    long FilePos;           
    nlist* SymTabEntry;
    char* StringTable;
    int SymTabEntries;

    // First, read the size of the string table from the file
    //
    int StrTabSize;
    FilePos = N_STROFF(header);
    fseek(infile, FilePos, 0);
    fread(&StrTabSize, sizeof(int), 1, infile);
    big_to_little_endian(&StrTabSize, sizeof(int), 1);

    // Now allocate memory from the string table and read it into 
    // memory from the file.
    //
    StringTable = new char[StrTabSize];
    FilePos = N_STROFF(header);
    fseek(infile, FilePos, 0);
    fread(StringTable, sizeof(char), StrTabSize, infile);

    // Read the Symbol Table entries from the file
    //
    SymTabEntries = int(header.a_syms / sizeof(nlist));
    SymTabEntry = new nlist[SymTabEntries+1];
    FilePos = N_SYMOFF(header);
    fseek(infile, FilePos, 0);
    fread(SymTabEntry, sizeof(nlist), SymTabEntries, infile);

    // Build up the return list and build the appropriate symbol tables
    //
    Tcl_HashTable *text_syms, *data_syms;
    if( strcmp( mode, "super" ) == 0 ) {
	text_syms = &super_text_syms;
	data_syms = &super_data_syms;
	if( super_strings != NULL ) {
	    delete super_strings;
	}
	super_strings = StringTable;
    } else {
	text_syms = &user_text_syms;
	data_syms = &user_data_syms;
	if( user_strings != NULL ) {
	    delete user_strings;
	}
	user_strings = StringTable;
    }

    clear_hash_table( text_syms );
    clear_hash_table( data_syms );

    for (int i = 0; i < SymTabEntries; i++) {
	big_to_little_endian(&SymTabEntry[i].n_un.n_name,
                             sizeof(SymTabEntry[i].n_un.n_name), 1);
	big_to_little_endian(&SymTabEntry[i].n_value,
                             sizeof(SymTabEntry[i].n_value), 1);

        char* str = &StringTable[int(SymTabEntry[i].n_un.n_name)];
	unsigned addr = SymTabEntry[i].n_value;

	char *sec;
	char *key = (char *)addr;
	Tcl_HashEntry *temp;
	Tcl_HashTable *tbl;
	if( (SymTabEntry[i].n_type & N_TYPE) == N_TEXT ) {
	    tbl = text_syms;
	    sec = "text";
	} else if( (SymTabEntry[i].n_type & N_TYPE) == N_DATA ) {
	    tbl = data_syms;
	    sec = "data";
	} else if( (SymTabEntry[i].n_type & N_TYPE) == N_BSS ) {
	    tbl = data_syms;
	    sec = "bss";
	} else if( (SymTabEntry[i].n_type & N_ABS) == N_ABS ) {
	    tbl = data_syms;
	    sec = "abs";
	} else {
	    printf( "oops! n_type = %d, str = %s\n", 
		    SymTabEntry[i].n_type & N_TYPE, str );
            continue;           // Should Assert(0,..) instead?
	}
	
	
	int is_new;
	temp = Tcl_CreateHashEntry( tbl, key, &is_new );

	// give preference to symbols that are legal identifiers
	// then to symbols that have the fewest leading underscores
	// finally, to symbols inserted earlier
	if( !is_new ) {
	    if( strspn(str,IDENT_CHARS) == strlen(str) ) {
		char *cur = (char *)Tcl_GetHashValue(temp);
		
		int j;
		for( j = 0 ; (cur[j]=='_') && (str[j]=='_') ; j++ ) {
		    /* nothing to do */
		}
		if( cur[j] == '_' ) {
		    Tcl_SetHashValue( temp, str );
		}
	    }
	} else {
	    Tcl_SetHashValue( temp, str );
	}

	char buf[256];
	sprintf( buf, "%s 0x%.8x %s", str, addr, sec );
	
	Tcl_AppendElement( interp, buf );
    }

    delete SymTabEntry;
}

//-----------------------------------------------------------------------------
int Isem_Load( ClientData, Tcl_Interp *interp, int argc, char *argv[] ) {
    //
    // usage:
    //    isem_load file [super]
    //
    long FilePos;            // File Position Placeholders
    UInt32 TxtLoadSpace;
    UInt32 DatLoadSpace;
    static int FirstLoad = 1;

    // Initialize the hash tables, so we can delete them as needed later
    if( FirstLoad ) {
	FirstLoad = 0;
	Tcl_InitHashTable( &super_text_syms, TCL_ONE_WORD_KEYS );
	Tcl_InitHashTable( &super_data_syms, TCL_ONE_WORD_KEYS );
	Tcl_InitHashTable( &user_text_syms, TCL_ONE_WORD_KEYS );
	Tcl_InitHashTable( &user_data_syms, TCL_ONE_WORD_KEYS );
    }

    // Determine address space to load to
    //
    if( argc != 3 ) {
	sprintf( interp->result, "isem_load:  invalid number of arguments: %d",
		 argc );
	return TCL_ERROR;
    } else if( strcmp(argv[2], "user") == 0 ) {
  	TxtLoadSpace = AS_U_TEXT;
	DatLoadSpace = AS_U_DATA;
	iu->PSR(iu->PSR() & ~0x80);	// clear supervisor mode bit
	iu->PSR(iu->PSR() |  0x20);	// enable traps
	user_laddr = 0;
	user_steps = 0;
	user_mems = 0;
    } else if( strcmp(argv[2], "super") == 0 ) {
  	TxtLoadSpace = AS_S_TEXT;
	DatLoadSpace = AS_S_DATA;
	iu->PSR(iu->PSR() |  0x80);	// set supervisor mode bit
	iu->PSR(iu->PSR() & ~0x20);	// disable traps
	super_laddr = 0;
	super_steps = 0;
	super_mems = 0;
    } else {
	sprintf( interp->result, "isem_load:  unknown option: %s", argv[2] );
	return TCL_ERROR;
    }

    FILE *infile;
    if( (infile=fopen(argv[1],"rb")) == NULL ) {
	sprintf( interp->result, "Error:  couldn't open %s for reading",
		 argv[1] );
        return TCL_ERROR;
    }

    // Pull out the header information
    //
    struct exec header_info;            // defined in <sys/exec.h>
    if( fread(&header_info,sizeof(header_info),1,infile) != 1 ) {
        sprintf( interp->result, "Error:  couldn't read SPARC header"
		 " information from %s", argv[1] );;
        return TCL_ERROR;
    }

    big_to_little_endian( &header_info, sizeof(header_info), 1 );

    // Check for proper executable type
    //
    if( N_MACHTYPE(header_info) != M_SPARC ) {
        sprintf( interp->result, "Error:  %s is not a SPARC executable\n"
		 "\tNMACH = %d, M_SPARC = %d\n", argv[1], 
		 N_MACHTYPE(header_info), M_SPARC );
        return TCL_ERROR;
    }

    // We can handle other formats, but not at this phase of testing
    //
    if( N_MAGIC(header_info) != ZMAGIC ) {
        sprintf( interp->result, "Error:  %s is not a page aligned executable"
		 " file", argv[1] );
        return TCL_ERROR;
    }

    // We could actually just use the header_info structure all
    // the time, but this makes things a bit more readable
    //
    int TxtSize = header_info.a_text;
    int TxtOffset = N_TXTOFF(header_info);
    int TxtAddr = N_TXTADDR(header_info);
    int TxtEnd = TxtAddr + TxtSize;

    int DatSize = header_info.a_data;
    int DatOffset = N_DATOFF(header_info);
    int DatAddr = N_DATADDR(header_info);

    int BssSize = header_info.a_bss;
    int BssAddr = N_BSSADDR(header_info);

    FilePos = TxtOffset;
    fseek(infile, FilePos, 0);

    // the first four elements of the return value are the text start,
    // text end, data start, and data end
    char buf[256];
    sprintf( buf, "0x%.8x", TxtAddr+sizeof(struct exec) );
    Tcl_AppendElement( interp, buf );
    sprintf( buf, "0x%.8x", TxtEnd-4 );
    Tcl_AppendElement( interp, buf );
    sprintf( buf, "0x%.8x", DatAddr );
    Tcl_AppendElement( interp, buf );
    sprintf( buf, "0x%.8x", BssAddr+BssSize-4 );
    Tcl_AppendElement( interp, buf );

    int k;
    for( k = TxtAddr; k < TxtEnd ; k+=4 ) {
        UInt32 wbuf;

        fread(&wbuf, sizeof(wbuf), 1, infile);
        big_to_little_endian(&wbuf, sizeof(wbuf), 1);

        mmu->write( TxtLoadSpace, k, 0xF, wbuf );
        Assert(mmu->read(TxtLoadSpace, k) == wbuf, "MMU write failed");
    }

    FilePos = DatOffset;
    fseek(infile, FilePos, 0);
    
    for ( k = 0; k < DatSize ; k+=4) {
         UInt32 wbuf;
         fread(&wbuf, sizeof(wbuf), 1, infile);
         big_to_little_endian(&wbuf, sizeof(wbuf), 1);

	 mmu->write(DatLoadSpace, k+DatAddr, 0xF, wbuf);
	 Assert(mmu->read(DatLoadSpace,k+DatAddr) == wbuf, "MMU write failed");
    }

    // Build the symbol table for use by the parser and the
    // disassembler.
    //
    CreateSymbolTable(interp, infile, header_info, argv[2] );

    iu->PC(header_info.a_entry);

    fclose(infile);
    return TCL_OK;
}

//
//-----------------------------------------------------------------------------
// Instruction Disassembly
//-----------------------------------------------------------------------------
//
static char **reg_name;

static char *win_rnames[] = {
    "%g0","%g1","%g2","%g3","%g4","%g5","%g6","%g7",
    "%o0","%o1","%o2","%o3","%o4","%o5","%o6","%o7",
    "%l0","%l1","%l2","%l3","%l4","%l5","%l6","%l7",
    "%i0","%i1","%i2","%i3","%i4","%i5","%i6","%i7" };

static char *reg_rnames[] = {
    "%r0","%r1","%r2","%r3","%r4","%r5","%r6","%r7","%r8","%r9",
    "%r10","%r11","%r12","%r13","%r14","%r15","%r16","%r17","%r18","%r19",
    "%r20","%r21","%r22","%r23","%r24","%r25","%r26","%r27","%r28","%r29",
    "%r30","%r31" };

static char *spc_reg[] = { "%y", "%psr", "%wim", "%tbr" };

static char *ticc_name[] = {
    "tn","te","tle","tl","tleu","tcs","tneg","tvs",
    "ta","tne","tg","tge","tgu","tcc","tpos","tvc" };

static char *bicc_name[] = {
    "bn","be","ble","bl","bleu","bcs","bneg","bvs",
    "ba","bne","bg","bge","bgu","bcc","bpos","bvc" };

static char *fmt2_name[] = {
    "unimp", "??","b","??","sethi","??","fb","cb"};

static char *fmt3_name[] = {
    "add","and","or","xor","sub","andn","orn","xnor",
    "addx","?","umul","smul","subx","?","udiv","sdiv",
    "addcc","andcc","orcc","xorcc","subcc","andncc","orncc","xnorcc",
    "addxcc","?","umulcc","smulcc","subxcc","?","udivcc","sdivcc",
    "taddcc","tsubcc","taddcctv","tsubcctv","mulscc","sll","srl","sra",
    "rdy","rdpsr","rdwim","rdtbr","I44?","I45?","I46?","I47?",
    "wr","wr","wr","wr","Fpop1","Fpop2","CPop1","CPop2",
    "jmpl","rett","t","iflush","save","restore","","",
    "ld", "ldub", "lduh", "ldd", "st", "stb", "sth", "std",
    "?","ldsb", "ldsh","?","?","ldstub","?","swap",
    "lda","lduba","lduha", "ldda", "sta", "stba", "stha", "stda",
    "?","ldsba","ldsha","?", "?","ldstuba","?","swapa",
    "ldf", "ldfsr", "?", "lddf","stf","stfsr","stdfq","stdf",
    "?","?","?","?","?","?","?","?", 
    "ldc", "ldcsr","?","lddc","stc","stcsr","stdcq","stdc" }; 

static inline char *inst_addr( Instruction instr ) {
    if( instr.i() ) {
	if( instr.rs1() == 0 ) {
	    sprintf( xtmp_buf, "%d", instr.simm13() );
	} else {
	    sprintf( xtmp_buf, "%s%+d", reg_name[instr.rs1()],
		     instr.simm13() );
	}
    } else if( instr.rs1() == 0 ) {
	    sprintf( xtmp_buf, "%s", reg_name[instr.rs2()] );
    } else if( instr.rs2() == 0 ) {
	    sprintf( xtmp_buf, "%s", reg_name[instr.rs1()] );
    } else {
	    sprintf( xtmp_buf, "%s+%s", reg_name[instr.rs1()], 
		     reg_name[instr.rs2()] );
    }
    return xtmp_buf;
}
	    
// Disassemble format 3 instructions - except load/store
//
void disassem3( Tcl_Interp *interp, Instruction& instr ) {
    int op3 = instr.op3();
    char *rs1 = reg_name[instr.rs1()];
    char *rs2 = reg_name[instr.rs2()];
    char *rd = reg_name[instr.rd()];
    int simm13 = instr.simm13();
    int immed = instr.i();

    if( op3 < Instruction::RDY ) {
	if( op3==Instruction::SUBcc && instr.rd()==0 ) {
	    Tcl_AppendElement( interp, "cmp" );
	    if( immed ) {
		sprintf( tmp_buf, "%s, %d", rs1, simm13 );
	    } else {
		sprintf( tmp_buf, "%s, %s", rs1, rs2 );
	    }
	} else if( op3==Instruction::ORcc && instr.rd()==0 
		   && !immed && instr.rs1()==0 ) {
	    Tcl_AppendElement( interp, "tst" );
	    sprintf( tmp_buf, "%s", rs2 );
	} else if( op3==Instruction::OR && instr.rs1()==0 ) {
	    if( immed ) {
		Tcl_AppendElement( interp, "set" );
		sprintf( tmp_buf, "%d, %s", simm13, rd );
	    } else {
		Tcl_AppendElement( interp, "mov" );
		sprintf( tmp_buf, "%s, %s", rs2, rd );
	    }
	} else if( (op3==Instruction::ADD || op3==Instruction::ADDcc
		    || op3==Instruction::SUB || op3==Instruction::SUBcc)
		   && immed &&  instr.rs1()==instr.rd() ) {
	    if( op3==Instruction::ADD ) {
		Tcl_AppendElement( interp, "inc" );
	    } else if( op3==Instruction::ADDcc ) {
		Tcl_AppendElement( interp, "inccc" );
	    } else if( op3==Instruction::SUB ) {
		Tcl_AppendElement( interp, "dec" );
	    } else if( op3==Instruction::SUBcc ) {
		Tcl_AppendElement( interp, "deccc" );
	    }

	    if( simm13 == 1 ) {
		sprintf( tmp_buf, "%s", rd );
	    } else {
		sprintf( tmp_buf, "%d, %s", simm13, rd );
	    }
	} else {
	    Tcl_AppendElement( interp, fmt3_name[op3] );
	    if( immed ) {
		sprintf( tmp_buf, "%s, %d, %s", rs1, simm13, rd );
	    } else {
		sprintf( tmp_buf, "%s, %s, %s", rs1, rs2, rd );
	    }
	}
	Tcl_AppendElement( interp, tmp_buf );
    } else if( Instruction::RDY<=op3 && op3<=Instruction::RDTBR ) {
	Tcl_AppendElement( interp, "mov" );
	sprintf( tmp_buf, "%s, %s", spc_reg[op3-Instruction::RDY], rd );
	Tcl_AppendElement( interp, tmp_buf );
    } else if( Instruction::WRY<=op3 && op3<=Instruction::WRTBR ) {
	char *spc = spc_reg[op3-Instruction::WRY];
	if( instr.rs1() == 0 ) {
	    Tcl_AppendElement( interp, "mov" );
	    if( immed ) {
		sprintf( tmp_buf, "%d, %s", instr.simm13(), spc );
	    } else {
		sprintf( tmp_buf, "%s, %s", rs2, spc );
	    }
	} else if( !immed && instr.rs2() == 0 ) {
	    Tcl_AppendElement( interp, "mov" );
	    sprintf( tmp_buf, "%s, %s", rs1, spc );
	} else {
	    Tcl_AppendElement( interp, "wr" );
	    if( immed ) {
		sprintf( tmp_buf, "%s, %d, %s", rs1, simm13, spc );
	    } else {
		sprintf( tmp_buf, "%s, %s, %s", rs1, rs2,spc );
	    }
	}
	Tcl_AppendElement( interp, tmp_buf );
    } else if( op3 == Instruction::JMPL ) {
	if( instr.rd() == 0 ) {
	    if( immed && simm13==8 && instr.rs1()==31 ) {
		Tcl_AppendElement( interp, "ret" );
		Tcl_AppendElement( interp, "" );
	    } else if( immed && simm13==8 && instr.rs1()==15 ) {
		Tcl_AppendElement( interp, "retl" );
		Tcl_AppendElement( interp, "" );
	    } else {
		Tcl_AppendElement( interp, "jmp" );
		Tcl_AppendElement( interp, inst_addr(instr) );
	    }
	} else if( instr.rd() == 15 ) {
	    Tcl_AppendElement( interp, "call" );
	    Tcl_AppendElement( interp, inst_addr(instr) );
	} else {
	    Tcl_AppendElement( interp, "jmpl" );
	    Tcl_AppendElement( interp, inst_addr(instr) );
	}
    } else if( op3 == Instruction::RETT ) {
	Tcl_AppendElement( interp, "rett" );
	Tcl_AppendElement( interp, inst_addr(instr) );
    } else if ( op3 == Instruction::Ticc ) {
	Tcl_AppendElement( interp, ticc_name[instr.cond()] );
	Tcl_AppendElement( interp, inst_addr(instr) );
    } else if( op3==Instruction::SAVE || op3==Instruction::RESTORE ) {
	Tcl_AppendElement( interp, fmt3_name[op3] );
	if( instr.rs1()==0 && instr.rd()==0 && !instr.i() && instr.rs2()==0 ) {
	    tmp_buf[0] = '\0';
	} else if( immed ) {
	    sprintf( tmp_buf, "%s, %d, %s", rs1, instr.simm13(), rd );
	} else {
	    sprintf( tmp_buf, "%s, %s, %s", rs1, rs2, rd );
	}
	Tcl_AppendElement( interp, tmp_buf );
    } else if( op3==Instruction::IFLUSH ) {
	Tcl_AppendElement( interp, "flush" );
	Tcl_AppendElement( interp, inst_addr(instr) );
    } else {
            Tcl_AppendElement( interp, fmt3_name[instr.op3()] );
            Tcl_AppendElement( interp, "" );
    }
}


// Disassemble Load Instructions
//
void dis_load( Tcl_Interp *interp, Instruction& instr ) {
    UInt32 ld_cmd  = instr.op3();
    UInt32 ld_type = (ld_cmd>>4) & 3;

    switch(ld_type) {
    case 3:	//CoProcessor Load
	Tcl_AppendElement( interp, "cpu_load" );
	Tcl_AppendElement( interp, "" );
	break;
    case 2:	// Floating Point Load
	Tcl_AppendElement( interp, "fpu_load" );
	Tcl_AppendElement( interp, "" );
	break;
    case 1:	// Alternate Space Integer Load
	Tcl_AppendElement( interp, fmt3_name[ld_cmd] );
	sprintf( tmp_buf, "[%s]%d, %s", inst_addr(instr), instr.asi(),
		 reg_name[instr.rd()] );
	Tcl_AppendElement( interp, tmp_buf );
	break;
    case 0:	// Integer Load
	Tcl_AppendElement( interp, fmt3_name[ld_cmd] );
	sprintf( tmp_buf, "[%s], %s", inst_addr(instr), reg_name[instr.rd()] );
	Tcl_AppendElement( interp, tmp_buf );
	break;
    }
}

// Disassemble Store Instructions
//
void dis_store( Tcl_Interp *interp, Instruction& instr ) {
    UInt32 st_cmd  = instr.op3();
    UInt32 st_type = (st_cmd>>4) & 3;

    switch(st_type) {
    case 3:	//CoProcessor Store
	Tcl_AppendElement( interp, "cpu_store" );
	Tcl_AppendElement( interp, "" );
	break;
    case 2:	// Floating Point Store
	Tcl_AppendElement( interp, "fpu_store" );
	Tcl_AppendElement( interp, "" );
	break;
    case 1:
	Tcl_AppendElement( interp, fmt3_name[st_cmd] ); 
	sprintf( tmp_buf, "%s, [%s]%d", reg_name[instr.rd()],
		 inst_addr(instr), instr.asi() );
	Tcl_AppendElement( interp, tmp_buf ); 
	break;
    case 0:
	Tcl_AppendElement( interp, fmt3_name[st_cmd] ); 
	sprintf( tmp_buf, "%s, [%s]", reg_name[instr.rd()], inst_addr(instr) );
	Tcl_AppendElement( interp, tmp_buf ); 
	break;
    }
}

//  Disassemble format 2 instructions
//
void disassem2( Tcl_Interp *interp, Instruction& instr, unsigned long addr,
		UInt32 TextSpace, Tcl_HashTable *text_syms,
		Tcl_HashTable *data_syms, int is_super ) {
    char* s;

    switch (instr.op2()) {
    case Instruction::UNIMP:    
	Tcl_AppendElement( interp, fmt2_name[0] ); 
	Tcl_AppendElement( interp, "" ); 
	break;
    case Instruction::Bicc: 	
	if( instr.a() ) {
	    sprintf( tmp_buf, "%s,a", bicc_name[instr.cond()] );
	    Tcl_AppendElement( interp, tmp_buf );
	} else {
	    Tcl_AppendElement( interp, bicc_name[instr.cond()] );
	}

	if( (s = get_label(text_syms, addr + (instr.disp22()<<2))) != NULL ) {
	    Tcl_AppendElement( interp, s );
	} else {
	    sprintf( tmp_buf, ".%+d", (instr.disp22()<<2) );
	    Tcl_AppendElement( interp, tmp_buf );
	}
	break;
    case Instruction::SETHI:
	if( instr.rd()==0 && instr.imm22()==0 ) {
	    Tcl_AppendElement( interp, "nop" );
	    Tcl_AppendElement( interp, "" );
	} else {
	    Tcl_AppendElement( interp, "sethi" );

	    unsigned labl = instr.imm22() << 10;
	    char *lbuf = is_super ? super_labl_buf : user_labl_buf;
	    unsigned long *ladr = is_super ? &super_laddr : &user_laddr;

	    // look at the next two instructions for the second half of a
	    // set
	    Instruction next( mmu->read(TextSpace,addr+4) );
	    Instruction nxtx( mmu->read(TextSpace,addr+8) );
	    if( next.op()==Instruction::FORMAT3 && next.op3()==Instruction::OR
		&& next.rs1()==next.rd() && next.rd()==instr.rd()
		&& next.i() ) {
		labl |= next.simm13() & 0x1fff;
		*ladr = addr+4;
	    } else if( nxtx.op()==Instruction::FORMAT3
		       && nxtx.op3()==Instruction::OR && nxtx.rs1()==nxtx.rd()
		       && nxtx.rd()==instr.rd() && nxtx.i() ) {
		labl |= nxtx.simm13() & 0x1fff;
		*ladr = addr+8;
	    }

	    char *l = get_label( data_syms, labl );
	    if( l == NULL ) {
		l = get_label( text_syms, labl );
	    }
	    if( l == NULL ) {
		sprintf( lbuf, "0x%x", labl );
	    } else {
		strcpy( lbuf, l );
	    }
	    sprintf( tmp_buf, "%%hi(%s), %s", lbuf, reg_name[instr.rd()] );
	    Tcl_AppendElement( interp, tmp_buf );
	}
	break;
#ifdef NYET
    case Instruction::FBfcc:
	UiDispMsg(fmt2_name[6]);
	UiDispMsg("\t");
        UiDispMsg(instruct.op2());
	UiDispEndl();
	break;
    case Instruction::CBccc:
	UiDispMsg(fmt2_name[7]);
	UiDispMsg("\t");
        UiDispMsg(instruct.op2());
	UiDispEndl();
	break;
#endif
    default:
	Tcl_AppendElement( interp, "????");
	Tcl_AppendElement( interp, "");
    }
}

static void disassm( Tcl_Interp *interp, unsigned long addr, UInt32 TextSpace,
		     Tcl_HashTable *text_syms, Tcl_HashTable *data_syms,
		     int is_super ) {
    char* s;

    Instruction instr( mmu->read(TextSpace,addr) );

    switch (instr.op()) {
    case Instruction::FORMAT1:
	Tcl_AppendElement( interp, "call" );

	if( (s = get_label(text_syms, addr + (instr.disp30()<<2))) != NULL ) {
	    Tcl_AppendElement( interp, s );
	} else {
	    sprintf( tmp_buf, ".%+d", (instr.disp30()<<2) );
	    Tcl_AppendElement( interp, tmp_buf );
	}
	break;
    case Instruction::FORMAT2:   
	disassem2( interp, instr, addr, TextSpace, text_syms, data_syms, 
		   is_super );
	break;
    case Instruction::FORMAT3: 
	if( is_super && super_laddr==addr ) {
	    Tcl_AppendElement( interp, "or" );
	    sprintf( tmp_buf, "%s, %%lo(%s), %s", reg_name[instr.rs1()],
		     super_labl_buf, reg_name[instr.rd()] );
	    Tcl_AppendElement( interp, tmp_buf );
	} else if( !is_super && user_laddr==addr ) {
	    Tcl_AppendElement( interp, "or" );
	    sprintf( tmp_buf, "%s, %%lo(%s), %s", reg_name[instr.rs1()],
		     user_labl_buf, reg_name[instr.rd()] );
	    Tcl_AppendElement( interp, tmp_buf );
	} else {
	    disassem3( interp, instr );
	}
	break;
    case Instruction::FORMAT3LS:     // Load/Store instruction
	if( (instr.op3()&4) != 0 ) {
	    int tmp = instr.op3();
	    if( tmp == Instruction::LDSTUB || tmp == Instruction::LDSTUBA || 
                tmp == Instruction::SWAP || tmp == Instruction::SWAPA ) {
	        dis_load( interp, instr );
	    } else {
	        dis_store( interp, instr );
            } 
	} else {
	    dis_load( interp, instr );
	}
	break;
    default:
	Tcl_AppendElement( interp, "?????" );
	Tcl_AppendElement( interp, "" );
    }
}

int Isem_Disasm( ClientData, Tcl_Interp *interp, 
		 int argc, char *argv[] ) {
    //
    // usage:
    //    isem_disasm addr (super | user)

    Tcl_HashTable *text_syms, *data_syms;
    UInt32 TextSpace;
    int is_super = 0;

    //
    // Determine address space to disassemble from
    //
    if( argc != 4 ) {
	sprintf( interp->result, "isem_disasm:  invalid number of"
		 " arguments: %d", argc );
	return TCL_ERROR;
    } else {
	if( strcmp(argv[2], "user") == 0 ) {
	    TextSpace = AS_U_TEXT;
	    text_syms = &user_text_syms;
	    data_syms = &user_data_syms;
	    is_super = 0;
	} else if( strcmp(argv[2], "super") == 0 ) {
	    TextSpace = AS_S_TEXT;
	    text_syms = &super_text_syms;
	    data_syms = &super_data_syms;
	    is_super = 1;
	} else {
	    sprintf( interp->result, "isem_disasm:  unknown option: %s",
		     argv[2] );
	    return TCL_ERROR;
	}
	if( strcmp( argv[3], "regular" ) == 0 ) {
	    reg_name = reg_rnames;
	} else if( strcmp( argv[3], "window" ) == 0 ) {
	    reg_name = win_rnames;
	} else {
	    sprintf( interp->result, "isem_disasm:  unknown option: %s",
		     argv[3] );
	    return TCL_ERROR;
	}
    }

    unsigned long addr = strtol( argv[1], NULL, 0 );

    // 
    // look for an instruction label
    //
    char *label = get_label( text_syms, addr );
    if( label != NULL ) {
	sprintf( tmp_buf, "%s:", label );
	Tcl_AppendElement( interp, tmp_buf );
    } else {
	Tcl_AppendElement( interp, "" );
    }

    disassm( interp, addr, TextSpace, text_syms, data_syms, is_super );

    return TCL_OK;
}
