/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue_select.c - select()-based event system
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "system.h"

#if HAVE_SYS_SELECT_H
#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/select.h>
#include <sys/time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "nqueue_select.h"
#include "types.h"
#include "thy.h"

/** Input/Output/Always-set queues. */
static fd_set input, output;

void
thy_nq_fd_control_select (int fd, thy_nq_event_t event, int set)
{
  if (fd < 0)
    return;

  if (fdstates[fd] && thy_nq_act_fds)
    thy_nq_act_fds--;

  if (trigger[fd])
    trigger_count--;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  fdstates[fd] = 0;
  trigger[fd] = 0;
  FD_CLR (fd, &input);
  FD_CLR (fd, &output);

  if (!set)
    return;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      FD_SET (fd, &input);
      break;
    case THY_NQ_EVENT_OUTPUT:
      FD_SET (fd, &output);
      break;
    case THY_NQ_EVENT_TRIGGER:
      trigger[fd] = 1;
      trigger_count++;
      break;
    default:
      break;
    }

  thy_nq_act_fds++;
  fdstates[fd] = 1;
}

int
thy_nq_init_select (void)
{
  FD_ZERO (&input);
  FD_ZERO (&output);
  return 0;
}

int
thy_nq_fd_check_select (int fd, thy_nq_event_t event)
{
  int rval = 0;

  if (fd < 0)
    return 0;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      if (FD_ISSET (fd, &input))
	rval = 1;
      break;
    case THY_NQ_EVENT_OUTPUT:
      if (FD_ISSET (fd, &output))
	rval = 1;
      break;
    case THY_NQ_EVENT_TRIGGER:
      if (trigger[fd])
	rval = 1;
    default:
      return 0;
    }

  if (rval && fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  return rval;
}

int
thy_nq_wait_select (long int timeout)
{
  struct timeval ts;

  ts.tv_sec = timeout;
  ts.tv_usec = 0;

  return select (thy_nq_max_fd + 1, &input, &output, NULL, &ts);
}
#endif

/* arch-tag: 751a6335-5fff-4647-aba4-8f131fb9215a */
