/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#ifndef ENTRY_H_INCLUDED
#define ENTRY_H_INCLUDED

#include <gdk--/font.h>
#include <gtk--/box.h>
#include <gtk--/checkbutton.h>
#include <gtk--/clist.h>
#include <gtk--/entry.h>
#include <gtk--/paned.h>
#include <gtk--/text.h>

#include "thoughttracker.h"
#include "linklist.h"

class TTApplication;

class TTEntryWidget : public Gtk::VBox {
  public:
    TTEntryWidget(TTApplication *application);
    virtual ~TTEntryWidget() {};

    /* tool bar button functionalities */
    virtual void btn_follow();
    virtual void btn_search();
    virtual void btn_add();
    virtual void btn_update();
    virtual void btn_del();
    virtual void btn_unlink();
    virtual void btn_clear();

    /* functions externally interfaced via menu items */
    virtual void import_details();
    virtual void export_details();
    virtual void revert();

    /* updates the widget's buttons */
    virtual void update_buttons();

    /* clears the widget's input fields */
    virtual void clear();

    /* saves the widget's data to a DB record, creates a new record if as_new
       is true; a FALSE return value indicates that things have not been
       saved and editing them ought to continue; to prevent accidental data
       loss, the user is asked for a confirmation before doing anything
       that overwrites existing data */
    virtual bool save_record(bool as_new, bool ask = true);

    /* loads the widget's data from the record with the specified ID */
    virtual void load_record(dbid_t);

    /* prepares widget state for it to be displayed */
    virtual void prepare_for_display() {
      update_buttons();
      w.summary->grab_focus();
    };

    /* sets the name of the font used as fixed-width font for the details
       widget */
    virtual void set_fixed_font(const string &name);

    /* sets/returns the pane widget to a specific position */
    virtual void set_pane_pos(gint pos) {
      w.pane->set_position(pos);
    }

    virtual gint get_pane_pos() {
      return w.pane->gtkobj()->child1_size;  // kludge
    }

  private:
    TTApplication *app;
    Gdk_Font fixed_font, normal_font;
    struct {  // relevant widgets
      Gtk::Paned *pane;
      Gtk::Entry *summary;
      Gtk::Text *details;
      Gtk::CheckButton *hub, *fixed;
      TTLinkList *list;
      // buttons
      struct { Gtk::Widget *follow, *update, *add, *del, *unlink; } b;
    } w;
    dbid_t selected;  // selected link's IDs
    bool modified;    // text fields modified?
    gint pane_pos, pane_max;  // to remember the position of the pane

    /* make [Return] on the link list follow the selected link */
    gint handle_ll_return(GdkEventKey *event) {
      if (event->keyval == 0xff0d && selected >= 0) btn_follow();
      return 0;
    }

    /* reacts to the selection of a result list row */
    virtual void handle_select_link(gint, gint, GdkEvent*);

    /* likewise */
    virtual void handle_unselect_link(gint, gint, GdkEvent*);

    /* sets the internal state of "modified" */
    virtual void handle_modification();

    /* sets up the details widget to use the appropriate font when the "fixed"
       button is toggled */
    virtual void handle_toggle_fixed(bool set_modified);

    /* handler called when the hub attribute is toggled */
    virtual void handle_toggle_hub();
};

#endif  /* ENTRY_H_INCLUDED */

