/***************************************************************************
                             th-app-window.c
                             ---------------
    begin                : Wed Feb 16 2005
    copyright            : (C) 200t by Tim-Philipp Müller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-app-window.h"
#include "th-job.h"
#include "th-job-config-dialog.h"
#include "th-job-progress-dialog.h"
#include "th-preferences-dialog.h"
#include "th-title-selector.h"
#include "th-utils.h"

#include <gst/gst.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include <string.h>
#include <unistd.h>

enum
{
  PROP_LAST_OUTPUT_DIRECTORY = 1
};

struct _ThAppWindowPrivate
{
	GtkActionGroup *action_group;

	ThPreferences  *prefs;

	GtkWidget   *notebook;
	GtkWidget   *menubar;
	GtkWidget   *top_vbox;

	GtkWidget   *ts;
	GtkWidget   *jcd;
	GtkWidget   *jpd;

	GtkWidget   *prefs_dialog;

	GList       *fake_dialogs;

	gchar       *last_output_dir;
};

static void             app_window_class_init           (ThAppWindowClass *klass);

static void             app_window_instance_init         (ThAppWindow *appwin);

static void             app_window_finalize              (GObject *object);

static void             app_window_add_menubar           (ThAppWindow *appwin);

static void             app_window_add_fake_dialog       (ThAppWindow *appwin, GtkWidget *fake_dialog);

static void             app_window_switch_to_fake_dialog (ThAppWindow *appwin, GtkWidget *fake_dialog);

static GtkWidget       *app_window_get_current_fake_dialog (ThAppWindow *appwin);

static void             app_window_update_action_sensitivity (ThAppWindow *appwin, GtkWidget *cur_dlg);

static void             appwin_action_about_cb      (GtkAction *action, ThAppWindow *appwin);

static void             appwin_action_quit_cb       (GtkAction *action, ThAppWindow *appwin);

static void             appwin_action_edit_prefs_cb (GtkAction *action, ThAppWindow *appwin);

static void             appwin_action_show_log_cb   (GtkAction *action, ThAppWindow *appwin);

static void             appwin_action_open_vob_folder_cb (GtkAction *action, ThAppWindow *appwin);

static void             appwin_action_open_iso_file_cb (GtkAction *action, ThAppWindow *appwin);

/* variables */

static GtkWindowClass  *appwin_parent_class; /* NULL */

static GtkActionEntry appwin_action_entries[] = 
{
  { "FileMenu",  
    NULL,
    N_("_File"),
    NULL,
    NULL,
    NULL,
  },
  { "EditMenu",  
    NULL,
    N_("_Edit"),
    NULL,
    NULL,
    NULL,
  },
  { "HelpMenu",
    NULL,
    N_("_Help"),
    NULL,
    NULL,
    NULL,
  },
  { "OpenVOBFolder", 
    GTK_STOCK_OPEN,  
    N_("_Open Folder with Video DVD Files"),  
    NULL,
    N_("Open a local folder containing .VOB and .IFO files from a video DVD"), 
    G_CALLBACK (appwin_action_open_vob_folder_cb)  
  },
  { "OpenISOFile", 
    GTK_STOCK_OPEN,  
    N_("Open .iso DVD image _file"),  
    NULL,
    N_("Open a file containing a DVD in form of an .iso image"),
    G_CALLBACK (appwin_action_open_iso_file_cb)
  },
  { "Quit", 
    GTK_STOCK_QUIT,  
    N_("_Quit"),
    "<control>Q",
    N_("Close this application."),
    G_CALLBACK (appwin_action_quit_cb)
  },
  { "EditPreferences",
    GTK_STOCK_PREFERENCES,
    N_("_Preferences"),
    NULL,
    N_("Edit your preferences"),
    G_CALLBACK (appwin_action_edit_prefs_cb)
  },
  { "About", 
    GTK_STOCK_ABOUT,
    N_("_About"),  
    NULL,         
    N_("About Thoggen"), 
    G_CALLBACK (appwin_action_about_cb)  
  },
  { "Log", 
    GTK_STOCK_DIALOG_INFO,
    N_("Show _Log"),  
    NULL,         
    N_("Show a window with information for debugging"), 
    G_CALLBACK (appwin_action_show_log_cb)  
  }
};

static const gchar *appwin_ui_description =
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu action='FileMenu'>"
"      <menuitem action='OpenVOBFolder'/>"
"      <menuitem action='OpenISOFile'/>"
"      <separator />"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='EditMenu'>"
"      <menuitem action='EditPreferences'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='Log'/>"
"      <separator />"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"</ui>";

/***************************************************************************
 *
 *   app_window_set_property
 *
 ***************************************************************************/

static void
app_window_set_property (GObject      *object,
                         guint         param_id,
                         const GValue *value,
                         GParamSpec   *pspec)
{
	ThAppWindow *appwin = (ThAppWindow*) object;

	g_return_if_fail (TH_IS_APP_WINDOW (appwin));
	
	switch (param_id)
	{
		case PROP_LAST_OUTPUT_DIRECTORY:
			g_free (appwin->priv->last_output_dir);
			appwin->priv->last_output_dir = g_value_dup_string (value);
			if (!appwin->priv->last_output_dir
			 || !g_file_test (appwin->priv->last_output_dir, G_FILE_TEST_IS_DIR))
			{
				g_warning ("NOT A DIRECTORY (%s), SETTING LAST-OUTPUT-DIR to %s!\n",
				           appwin->priv->last_output_dir, g_get_home_dir ());
				g_free (appwin->priv->last_output_dir);
				appwin->priv->last_output_dir = g_strdup (g_get_home_dir ());
			}
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   app_window_get_property
 *
 ***************************************************************************/

static void
app_window_get_property (GObject      *object,
                         guint         param_id,
                         GValue       *value,
                         GParamSpec   *pspec)
{
	ThAppWindow *appwin = (ThAppWindow*) object;

	g_return_if_fail (TH_IS_APP_WINDOW (appwin));
	
	switch (param_id)
	{
		case PROP_LAST_OUTPUT_DIRECTORY:
			g_value_set_string (value, appwin->priv->last_output_dir);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   app_window_class_init
 *
 ***************************************************************************/

static void
app_window_class_init (ThAppWindowClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	appwin_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = app_window_finalize;
	object_class->set_property = app_window_set_property;
	object_class->get_property = app_window_get_property;

	g_object_class_install_property (object_class, PROP_LAST_OUTPUT_DIRECTORY,
	                                 g_param_spec_string ("last-output-directory", 
	                                                      "last-output-directory", 
	                                                      "last-output-directory", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE));
}

/***************************************************************************
 *
 *   app_window_do_quit
 *
 ***************************************************************************/

static void
app_window_do_quit (ThAppWindow *appwin)
{
	g_assert (TH_IS_APP_WINDOW (appwin));

	if (appwin->priv->jpd)
		gtk_widget_destroy (appwin->priv->jpd);
	if (appwin->priv->jcd)
		gtk_widget_destroy (appwin->priv->jcd);
	if (appwin->priv->ts)
		gtk_widget_destroy (appwin->priv->ts);

	appwin->priv->jpd = NULL;
	appwin->priv->jcd = NULL;
	appwin->priv->ts = NULL;
}

/***************************************************************************
 *
 *   app_window_delete_event_cb
 *
 ***************************************************************************/

static gboolean
app_window_delete_event_cb (ThAppWindow *appwin)
{
	guint ret = GTK_RESPONSE_ACCEPT;

	if (app_window_get_current_fake_dialog (appwin) == appwin->priv->jpd)
	{
		GtkWidget *dlg;

		dlg = gtk_message_dialog_new_with_markup (GTK_WINDOW (appwin), 
		                                          GTK_DIALOG_MODAL,
		                                          GTK_MESSAGE_QUESTION,
		                                          GTK_BUTTONS_YES_NO,
		                                          _("Do you really want to quit?"));

		gtk_window_set_title (GTK_WINDOW (dlg), _("Do you really want to quit?"));

		ret = gtk_dialog_run (GTK_DIALOG (dlg));
	
		gtk_widget_destroy (dlg);
	}
	
	if (ret == GTK_RESPONSE_ACCEPT || ret == GTK_RESPONSE_YES)
		app_window_do_quit (appwin);

	return TRUE; /* don't destroy window (we do that later) */
}

/***************************************************************************
 *
 *   appwin_setup_preferences
 *
 ***************************************************************************/

static void
appwin_setup_preferences (ThAppWindow *appwin)
{
	appwin->priv->prefs = th_preferences_new ("preferences");

	th_preferences_register_proxy (appwin->priv->prefs,
	                               G_OBJECT (appwin->priv->jpd),
	                               "delete-partial-file-on-cancel",
	                               "delete-partial-file-on-cancel",
	                               NULL);

	th_preferences_register_proxy (appwin->priv->prefs,
	                               G_OBJECT (appwin),
	                               "last-output-directory",
	                               "last-output-directory",
	                               NULL);

	th_preferences_load (appwin->priv->prefs);
}

/***************************************************************************
 *
 *   app_window_destroy_cb
 *
 ***************************************************************************/

static void
app_window_destroy_cb (ThAppWindow *appwin)
{
	th_preferences_save (appwin->priv->prefs);
}

/***************************************************************************
 *
 *   app_window_instance_init
 *
 ***************************************************************************/

static void
app_window_instance_init (ThAppWindow *appwin)
{
	GtkNotebook *notebook;

	appwin->priv = g_new0 (ThAppWindowPrivate, 1);

	appwin->priv->top_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (appwin), appwin->priv->top_vbox);

	app_window_add_menubar (appwin);

	notebook = GTK_NOTEBOOK (gtk_notebook_new ());
	appwin->priv->notebook = GTK_WIDGET (notebook);

	gtk_box_pack_start (GTK_BOX (appwin->priv->top_vbox), 
	                    appwin->priv->notebook, 
	                    TRUE, TRUE, 0);

	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (appwin->priv->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (appwin->priv->notebook), FALSE);

	gtk_window_set_type_hint (GTK_WINDOW (appwin), GDK_WINDOW_TYPE_HINT_NORMAL);

	gtk_window_set_icon_from_file (GTK_WINDOW (appwin),
	                               DATADIR "/ui/icons/thoggen.png",
	                               NULL);

	appwin->priv->ts = th_title_selector_new ();
	appwin->priv->jcd = th_job_config_dialog_new ();
	appwin->priv->jpd = th_job_progress_dialog_new ();

	app_window_add_fake_dialog (appwin, appwin->priv->ts);
	app_window_add_fake_dialog (appwin, appwin->priv->jcd);
	app_window_add_fake_dialog (appwin, appwin->priv->jpd);

	g_signal_connect (appwin, "delete-event",
	                  G_CALLBACK (app_window_delete_event_cb), 
	                  NULL);

	g_signal_connect (appwin, "destroy",
	                  G_CALLBACK (app_window_destroy_cb), 
	                  NULL);

	gtk_widget_show_all (GTK_WIDGET (appwin));

	appwin->priv->last_output_dir = g_strdup (g_get_home_dir());

	appwin_setup_preferences (appwin);

	appwin->priv->prefs_dialog = th_preferences_dialog_new (appwin->priv->prefs);
}

/***************************************************************************
 *
 *   app_window_finalize
 *
 ***************************************************************************/

static void
app_window_finalize (GObject *object)
{
	ThAppWindow *appwin;

	appwin = (ThAppWindow*) object;

	th_log ("ThAppWindow: finalize\n");

	g_object_unref (appwin->priv->prefs);

	g_list_free (appwin->priv->fake_dialogs);
	appwin->priv->fake_dialogs = NULL;

	g_free (appwin->priv->last_output_dir);

	memset (appwin->priv, 0xab, sizeof (ThAppWindowPrivate));
	g_free (appwin->priv);
	
	/* chain up */
	G_OBJECT_CLASS (appwin_parent_class)->finalize (object);
}


/***************************************************************************
 *
 *   th_app_window_get_type
 *
 ***************************************************************************/

GType
th_app_window_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThAppWindowClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) app_window_class_init,
			NULL, NULL,
			sizeof (ThAppWindow),
			0,
			(GInstanceInitFunc) app_window_instance_init
		};

		type = g_type_register_static (GTK_TYPE_WINDOW, "ThAppWindow", &info, 0);
	}

	return type;
}

/***************************************************************************
 *
 *   app_window_add_menubar
 *
 ***************************************************************************/

static void
app_window_add_menubar (ThAppWindow *appwin)
{
	GtkActionGroup  *action_group;
	GtkAccelGroup   *accel_group;
	GtkUIManager    *ui_manager;
	GtkWidget       *menubar;
	GError          *error = NULL;
	
	ui_manager = gtk_ui_manager_new ();

	action_group = gtk_action_group_new ("ThAppWindowActions");
	
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);

	gtk_action_group_add_actions (action_group,
	                              appwin_action_entries,
	                              G_N_ELEMENTS (appwin_action_entries),
	                              appwin);
	
	gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);

	gtk_ui_manager_ensure_update (ui_manager);

	if (!gtk_ui_manager_add_ui_from_string (ui_manager, appwin_ui_description, -1, &error))
	{
		g_warning ("Building title selector menu failed: %s", error->message);
		g_error_free (error);
		return;
	}

	menubar = gtk_ui_manager_get_widget (ui_manager, "/MainMenu");
	gtk_box_pack_start (GTK_BOX (appwin->priv->top_vbox), menubar, FALSE, FALSE, 0);
	gtk_widget_show (menubar);

	accel_group = gtk_ui_manager_get_accel_group (ui_manager);
	gtk_window_add_accel_group (GTK_WINDOW (appwin), accel_group);

	appwin->priv->menubar = menubar;
	appwin->priv->action_group = action_group;
}

/***************************************************************************
 *
 *   th_app_window_new
 *
 ***************************************************************************/

GtkWidget *
th_app_window_new (void)
{
	ThAppWindow *appwin;
	
	appwin = (ThAppWindow *) g_object_new (TH_TYPE_APP_WINDOW, NULL);
	
	gtk_widget_show (GTK_WIDGET (appwin));

	return GTK_WIDGET (appwin);
}

/***************************************************************************
 *
 *   appwin_action_quit_cb
 *
 ***************************************************************************/

static void
appwin_action_quit_cb (GtkAction *action, ThAppWindow *appwin)
{
	app_window_do_quit (appwin);
}

/***************************************************************************
 *
 *   appwin_action_show_log_cb
 *
 ***************************************************************************/

static void
appwin_action_show_log_cb (GtkAction *action, ThAppWindow *appwin)
{
	th_log_show ();
}

/***************************************************************************
 *
 *   appwin_action_open_vob_folder_cb
 *
 ***************************************************************************/

static void
appwin_action_open_vob_folder_cb (GtkAction *action, ThAppWindow *appwin)
{
	GtkWidget *fc;
	gboolean   looksfine;
	gchar     *fn, *fn_video_ts_ifo;

	fc = gtk_file_chooser_dialog_new (NULL, /* "Load folder with VOB files ..." */
	                                  GTK_WINDOW (appwin),
	                                  GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, 
	                                  GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
	                                  NULL);

	if (gtk_dialog_run (GTK_DIALOG (fc)) != GTK_RESPONSE_ACCEPT)
	{
		gtk_widget_destroy (fc);
		return;
	}

	fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (fc));
	gtk_widget_destroy (fc);

	looksfine = TRUE;

	fn_video_ts_ifo = g_strdup_printf ("%s/video_ts.ifo", fn);

	if (!g_file_test (fn_video_ts_ifo, G_FILE_TEST_EXISTS))
	{
		g_free (fn_video_ts_ifo);
		fn_video_ts_ifo = g_strdup_printf ("%s/VIDEO_TS.IFO", fn);
		if (!g_file_test (fn_video_ts_ifo, G_FILE_TEST_EXISTS))
			looksfine = FALSE;
	}
	g_free (fn_video_ts_ifo);

	if (looksfine)
	{
		th_title_selector_add_vob_folder (TH_TITLE_SELECTOR (appwin->priv->ts), fn);
	}
	else
	{
		GtkWidget *dlg;
		dlg = gtk_message_dialog_new (GTK_WINDOW (appwin),
		                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		                GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, 
		                _("No VIDEO_TS.IFO file"));
		gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dlg), 
		                "The folder\n\n%s\n\n"
		                "does not really look like it contains a full\n"
		                "video DVD. There is no video_ts.ifo file.\n", fn);

		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}

	g_free (fn);
}

/***************************************************************************
 *
 *   appwin_action_open_iso_file_cb
 *
 ***************************************************************************/

static void
appwin_action_open_iso_file_cb (GtkAction *action, ThAppWindow *appwin)
{
	GtkFileFilter *filter;
	GtkWidget     *fc;
	gchar         *fn;

	fc = gtk_file_chooser_dialog_new (NULL, /* "Load DVD .iso image ..." */
	                                  GTK_WINDOW (appwin),
	                                  GTK_FILE_CHOOSER_ACTION_OPEN, 
	                                  GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
	                                  NULL);

	filter = gtk_file_filter_new ();
	gtk_file_filter_add_mime_type (filter, "application/x-cd-image");
	gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (fc), filter);

	if (gtk_dialog_run (GTK_DIALOG (fc)) != GTK_RESPONSE_ACCEPT)
	{
		gtk_widget_destroy (fc);
		return;
	}

	fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (fc));
	gtk_widget_destroy (fc);

	if (fn == NULL)
		return;

	th_title_selector_add_iso_file (TH_TITLE_SELECTOR (appwin->priv->ts), fn);
	g_free (fn);
}

/***************************************************************************
 *
 *   appwin_about_url_func
 *
 ***************************************************************************/

static void
appwin_about_url_func (GtkAboutDialog *about_dlg, const gchar *url, gpointer foobar)
{
	g_print ("Open browser with URL '%s'\n", url);
	(void) th_utils_invoke_browser_with_url (url);
}

/***************************************************************************
 *
 *   appwin_about_dialog_get_comments
 *
 ***************************************************************************/

static gchar *
appwin_about_dialog_get_comments (ThAppWindow *appwin)
{
	GstPlugin *core, *base, *good, *ugly;
	gchar *versions, *ret;

	core = gst_default_registry_find_plugin ("coreelements");
	base = gst_default_registry_find_plugin ("audioconvert");
	good = gst_default_registry_find_plugin ("videobox");
	ugly = gst_default_registry_find_plugin ("dvdread");

	versions = g_strdup_printf ("GStreamer core %s\n"
	    "GStreamer base %s\nGStreamer good %s\nGStreamer ugly %s",
	    gst_plugin_get_version (core), gst_plugin_get_version (base),
	    gst_plugin_get_version (good), gst_plugin_get_version (ugly));

	gst_object_unref (core);
	gst_object_unref (base);
	gst_object_unref (good);
	gst_object_unref (ugly);

	ret = g_strdup_printf ("%s\n\n%s",
	    _("DVD video extraction and backup tool"), versions);

	g_free (versions);

	return ret;
}

/***************************************************************************
 *
 *   appwin_action_about_cb
 *
 ***************************************************************************/

static void
appwin_action_about_cb (GtkAction *action, ThAppWindow *appwin)
{
	static const gchar *authors[] =
	{
		"Tim-Philipp M\303\274ller <tim@centricular.net>", NULL
	};
	
	GtkAboutDialog *dlg;
	GString        *fn;
	gchar          *comments;
	gchar          *license = NULL;
	
	dlg = (GtkAboutDialog*) gtk_about_dialog_new ();
	
	gtk_about_dialog_set_url_hook (appwin_about_url_func, NULL, NULL);
	
	gtk_about_dialog_set_name (dlg, PACKAGE);
	gtk_about_dialog_set_version (dlg, VERSION);
	gtk_about_dialog_set_authors (dlg, authors);
	gtk_about_dialog_set_copyright (dlg, "Copyright \302\251 2004-2005 Tim-Philipp M\303\274ller <tim@centricular.net>");
	gtk_about_dialog_set_website (dlg, "http://thoggen.net");
	gtk_about_dialog_set_translator_credits (dlg, _("translator-credits"));

	fn = g_string_new (DATADIR);
	if (strstr (fn->str, "thoggen"))
	{
		gsize llen;
		g_string_truncate (fn, strstr (fn->str, "thoggen") - fn->str);
		g_string_append (fn, "doc/thoggen/COPYING");
		if (!g_file_get_contents (fn->str, &license, &llen, NULL))
		{
			gchar *cmd, *stderr = NULL;
			gint   ret;

			g_warning ("Failed to load license text '%s'\n", fn->str);
			license = NULL;
			
			/* package managers might gzip it */
			g_string_append (fn, ".gz");
			cmd = g_strdup_printf ("zcat '%s'", fn->str);
			if (!g_spawn_command_line_sync (cmd, &license, &stderr, &ret, NULL) 
			 || ret != 0 || (stderr != NULL && *stderr != 0x00))
			{
				g_free (license);
				license = NULL;
			}
			g_free (cmd);
			g_free (stderr);
		}
	}

	if (license)
		gtk_about_dialog_set_license (dlg, license);
	else
		gtk_about_dialog_set_license (dlg, "GNU General Public License Version 2,\n"
		                                  " as published by the Free Software Foundation (FSF).\n"
		                                  " See http://www.gnu.org/copyleft/gpl.html");

	if (strstr (fn->str, "share/doc"))
	{
		GdkPixbuf *pixbuf;
		g_string_truncate (fn, strstr (fn->str, "share/doc") - fn->str);
		g_string_append (fn, "share/thoggen/ui/icons/thoggen.png");
		pixbuf = gdk_pixbuf_new_from_file (fn->str, NULL);
		if (pixbuf)
		{
			gtk_about_dialog_set_logo (dlg, pixbuf);
			g_object_unref (pixbuf);
		}
		else 
		{
			g_warning ("Failed to load logo '%s's\n", fn->str);
		}
	}

	comments = appwin_about_dialog_get_comments (appwin);
	gtk_about_dialog_set_comments (dlg, comments);
	g_free (comments);

	(void) gtk_dialog_run (GTK_DIALOG (dlg));
	
	gtk_widget_destroy (GTK_WIDGET (dlg));
	g_string_free (fn, TRUE);
	g_free (license);
}

/***************************************************************************
 *   
 *   appwin_action_edit_prefs_cb
 *
 ***************************************************************************/

static void
appwin_action_edit_prefs_cb (GtkAction *action, ThAppWindow *appwin)
{
	gtk_window_present (GTK_WINDOW (appwin->priv->prefs_dialog));
}

/***************************************************************************
 *
 *   app_window_check_free_disk_space
 *
 ***************************************************************************/

static gboolean
app_window_check_free_disk_space (ThAppWindow *appwin, GList *jobs)
{
	GtkWidget *dlg;
	GString   *errmsg;
	GList     *mountpoints, *problems, *l;
	gint       ret;

	if (!th_utils_free_space_check_init (&mountpoints))
	{
		g_warning ("Cannot check free disk space.");
		return TRUE; /* just continue */
	}
	
	for (l = jobs;  l;  l = l->next)
	{
		gdouble  output_size = 0.0;
		gchar   *output_fn;
		gint     output_quality;
		
		g_object_get (TH_JOB (l->data), 
		              "output-fn", &output_fn, 
		              "target-quality", &output_quality,
		              "target-output-size", &output_size,
		              NULL);

		/* when encoding with a quality target, 
		 *  we have no idea how much disk space 
		 *  we are going to need */
		if (output_quality > 0)
			output_size = 0.1;

		if (output_size > 0.0 && !th_utils_free_space_check_add_filename (&mountpoints, output_fn, output_size))
		{
			g_warning ("Cannot check free disk space for file '%s'", output_fn);
			/* just continue */
		}
	}
	
	if (mountpoints == NULL)
		return TRUE; /* we have already printed warnings, just continue */

	problems = th_utils_free_space_check (&mountpoints);
	
	if (problems == NULL)
	{
		th_utils_free_space_check_deinit (&mountpoints);
		return TRUE;
	}
	
	errmsg = g_string_new (NULL);
	
	g_string_append_printf (errmsg, "<b>%s\n%s</b>", 
	                        _("Not enough disk space available."),
	                        _("Do you want to continue anyway?"));
	
	g_string_append_printf (errmsg, "\n\n%s", _("Details:"));
	
	for (l = problems;  l;  l = l->next)
	{
		ThMountPoint *mp = (ThMountPoint*) l->data;
		
		g_string_append (errmsg, "\n\n");
		g_string_append_printf (errmsg, 
		                        _("Not enough space available on '%s'\n"
		                          "  Space available:\t%.1f MB\n"
		                          "  Space required:\t%.1f MB\n"),
		                         mp->mnt_point, mp->space_avail,
		                         mp->space_needed);
	}

	dlg = gtk_message_dialog_new_with_markup (GTK_WINDOW (appwin), 
	                                          GTK_DIALOG_MODAL,
	                                          GTK_MESSAGE_WARNING,
	                                          GTK_BUTTONS_YES_NO,
	                                          errmsg->str);

	gtk_window_set_title (GTK_WINDOW (dlg), _("Not enough disk space - continue anyway?"));

	app_window_switch_to_fake_dialog (appwin, appwin->priv->jpd);

	ret = gtk_dialog_run (GTK_DIALOG (dlg));
	
	gtk_widget_destroy (dlg);

	g_string_free (errmsg, TRUE);
	
	g_list_free (problems);

	th_utils_free_space_check_deinit (&mountpoints);
	
	return (ret == GTK_RESPONSE_YES);
}

/***************************************************************************
 *
 *   app_window_add_fake_dialog
 *
 ***************************************************************************/

static void
app_window_add_fake_dialog (ThAppWindow *appwin, GtkWidget *fake_dialog)
{
	gint page_num;

	g_assert (TH_IS_FAKE_DIALOG (fake_dialog));

	page_num = gtk_notebook_append_page (GTK_NOTEBOOK (appwin->priv->notebook), 
	                                     fake_dialog, 
	                                     NULL);

	g_object_set_data (G_OBJECT (fake_dialog), 
	                   "th-app-win-page-num",
	                   GUINT_TO_POINTER (page_num));

	appwin->priv->fake_dialogs = g_list_append (appwin->priv->fake_dialogs, fake_dialog);
}

/***************************************************************************
 *
 *   app_window_resize_to_fit
 *
 ***************************************************************************/

static void
app_window_resize_to_fit (ThAppWindow *appwin, GtkWidget *fake_dialog)
{
	GtkRequisition req_menu, req_dlg;

	gtk_widget_size_request (appwin->priv->menubar, &req_menu);
	gtk_widget_size_request (fake_dialog, &req_dlg);

	gtk_window_resize (GTK_WINDOW (appwin),
	                   req_menu.width + req_dlg.width,
	                   req_menu.height + req_dlg.height);
}

/***************************************************************************
 *
 *   app_window_get_current_fake_dialog
 *
 ***************************************************************************/

static GtkWidget *
app_window_get_current_fake_dialog (ThAppWindow *appwin)
{
	GList *l;
	gint   page_num, p;

	page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (appwin->priv->notebook));

	for (l = appwin->priv->fake_dialogs; l; l = l->next)
	{
		p = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (l->data), 
		                                         "th-app-win-page-num"));
		if (p == page_num)
			return GTK_WIDGET (l->data);
	}

	return NULL;
}

/***************************************************************************
 *
 *   app_window_update_action_sensitivity
 *
 *   sets actions sensitive/insensitive depending on which dialog is
 *    currently active (our code isn't complex enough to warrant
 *    action groups)
 *
 ***************************************************************************/

static void
app_window_update_action_sensitivity (ThAppWindow *appwin, GtkWidget *cur_dlg)
{
	GtkAction *action_openvobfolder;

	action_openvobfolder = gtk_action_group_get_action (appwin->priv->action_group, 
	                                                    "OpenVOBFolder");

	if (cur_dlg == appwin->priv->ts)
	{
		gtk_action_set_sensitive (action_openvobfolder, TRUE);
	}
	else
	{
		gtk_action_set_sensitive (action_openvobfolder, FALSE);
	}
}

/***************************************************************************
 *
 *   app_window_switch_to_fake_dialog
 *
 ***************************************************************************/

static void
app_window_switch_to_fake_dialog (ThAppWindow *appwin, GtkWidget *fake_dialog)
{
	gint page_num;

	g_assert (TH_IS_FAKE_DIALOG (fake_dialog));

	page_num = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (fake_dialog), 
	                                                "th-app-win-page-num"));

	{
		GList *l;
		for (l = appwin->priv->fake_dialogs; l; l = l->next)
		{
			if (GTK_WIDGET (l->data) != fake_dialog)
			{
				gtk_widget_hide (GTK_WIDGET (l->data));
				gtk_widget_set_sensitive (GTK_WIDGET (l->data), FALSE);
			}
		}
	}

	gtk_widget_show (fake_dialog);
	gtk_widget_set_sensitive (fake_dialog, TRUE);

	gtk_notebook_set_current_page (GTK_NOTEBOOK (appwin->priv->notebook), 
	                               page_num);

	app_window_resize_to_fit (appwin, fake_dialog);

	app_window_update_action_sensitivity (appwin, fake_dialog);
}

/***************************************************************************
 *
 *   app_window_run_jobs
 *
 ***************************************************************************/

static gboolean
app_window_run_jobs (ThAppWindow *appwin, GList *jobs)
{
	ThJobProgressDialog *jpd;

	app_window_switch_to_fake_dialog (appwin, appwin->priv->jpd);

	jpd = TH_JOB_PROGRESS_DIALOG (appwin->priv->jpd);

	return th_job_progress_dialog_run_jobs (jpd, jobs);
}

/***************************************************************************
 *
 *   app_window_configure_jobs
 *
 ***************************************************************************/

static gboolean
app_window_configure_jobs (ThAppWindow *appwin, GList *jobs)
{
	ThJobConfigDialog *jcd;

	app_window_switch_to_fake_dialog (appwin, appwin->priv->jcd);

	jcd = TH_JOB_CONFIG_DIALOG (appwin->priv->jcd);
	if (!th_job_config_dialog_configure_jobs (jcd, jobs))
		return FALSE;
	
	if (!app_window_check_free_disk_space (appwin, jobs))
		return FALSE;

	return app_window_run_jobs (appwin, jobs);
}

/***************************************************************************
 *
 *   app_window_run_title_selector
 *
 ***************************************************************************/

static void
app_window_run_title_selector (ThAppWindow *appwin)
{
	GList *jobs;
	gint   ret;

	do
	{
		app_window_switch_to_fake_dialog (appwin, appwin->priv->ts);

		jobs = th_title_selector_run (TH_TITLE_SELECTOR (appwin->priv->ts));

		if (jobs == NULL)
			break;

		ret = app_window_configure_jobs (appwin, jobs);

		g_list_foreach (jobs, (GFunc) g_object_unref, NULL);
		g_list_free (jobs);
	}
	while (ret == FALSE  &&  appwin->priv->ts != NULL);
}

/***************************************************************************
 *
 *   th_app_window_run
 *
 ***************************************************************************/

void
th_app_window_run (ThAppWindow *appwin)
{
	gtk_widget_show (GTK_WIDGET (appwin));

	app_window_run_title_selector (appwin);
}

