
/******************************************************************************
* MODULE     : tm_menus.gen.cc
* DESCRIPTION: Dynamical menus
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_layout.gen.h>
#include <Widget/make_widget.gen.h>

#module code_tm_menus
#import tm_layout
#import make_widget

widget make_menu_widget (server sv, display dis, scheme_tree p);
scheme_tree menu_merge (scheme_tree p, scheme_tree q);

/******************************************************************************
* Constructor and destructor
******************************************************************************/

tm_layout_rep::tm_layout_rep () {}
tm_layout_rep::~tm_layout_rep () {}

/******************************************************************************
* Compilation routines
******************************************************************************/

class server_command_rep: public command_rep {
  server  sv;
  scheme_tree p;
public:
  server_command_rep (server sv2, scheme_tree p2): sv (sv2), p(p2) {}
  void apply () {
    sv->get_editor()->buf->mark_undo_block ();
    sv->get_editor()->set_input_normal ();
    sv->exec_delayed (p);
    sv->get_editor()->notify_change (THE_DECORATIONS);
  }
  ostream& print (ostream& out) { return out << p; }
};

/******************************************************************************
* Abstract function for creating menu widgets
******************************************************************************/

class make_menu_widget_rep: public make_widget_rep {
  server  sv;
  scheme_tree p;
public:
  make_menu_widget_rep (server sv2, scheme_tree p2): sv(sv2), p(p2) {}
  ostream& print (ostream& out) { return out << p; }
  widget get_widget (display dis);
};

widget
make_menu_widget_rep::get_widget (display dis) {
  return make_menu_widget (sv, dis, p);
}

static make_widget
make_menu_widget (server sv, scheme_tree p) {
  return new make_menu_widget_rep (sv, p);
}

/******************************************************************************
* Subroutines
******************************************************************************/

static inline bool
is_quoted (tree t) {
  return is_atomic (t) && is_quoted (as_string (t));
}

static inline string
unquote (tree t) {
  return unquote (as_string (t));
}

static bool
is_menu_label (scheme_tree p) {
  return
    (is_tuple (p, "balloon", 2) && is_menu_label (p[1]) && is_quoted (p[2])) ||
    (is_tuple (p, "text", 2) && is_tuple (p[1]) && is_quoted (p[2])) ||
    is_quoted (p) || (is_tuple (p, "icon", 1) && is_quoted (p[1]));
}

/******************************************************************************
* Menu creation routines
******************************************************************************/

static widget
make_menu_label (server sv, display dis, scheme_tree p, bool tt= FALSE) {
  if (is_tuple (p, "balloon", 2))
    return make_menu_label (sv, dis, p[1], tt);
  else if (is_tuple (p, "text", 2))
    return box_widget (dis, p[1], unquote (p[2]), TRUE, TRUE);
  else if (is_quoted (p)) {
    string s= unquote (p);
    return text_widget (dis, s, TRUE, sv->get_input_language(), tt);
  }
  else {
    string s= unquote (p[1]);
    return xpm_widget (dis, s);
  }
}

static widget
make_menu_button (server sv, widget but, scheme_tree lab) {
  if (is_tuple (lab, "balloon", 2)) {
    string lan = sv->get_input_language();
    widget help= text_widget (but->dis, unquote (lab[2]), TRUE, lan);
    return balloon_widget (but, help);
  }
  else return but;
}

static void
insert_menu (server sv, display dis, array<widget>& a, scheme_tree p) {
  if (is_atomic (p)) {
    if (p == "---") {
      widget sep= separator_widget (dis, 2*PIXEL, 2*PIXEL);
      a << sep;
      return;
    }

    if (p == "|") {
      widget sep= separator_widget (dis, 2*PIXEL, 2*PIXEL, TRUE);
      a << sep;
      return;
    }
  }
  else if (!is_tuple (p));
  else {
    int i, n= N(p);
    
    if ((n == 0) || (is_tuple (p[0]) && (!is_menu_label (p[0])))) {
      for (i=0; i<n; i++)
	insert_menu (sv, dis, a, p[i]);
      return;
    }

    if ((p[0] == "if") && (n >= 2)) {
      bool flag;
      if (!sv->eval_bool (p[1], flag))
	if (flag)
	  for (i=2; i<n; i++)
	    insert_menu (sv, dis, a, p[i]);
      return;
    }

    if ((p[0] == "=>") && (n >= 2) && is_menu_label (p[1])) {
      scheme_tree q (TUPLE, n-1);
      q[0]= "ver-menu";
      for (i=0; i<n-2; i++) q[i+1]= p[i+2];
      make_widget mw= make_menu_widget (sv, q);
      
      widget wid= make_menu_label (sv, dis, p[1]);
      widget but= pulldown_button (wid, mw);
      a << make_menu_button (sv, but, p[1]);
      return;
    }

    if ((p[0] == "->") && (n >= 2) && is_menu_label (p[1])) {
      scheme_tree q (TUPLE, n-1);
      q[0]= "ver-menu";
      for (i=0; i<n-2; i++) q[i+1]= p[i+2];
      make_widget mw= make_menu_widget (sv, q);

      widget wid= make_menu_label (sv, dis, p[1]);
      widget but= pullright_button (wid, mw);
      a << make_menu_button (sv, but, p[1]);
      return;
    }

    if ((p[0] == "tile") && (n >= 2) && is_atomic (p[1])) {
      array<widget> b (0);
      for (i=2; i<n; i++)
	insert_menu (sv, dis, b, p[i]);
      a << tile (dis, b, as_int (p[1]));
      return;
    }

    if ((p[0] == "link") && (n == 2)) {
      scheme_tree r;
      if (!sv->eval_scheme_tree (p[1], r))
	insert_menu (sv, dis, a, r);
      return;
    }

    if ((n == 2) && is_menu_label (p[0])) {
      widget wid= make_menu_label (sv, dis, p[0]);
      widget but= command_button (wid, new server_command_rep (sv, p[1]));
      a << make_menu_button (sv, but, p[0]);
      return;
    }

    if ((n == 3) && is_menu_label (p[0]) && is_menu_label (p[1])) {
      scheme_tree rt= p[1];
      // if (is_quoted (p[1])) rt= "\"(" * unquote (p[1]) * ")\"";
      widget lw = make_menu_label (sv, dis, p[0]);
      widget rw = make_menu_label (sv, dis, rt, TRUE);
      widget but= command_button (lw, rw, new server_command_rep (sv, p[2]));
      a << make_menu_button (sv, but, p[0]);
      /*
      scheme_tree rt= p[1];
      if (is_quoted (rt))
	rt= tree ("\"Keyboard equivalent: " * unquote (rt) * "\"");
      widget lw = make_menu_label (sv, dis, p[0]);
      widget rw = make_menu_label (sv, dis, rt, TRUE);
      widget but= command_button (lw, new server_command_rep (sv, p[2]));
      a << balloon_widget (make_menu_button (sv, but, p[0]), rw);
      */
      return;
    }

    if ((p[0] == "symbol") && (n >= 2) && (n <= 4) && is_quoted (p[1])) {
      string pps= "(insert-string \"" * unquote (p[1]) * "\")";
      scheme_tree pp = string_to_scheme_tree (pps);
      if ((n >= 3) && is_compound (p[2]))
	pp= tuple ("insert-object", tuple ("'", copy (p[2])));
      widget      bw = box_widget (dis, tree (TUPLE), unquote (p[1]), TRUE);
      widget      but= command_button (bw, new server_command_rep (sv, pp));
      if ((n >= 3) && is_quoted (p[n-1])) {
	tree rt= tree ("\"Keyboard equivalent: " * unquote (p[n-1]) * "\"");
	widget rw = make_menu_label (sv, dis, rt);
	but= balloon_widget (but, rw);
      }
      a << but;
      return;
    }

    for (i=0; i<n; i++)
      insert_menu (sv, dis, a, p[i]);
    return;
  }

  cerr << "TeXmacs] inserted menu has bad format in " << p << "\n";
}

widget
make_menu_widget (server sv, display dis, scheme_tree p) {
  if (is_tuple (p)) {
    int i, n= N(p);

    if (is_tuple (p, "ver-menu")) {
      array<widget> a (0);
      for (i=1; i<n; i++)
	insert_menu (sv, dis, a, p[i]);
      return vertical_menu (dis, a);
    }

    if (is_tuple (p, "link", 1)) {
      scheme_tree r;
      if (sv->eval_scheme_tree (p[1], r))
	return text_widget (dis, "Error");
      else return make_menu_widget (sv, dis, r);
    }

    array<widget> a (0);
    for (i=0; i<n; i++)
      insert_menu (sv, dis, a, p[i]);
    return horizontal_array (dis, a);
  }

  cerr << "TeXmacs] menu has bad format in " << p << "\n";
  array<widget> a (0);
  return horizontal_array (dis, a);
}

/******************************************************************************
* Operations on menus
******************************************************************************/

static void
menu_append (scheme_tree& a, scheme_tree p) {
  if (is_tuple (a) && is_tuple (p)) {
    int i, k= N(a), n= N(p);

    if (((n == 2) || (n == 3)) && is_quoted (p[0]))
      for (i=0; i<k; i++)
	if (is_tuple (a[i]) && ((N(a[i]) == 2) || (N(a[i]) == 3)) &&
	    (a[i][0]==p[0]))
	  {
	    a[i]= p;
	    return;
	  }

    if ((is_tuple (p, "=>") || is_tuple (p, "->")) &&
	(n >= 2) && is_atomic (p[1]))
      for (i=0; i<k; i++)
	if (is_tuple (a[i]) && (N(a[i]) >= 2) &&
	    (a[i][0]==p[0]) && (a[i][1]==p[1]))
	  {
	    int j;
	    scheme_tree b (TUPLE, n-2);
	    for (j=0; j<n-2; j++) b[j]= p[j+2];
	    a[i]= menu_merge (a[i], scheme_tree (b));
	    return;
	  }
  }

  a << p;
}

scheme_tree
menu_merge (scheme_tree p, scheme_tree q) {
  if (is_atomic (p)) return scheme_tree (TUPLE);
  else {
    int i, k= N(p);
    scheme_tree a (TUPLE, k);
    for (i=0; i<k; i++) a[i]= p[i];

    if (is_atomic (q)) menu_append (a, q);
    else {
      int n= N(q);
      if ((n > 0) && ((q[0] == "---") || (q[0] == "|") || is_tuple (q[0])))
	for (i=0; i<n; i++) menu_append (a, q[i]);
      else menu_append (a, q);
    }

    return a;
  }
}

/******************************************************************************
* Interface
******************************************************************************/

void
tm_layout_rep::menu_widget (scheme_tree p, widget& w) {
  tm_widget meta= get_meta ();
  w= make_menu_widget (server (this), meta->dis, p);
}

void
tm_layout_rep::menu_main (scheme_tree p) {
  tm_widget meta= get_meta ();
  meta->set_main_menu (make_menu_widget (server (this), meta->dis, p));
}

void
tm_layout_rep::menu_icons (int which, scheme_tree p) {
  tm_widget meta= get_meta ();
  meta->set_icon_menu (which, make_menu_widget (server (this), meta->dis, p));
}

void
tm_layout_rep::menu_icons_on (int which, bool flag) {
  tm_widget meta= get_meta ();
  meta->set_icon_flag (which, flag);
}

bool
tm_layout_rep::menu_icons_status (int which) {
  tm_widget meta= get_meta ();
  return meta->get_icon_flag (which);
}

#endmodule // code_tm_menus
