
/******************************************************************************
* MODULE     : gg_cc.cc
* DESCRIPTION: generic c++ preprocessor
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include "gg_file.h"

string gencc_include ("");
bool   gencc_debug= TRUE;
bool   gencc_error= FALSE;

/******************************************************************************
* Useful subroutines
******************************************************************************/

bool
is_num (char c) {
  return (c>='0') && (c<='9');
}

bool
is_alpha_num (char c) {
  return (c=='_') ||
    ((c>='0') && (c<='9')) ||
    ((c>='a') && (c<='z')) ||
    ((c>='A') && (c<='Z'));
}

/******************************************************************************
* The module class and module importation
******************************************************************************/

class module {
public:
  string        name;
  array_string  args;
  string        body;
};

string
import (module m, array_string args) {
  int i;
  string s (m.body), out;

  for (i=0; i<N(s); ) {
    if (is_alpha_num (s[i])) {
      int j;
      for (j=0; j<N(m.args); j++) {
	string& arg (m.args[j]);
	if (((N(s)-i) >= N(arg)) &&
	    (s (i, i+ N(arg)) == arg) &&
	    ((!is_alpha_num (s [i+N(arg)])) || (N(s)==i+N(arg))))
	  {
	    out << args[j];
	    i+=N(arg);
	    break;
	  }
      }
      if ((i==N(s)) || (!is_alpha_num (s[i]))) continue;
      int start= i;
      while ((i<N(s)) && is_alpha_num (s[i])) i++;
      out << s (start, i);
      continue;
    }
    out << s[i++];
  }


  return out;
}

string
unbracket (string s) {
  bool flag=FALSE;
  string out;
  int i;

  for (i=0; i<N(s); ) {
    if (is_alpha_num (s[i])) {
      int start= i;
      while ((i<N(s)) && is_alpha_num (s[i])) i++;
      while ((i<N(s)) && ((s[i]==' ') && (s[i]=='\t'))) i++;
      if ((i<N(s)) && (s[i]=='<')) {
	i++;
	int end= i;
	while ((i<N(s)) &&
	       ((s[i]==' ') || (s[i]=='\t') || (s[i]==',') ||
		is_alpha_num (s[i]))) i++;
	if ((i<N(s)) && (s[i]=='>')) {
	  i=start;
	  while (s[i]!='>') {
	    if ((s[i]==' ') || (s[i]=='\t')) i++;
	    else if ((s[i]=='<') || (s[i]==',')) { out << '_'; i++; }
	    else out << s[i++];
	  }
	  i++;
	  flag= TRUE;
	  continue;
	}
	else i= end;
      }
      out << s (start, i);
      continue;
    }
    out << s[i++];
  }
  if (flag) return unbracket (out);
  return out;
}

/******************************************************************************
* The gencc class
******************************************************************************/

class gencc {
public:
  int      M_nr;
  int      M_allocated;
  module*  M;
  int      I_nr;
  int      I_allocated;
  module*  I;
  
  string   s;
  int      i;
  string   file_name;
  int      line_nr;

  string   out;

public:
  gencc ();
  ~gencc ();
  void error (string message);
  friend int main (int, char **);

  void skip_spaces ();
  void skip_until_CR ();
  int read_number ();
  string read_string ();
  string read_id ();
  array_string read_args ();

  void define_module (string name, array_string args, string body);
  void make_modules (string s2);
  void process (string s2);
  string import (string name, array_string args);
};

gencc::gencc () {
  M_nr       = 0;
  M_allocated= 1;
  M          = new module[M_allocated];

  I_nr       = 0;
  I_allocated= 1;
  I          = new module[I_allocated];
}

gencc::~gencc () {
  delete[] M;
  delete[] I;
}

void
gencc::error (string message) {
  cerr << file_name << ":";
  if (line_nr!=0) cerr << line_nr << ":";
  cerr << " " << message << "\n";
  gencc_error= TRUE;
}

/******************************************************************************
* Reading information
******************************************************************************/

void
gencc::skip_spaces () {
  while ((i<N(s)) && ((s[i]==' ') || (s[i]=='\t'))) i++;
}

void
gencc::skip_until_CR () {
  while ((i<N(s)) && (s[i]!='\n')) i++;
}

int
gencc::read_number () {
  skip_spaces ();
  int start= i;
  while ((i<N(s)) && is_num (s[i])) i++;
  return as_int (s (start, i));
}

string
gencc::read_string () {
  skip_spaces ();
  if (i==N(s)) return "";
  if (s[i]!='\"') return ""; // "
  int start= (++i);
  while ((i<N(s)) && (s[i]!='\"')) i++; // "
  string r= s (start, i);
  if (i<N(s)) i++;
  return r;
}

string
gencc::read_id () {
  skip_spaces ();
  int start= i;
  while ((i<N(s)) && is_alpha_num (s[i])) i++;
  return s (start, i);
}

array_string
gencc::read_args () {
  array_string a;
  skip_spaces ();
  if ((i==N(s)) || (s[i]=='\n')) return a;
  if (s[i]!='(') {
    error ("arguments should be enclosed between ()");
    return a;
  }

  i++;
  skip_spaces ();
  while ((i<N(s)) && (s[i]!=')') && (s[i]!='\n')) {
    a << read_id ();
    skip_spaces ();
    if (s[i]==',') { i++; skip_spaces (); }
  }
  return a;
}

/******************************************************************************
* Make modules
******************************************************************************/

void
gencc::define_module (string name, array_string args, string body) {
  if (info>=2) {
    cerr << "Info: " << file_name << ":";
    if (line_nr!=0) cerr << line_nr << ":";
    cerr << " define " << name << args << "\n";
  }

  int i;
  for (i=0; i<M_nr; i++)
    if ((name==M[i].name) && (N(args)==N(M[i].args))) {
      error ("module " * name * " with " * as_string (N(args)) *
	     " arguments already defined");
      return;
    }

  if (M_nr==M_allocated) {
    int i;
    module* M2= new module[2*M_nr];
    for (i=0; i<M_nr; i++) M2[i]=M[i];
    delete[] M;
    M= M2;
    M_allocated *= 2;
  }
  
  M[M_nr].name= name;
  M[M_nr].args= args;
  M[M_nr].body= body;

  M_nr++;
}

void
gencc::make_modules (string s2) {
  string       M_name; 
  array_string M_args;
  int          M_start= -1;
  
  s      = s2;
  i      = 0;
  line_nr= 0;

  while (i<N(s)) {
    if (s[i]=='\n') {
      out << s[i++];
      line_nr++;
      continue;
    }

    if (s ("#module ", i)) {
      i+=8;
      M_name = read_id ();
      M_args = read_args ();
      skip_until_CR ();
      if (s[i]!='\n') error ("missing #endmodule for module " * M_name);
      else M_start= i+1;
      continue;
    }

    if (s ("#endmodule ", i)) {
      if (M_start==-1) error ("no corresponding #module");
      else {
	define_module (M_name, M_args, s (M_start, i));
	M_start= -1;
      }
      skip_until_CR ();
      continue;
    }

    if (s ("#line ", i)) {
      i+=6;
      line_nr  = read_number ();
      file_name= read_string ();
      skip_until_CR ();
      continue;
    }

    skip_until_CR ();
  }
}

/******************************************************************************
* Importation and processing of the importation
******************************************************************************/

void
gencc::process (string s2) {
  string old_s= s;
  int    old_i= i;
  string old_n= file_name;
  int    old_l= line_nr;

  s        = s2;
  i        = 0;
  file_name= "";
  line_nr  = 0;

  while (i<N(s)) {
    if (s[i]=='\n') {
      out << s[i++];
      line_nr++;
      continue;
    }

    if (s ("#import ", i)) {
      i+=8;
      string name       = read_id ();
      array_string args = read_args ();
      string r          = import (name, args);
      process (r);
      skip_until_CR ();
      continue;
    }

    if (s ("#line ", i)) {
      int start= i;
      i+=6;
      line_nr  = read_number ();
      file_name= read_string ();
      skip_until_CR ();
      if (gencc_debug) out << s (start, i);
      continue;
    }

    int start= i;
    skip_until_CR ();
    out << s (start, i);
  }

  s        = old_s;
  i        = old_i;
  file_name= old_n;
  line_nr  = old_l;
}

string
gencc::import (string name, array_string args) {
  int i;
  for (i=0; i<I_nr; i++)
    if ((I[i].name==name) && (I[i].args==args))
      return "";
  if (info>=2) {
    cerr << "Info: " << file_name << ":";
    if (line_nr!=0) cerr << line_nr << ":";
    cerr << " import " << name << args << "\n";
  }

  if (I_nr==I_allocated) {
    int i;
    module* I2= new module[2*I_nr];
    for (i=0; i<I_nr; i++) I2[i]=I[i];
    delete[] I;
    I= I2;
    I_allocated *= 2;
  }
  I[I_nr].name= name;
  I[I_nr].args= args;
  I[I_nr].body= "";
  I_nr++;

  for (i=0; i<M_nr; i++)
    if ((M[i].name==name) && (N(M[i].args)==N(args)))
      return unbracket (::import (M[i], args));
  error ("module " * name * " with " * as_string (N(args)) *
	 " arguments not found");
  return "";
}

/******************************************************************************
* Main program
******************************************************************************/

int
main (int argc, char **argv)
{
  char* temp;
  temp= getenv ("GENCC_INCLUDE");
  if (temp!=NULL) gencc_include= string(temp);

  int i, arg;
  if ((argc<=1) || ((argv[1][0]=='-') && (argv[1][1]=='h'))) {
    cout << "--------------------------------------------------------------\n";
    cout << "Generic c, c++ preprocessor options:\n";
    cout << "--------------------------------------------------------------\n";
    cout << "  -I path : Path(s) for include files\n";
    cout << "  -o file : Write output to file\n";
    cout << "  -ds     : Debugging information for source files (default)\n";
    cout << "  -dg     : Debugging information for generated .cc files\n";
    cout << "  -i0     : Compile silent\n";
    cout << "  -i1     : Display executed processes (default)\n";
    cout << "  -i2     : Maximum information\n";
    cout << "  -h      : display this help message\n";
    cout << "--------------------------------------------------------------\n";
    exit(0);
  }
  
  string output_name;
  for (arg=1; arg<argc; arg++) {
    string name=argv[arg];
    if (name ("-i0")) { info=0; continue; }
    if (name ("-i1")) { info=1; continue; }
    if (name ("-i2")) { info=2; continue; }
    if (name ("-i3")) { info=3; continue; }
    if (name ("-ds")) { gencc_debug= TRUE; continue; }
    if (name ("-dg")) { gencc_debug= FALSE; continue; }
    if (arg+1<argc) {
      string next=argv[arg+1];
      if (name ("-I")) {
	if (N(gencc_include)==0) gencc_include= next;
	else gencc_include << ":" << next;
	arg++;
	continue;
      }
      if (name ("-o")) { output_name= next; arg++; continue; }
    }

    file_reader F;
    F.read ("", name);
    gencc G;
    G.make_modules (F.in);
    G.file_name= "command line";
    G.line_nr= 0;
    string r= G.import ("main", array_string ());
    G.process (r);
    string rad= radical (radical (name));
    if (N(output_name)==0) output_name= rad * ".cc";
    if (!gencc_error) save (output_name, G.out);
    gencc_error= FALSE;
    output_name= "";
  }

  return 0;
}
