/* 
 * simple.c - a sample program for DVIlib
 * by Hirotsugu Kakugawa
 *
 * 21 Nov 1996    Version 1.0
 * 21 Feb 1997    Version 1.1
 * 21 Feb 1997    Version 1.2
 * 26 Oct 1999    Version 1.3  Added paper and dev mode list
 *
 */
/*
 * Copyright (C) 1996-1999 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "dvi-2_6.h"

#define VERSION         "1.3"

#define DPI             "300"
#define KPATHSEA_MODE   "cx"
#define VFLIBCAP        "vflibcap-tex"

char     *dpi;
char     *kpathsea_mode;
char     *vflibcap;
int       show_papers        = 0;   
int       show_device_modes  = 0;  

void        usage(void);
DVI_DEVICE  device_create(void);



int
main(int argc, char** argv)
{
  DVI            dvi;
  DVI_DEVICE     dev;
  DVI_PROPERTY   prop;
  int         page;
  double      shrink;
  char        *dvi_file;
  char        vars[1024];

  shrink = 1.0;
  dvi_file = NULL;
  vflibcap      = VFLIBCAP;
  dpi           = DPI;
  kpathsea_mode = KPATHSEA_MODE;

  printf("** Simple DVI Printer Version: %s\n", VERSION);
  argc--;
  argv = &argv[1];
  while (argc >= 1){
    if (strcmp(argv[0], "-s") == 0){
      shrink = atof(argv[1]);
      argc -= 2;
      argv = &argv[2];
    } else if ((strcmp(argv[0], "-h") == 0)
	       || (strcmp(argv[0], "--h") == 0)
	       || (strcmp(argv[0], "-help") == 0)
	       || (strcmp(argv[0], "--help") == 0)){
      usage();
      exit(0);
    } else if (strcmp(argv[0], "-v") == 0){
      vflibcap = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-dpi") == 0){
      dpi = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-mode") == 0){
      kpathsea_mode = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-papers") == 0){
      show_papers = 1;
      argc -= 1;
      argv = &argv[1];
    } else if (strcmp(argv[0], "-modes") == 0){
      show_device_modes = 1;
      argc -= 1;
      argv = &argv[1];
    } else {
      dvi_file = argv[0];
      break;
    }
  }


  DVI_SETUP();

  /*
   * Paper List 
   */
  if (show_papers == 1){
    char  **list, **p;
    list = DVI_get_paper_size_list();
    if (list == NULL){
      fprintf(stderr, "Failed to get paper list\n");
    } else {
      printf("Papers:\n");
      for (p = list; *p != NULL; p++)
	printf("%s\n", *p);
    }
    exit(1);
  }

  /*
   * Device Mode List 
   */
  if (show_device_modes == 1){
    char  **list, **p, *desc;
    list = DVI_get_device_mode_list();
    if (list == NULL){
      fprintf(stderr, "Failed to get device mode list\n");
    } else {
      printf("Device Modes:\n");
      for (p = list; *p != NULL; p++){
	DVI_parse_device_mode(*p, NULL, NULL, &desc);
	printf("%12s   %s\n", *p, desc);
      }
    }
    exit(1);
  }


  /*
   * Initialize DVILIB 
   */
  sprintf(vars, "TeX_DPI=%s, TeX_KPATHSEA_MODE=%s",
	  dpi, kpathsea_mode);
  if (DVI_INIT(vflibcap, vars) < 0){
    fprintf(stderr, "Error: Cannot initialize dvilib.\n");
    exit(1);
  }
  
  if (dvi_file == NULL){
    fprintf(stderr, "Error: No DVI file.\n");
    usage();
    exit(1);
  }


  /*
   * Create Device Object 
   */
  if ((dev = device_create()) == NULL){
    fprintf(stderr, "Error: Cannot initialize device.\n");
    exit(1);
  }

  /*
   * Define DVI property
   */
  prop = DVI_PROPERTY_ALLOC_DEFAULT();

  /*
   * Create DVI Object
   */
  if ((dvi = DVI_CREATE(dev, dvi_file, prop)) == NULL){
    printf("Error: Cannot open DVI file.\n");
    exit(1);
  }
  printf("** DVIlib Version: %s\n\n", DVI_VERSION());

  /*
   * Open fonts used int the DVI file
   */
  printf("** OPEINING FONTS...\n");
  if (DVI_OPEN_FONT(dvi, dev) < 0){
    printf("Error: Cannot open all fonts\n");
    exit(1);
  }

  /*
   * Draw Each Page
   */
  for (page = 1; page <= dvi->pages; page++){
    printf("** PAGE %d\n", page);
    DVI_DRAW_PAGE(dvi, dev, page, shrink);
  }

  /* 
   * Finish!
   */
  DVI_DISPOSE(dvi, dev);

  return 0;
}

void usage(void)
{
  printf("Usage: simple [options] dvi_file\n");
  printf("Options: -s SHRINK\n");
  printf("         -v VFLIBCAP\n");
  printf("         -dpi DPI\n");
  printf("         -mode MODE  kpathsea device mode name (e.g., \"cx\")\n");
  printf("         -papers    show list of paper size\n");
  printf("         -modes     show list of kpathsea device modes \n");
}



void
dvi_dump_bitmap(DVI_BITMAP bm)
{
  static unsigned char bit_table[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
  unsigned char *p, *p0;
  int           x, y;
  
  if ((p = bm->bitmap) == NULL)
    return;
  for (x = 0; x < bm->width+2; x++)
    putchar('-');
  putchar('\n');
  for (y = 0; y < bm->height; y++){
    p0 = p;
    --p;
    putchar('|');
    for (x = 0; x < bm->width; x++){
      if (x%8 == 0)
	p++;
      if ((*p & bit_table[x%8]) != 0)
	putchar('@');
      else
	putchar('.');
    }
    p = p0 + bm->raster;
    putchar('|');
    putchar('\n');
  }
  for (x = 0; x < bm->width+2; x++)
    putchar('-');
  putchar('\n');
}



int 
dvidev_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  return 0; /* no interruption */
}

void 
dvidev_message(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  va_end(ap);
  printf("\n");
}

void 
dvidev_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		     long x, long y, long w, long h)
{
  printf("Put Rectangle (%ld,%ld)-(%ld,%ld) W=%ld,H=%ld\n", 
	 x, y, x+w, y+h, w, h);
}
void 
dvidev_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
		  int font_id, long id2, long code_point,
		  long x, long y)
{
  if (bm != NULL){
    printf("Put Bitmap (%ld,%ld)  W=%ld, H=%ld\n",
	   x, y, bm->width, bm->height);
    dvi_dump_bitmap(bm);
  }
}

int 
dvidev_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  return 1;
}
void 
dvidev_before_ps_figure(DVI_DEVICE dev, DVI dvi, char* p, long x, long y)
{
  printf("PS Figure Start\n");
}
void 
dvidev_after_ps_figure(DVI_DEVICE dev, DVI dvi)
{
  printf("PS Figure End\n");
}
int
dvidev_special_command_hook(DVI_DEVICE dev, DVI dvi, char *p, long x, long y)
{
  printf("Special Command\n");
  return 0;
}

int
dvidev_gs_timeout_value(DVI_DEVICE dev, DVI dvi)
{
  return 10;
}

int
dvidev_gs_timeout_giveup(DVI_DEVICE dev, DVI dvi)
{
  return 0;
}

char*
dvidev_gs_path(DVI_DEVICE dev, DVI dvi)
{
  return "/usr/local/bin/gs";
}

DVI_DEVICE
device_create(void)
{
  DVI_DEVICE  dev;

  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = atoi(dpi);
  dev->v_dpi = atoi(dpi);
  dev->put_rectangle        = dvidev_put_rectangle;
  dev->put_bitmap           = dvidev_put_bitmap;
  dev->draw_ps_figures      = dvidev_draw_ps_figures;
  dev->before_ps_figure     = dvidev_before_ps_figure;
  dev->after_ps_figure      = dvidev_after_ps_figure;
  dev->special_command_hook = dvidev_special_command_hook;
  dev->gs_program_path      = dvidev_gs_path;
  dev->gs_timeout_value     = dvidev_gs_timeout_value;
  dev->gs_timeout_give_up   = dvidev_gs_timeout_giveup;
  dev->message_advice       = dvidev_message;
  dev->message_warning      = dvidev_message;
  dev->message_error        = dvidev_message;
  dev->message_fatal        = dvidev_message;
  dev->device_polling       = dvidev_poll;
  return dev;
}

/*EOF*/
