/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://terraform.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * This file is copyright (c) 2000 David A. Bartold
 */


#include <sys/types.h>	/* OS/X requires this */
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include "filelist.h"
#include "filenameops.h"

static int 
str_cmp_func (const void *_v1_, const void *_v2_)
{
  gchar *s1 = (gchar *)_v1_;
  gchar *s2 = (gchar *)_v2_;

  return g_strcasecmp (s1, s2);
}

static GList *
file_list_add_file_sorted (GList   *list,
                           gchar   *file,
                           gchar   *base,
		           gboolean do_sort)
{
  struct stat stat_buf;
  int    (*fnptr) (const void *, const void *) = str_cmp_func;

  stat (file, &stat_buf);
  if (S_ISDIR (stat_buf.st_mode))
    return list;

  if (!S_ISREG (stat_buf.st_mode))
    {
      g_print ("Ignoring file '%s'.  It might be a symlink.\n", file);
      return list;
    }

  if(g_list_find_custom(list, g_strdup (base), fnptr))
    return list;

  if (do_sort)
    return g_list_insert_sorted (list, g_strdup (base), fnptr);

  return g_list_append (list, g_strdup (base));
}

static GList *
file_list_add_file (GList   *list,
                    gchar   *file,
                    gchar   *base)
{
  return file_list_add_file_sorted (list, file, base, TRUE);
}

static GList *
file_list_add_path (GList *list,
                    gchar *path)
{
  DIR   *dir;
  struct dirent *dirent;

  dir = opendir (path);
  if (dir == NULL)
    return list;

  dirent = readdir (dir);
  while (dirent != NULL)
    {
      gchar *filename, *base;

      filename = g_strdup_printf ("%s/%s", path, dirent->d_name);
      base = filename_get_base (filename);
      list = file_list_add_file (list, filename, base);
      g_free (base);
      g_free (filename);

      dirent = readdir (dir);
    }
  
  closedir (dir);

  return list;
}

GList *
theme_list_new ()
{
  gchar *home_dir;

  home_dir = getenv ("HOME");
  if (home_dir != NULL){
    GList *themes_list;
    gchar *themes_path;

    themes_path = g_strdup_printf("%s/.terraform/themes", home_dir);
    themes_list = file_list_add_path (NULL, themes_path);
    g_free (themes_path);
    return file_list_add_path (themes_list, TERRAFORM_DATA_DIR "/themes");
  }
  else
    return file_list_add_path (NULL, TERRAFORM_DATA_DIR "/themes");
}

GList *
object_list_new ()
{
  gchar *home_dir;
  
  home_dir = getenv ("HOME");
  if (home_dir != NULL){
    GList *object_list;
    gchar *object_path;

    object_path = g_strdup_printf("%s/.terraform/objects", home_dir);
    object_list = file_list_add_path (NULL, object_path);
    g_free (object_path);
    return file_list_add_path (object_list, TERRAFORM_DATA_DIR "/objects");
  }
  else
    return file_list_add_path (NULL, TERRAFORM_DATA_DIR "/objects");
}

void
file_list_free (GList *list)
{
  g_list_foreach (list, (GFunc) g_free, NULL);
  g_list_free (list);
}

GList *
file_list_copy (GList *list)
{
  GList *out;

  out = NULL;
  for (list = g_list_first (list); list != NULL; list = list->next)
    out = g_list_append (out, g_strdup (list->data));

  return out;
}
