/*   FILE: fmt_ptrn.h -- 
 * AUTHOR: W. Michael Petullo <new@flyn.org>
 *   DATE: 08 January 2000
 */

#ifndef _FMT_PTRN_H
#define _FMT_PTRN_H

#ifdef __cplusplus
extern "C" {
#endif
#include <config.h>
#include <stdio.h> /* For BUFSIZ. */
#include <pair.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#include <limits.h>
#include <stdlib.h>
#include <queue.h>
#include <bistree.h>

/* ============================ buffer_t =================================== */
typedef struct buffer_t {
	char *data;
	size_t size;
} buffer_t;

/* ============================ fmt_ptrn_t ================================= */
    typedef struct fmt_ptrn_t {
#ifdef HAVE_LIBZ
	gzFile template_fp;
#else
	FILE *template_fp;
#endif
	char template_path[PATH_MAX + 1];
	long line_num;
	buffer_t raw_buf; /* Buffer for unfilled data. */
	buffer_t filled_buf; /* Buffer for filled data. */
	buffer_t lookup_buf; /* Lookup buffer; here so it is persistent. */
	pair_t *kv; /* Here so it is persistent; saves on a malloc. */
	char errmsg[BUFSIZ + 1]; /* General errors. */
	queue_t parse_errmsg; /* Parse errors. */
	bistree_t fillers; /* Format pattern / value pairs. */
    } fmt_ptrn_t;

/* ============================ fmt_ptrn_open () =========================== */
/* Open the template at path and prepare to fill it. */
    int fmt_ptrn_open(const char *path, fmt_ptrn_t *x);

/* ============================ fmt_ptrn_init () =========================== */
/* Similar to fmt_ptrn_open but does not open a template file.  call this 
 * before fmt_ptrn_filled to fill a string. 
 */
    int fmt_ptrn_init(fmt_ptrn_t *x);

/* ============================ fmt_ptrn_gets () =========================== */
/* Read a filled line from a template. */
    char *fmt_ptrn_gets(char *buf, size_t size, fmt_ptrn_t *x);

/* ============================ fmt_ptrn_close () ========================== */
/* Close a template. */
    int fmt_ptrn_close(fmt_ptrn_t *x);

/* ============================ fmt_ptrn_update_kv_1 () ==================== */
/* Add a key / value mapping for use in filling format patterns. */
    void fmt_ptrn_update_kv_1(fmt_ptrn_t *x, pair_t *p);

/* ============================ fmt_ptrn_update_kv ()======================= */
/* Add a key / value mapping for use in filling format patterns. */
    void fmt_ptrn_update_kv(fmt_ptrn_t *x, char *key, char *val);

/* ============================ fmt_ptrn_filled () ========================= */
/* Takes a string, p, and returns p with its format patterns filled. */
    char *fmt_ptrn_filled(fmt_ptrn_t *x, const char *p);

/* ============================ fmt_ptrn_parse_err () ====================== */
/* Returns true if a parse error has occured while processing. */
    int fmt_ptrn_parse_err(const fmt_ptrn_t *x);

/* ============================ fmt_ptrn_parse_strerror () ================= */
/* Dequeues and returns a parse error. */
    char *fmt_ptrn_parse_strerror(fmt_ptrn_t *x);

/* ============================= fmt_ptrn_parse_perror () ================== */
/* Prints the last parse error. */
    void fmt_ptrn_parse_perror(fmt_ptrn_t *x, const char *msg);

/* ============================ fmt_ptrn_perror () ========================= */
/* Prints the last non-parse error. */
    void fmt_ptrn_perror(const fmt_ptrn_t *x, const char *msg);

/* ============================ fmt_ptrn_strerror () ======================= */
/* Returns the last non-parse error. */
    const char *fmt_ptrn_strerror(const fmt_ptrn_t *x);

#ifdef __cplusplus
}
#endif
#endif				/* _FMT_PTRN_H */
