/* TODO: clean up this file and move functions to their own files */

/* system includes */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

/* lua includes */
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

/* UNIX includes. needed? */
#include <sys/types.h>
#include <sys/stat.h>

/* program includes */
#include "mem.h"
#include "node.h"
#include "path.h"
#include "support.h"
#include "context.h"
#include "platform.h"

/* internal base.bam file */
#include "internal_base.h"

/* why is this needed? -kma
   because of getcwd() -jmb */
#ifdef BAM_FAMILY_UNIX
#include <unistd.h>
#endif
#ifdef BAM_FAMILY_WINDOWS
#include <direct.h>
#endif

#define lua_register(L,n,f) \
	   (lua_pushstring(L, n), \
		lua_pushcfunction(L, f), \
		lua_settable(L, LUA_GLOBALSINDEX))

struct OPTION
{
	const char **s;
	int *v;
	const char *sw;
	const char *desc;
};

/* options passed via the command line */
static int option_clean = 0;
static int option_verbose = 0;
static int option_force = 0;
static int option_dry = 0;
static int option_threads = 0;
static int option_debug_tree = 0;
static int option_debug_nodes = 0;
static int option_debug_memstats = 0;
static int option_debug_buildtime = 0;
static int option_dump_internal = 0;
static int option_print_help = 0;
static int option_cppdep_includewarnings = 0;
static const char *option_script = "default.bam"; /* -f filename */
static const char *option_basescript = 0x0; /* -b filename or BAM_BASE */
static const char *option_threads_str = "0";
static const char *option_targets[128] = {0};
static int option_num_targets = 0;

static struct OPTION options[] =
	{
		{&option_basescript,0		, "-b", "base file to use"},
		{&option_script,0			, "-s", "bam file to use"},
/*		{&option_target,0			, "-t", "target to build"},*/
		{0, &option_clean			, "-c", "clean"},
		{0, &option_force			, "-f", "force"},
		{0, &option_verbose			, "-v", "verbose"},
		{0, &option_print_help		, "-h", "prints this help"},
		{&option_threads_str,0		, "-j", "sets the number of threads to use. 0 disables threading. (EXPRIMENTAL)"},
		{0, &option_dry				, "--dry", "dry run"},
		{0, &option_cppdep_includewarnings, "--cppdep-includewarnings", "the c++ dependency will be very bitchy"},
		{0, &option_print_help		, "--help", "prints this help"},
		{0, &option_debug_nodes		, "--debug-nodes", "prints all the nodes with dependencies"},
		{0, &option_debug_tree		, "--debug-tree", "prints the dependency tree"},
		{0, &option_debug_memstats	, "--debug-memstats", "prints the memory status after build"},
		{0, &option_debug_buildtime	, "--debug-build-time", "prints the build time"},
		{0, &option_dump_internal	, "--dump-internal", "dumps the internal base script"},
		{0, 0, (const char*)0, (const char*)0},
	};

/****/
static const char *program_name = "bam";
#define L_FUNCTION_PREFIX "bam_"

/****/
void debug_print_lua_value(lua_State *L, int i)
{
	if(lua_type(L,i) == LUA_TNIL)
		printf("nil");
	else if(lua_type(L,i) == LUA_TSTRING)
		printf("'%s'", lua_tostring(L,i));
	else if(lua_type(L,i) == LUA_TNUMBER)
		printf("%f", lua_tonumber(L,i));
	else if(lua_type(L,i) == LUA_TBOOLEAN)
	{
		if(lua_toboolean(L,i))
			printf("true");
		else
			printf("false");
	}
	else if(lua_type(L,i) == LUA_TTABLE)
		printf("{...}");
	else
		printf("%p (%s)", lua_topointer(L,i), lua_typename(L,lua_type(L,i)));
}

/****/
void debug_dump_lua_stack(lua_State *L)
{
	int n;
	int i;
	n = lua_gettop(L);
	for(i = 1; i <= n; ++i)
	{
		printf("%d %10s:", i, lua_typename(L,lua_type(L,i)));
		debug_print_lua_value(L,i);
		printf("\n");
	}
}


/*         str      str      *        */
/* add_job(tool, filename, arguments) */
static int lf_add_job(lua_State *L)
{
	struct NODE *node;
	struct CONTEXT *context;
	int n = lua_gettop(L);
	int i;
	if(n < 2)
	{
		lua_pushstring(L, "add_job: incorrect number of arguments");
		lua_error(L);
	}
	
	/* fetch contexst from lua */
	context = context_get_pointer(L);

	/* create the node */
	i = node_create(&node, context->graph, lua_tostring(L,2), lua_tostring(L,1));
	if(i == NODECREATE_NOTNICE)
	{
		printf("%s: '%s' is not nice\n", program_name, lua_tostring(L,2));
		lua_pushstring(L, "add_job: path is not nice");
		lua_error(L);
	}
	else if(i == NODECREATE_EXISTS)
	{
		printf("%s: '%s' already exists\n", program_name, lua_tostring(L,2));
		lua_pushstring(L, "add_job: node already exists");
		lua_error(L);
	}
	else if(i != NODECREATE_OK)
	{
		lua_pushstring(L, "add_job: unknown error creating node");
		lua_error(L);
	}
	
	/* save arguments */
	{
		/* fetch global options table */
		lua_pushstring(L, CONTEXT_LUA_OPTIONS_TABLE);
		lua_gettable(L, LUA_GLOBALSINDEX);
		
		/* create table that will hold all the arguments */
		lua_pushnumber(L, node->id); /* id in the options table */
		lua_newtable(L); /* argument table */
		
		/* add arguments to table */
		for(i = 3; i <= n; ++i)
		{
			lua_pushnumber(L, i-2); /* 1 .. */
			lua_pushvalue(L, i); /* push value */
			lua_settable(L, -3);
		}
		
		/* save table to */
		lua_settable(L, -3); /* L_OPTIONS_TABLE[node->id] = arguments */
	}
	
	return 0;
}

/************/
static int lf_add_dependency(lua_State *L)
{
	struct NODE *node;
	struct CONTEXT *context;

	int n = lua_gettop(L);
	int i;
	
	if(n < 2)
	{
		lua_pushstring(L, "add_dep: to few arguments");
		lua_error(L);
	}

	context = context_get_pointer(L);

	node = node_find(context->graph, lua_tostring(L,1));
	if(!node)
	{
		char buf[512];
		sprintf(buf, "add_dep: couldn't find node with name '%s'", lua_tostring(L,1));
		lua_pushstring(L, buf);
		lua_error(L);
	}
	
	/* seek deps */
	for(i = 2; i <= n; ++i)
	{
		if(lua_isstring(L,n))
		{
			if(!node_add_dependency(node, lua_tostring(L,n)))
			{
				lua_pushstring(L, "add_dep: could not add dependency");
				lua_error(L);
			}
		}
		else
		{
			lua_pushstring(L, "add_dep: dependency is not a string");
			lua_error(L);
		}
	}
	
	return 0;
}


/*             str      */
/* default_target(filename) */
static int lf_default_target(lua_State *L)
{
	struct NODE *node;
	struct CONTEXT *context;

	int n = lua_gettop(L);
	if(n != 1)
	{
		lua_pushstring(L, "default_target: incorrect number of arguments");
		lua_error(L);
	}
	
	if(!lua_isstring(L,1))
	{
		lua_pushstring(L, "default_target: expected string");
		lua_error(L);
	}

	/* fetch context from lua */
	context = context_get_pointer(L);

	/* search for the node */
	node = node_find(context->graph, lua_tostring(L,1));
	if(!node)
	{
		lua_pushstring(L, "default_target: node not found");
		lua_error(L);
	}
	
	/* set target */
	context_default_target(context, node);
	return 0;
}

/*             str      */
/* add_target(filename) */
static int lf_add_target(lua_State *L)
{
	struct NODE *node;
	struct CONTEXT *context;

	int n = lua_gettop(L);
	if(n != 1)
	{
		lua_pushstring(L, "add_target: incorrect number of arguments");
		lua_error(L);
	}
	
	if(!lua_isstring(L,1))
	{
		lua_pushstring(L, "add_target: expected string");
		lua_error(L);
	}

	/* fetch context from lua */
	context = context_get_pointer(L);

	/* search for the node */
	node = node_find(context->graph, lua_tostring(L,1));
	if(!node)
	{
		lua_pushstring(L, "add_target: node not found");
		lua_error(L);
	}
	
	/* add target */
	if(context_add_target(context, node))
	{
		lua_pushstring(L, "add_target: target already exists");
		lua_error(L);
	}
	
	/* we need to save the options some where */
	return 0;
}

/* external, found in depend.c */
extern int dependency_cpp_run(const char *filename,
	int (*callback)(void *, const char *, int), void *userdata);

struct CPPDEPPATH
{
	char *path;
	struct CPPDEPPATH *next;
};

struct CPPDEPINFO
{
	struct NODE *node;
	struct CPPDEPPATH *first_path;
};

/* */
static int dependency_cpp_callback(void *user, const char *filename, int sys)
{
	struct CPPDEPINFO *depinfo = (struct CPPDEPINFO *)user;
	struct NODE *node = depinfo->node;
	struct NODE *depnode;
	struct CPPDEPINFO recurseinfo;
	char buf[512];
	char normal[512];
	int check_system = sys;
	int found = 0;
	
	if(!sys)
	{
		/* "normal.header" */
		int flen = strlen(node->filename)-1; 
		int flen2 = strlen(filename);
		while(flen)
		{
			if(node->filename[flen] == '/')
				break;
			flen--;
		}
		
		memcpy(buf, node->filename, flen+1);
		memcpy(buf+flen+1, filename, flen2);
		buf[flen+flen2+1] = 0;
		
		if(!file_exist(buf) && !node_find(node->graph, filename))
		{
			/* file does not exist */
			memcpy(normal, buf, 512);
			check_system = 1;
		}
		else
			found = 1;
	}

	if(check_system)
	{
		/* <system.header> */
		if(path_isabs(filename))
		{
			if(file_exist(filename) || node_find(node->graph, filename))
			{
				strcpy(buf, filename);
				found = 1;
			}
		}
		else
		{
			struct CPPDEPPATH *cur;
			int flen = strlen(filename);
			int plen;

			for(cur = depinfo->first_path; cur; cur = cur->next)
			{
				plen = strlen(cur->path);
				memcpy(buf, cur->path, plen);
				memcpy(buf+plen, filename, flen+1); /* copy the 0-term aswell */

				if(file_exist(buf) || node_find(node->graph, buf))
				{
					if(option_cppdep_includewarnings && !sys)
					{
						printf("%s: c++ dependency error. could not find \"%s\" as a relative file but as a system header.\n", program_name, filename);
						return -1;
					}

					found = 1;

					/* printf("found system header header %s at %s\n", filename, buf); */
					break;
				}
			}
		}
	
		/* no header found */
		if(!found)
		{
			/* if it was a <system.header> we are just gonna ignore it and hope that it never */
			/* changes */
			if(!sys && option_cppdep_includewarnings)
			{
				/* not a system header so we gonna bitch about it */
				char a[] = {'"', '<'};
				char b[] = {'"', '>'};
				printf("%s: C++ DEPENDENCY WARNING: header not found. %c%s%c\n", program_name, a[sys], filename, b[sys]);
			}

			if(sys)
				return 0;
			else
				memcpy(buf, normal, 512);

			/*
			if(node_add_dependency(node, buf) != 0)
				return 4;
			*/
			/*return 0;*/
		}
	}
		
	/* */
	path_normalize(buf);
	depnode = node_add_dependency(node, buf);
	if(!depnode)
		return 2;
	
	node->depchecked = 1;
	
	/* do the dependency walk */
	if(found && !depnode->depchecked)
	{
		recurseinfo.first_path = depinfo->first_path;
		recurseinfo.node = depnode;
		
		if(dependency_cpp_run(buf, dependency_cpp_callback, &recurseinfo) != 0)
			return 3;
	}
	
	return 0;
}

/* */
static int lf_dependency_cpp(lua_State *L)
{
	struct NODE *node;
	struct CONTEXT *context;
	struct HEAP *includeheap;
	struct CPPDEPINFO depinfo;
	struct CPPDEPPATH *cur_path;
	struct CPPDEPPATH *prev_path;
	int n = lua_gettop(L);
	int i;
	
	if(n != 2)
	{
		lua_pushstring(L, "dependency_cpp: incorrect number of arguments");
		lua_error(L);
	}

	if(!lua_isstring(L,1))
	{
		lua_pushstring(L, "dependency_cpp: expected string");
		lua_error(L);
	}

	if(!lua_istable(L,2))
	{
		lua_pushstring(L, "dependency_cpp: expected table");
		lua_error(L);
	}
	
	/* no need to do the depenceny stuff if we only going to clean */
	if(option_clean)
		return 0;
	
	/* fetch context */
	context = context_get_pointer(L);
	
	/* create a heap to store the includes paths in */
	includeheap = mem_create();

	/* fetch the system include paths */
	lua_pushnil(L);
	cur_path = 0x0;
	prev_path = 0x0;
	depinfo.first_path = 0x0;
	while(lua_next(L, 2))
	{
		if(lua_isstring(L,-1))
		{
			/* allocate the path */
			i = strlen(lua_tostring(L,-1));
			cur_path = (struct CPPDEPPATH*)mem_allocate(includeheap, sizeof(struct CPPDEPPATH)+i+2);
			cur_path->path = (char *)(cur_path+1);
			cur_path->next = 0x0;
		
			/* copy path and terminate with a / */
			memcpy(cur_path->path, lua_tostring(L,-1), i);
			cur_path->path[i] = '/';
			cur_path->path[i+1] = 0;
		
			/* add it to the chain */
			if(prev_path)
				prev_path->next = cur_path;
			else
				depinfo.first_path = cur_path;
			prev_path = cur_path;
		
		}
		
		/* pop the value, keep the key */
		lua_pop(L, 1);
	}

	/*
	for(cur_path = depinfo.first_path; cur_path; cur_path = cur_path->next)
		printf("p: %s\n", cur_path->path);
	*/
	
	
	/**/
	node = node_find(context->graph, lua_tostring(L,1));
	if(!node)
	{
		mem_destroy(includeheap);
		lua_pushstring(L, "dependency_cpp: node not found");
		lua_error(L);
	}
	
	/* do the dependency walk */
	/* TODO: caching system */
	depinfo.node = node;
	if(dependency_cpp_run(node->filename, dependency_cpp_callback, &depinfo) != 0)
	{
		mem_destroy(includeheap);
		lua_pushstring(L, "dependency_cpp: error during depencency check");
		lua_error(L);
	}

	/* free the include heap */
	mem_destroy(includeheap);
	
	return 0;
}

static int run_node(struct NODE *node, lua_State *L)
{
	int numargs = 0;
	int retvalue = 0;
	
	/* fetch tool */
	lua_pushstring(L, node->tool);
	lua_gettable(L, LUA_GLOBALSINDEX);
	if(!lua_isfunction(L, -1))
	{
		printf("couldn't find tool '%s', not a function.\n", node->tool);
		return 1;
	}
	
	/* resulting file is the first argument */
	lua_pushstring(L, node->filename); 

	/* fetch arguments table */
	lua_pushstring(L, CONTEXT_LUA_OPTIONS_TABLE);
	lua_gettable(L, LUA_GLOBALSINDEX);
	lua_pushnumber(L, node->id);
	lua_gettable(L,-2);

	lua_insert(L,-2); /* remove the options table */
	lua_pop(L,1);
	
	/* push the rest */
	lua_pushnil(L);
	numargs = 0;
	while(lua_next(L,-2) != 0)
	{
		numargs++;
		lua_insert(L,-3); /* is this correct ?? */
	}
	
	lua_pop(L,1); /* remove the argument table */

	/* call the tool */
	if(lua_pcall(L, numargs+1, 1, 0) != 0)
	{
		printf("%s: error during call to tool: %s\n", program_name, 
			lua_tostring(L,-1));
		lua_pop(L,1);
		return 1;
	}

	/* check the return value */
	if(lua_isnumber(L,-1))
		retvalue = (int)lua_tonumber(L,-1);

	if(retvalue)
		printf("%s: %s returned error number %d\n", program_name, node->tool, retvalue);

	/* pop the return value */
	lua_pop(L,1);
	return retvalue;
}

struct THREADINFO
{
	int id;
	lua_State *lua;
	struct NODE *node; /* the top node */
	int errorcode;
};

static int threads_run_callback(struct NODE *node, void *u)
{
	struct THREADINFO *info = (struct THREADINFO *)u;
	struct DEPENDENCY *dep;
	int errorcode = 0;

	/* if it doesn't have a tool, just mark it as done and continue the search */
	if(!node->tool)
	{
		node->workstatus = NODESTATUS_DONE;
		return 0;
	}
	
	/* make sure that all deps are done */
	for(dep = node->firstdep; dep; dep = dep->next)
	{
		if(dep->node->dirty && dep->node->workstatus != NODESTATUS_DONE)
			return 0;
	}

	/* mark the node as its in the working */
	node->workstatus = NODESTATUS_WORKING;
	
	/* unlock the dependency graph, run the node and then lock it again */
	criticalsection_leave();
	errorcode = run_node(node, info->lua);
	criticalsection_enter();
	
	/* this node is done, mark it so and return the error code */
	node->workstatus = NODESTATUS_DONE;
	return errorcode;
}

static void threads_run(void *u)
{
	struct THREADINFO *info = (struct THREADINFO *)u;
	int flags = NODEWALK_BOTTOMUP|NODEWALK_UNDONE|NODEWALK_QUICK;
	
	if(option_force)
		flags |= NODEWALK_FORCE;
	
	info->errorcode = 0;
	
	/* lock the dependency graph */
	criticalsection_enter();
	
	if(info->node->dirty)
	{
		while(1)
		{
			info->errorcode = node_walk(info->node, flags, threads_run_callback, info);
			
			/* check if we are done */
			if(info->node->workstatus == NODESTATUS_DONE || info->errorcode != 0)
				break;
				
			/* let the others have some time */
			criticalsection_leave();
			threads_yield();
			criticalsection_enter();
		}
	}
	criticalsection_leave();
}

static int run_callback(struct NODE *node, void *u)
{
	/* no tool, no processing */
	if(!node->tool)
		return 0;
	
	if(!node->dirty)
		return 0;
	
	node->dirty = 0;
	return run_node(node, (lua_State *)u);
}

static int run(struct NODE *node, lua_State *L)
{
	int flags;

	if(option_threads > 0)
	{
		/* multithreaded */
		struct THREADINFO info[64];
		void *threads[64];
		int i;
		
		/* clamp number of threads */
		if(option_threads > 64)
			option_threads = 64;
		
		/* make lua run the GC */
		lua_setgcthreshold(L, 0);
		lua_setgcthreshold(L, 0x7fffffff);
		
		for(i = 0; i < option_threads; i++)
		{
			info[i].lua = lua_newthread(L);
			info[i].node = node;
			info[i].id = i;
			info[i].errorcode = 0;
		}

		/* start threads */
		for(i = 0; i < option_threads; i++)
			threads[i] = threads_create(threads_run, &info[i]);
		
		/* wait for threads */
		for(i = 0; i < option_threads; i++)
		{
			threads_wait(threads[i]);
			threads_destroy(threads[i]);
		}
		
		for(i = 0; i < option_threads; i++)
		{
			if(info[i].errorcode)
				return info[i].errorcode;
		}
		return 0;
	}
	else
	{
		/* no threading */
		flags = NODEWALK_BOTTOMUP|NODEWALK_QUICK;
		if(option_force)
			flags |= NODEWALK_FORCE;
		return node_walk(node, flags, run_callback, L);
	}
}

static int clean_callback(struct NODE *node, void *u)
{
	(void)u;
	
	/* no tool, no processing */
	if(!node->tool)
		return 0;

	if(node->timestamp)
	{
		if(remove(node->filename) == 0)
			printf("%s: removed '%s'\n", program_name, node->filename);
	}
	return 0;
}

static void clean(struct NODE *node)
{
	node_walk(node, NODEWALK_BOTTOMUP|NODEWALK_FORCE, clean_callback, 0);
}

static int do_target(lua_State *L, struct NODE *node)
{
	if(option_clean)
	{
		printf("%s: cleaning '%s'\n", program_name, path_filename(node->filename));
		clean(node);
		return 0;
	}
	else
	{
		printf("%s: building '%s'\n", program_name, path_filename(node->filename));
		return run(node, L);
	}
}

/* *** */
static int process_targets(struct CONTEXT *context)
{
	struct TARGET *target = context->firsttarget;
	if(!target)
	{
		printf("%s: no targets\n", program_name);
		return -1;
	}
	
	while(target)
	{
		int errorcode = do_target(context->lua, target->node);
		if(errorcode)
			return errorcode;
		target = target->next;
	}
	
	return 0;
}

/* dirty marking */
static int dirty_mark_callback(struct NODE *node, void *u)
{
	struct DEPENDENCY *dep;
	unsigned int time = timestamp();
	(void)u;
	
	for(dep = node->firstdep; dep; dep = dep->next)
	{
		if(node->timestamp > time)
			printf("%s: WARNING:'%s' comes from the future\n", program_name, node->filename);
		
		if(dep->node->dirty)
		{
			node->dirty = 1;
			break;
		}
		else if(node->timestamp < dep->node->timestamp)
		{
			if(node->tool)
			{
				node->dirty = 1;
				break;
			}
			else
				node->timestamp = dep->node->timestamp;
		}
	}
	return 0;
}

void dirty_mark(struct CONTEXT *context)
{
	struct TARGET *target = context->firsttarget;
	while(target)
	{
		node_walk(target->node, NODEWALK_BOTTOMUP|NODEWALK_FORCE, dirty_mark_callback, 0);
		target = target->next;
	}
}

/* view dependency graph */
static int dump_callback(struct NODE *node, void *u)
{
	const char *tool = "";
	const char *dirty = "";
	
	(void)u;
	
	if(node->tool)
		tool = node->tool;
	if(node->dirty)
		dirty = "D";
	
	printf("%-15s %2s %8x ", tool, dirty, node->timestamp);

	printf("%s\n", node->filename);
	return 0;
}

void dump_dependecy_graph(struct CONTEXT *context, lua_State *L)
{
	struct TARGET *target = context->firsttarget;
	if(target)
	{
		while(target)
		{
			node_walk(target->node, NODEWALK_FORCE|NODEWALK_TOPDOWN, dump_callback, L);
			target = target->next;
		}
	}
	else
		printf("%s: no targets\n", program_name);
}

/* error function */
static int lf_errorfunc(lua_State *L)
{
	int depth = 0;
	lua_Debug frame;
	printf("%s: %s\n", program_name, lua_tostring(L,-1));
	printf("strack traceback:\n");
	
	while(lua_getstack(L, depth, &frame) == 1)
	{
		lua_getinfo(L, "nlSf", &frame);
		printf("\t%s(%d): %s\n", frame.short_src, frame.currentline, frame.name);
		{
			int i;
			const char *name = 0;
			
			i = 1;
			while((name = lua_getlocal(L, &frame, i++)) != NULL)
			{
				printf("\t\t%s=", name);
				debug_print_lua_value(L,i);
				printf("\n");
				lua_pop(L,1);
			}
			
			i = 1;
			while((name = lua_getupvalue(L, -1, i++)) != NULL)
			{
				/* TODO: parse value as well*/
				printf("\t\tupvalue: %d %s\n", i-1, name);
				lua_pop(L,1);
			}
		}

		depth++;
	}
	return 1;
}


/* *** */
/* TODO: remove this */
extern int luaopen_support(lua_State *L);

int register_lua_globals(struct CONTEXT *context)
{
	/* create the clone table */
	/* TODO: make C code for this */
	static const char clonecreate[] = 
		"_bam_clone = {}\n"
		"for i,v in _G do\n"
		"	_bam_clone[i] = v\n"
		"end\n";
		
	/* add standard libs */
	luaopen_base(context->lua);
	luaopen_table(context->lua);
	luaopen_io(context->lua);
	luaopen_string(context->lua);
	luaopen_math(context->lua);
	luaopen_debug(context->lua);
	luaopen_loadlib(context->lua);

	support_luaopen(context->lua);
	
	/* add specific functions */
	lua_register(context->lua, L_FUNCTION_PREFIX"add_job", lf_add_job);
	lua_register(context->lua, L_FUNCTION_PREFIX"add_dependency", lf_add_dependency);
	lua_register(context->lua, L_FUNCTION_PREFIX"add_target", lf_add_target);
	lua_register(context->lua, L_FUNCTION_PREFIX"default_target", lf_default_target);

	lua_register(context->lua, L_FUNCTION_PREFIX"path_join", lf_path_join);
	lua_register(context->lua, L_FUNCTION_PREFIX"path_fix", lf_path_fix);
	lua_register(context->lua, L_FUNCTION_PREFIX"path_isnice", lf_path_isnice);
	
	lua_register(context->lua, "errorfunc", lf_errorfunc);
	
	/* accelerator functions */
	lua_register(context->lua, L_FUNCTION_PREFIX"dependency_cpp", lf_dependency_cpp);

	/* create options table */
	lua_pushstring(context->lua, CONTEXT_LUA_OPTIONS_TABLE);
	lua_newtable(context->lua);
	lua_settable(context->lua, LUA_GLOBALSINDEX);
	
	/* set paths */
	{
		char cwd[512];
		getcwd(cwd, 512);
		
		lua_pushstring(context->lua, CONTEXT_LUA_PATH);
		lua_pushstring(context->lua, context->script_directory);
		lua_settable(context->lua, LUA_GLOBALSINDEX);

		lua_pushstring(context->lua, CONTEXT_LUA_WORKPATH);
		lua_pushstring(context->lua, cwd);
		lua_settable(context->lua, LUA_GLOBALSINDEX);
	}

	/* set context */
	lua_pushstring(context->lua, CONTEXT_LUA_CONTEXT_POINTER);
	lua_pushlightuserdata(context->lua, context);
	lua_settable(context->lua, LUA_GLOBALSINDEX);

	/* set family */
	lua_pushstring(context->lua, "family");
	lua_pushstring(context->lua, BAM_FAMILY_STRING);
	lua_settable(context->lua, LUA_GLOBALSINDEX);

	/* set platform */
	lua_pushstring(context->lua, "platform");
	lua_pushstring(context->lua, BAM_PLATFORM_STRING);
	lua_settable(context->lua, LUA_GLOBALSINDEX);

	/* set arch */
	lua_pushstring(context->lua, "arch");
	lua_pushstring(context->lua, BAM_ARCH_STRING);
	lua_settable(context->lua, LUA_GLOBALSINDEX);

	/* set verbosity level */
	lua_pushstring(context->lua, "verbose");
	lua_pushnumber(context->lua, option_verbose);
	lua_settable(context->lua, LUA_GLOBALSINDEX);

	/* load base script */
	if(option_basescript)
	{
		if(option_verbose)
			printf("%s: reading base script from '%s'\n", program_name, option_basescript);
		if(lua_dofile(context->lua, option_basescript))
		{
			printf("%s: error in base script '%s'\n", program_name, option_basescript);
			return 1;
		}
	}
	else
	{
		if(option_verbose)
			printf("%s: reading internal base script\n", program_name);
		
		if(lua_dobuffer(context->lua, internal_base, sizeof(internal_base)-1, "internal base"))
		{
			printf("%s: error in internal base script\n", program_name);
			return 1;
		}
	}
	
	if(lua_dobuffer(context->lua, clonecreate, sizeof(clonecreate)-1, "internal clone creation"))
	{
		printf("%s: error in internal clone creation script\n", program_name);
		return 1;
	}

	return 0;
}

/* *** */
static int bam(const char *scriptfile, const char **targets, int num_targets)
{
	struct CONTEXT context;
	int error = 0;
	int i;

	/* build time */
	time_t starttime;

	if(option_debug_buildtime)
		starttime = time(0x0);
		
	/* clean the context structure */
	memset(&context, 0, sizeof(struct CONTEXT));

	/* create memory heap */
	context.heap = mem_create();
	
	/* create graph */
	context.graph = node_create_graph(context.heap);

	/* set filename */
	context.filename = scriptfile;
	context.filename_short = path_filename(scriptfile);
	
	{
		char cwd[512];
		char path[512];

		getcwd(cwd, 512);
		if(path_directory(context.filename, path, 512))
		{
			printf("crap error1\n");
			*((int*)0) = 0;
		}
		
		if(path_join(cwd, path, context.script_directory, 512))
		{
			printf("crap error2\n");
			*((int*)0) = 0;
		}
	}
	
	/* create lua context */
	context.lua = lua_open();
	
	/* register all functions */
	if(register_lua_globals(&context) != 0)
		return 1;
	
	/* load script */	
	if(option_verbose)
		printf("%s: reading script from '%s'\n", program_name, scriptfile);

	if(1)
	{
		int ret;
		lua_getglobal(context.lua, "errorfunc");
		
		/* push error function to stack */
		ret = luaL_loadfile(context.lua, scriptfile);
		if(ret != 0)
		{
			if(ret == LUA_ERRSYNTAX)
				printf("%s: syntax error\n", program_name);
			else if(ret == LUA_ERRMEM)
				printf("%s: memory allocation error\n", program_name);
			else if(ret == LUA_ERRFILE)
				printf("%s: error opening '%s'\n", program_name, scriptfile);
			lf_errorfunc(context.lua);
			error = 1;
		}
		else if(lua_pcall(context.lua, 0, LUA_MULTRET, -2) != 0)
			error = 1;
	}

	/* mark dirty nodes */
	dirty_mark(&context);

	/* debug */
	if(option_debug_nodes)
	{
		/* debug dump all nodes */
		node_debug_dump(context.graph);
	}
	else if(option_debug_tree)
	{
		/* debug dump the tree */
		dump_dependecy_graph(&context, context.lua);
	}
	else if(option_dry)
	{
		/* do NADA */
	}
	else
	{
		/* run */
		if(error == 0)
		{
			if(num_targets)
			{
				for(i = 0; i < num_targets; i++)
				{
					if(strcmp(targets[i], "all") == 0)
					{
						error = process_targets(&context);
						break;
					}
				}
				
				if(i == num_targets) /* no all target was found */
				{
					for(i = 0; i < num_targets; i++)
					{
						char cwd[512];
						char nice_target[512];
						struct NODE *node;
						getcwd(cwd, 512);
						if(path_join(context.script_directory, targets[i], nice_target, 512))
						{
							printf("crap error1\n");
							*((int*)0) = 0;
						}
					
						node = node_find(context.graph, nice_target);
						if(node)
						{
							error = do_target(context.lua, node);
						}
						else
						{
							printf("%s: target not found, '%s' ('%s')\n", program_name, targets[i], nice_target);
							error = -1;
						}

						if(error)
							break;
					}
				}
			}
			else
			{
				if(context.defaulttarget)
					error = do_target(context.lua, context.defaulttarget);
				else
				{
					printf("%s: no default target\n", program_name);
					error = -1;
				}
			}
		}
	}

	/* */
	if(option_debug_memstats)
		mem_dumpstats(context.heap);

	/* clean up */
	lua_close(context.lua);
	mem_destroy(context.heap);

	/* print and return */
	if(error)
		printf("%s: error during build\n", program_name);
	else
		printf("%s: done\n", program_name);

	/* print build time */
	if(option_debug_buildtime)
	{
		int s = time(0x0) - starttime;
		printf("\nbuild time: %d:%.2d\n", s/60, s%60);
	}

	return error;
}


/* */
static void print_help()
{
	int j;
	printf("bam version X. built "__DATE__" "__TIME__" using " LUA_VERSION "\n");
	printf("mainly coded by matricks. direct anger to matricks (at) teewars.com\n");
	printf("\n");
	printf("%s [OPTIONS] [TARGETS]\n", program_name);
	printf("\n");

	for(j = 0; options[j].sw; j++)
		printf("  %-20s %s\n", options[j].sw, options[j].desc);

	printf("\n");
}

/* signal handler */
static void abortsignal(int i)
{
	(void)i;
	printf("signal cought! exiting.\n");
	exit(1);
}

/***********/
int main(int argc, char **argv)
{
	int i, j, error;

	/* init platform */
	install_signals(abortsignal);
	platform_init();

	/* fetch program name, if we can */
	for(i = 0; argv[0][i]; ++i)
	{
		if(argv[0][i] == '/' || argv[0][i] == '\\')
			program_name = &argv[0][i+1];
	}

	/* get basescript */
	if(getenv("BAM_BASESCRIPT"))
		option_basescript = getenv("BAM_BASESCRIPT");

	/* parse parameters */
	for(i = 1; i < argc; ++i)
	{
		for(j = 0; options[j].sw; j++)
		{
			if(strcmp(argv[i], options[j].sw) == 0)
			{
				if(options[j].s)
				{
					if(i+1 >= argc)
					{
						printf("%s: you must supply a argument with %s\n", program_name, options[j].sw);
						return 1;
					}
					
					i++;
					*options[j].s = argv[i];
				}
				else
					*options[j].v = 1;

				j = -1;
				break;
			}
		}
		
		if(j != -1)
		{
			/* TODO: there should be some sort of way to pass arguments to the
				spellbook itself to select target and such */

			if(argv[i][0] == '-')
			{
				printf("%s: unknown switch '%s'\n", program_name, argv[i]);
				return 1;
			}
			else
			{
				option_targets[option_num_targets] = argv[i];
				option_num_targets++;
			}
		}
	}
	
	/* convert the threads string */
	option_threads = atoi(option_threads_str);
	if(option_threads < 0)
	{
		printf("%s: invalid number of threads supplied\n", program_name);
		return 1;
	}
	
	/* check for help argument */
	if(option_print_help)
	{
		print_help();
		return 0;
	}
	
	/* check if we should dump the internal base script */
	if(option_dump_internal)
	{
		printf("%s", internal_base);
		return 0;
	}
	
	/* check if a script exist */
	if(!file_exist(option_script))
	{
		printf("%s: no project named '%s'\n", program_name, option_script);
		return 1;
	}
	
	/* init the context */
	/* TODO: search for a bam file */
	error = bam(option_script, option_targets, option_num_targets);

	platform_shutdown();

	/* error could be some high value like 256 */
	/* seams like this could be clamped down to a */
	/* unsigned char and not be an error anymore */
	if(error)
		return 1;
	return 0;
}
