package Boot::Grub;

#   $Header: /cvsroot/systemconfig/systemconfig/lib/Boot/Grub.pm,v 1.15 2001/10/26 04:19:42 donghwajohnkim Exp $


#   Copyright (c) 2001 International Business Machines

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

#   See the
#   GNU General Public License for more details.
 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#   Donghwa John Kim <johkim@us.ibm.com>

=head1 NAME

Boot::Grub - Grub bootloader configuration module.

=head1 SYNOPSIS

  my $bootloader = new Boot::Grub(%bootvars);

  if($bootloader->footprint_loader()) {
      $bootloader->install_config();
  }
  
  if($bootloader->footprint_config() && $bootloader->footprint_loader()) {
    $boot->install_loader();
  }

  my @fileschanged = $bootloader->files();

=cut

use strict;
use Carp;
use vars qw($VERSION);
use Boot;
use Util::Cmd qw(:all);
use Util::Log qw(:print);

$VERSION = sprintf("%d.%02d", q$Revision: 1.15 $ =~ /(\d+)\.(\d+)/);

push @Boot::boottypes, qw(Boot::Grub);

sub new {
    my $class = shift;
    my %this = (
                root => "",
                filesmod => [],
		boot_rootdev => "",          ### Global root device 
		boot_timeout => 50,   
		boot_defaultboot => "",      ### Label identifying default image to boot with.
                @_,                          ### Overwrite default values.
		bootloader_exe => "",        ### Path to Grub executable.
		default_root => "",          ### The default root device.
		defaultbootnum => "",        ### An integer identifying I-th kernel image which corresponds to
					     ### the default image to boot with.
		boot_bootdev => "(hd0)",     ### Device to which to install boot image.
		                             ### NOTICE: This will overwrite user specified boot device.
		                             ### Grub will always install boot images to MBR of the device 
		                             ### identified by BIOS as the first bootable disk drive. 
		                             ### This may change in the future.
               );

    ### Let's see if Grub is installed in the system. 
    $this{bootloader_exe} = which("grub");
    verbose("Grub executable set to: $this{bootloader_exe}.");
    $this{device_map_file} = "$this{root}/boot/grub/device.map";
    $this{config_file} = "$this{root}/boot/grub/menu.lst";

    ### Initially set path to the default root device
    if ($this{boot_rootdev}) {
        $this{default_root} = $this{boot_rootdev};
    }
    ### Or should we overwrite it with the one specified under the default kernel image.
    undef $this{defaultbootnum};
    my $counter = 0;
    DBOUT: foreach my $key (sort keys %this) {
        if ($key =~ /^(kernel\d+)_label/) {
            if ($this{$key} eq $this{boot_defaultboot}) {
                ### Order of default kernel image.
                $this{defaultbootnum} = $counter; ### This is the i-th image
                                                  ### from the top of the config file.
		verbose("Default boot number set to: $1");
		if ($this{"$1_rootdev"}) {
                $this{default_root} = $this{"$1_rootdev"};
		}
                last DBOUT;
            }
            $counter++;
        }
    }

    bless \%this, $class;
}

=head1 METHODS

The following methods exist in this module:

=over 4

=item files()

The files() method is merely an accessor method for the all files
touched by the instance during its run.

=cut

sub files {
    my $this = shift;
    return @{$this->{filesmod}};
}

=item footprint_loader()

This method returns "TRUE" if executable Grub bootloader is installed. 

=cut

sub footprint_loader {
    my $this = shift;
    return $this->{bootloader_exe};
}

=item footprint_config()

This method returns "TRUE" if Grub's configuration file, i.e. "/boot/grub/menu.lst", exists. 

=cut

sub footprint_config {
    my $this = shift;
    return -e $this->{config_file};
}

=item dev2bios()

This method is used as an internal method.
It takes a path to the device as the argument and returns device syntax ( 
matching an entry in the device map file) used by Grub 

=cut

sub dev2bios {
# Searches for and converts "/dev/ ..."  to device syntax used by Grub from the device map file.
# Arg: path to the device.
    
    my ($device, $devpart);
    my $this = shift;
    my $devpath = shift;
    my $biosdev = "";
    
    $devpath =~ /^(\/dev\/.+?)(\d*?)$/; # extract partition number and device name
    $device = $1;                     # from the device path.  
    $devpart = $2;

    ### First, check to see if the device map file exists, if it doesn't
    ### let's create one. It will make our job much easier.
    unless (-e $this->{device_map_file}) { 
        my ($output, $exitval);
	my $command = "" . <<"EOL";
$this->{bootloader_exe} --device-map=$this->{device_map_file} 2>&1 << ETO 
ETO
EOL
        $output = qx/$command/;
	$exitval = $? >> 8;
	if ($exitval) {
	    croak("Error: cannot create the device map file!\n$output\n");
	}
	### Add device map file to the exclusion list.
	push @{$this->{filesmod}}, "$this->{device_map_file}";
    }


    if ($device) { ### Empty string could have been passed as the input param
	open(MAPFH,"<$this->{device_map_file}") 
	    or croak("Couldn't open $this->{device_map_file} for reading");
	
	while(<MAPFH>) {
	    if (/(\(.+)\)\s+$device/) {
		if ($devpart) {
		    $devpart--; # Partition number starts from 0
		    $biosdev = "$1,$devpart)";
		}
		else {
		    $biosdev = "$1)";
		}
		last;
	    }
	}
	close(MAPFH);
    }
    $biosdev;
}

=item install_loader()
    
install_loader() is an internal method. 

This method invokes the Grub executable.
Grub write the boot image onto the MBR of the bootable disk.  

=cut

sub install_loader {
    my $this = shift;
    my ($setupcmd, $grubdirhash);
    my ($output, $exitval);

    $grubdirhash = $this->getdevhash("/boot/grub");

    unless ($grubdirhash->{"device"}) {
	croak("Error: Cannot derive device on which grub dir exists");
    }
    verbose("According to /etc/fstab, grub directory is on: $grubdirhash->{'device'}.");
    
    $setupcmd = "" . <<"EOL";
$this->{bootloader_exe} 2>&1 << ETO 
setup $this->{boot_bootdev} $grubdirhash->{device}
quit
ETO
EOL
    
    $output = qx/$setupcmd/;
    $exitval = $? >> 8;
    if ($exitval) {
	croak("Error: Setting up of Grub failed!\n$output\n");
    }    

    1;
}

=item getdevhash()
    
getdevhash() is an internal method. 

This method takes full path to a file on disk as input, and breaks it
down into Grub readable device name and the remaining part.

=cut

sub getdevhash {
    ### Arguments: $path => Full path name to a file or a directory  which is to be broken down into
    ###               Grub readable device name and remaining path thereafter.
    ### Returns: a hash containing Grub readable device name and remaining path.
    ### Must be able to open "/etc/fstab" and "/proc/partitions".
    ### This might not be the most efficient solution, i.e. opening fstab file
    ###    each time, this will certainly make the code cleaner.
    ### Based on a code segment from Palo.pm (Dann Frazier). Thanks Dann!  

    ### This will most likely be a separate class very soon.
    
  my ($this, $path) = @_;
  my $pathbase = "";
  my @patharray;
  my %returnval;
  my @all_part_array;
  my %fstab_hash;
  my $e2label_exec;
  my ($label2partition, $label_mount);
  my ($major, $minor, $blocks, $name, $junk1);
  my ($partition, $mount, $junk2);
  my $tempstring;

  # Support for e2label.
  # First define all the partitions in the system.
  # Read from "/proc/partitions  
  open(PROCPART, "$this->{root}/proc/partitions")
    or croak("Couldn't open $this->{root}/proc/partitions for reading");
  while(<PROCPART>){
    unless(/\s*\d/) {
      next;
    }
    ($major, $minor, $blocks, $name, $junk1) = split " ", $_, 5;
    push @all_part_array, $name;
  }
  
  open(FSTAB, "$this->{root}/etc/fstab")
    or croak("Couldn't open $this->{root}/etc/fstab for reading");
  while(<FSTAB>){
    unless((/^\s*\/dev\//) || (/^\s*LABEL/)) {
      next;
    }
    ($partition, $mount, $junk2) = split " ", $_, 3;
    $fstab_hash{"$partition"} = $mount;
  }
  
  foreach $partition (@all_part_array){
    # If the whole disk, i.e. hda, sda, skip.
    unless($partition =~ /[a-zA-Z]+\d+/) {
      next;
    }
    # In case of proc partition, skip.
    if((defined $fstab_hash{"/dev/$partition"}) && 
       ($fstab_hash{"/dev/$partition"} eq "/proc")) {
      next;
    }   
    # At this point we assume that if there is an entry in
    # "/proc/partitions which does not have a value in %fstab_hash,
    # it is most likely be e2labeled.
    unless(defined $fstab_hash{"/dev/$partition"}) {
      unless(which("e2label")) {
	croak("ERROR: e2label executable not found.");
      }
      $label2partition = qx[e2label /dev/$partition];
      if ($?) {
	verbose("WARNING: Error while issuing e2label on /dev/$partition.");
	next;
      }
      else {
	  $label2partition = "LABEL=$label2partition";
	  chomp $label2partition;
      }
      $label_mount = $fstab_hash{"$label2partition"};
      $fstab_hash{"/dev/$partition"} = $label_mount;
      delete $fstab_hash{"$label2partition"};
    }
  }
  
  if ($this->{root}) {
    if ($path =~ /^$this->{root}(.*)$/) {
      $path = $1; ### remove temporary root from the path
    }
  }

  @patharray = split /\//, $path;

  unless (-e $this->{root} . $path) {
    verbose("Warning: $this->{root}$path does not exist");
    return 0;
  }

  if (-f $this->{root} . $path) {  ### If the full path name is not a directory, then move down a level
    pop @patharray;
  }
  
  $returnval{device} = "";
  while(!$returnval{device}) {
    unless (@patharray) {  ### We moved down too far.
      verbose("ERROR: \@patharray has reached zero length.");
      croak("ERROR: Internal error Boot::Grub::getdevhash.");
    }

    $pathbase = join '/', @patharray;     ### Reconstruct the path 
    if (!$pathbase) { $pathbase = "/"; }  ### File/directory on "/".
    foreach $partition (keys %fstab_hash) {
      # In case of proc partition, skip.
      if($fstab_hash{"$partition"} eq "/proc") {
	next;
      }   
      if ($fstab_hash{"$partition"} eq $pathbase) { 
	verbose("$path found to be on $partition}.");
	$returnval{device} = $this->dev2bios($partition);
	$path =~ /^$pathbase(.*)$/;
	if ($pathbase eq "/") {           ### If the file is directly under "/"
	  $returnval{remnant} = "/" . $1; ###    we need to prepend the remnant with "/". 
	}
	else {
	  $returnval{remnant} = $1;
	}
	last;
      }
    }
    pop @patharray;           ### One level down and reiterate.
  }
  close(FSTAB);
  verbose("Device on which $path resides cannot be derived from $this->{root}/etc/fstab.") if !$returnval{device};
  return \%returnval;
}

=item install_config()

This method read the System Configurator's config file and creates Grub's 
menu file, i.e. "/boot/grub/menu.lst". 

=cut

sub install_config {
    my $this = shift;
    my ($timeout, $output, $exitval);
    my $kernelnum;
    my ($kernelhash, $initrdhash); ### Hash references returned from getdevhash subroutine. 

    if(!$this->{boot_defaultboot}) 
    {
	croak("Error: DEFAULTBOOT must be specified.\n");;
    }

    ### Open the native bootloader config file for write.
    open(OUT,">$this->{config_file}") or croak("Couldn't open $this->{config_file} for writing!");
    print OUT "##################################################\n";
    print OUT "# This file is generated by System Configurator. #\n";
    print OUT "##################################################\n";
    print OUT "\n";
    
    print OUT "#----- Global Options -----#\n";
    
    ### Timeout
    $timeout = $this->{boot_timeout} / 10; # User specifies timeout in deciseconds
    print OUT "# The number of seconds to wait before booting. \n";
    print OUT "timeout " . $timeout . "\n";
    
    ### default kernel image to boot with
    print OUT "# The default kernel image to boot. \n";
    
    unless (defined $this->{defaultbootnum}) {
	croak("Error: default kernel image cannot be identified. \n"); 
    }
    print OUT "default " . $this->{defaultbootnum} . "\n";
    print OUT "\n";

    ### Set up kernel image options
    foreach my $key (sort keys %$this) {
      if ($key =~ /^(kernel\d+)_path/) {
	$kernelnum = $1;
	### Notice that we have to prepend $this->{root}, since getdevhash() first tries
	### to look for the existence of the file specified as the input.
	$kernelhash = $this->getdevhash($this->{root} . $this->{$kernelnum . "_path"});
	if ($this->{$kernelnum . "_initrd"}) {
	  $initrdhash = $this->getdevhash($this->{root} . $this->{$kernelnum. "_initrd"});
	}
	$this->setup_kernel($kernelnum, \*OUT, $kernelhash, $initrdhash);
      }
    }

    close(OUT); ### "$this->{root}/boot/grub/menu.lst"
    
    ### To be part of the exclusion files
    push @{$this->{filesmod}}, "$this->{config_file}";
    
    1;
  } 

=item setup_kernel()

An internal method.
This method sets up a kernel image as specified in the config file.

=cut

sub setup_kernel {
    my ($this, $image, $outfh, $kernelhash, $initrdhash) = @_;
    my ($append, $deviceline, $rootpath, $kernelcom);

    ### Now set up the options.
    print $outfh "#----- Options for \U$image -----#\n";

    ### label
    print $outfh "### Label for $image. \n";
    print $outfh "title " . $this->{$image . "_label"} . "\n";    

    ### Specify device to be mounted as the root
    undef $rootpath;
    if ($this->{$image . "_rootdev"}) { 
	$rootpath = $this->{$image . "_rootdev"}; #rootdev option specified for the image
    }
    elsif ($this->{boot_rootdev}) {
	$rootpath = $this->{boot_rootdev}; #rootdev global option
    }
    else {
	croak("ROOTDEV must be specified either globally or for $image.\n");
    }

    ### kernel line
    $kernelcom = "kernel " . $kernelhash->{device} . $kernelhash->{remnant} .
                          " root=" . $rootpath . " ro";

    ### Check for command line kernel options.
    if ($this->{$image . "_append"}) 
    {
	$kernelcom = $kernelcom . " " . $this->{$image . "_append"} . "\n";
    }
    elsif ($this->{boot_append}) {
	$kernelcom = $kernelcom . " " . $this->{boot_append} . " \n";
    }
    else {
	$kernelcom = $kernelcom . "\n";
    }
    print $outfh "### Kernel command. \n";
    print $outfh $kernelcom;

    ### Initrd image
    if ($this->{$image . "_initrd"})
    {
	print $outfh "### initrd \n";
	print $outfh "initrd " . $initrdhash->{device} . $initrdhash->{remnant} . "\n";
    }        
    print $outfh "\n";
}

=back

=head1 AUTHOR

  Donghwa John Kim <donghwajohnkim@yahoo.com>

=head1 SEE ALSO

L<Boot>, L<perl>

=cut

1;




















