/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TOUCHPAD_H
#define TOUCHPAD_H

#include "qxdevice.h"
#include <QtDBus/QDBusContext>
#include <QtCore/QObject>
#include <QtCore/QSharedPointer>

/**
 * @file
 *
 * Provides a class to control a synaptics touchpad.
 */

/**
 * @brief Provides the core library classes of @b synaptiks.
 *
 * The central class is the Touchpad class, which provides means to
 * configure the touchpad directly.
 *
 * Moreover KeyboardMonitor and MouseDevicesMonitor are provided to monitor
 * the environment.
 *
 * The TouchpadManager class aggregates these classes to implement
 * sophisticated touchpad management.
 */
namespace synaptiks {

    class TouchpadPrivate;

    /**
     * @brief Control and configure a synaptics touchpad.
     *
     * \section usage Usage
     *
     * This class uses input device properties to access the touchpad, which
     * are supported as of Xorg 1.6.
     *
     * Usage is quite simple, just create an object of this class:
     * @code
     *
     * Touchpad *touchpad = Touchpad::findTouchpad(parent);
     * if (touchpad) {
     *     # a touchpad was found, use it
     * }
     * @endcode
     *
     * If a touchpad was found, you can easily access and change its
     * properties, but be sure to catch errors!  The Touchpad class simply
     * passes errors from QXDevice to the caller, so handle QXDeviceError
     * and its subclasses:
     *
     * @code
     * try {
     *     touchpad->setOn(true);
     *     touchpad->setCircularScrolling(true);
     *     kDebug() << "your touchpad supports" << touchpad->fingerDetection()
     *              << "fingers";
     * } catch (const QXDeviceError &error) {
     *     kWarning() << error.toString();
     * }
     * @endcode
     *
     * \section dbus D-Bus
     *
     * All members marked as @c Q_SCRIPTABLE (except properties, which are
     * @em not exported) are exported on D-Bus on the interface @c
     * org.kde.Touchpad.  The adaptor catches all QXDeviceError exceptions
     * and turns them into the D-Bus error @c org.kde.TouchpadError.  The
     * error message is the same as for QXDeviceError::toString().
     */
    class Touchpad: public QObject, public QDBusContext {
        Q_OBJECT
        Q_ENUMS(CircularScrollingTrigger)

        /**
         * @brief The touchpad name
         *
         * @see name() const
         */
        Q_PROPERTY(QString name READ name DESIGNABLE false)

        /**
         * @brief Is the touchpad on?
         *
         * @see setOn(bool)
         * @see isOn() const
         */
        Q_PROPERTY(bool on READ isOn WRITE setOn
                   DESIGNABLE false)

        /**
         * @brief The minimum speed of cursor motion.
         *
         * This property defines the scaling between touchpad coordinates
         * and screen coordinates at slow finger movement.  In other words,
         * if the finger is moved very slow across the touchpad, this value
         * will be used to scale touchpad coordinates to screen coordinates.
         *
         * @see setMinimumSpeed(float)
         * @see minimumSpeed() const
         * @see maximumSpeed
         * @see accelerationFactor
         */
        Q_PROPERTY(float minimumSpeed READ minimumSpeed
                   WRITE setMinimumSpeed DESIGNABLE false)

        /**
         * @brief The maximum speed of cursor motion.
         *
         * This property defines the scaling between touchpad coordinates
         * and screen coordinates at fast finger movement.  In other words,
         * if the finger is moved very fast across the touchpad, this value
         * will be used to scale touchpad coordinates to screen coordinates.
         *
         * @see setMaximumSpeed(float)
         * @see maximumSpeed() const
         * @see minimumSpeed
         * @see accelerationFactor
         */
        Q_PROPERTY(float maximumSpeed READ maximumSpeed
                   WRITE setMaximumSpeed DESIGNABLE false)

        /**
         * @brief The scaling between #minimumSpeed and #maximumSpeed.
         *
         * This property defines, how the pointer is accelerated between
         * #minimumSpeed and #maximumSpeed as the finger speed increases on
         * the touchpad.
         *
         * If #minimumSpeed and #maximumSpeed have the same value,
         * acceleration is disabled, and this property has no effect.
         *
         * @see setAccelerationFactor(float)
         * @see accelerationFactor() const
         * @see minimumSpeed
         * @see maximumSpeed
         */
        Q_PROPERTY(float accelerationFactor READ accelerationFactor
                   WRITE setAccelerationFactor DESIGNABLE false)

        /**
         * @brief Use edge motion always.
         *
         * Cursor movement continues, if the finger hits the touchpad edge.
         * This feature is called "edge motion".  By default, it is only
         * active when dragging.  If this property is set to @c true, edge
         * motion is always active, even for normal movement.
         *
         * @see setEdgeMotionAlways(bool)
         * @see edgeMotionAlways() const
         */
        Q_PROPERTY(bool edgeMotionAlways READ edgeMotionAlways
                   WRITE setEdgeMotionAlways DESIGNABLE false)

        /**
         * @brief Enable or disable the tap-and-drag gesture.
         *
         * The tap-and-drag gesture provides an alternative way of dragging.
         * A tap-and-drag gesture is performed by tapping the touchpad, and
         * then immediately touching the touchpad and moving the finger on
         * it.  With this gesture, items can be dragged by tapping only,
         * without using the mouse buttons of the touchpad.
         *
         * @see setTapAndDragGesture(bool)
         * @see tapAndDragGesture() const
         * @see lockedDrags
         * @see lockedDragsTimeout
         */
        Q_PROPERTY(bool tapAndDragGesture READ tapAndDragGesture
                   WRITE setTapAndDragGesture DESIGNABLE false)

        /**
         * @brief Continue a tap-and-drag gesture after releasing the finger.
         *
         * If this property is @c false, a tap-and-drag gesture ends, when
         * the finger is released from the touchpad (just like drags end,
         * when the mouse button is released).  By setting this property to
         * @c true, the gesture continues even after releasing the finger
         * until the touchpad is tapped again or #lockedDragsTimeout
         * expires.
         *
         * @see setLockedDrags(bool)
         * @see lockedDrags() const
         * @see tapAndDragGesture
         * @see lockedDragsTimeout
         */
        Q_PROPERTY(bool lockedDrags READ lockedDrags WRITE setLockedDrags
                   DESIGNABLE false)

        /**
         * @brief The timeout of locked drag mode in milliseconds.
         *
         * If this timeout expires in locked drag mode, the tap-and-drag
         * gesture will automatically end, even if the touchpad was not yet
         * tapped.
         *
         * If #lockedDrags is @c false, this property has no effect.
         *
         * @see setLockedDragsTimeout(int)
         * @see lockedDragsTimeout() const
         * @see tapAndDragGesture
         * @see lockedDrags
         */
        Q_PROPERTY(int lockedDragsTimeout READ lockedDragsTimeout
                   WRITE setLockedDragsTimeout DESIGNABLE false)

        /**
         * @brief Whether circular scrolling is enabled or not.
         *
         * If circular scrolling is enabled, the user can scroll by moving
         * the finger in circles over the touchpad.
         *
         * @see setCircularScrolling(bool)
         * @see circularScrolling() const
         * @see circularScrollingTrigger
         * @see circularScrollingDistance
         */
        Q_PROPERTY(bool circularScrolling READ circularScrolling
                   WRITE setCircularScrolling DESIGNABLE false)

        /**
         * @brief The touchpad area, which triggers circular scrolling.
         *
         *
         * To avoid interference with normal movements, circular movements
         * of across the touchpad don't result in any scrolling, even is
         * #circularScrolling is @c true.  Instead, the user has to
         * move his finger into a certain touchpad area to trigger circular
         * scrolling.
         *
         * @see setCircularScrollingTrigger(CircularScrollingTrigger)
         * @see circularScrollingTrigger() const
         * @see circularScrolling
         * @see circularScrollingDistance
         */
        Q_PROPERTY(CircularScrollingTrigger circularScrollingTrigger
                   READ circularScrollingTrigger
                   WRITE setCircularScrollingTrigger DESIGNABLE false)

        /**
         * @brief Move angle in radians of the finger to generate a
         * scrolling event in circular scrolling mode.
         *
         * This property defines, by which angle the finger needs to move in
         * circular scrolling mode for a scroll event to be generated.  The
         * smaller this value is, the more scroll events are generated for a
         * specific move angle of the finger and the faster circular
         * scrolling gets.
         *
         * @see setCircularScrollingDistance(float)
         * @see circularScrollingDistance() const
         * @see circularScrolling
         * @see circularScrollingTrigger
         */
        Q_PROPERTY(float circularScrollingDistance
                   READ circularScrollingDistance
                   WRITE setCircularScrollingDistance
                   DESIGNABLE false)

        /** @brief Whether horizontal scrolling with two fingres is enabled
         * or not.
         *
         * If this property is @c true, one can scroll horizontally by
         * dragging with two fingers anywhere on the touchpad.
         *
         * @see setHorizontalTwoFingerScrolling(bool)
         * @see horizontalTwoFingerScrolling() const
         * @see verticalTwoFingerScrolling
         */
        Q_PROPERTY(bool horizontalTwoFingerScrolling
                   READ horizontalTwoFingerScrolling
                   WRITE setHorizontalTwoFingerScrolling DESIGNABLE false);

        /** @brief Whether vertical scrolling with two fingres is enabled
         * or not.
         *
         * If this property is @c true, one can scroll vertically by
         * dragging with two fingers anywhere on the touchpad.
         *
         * @see setVerticalTwoFingerScrolling(bool)
         * @see verticalTwoFingerScrolling() const
         * @see horizontalTwoFingerScrolling
         */
        Q_PROPERTY(bool verticalTwoFingerScrolling
                   READ verticalTwoFingerScrolling
                   WRITE setVerticalTwoFingerScrolling DESIGNABLE false);

        /**
         * @brief Whether horizontal scrolling at the bottom edge of the
         * touchpad is enabled or not.
         *
         * @see setHorizontalEdgeScrolling(bool)
         * @see horizontalEdgeScrolling() const
         * @see verticalEdgeScrolling
         * @see cornerCoasting
         * @see coastingSpeed
         */
        Q_PROPERTY(bool horizontalEdgeScrolling READ horizontalEdgeScrolling
                   WRITE setHorizontalEdgeScrolling DESIGNABLE false)

        /**
         * @brief Whether vertical scrolling at the right edge of the
         * touchpad is enabled or not.
         *
         * @see setVerticalEdgeScrolling(bool)
         * @see verticalEdgeScrolling() const
         * @see horizontalEdgeScrolling
         * @see cornerCoasting
         * @see coastingSpeed
         */
        Q_PROPERTY(bool verticalEdgeScrolling READ verticalEdgeScrolling
                   WRITE setVerticalEdgeScrolling DESIGNABLE false)

        /**
         * @brief Whether corner coasting is enabled or not.
         *
         * If corner coasting is enabled, edge scrolling will continue as
         * long as the finger remains in an edge corner.
         *
         * @note
         *
         * Setting this to @c true doesn't mean, that the user can really
         * use corner coasting.  The driver only really enables corner
         * coasting, if this property is @c true @em and #coastingSpeed is
         * not @c 0.
         *
         * @see setCornerCoasting(bool)
         * @see cornerCoasting() const
         * @see verticalEdgeScrolling
         * @see horizontalEdgeScrolling
         * @see coastingSpeed
         */
        Q_PROPERTY(bool cornerCoasting READ cornerCoasting
                   WRITE setCornerCoasting DESIGNABLE false)

        /**
         * @brief The threshold speed (in scrolling events per second) for
         * conventional coasting.
         *
         * Setting this to a non-zero positive value enables coasting.
         * Coasting comes in two different flavors:
         *
         * <em>Conventional coasting</em> is used, if #cornerCoasting is @c
         * false.  In this case, edge scrolling continues with the current
         * scrolling speed, if the finger leaves the touchpad, while the
         * current scrolling speed is above the value of this property.  If
         * the finger touches the touchpad again, scrolling is stopped.
         *
         * However, if #cornerCoasting is @c true, <em>corner coasting</em>
         * is used.  See #cornerCoasting for a description of corner
         * coasting.
         *
         * If this property is @c 0, coasting is completely disabled,
         * regardless of the value of #cornerCoasting.
         *
         * @see setCoastingSpeed(float)
         * @see coastingSpeed() const
         * @see verticalEdgeScrolling
         * @see horizontalEdgeScrolling
         * @see cornerCoasting
         */
        Q_PROPERTY(float coastingSpeed READ coastingSpeed
                   WRITE setCoastingSpeed DESIGNABLE false)

        /**
         * @brief The speed of horizontal scrolling.
         *
         * This property holds the move distance of the finger to generate a
         * single horizontal scroll event.
         *
         * @see setHorizontalScrollingDistance(int)
         * @see horizontalScrollingDistance() const
         * @see horizontalEdgeScrolling
         * @see horizontalTwoFingerScrolling
         */
        Q_PROPERTY(int horizontalScrollingDistance
                   READ horizontalScrollingDistance
                   WRITE setHorizontalScrollingDistance DESIGNABLE false)

        /**
         * @brief The speed of vertical scrolling.
         *
         * This property holds the move distance of the finger to generate a
         * single vertical scroll event.
         *
         * @see setVerticalScrollingDistance(int)
         * @see verticalScrollingDistance() const
         * @see verticalEdgeScrolling
         * @see verticalTwoFingerScrolling
         */
        Q_PROPERTY(int verticalScrollingDistance
                   READ verticalScrollingDistance
                   WRITE setVerticalScrollingDistance DESIGNABLE false)

        /**
         * @brief The mouse buttons triggered by tapping the touchpad
         * corner.
         *
         * This property has four elements, interpreted as follows:
         *
         * @li index @c 0:  Top right corner
         * @li index @c 1:  Bottom right corner
         * @li index @c 2:  Top left corner
         * @li index @c 3:  Bottom left corner
         *
         * Each element is a numeric value of a mouse button.  These are
         * excatly the same the xserver sends to client applications, and
         * that you can monitor with tools like @c xev.  The following
         * values are the most interesting:
         *
         * @li value @c 1: Left mouse button
         * @li value @c 2: Middle mouse button
         * @li value @c 3: Right mouse button
         *
         * A value of @c 0 disables any special button set for the
         * corresponding corner.  Tapping this corner will be interpreted
         * according to the #fingerButtons setting.
         *
         * For instance, the list <tt>(3, 0, 2, 0)</tt> will have the
         * following effects:
         *
         * @li Tapping the top right corner is like pressing the right mouse
         *     button.
         * @li Tapping the top left corner is like pressing the middle mouse
         *     button
         * @li When tapping any of the bottom corners, nothing special will
         *     happen.
         *
         * @see setCornerButtons(const QByteArray&)
         * @see cornerButtons() const
         * @see fingerButtons
         */
        Q_PROPERTY(QByteArray cornerButtons
                   READ cornerButtons WRITE setCornerButtons
                   DESIGNABLE false)

        /**
         * @brief The mouse buttons triggered by tapping the touchpad with a
         * given amount of fingers concurrently.
         *
         * The list has three elements, interpreted as follows:
         *
         * @li index @c 0:  Tapping the touchpad with one finger
         * @li index @c 1:  Tapping the touchpad with two fingers
         * @li index @c 2:  Tapping the touchpad with three fingers
         *
         * Each element is a numeric value of a mouse button.  These are
         * excatly the same the xserver sends to client applications, and
         * that you can monitor with tools like @c xev.  The following
         * values are the most interesting:
         *
         * @li value @c 1: Left mouse button
         * @li value @c 2: Middle mouse button
         * @li value @c 3: Right mouse button
         *
         * A value of @c 0 disables taps with the corresponding amount of
         * fingers.
         *
         * For instance, the list <tt>(1, 3, 0)</tt> will have the following
         * effects:
         *
         * @li Tapping with one finger is like pressing the left mouse
         *     button.
         * @li Tapping with two fingers is like pressing the right mouse
         *     button.
         * @li When tapping the touchpad with three fingers, nothing will
         *     happen.
         *
         * @note
         *
         * Multiple finger taps require special hardware support, which is
         * not provided by all touchpads.  Use #capabilities or
         * #fingerDetection to check, how well the touchpad support multiple
         * finger taps.
         *
         * @see setFingerButtons(const QByteArray&)
         * @see fingerButtons() const
         * @see capabilities
         * @see fingerDetection
         * @see cornerButtons
         */
        Q_PROPERTY(QByteArray fingerButtons
                   READ fingerButtons WRITE setFingerButtons
                   DESIGNABLE false)

        /**
         * @brief Whether or not fast tapping is enabled.
         *
         * Fast tapping makes the driver react faster on single taps, but
         * slower to double clicks caused by tapping.
         *
         * @see setFastTaps(bool)
         * @see fastTaps() const
         */
        Q_PROPERTY(bool fastTaps
                   READ fastTaps WRITE setFastTaps
                   DESIGNABLE false)

        /**
         * @brief Whether or not the touchpad is considered as circular
         * touchpad.
         *
         * If this property is @c true, the touchpad drivers considered the
         * touchpad as circular pad and adjusts edges and coordinates
         * accordingly.
         *
         * The driver does @e not detect circular touchpads automatically,
         * therefore this property must be set manually, if a circular
         * touchpad is present.
         *
         * @see setCircularTouchpad(bool)
         * @see circularTouchpad() const
         */
        Q_PROPERTY(bool circularTouchpad
                   READ circularTouchpad WRITE setCircularTouchpad
                   DESIGNABLE false)

        /**
         * @brief The capabilities of the touchpad.
         *
         * This property provides a combination of #Capability flags, which
         * indicate the capabilities of this touchpad.
         *
         * @warning
         *
         * synaptics supports capabilities reporting as of 1.1.  With
         * synaptics 1.0 access to this property raises
         * QXNoSuchPropertyError.
         *
         * @see capabilities() const
         * @see hasLeftButton
         * @see hasMiddleButton
         * @see hasRightButton
         * @see fingerDetection
         */
        Q_PROPERTY(Capabilities capabilities READ capabilities)

        /**
         * @brief Whether this touchpad provides a left mouse button.
         *
         * @warning
         *
         * synaptics supports capabilities reporting as of 1.1.  With
         * synaptics 1.0 access to this property raises
         * QXNoSuchPropertyError.
         *
         * @see hasLeftButton() const
         * @see hasMiddleButton
         * @see hasRightButton
         * @see capabilities
         */
        Q_PROPERTY(bool hasLeftButton READ hasLeftButton DESIGNABLE false)

        /**
         * @brief Whether this touchpad provides a middle mouse button.
         *
         * @note
         *
         * synaptics supports capabilities reporting as of 1.1.  With
         * synaptics 1.0 access to this property causes an error.
         *
         * @see hasMiddleButton() const
         * @see hasLeftButton
         * @see hasRightButton
         * @see capabilities
         */
        Q_PROPERTY(bool hasMiddleButton READ hasLeftButton DESIGNABLE false)

        /**
         * @brief Whether this touchpad provides a right mouse button.
         *
         * @warning
         *
         * synaptics supports capabilities reporting as of 1.1.  With
         * synaptics 1.0 access to this property raises
         * QXNoSuchPropertyError.
         *
         * @see hasRightButton() const
         * @see hasLeftButton
         * @see hasMiddleButton
         * @see capabilities
         */
        Q_PROPERTY(bool hasRightButton READ hasLeftButton DESIGNABLE false)

        /**
         * @brief How many fingers are detected by the touchpad.
         *
         * Some touchpads provide support for detecting taps with multiple
         * fingers.  If such a touchpad is used, this property provides the
         * amount of fingers, which are detected.  Otherwise this property
         * just returns @c 1.
         *
         * @warning
         *
         * synaptics supports capabilities reporting as of 1.1.  With
         * synaptics 1.0 access to this property raises
         * QXNoSuchPropertyError.
         *
         * @see fingerDetection() const
         * @see capabilities
         */
        Q_PROPERTY(int fingerDetection READ fingerDetection
                   DESIGNABLE false)

    public:
        /**
         * @brief The touchpad areas, which trigger circular scrolling.
         */
        enum CircularScrollingTrigger {
            // enum values correspond to synaptics property values
            /** Any edge */
            AnyEdgeTrigger = 0,
            /** Top edge */
            TopEdgeTrigger = 1,
            /** Top right corner */
            TopRightCornerTrigger = 2,
            /** Right edge */
            RightEdgeTrigger = 3,
            /** Bottom right corner */
            BottomRightCornerTrigger = 4,
            /** Bottom edge */
            BottomEdgeTrigger = 5,
            /** Bottom left corner */
            BottomLeftCornerTrigger = 6,
            /** Left edge */
            LeftEdgeTrigger = 7,
            /** Top left corner */
            TopLeftCornerTrigger = 8
        };

        /**
         * @brief The touchpad capabilities
         */
        enum Capability {
            /** The touchpad has a left button */
            LeftButtonCapability = 0x1,
            /** The touchpad has a middle button */
            MiddleButtonCapability = 0x2,
            /** The touchpad has a right button */
            RightButtonCapability = 0x4,
            /** The touchpad can detect two fingers */
            TwoFingersCapability = 0x8,
            /** The touchpad can detect three fingers */
            ThreeFingersCapability = 0x16
        };

        Q_DECLARE_FLAGS(Capabilities, synaptiks::Touchpad::Capability)

        /**
         * @brief Find the touchpad on this system.
         *
         * If the system doesn't provide a configurable touchpad, a null
         * pointer is returned.  This happens, if there is no physical
         * touchpad on this system, if the touchpad is not controlled by the
         * synaptics driver or if the driver is too old.
         *
         * @param parent the parent object
         * @return A synaptiks::Touchpad instance representing the installed
         * touchpad or @c NULL, if there is no touchpad.
         */
        static Touchpad *findTouchpad(QObject *parent=0);

        /**
         * Destroy this instance.
         */
        ~Touchpad();

        /**
         * @brief Check, whether device properties are supported.
         *
         * This class will @b not work, if @c false is returned!
         *
         * @return @c true, if properties are supported, @c false otherwise
         */
        static bool isSupported();

    public Q_SLOTS:
        /**
         * @brief Return the human-readable name of this touchpad.
         *
         * @return the product name of the touchpad
         * @see name
         */
        Q_SCRIPTABLE QString name() const;

        /**
         * @brief Is the touchpad on?
         *
         * @return @c true, if the touchpad is on, @c false otherwise
         * @see on
         */
        Q_SCRIPTABLE bool isOn() const;

        /**
         * @brief Switch the touchpad according to the given parameter.
         *
         * @param on if @c true, switch the touchpad on, otherwise switch it
         *        off.
         * @return @c true, if successful, @c false otherwise
         * @see on
         */
        Q_SCRIPTABLE void setOn(bool on);

        /**
         * @brief Return the minimum motion speed.
         *
         * @return the minimum motion speed
         * @see minimumSpeed
         */
        Q_SCRIPTABLE float minimumSpeed() const;

        /**
         * @brief Set the minimum motion speed
         *
         * @param speed the speed value
         * @see minimumSpeed
         */
        Q_SCRIPTABLE void setMinimumSpeed(float speed);

        /**
         * @brief Return the maximum motion speed.
         *
         * @return the maximum motion speed
         * @see maximumSpeed
         */
        Q_SCRIPTABLE float maximumSpeed() const;

        /**
         * @brief Set the maximum motion speed
         *
         * @param speed the speed value
         * @see maximumSpeed
         */
        Q_SCRIPTABLE void setMaximumSpeed(float speed);

        /**
         * @brief Get the acceleration factor.
         *
         * @return the acceleration factor
         * @see accelerationFactor
         */
        Q_SCRIPTABLE float accelerationFactor() const;

        /**
         * @brief Set the acceleration factor.
         *
         * @param accel the acceleration factor
         * @see accelerationFactor
         */
        Q_SCRIPTABLE void setAccelerationFactor(float accel);

        /**
         * @brief Is edge motion always active, or only when dragging?
         *
         * @return @c true, if edge motion is always active, or @c false, if
         *         it is only active when dragging.
         * @see edgeMotionAlways
         */
        Q_SCRIPTABLE bool edgeMotionAlways() const;

        /**
         * @brief Enable edge motion for all movements.
         *
         * @param enabled if @c true, edge motion will always be active, if
         *        @c false, edge motion is only used when dragging
         * @see edgeMotionAlways
         */
        Q_SCRIPTABLE void setEdgeMotionAlways(bool enabled);

        /**
         * @brief Is the tap-and-drag gesture enabled?
         *
         * @return @c true, if the tap-and-drag gesture is enabled, @c false
         *         otherwise
         * @see tapAndDragGesture
         */
        Q_SCRIPTABLE bool tapAndDragGesture() const;

        /**
         * @brief Enable or disable the tap-and-drag gesture.
         *
         * @param enabled if @c true, enable the gesture, otherwise disable
         *        it
         * @see tapAndDragGesture
         */
        Q_SCRIPTABLE void setTapAndDragGesture(bool enabled);

        /**
         * @brief Are locked drags enabled?
         *
         * @return @c true, if locked drags are enabled, @c false otherwise
         * @see lockedDrags
         */
        Q_SCRIPTABLE bool lockedDrags() const;

        /**
         * @brief Enable or disable locked drags.
         *
         * @param enabled if @c true, enable locked drags, otherwise disable
         *        them
         * @see lockedDrags
         */
        Q_SCRIPTABLE void setLockedDrags(bool enabled);

        /**
         * @brief Get the timeout for a lock drag to expire.
         *
         * @return the timeout in milliseconds
         * @see lockedDragsTimeout
         */
        Q_SCRIPTABLE int lockedDragsTimeout() const;

        /**
         * @brief Set the timeout for a locked drag to expire.
         *
         * @param timeout the timeout in milliseconds
         * @see lockedDragsTimeout
         */
        Q_SCRIPTABLE void setLockedDragsTimeout(int timeout);

        /**
         * @brief Is circular scrolling enabled?
         *
         * @return @c true, if circular scrolling is enabled, @c false
         *         otherwise
         * @see circularScrolling
         */
        Q_SCRIPTABLE bool circularScrolling() const;

        /**
         * @brief Enabled or disable circular scrolling.
         *
         * @param enabled if @c true, enable circular scrolling, otherwise
         *        disable it.
         * @return @c true, if successful, @c false otherwise
         * @see circularScrolling
         */
        Q_SCRIPTABLE void setCircularScrolling(bool enabled);

        /**
         * @brief Return the part of the touchpad, that triggers circular
         * scrolling.
         *
         * @return the trigger
         * @see circularScrollingTrigger
         */
        Q_SCRIPTABLE CircularScrollingTrigger circularScrollingTrigger() const;

        /**
         * @brief Set the trigger for circular scrolling.
         *
         * @param trigger the trigger
         * @return @c true, if successful, @c false otherwise
         * @see circularScrollingTrigger
         */
        Q_SCRIPTABLE void setCircularScrollingTrigger(
            CircularScrollingTrigger trigger);

        /**
         * @brief Return the move angle of the finger to generate a scroll
         * event in circular scrolling mode.
         *
         * @return the move angle in radians
         * @see circularScrollingDistance
         */
        Q_SCRIPTABLE float circularScrollingDistance() const;

        /**
         * @brief Set the move angle of the finger to generate a scroll
         * event in circular scrolling mode.
         *
         * @param angle the move angle in radians
         * @see circularScrollingDistance
         */
        Q_SCRIPTABLE void setCircularScrollingDistance(float angle);

        /**
         * @brief Is horizontal two-finger scrolling enabled?
         *
         * @return @c true, if horizontal two-finger scrolling is enabled,
         *         @c false otherwise.
         * @see horizontalTwoFingerScrolling
         */
        Q_SCRIPTABLE bool horizontalTwoFingerScrolling() const;

        /**
         * @brief Enable or disable horizontal two-finger scrolling.
         *
         * @param enabled if @c true, enable horizontal two-finger
         *        scrolling, otherwise disable it
         * @see horizontalTwoFingerScrolling
         */
        Q_SCRIPTABLE void setHorizontalTwoFingerScrolling(bool enabled);

        /**
         * @brief Is vertical two-finger scrolling enabled?
         *
         * @return @c true, if vertical two-finger scrolling is enabled,
         *         @c false otherwise.
         * @see verticalTwoFingerScrolling
         */
        Q_SCRIPTABLE bool verticalTwoFingerScrolling() const;

        /**
         * @brief Enable or disable vertical two-finger scrolling.
         *
         * @param enabled if @c true, enable vertical two-finger
         *        scrolling, otherwise disable it
         * @see verticalTwoFingerScrolling
         */
        Q_SCRIPTABLE void setVerticalTwoFingerScrolling(bool enabled);

        /**
         * @brief Is horizontal scrolling at the bottom edge enabled?
         *
         * @return @c true, if horizontal edge scrolling is enabled, @c
         *         false otherwise.
         * @see horizontalEdgeScrolling
         */
        Q_SCRIPTABLE bool horizontalEdgeScrolling() const;

        /**
         * @brief Enable or disable horizontal scrolling at the bottom edge.
         *
         * @param enabled if @c true, enable horizontal edge scrolling,
         *        otherweise disable it.
         * @return @c true, if successful, @c false otherwise.
         * @see horizontalEdgeScrolling
         */
        Q_SCRIPTABLE void setHorizontalEdgeScrolling(bool enabled);

        /**
         * @brief Is vertical scrolling at the right edge enabled?
         *
         * @return @c true, if vertical edge scrolling is enabled, @c
         *         false otherwise.
         * @see verticalEdgeScrolling
         */
        Q_SCRIPTABLE bool verticalEdgeScrolling() const;

        /**
         * @brief Enable or disable vertical scrolling at the right edge.
         *
         * @param enabled if @c true, enable vertical edge scrolling,
         *        otherweise disable it.
         * @return @c true, if successful, @c false otherwise.
         * @see verticalEdgeScrolling
         */
        Q_SCRIPTABLE void setVerticalEdgeScrolling(bool enabled);

        /**
         * @brief Is corner coasting enabled?
         *
         * @return @c true, if corner coasting is enabled, @c false
         *         otherwise
         * @see cornerCoasting
         */
        Q_SCRIPTABLE bool cornerCoasting() const;

        /**
         * @brief Enable or disable corner coasting.
         *
         * @param enabled if @c true, enable corner coasting, otherwise
         *        disable it
         * @return @c true, if successful, @c false otherwise.
         * @see cornerCoasting
         */
        Q_SCRIPTABLE void setCornerCoasting(bool enabled);

        /**
         * @brief Get the coasting speed
         *
         * If this returns 0, coasting is completely disabled, including
         * corner coasting.
         *
         * @return the coasting speed in scroll events per second, or -1 in
         *         case of an error.
         * @see coastingSpeed
         */
        Q_SCRIPTABLE float coastingSpeed() const;

        /**
         * @brief Set the coasting speed.
         *
         * @param speed the coasting speed in scroll events per second
         * @return @c true, if successful, @c false otherwise.
         * @see coastingSpeed
         */
        Q_SCRIPTABLE void setCoastingSpeed(float speed);

        /**
         * @brief Return the move distance of the finger to generate a
         * horizontal scroll event.
         *
         * @return the move distance for a horizontal scroll event
         * @see horizontalScrollingDistance
         */
        Q_SCRIPTABLE int horizontalScrollingDistance() const;

        /**
         * @brief Set the move distance of the finger to generate a
         * horizontal scroll event.
         *
         * @param distance the move distance for a horizontal scroll event
         * @see horizontalScrollingDistance
         */
        Q_SCRIPTABLE void setHorizontalScrollingDistance(int distance);

        /**
         * @brief Return the move distance of the finger to generate a
         * vertical scroll event.
         *
         * @return the move distance for a vertical scroll event
         * @see verticalScrollingDistance
         */
        Q_SCRIPTABLE int verticalScrollingDistance() const;

        /**
         * @brief Set the move distance of the finger to generate a vertical
         * scroll event.
         *
         * @param distance the move distance for a vertical scroll event
         * @see verticalScrollingDistance
         */
        Q_SCRIPTABLE void setVerticalScrollingDistance(int distance);

        /**
         * @brief Query the mouse buttons triggered by tapping the touchpad
         * corners.
         *
         * In case of an error the returned list is empty.  Otherwise it is
         * guaranteed to contain four items.
         *
         * @return a list of buttons, empty in case of error.
         * @see cornerButtons
         */
        Q_SCRIPTABLE QByteArray cornerButtons() const;

        /**
         * @brief Set the mouse buttons triggered by tapping the touchpad
         * corners.
         *
         * The list of @p buttons must have at least four elements, less
         * elements cause an error, more elements are silently ignored.
         *
         * @param buttons the list of buttons
         * @return @c true, if successful, @c false otherwise.
         * @see cornerButtons
         */
        Q_SCRIPTABLE void setCornerButtons(const QByteArray &buttons);

        /**
         * @brief Query the mouse buttons triggered by tapping the touchpad.
         *
         * In case of an error the returned list is empty.  Otherwise it is
         * guaranteed to contain three items.
         *
         * @return a list of buttons, empty in case of error
         * @see fingerButtons
         */
        Q_SCRIPTABLE QByteArray fingerButtons() const;

        /**
         * @brief Set the mouse buttons triggered by tapping the touchpad.
         *
         * The list of @p buttons must have at least three elements, less
         * elements cause an error, more elements are silently ignored.
         *
         * @param buttons the list of buttons
         * @return @c true, if successful, @c false otherwise.
         * @see fingerButtons
         */
        Q_SCRIPTABLE void setFingerButtons(const QByteArray &buttons);

        /**
         * @brief Whether fast taps are enabled.
         *
         * @return @c true, if fast taps are enabled, @c false otherwise
         * @see fastTaps
         */
        Q_SCRIPTABLE bool fastTaps() const;

        /**
         * @brief Enable or disable fast taps.
         *
         * @param enabled If @c true, enable fast taps, otherwise disable
         *        them.
         * @see fastTaps
         */
        Q_SCRIPTABLE void setFastTaps(bool enabled);

        /**
         * @brief Whether the touchpad is considered as circular touchpad.
         *
         * @return @c true, if the touchpad is circular, @c false otherwise
         * @see circularTouchpad
         */
        Q_SCRIPTABLE bool circularTouchpad() const;

        /**
         * @brief Configure, if the touchpad is circular or not.
         *
         * @param circular if @c true, the touchpad is considered as
         *        circular touchpad by the driver, if @c false, the touchpad
         *        is considered as rectangular touchpad.
         * @see circularTouchpad
         */
        Q_SCRIPTABLE void setCircularTouchpad(bool circular);

        /**
         * @brief Query the touchpad capabilities.
         *
         * The returns a combination of flags, which indicate the touchpad
         * capabilities.
         *
         * @return the capabilities
         * @exception QXNoSuchPropertyError when used with synaptics 1.0,
         *            which does not report capabilities
         */
        Capabilities capabilities() const;

        /**
         * @brief Does the touchpad have a left button?
         *
         * @return @c true, if the touchpad has a left button, @c false
         *         otherwise
         * @exception QXNoSuchPropertyError when used with synaptics 1.0,
         *            which does not report capabilities
         */
        Q_SCRIPTABLE bool hasLeftButton() const;

        /**
         * @brief Does the touchpad have a middle button?
         *
         * @return @c true, if the touchpad has a middle button, @c false
         *         otherwise
         * @exception QXNoSuchPropertyError when used with synaptics 1.0,
         *            which does not report capabilities
         */
        Q_SCRIPTABLE bool hasMiddleButton() const;

        /**
         * @brief Does the touchpad have a right button?
         *
         * @return @c true, if the touchpad has a right button, @c false
         *         otherwise
         * @exception QXNoSuchPropertyError when used with synaptics 1.0,
         *            which does not report capabilities
         */
        Q_SCRIPTABLE bool hasRightButton() const;

        /**
         * @brief How many fingers are detected by this touchpad?
         *
         * @return the amount of fingers, which can be detected
         * @exception QXNoSuchPropertyError when used with synaptics 1.0,
         *            which does not report capabilities
         */
        Q_SCRIPTABLE int fingerDetection() const;

    private:
        Q_DISABLE_COPY(Touchpad)
        Q_DECLARE_PRIVATE(Touchpad)

        /**
         * @brief Create a new touchpad instance.
         *
         * @param device the QXDevice instance representing the
         *        touchpad
         * @param parent same as for QObject()
         */
        Touchpad(QSharedPointer<QXDevice> device, QObject *parent=0);

        TouchpadPrivate *const d_ptr;
    };
}

#endif /* TOUCHPAD_H */

