/*==================================================================
 * sample.c - Sample data routines
 *
 * libInstPatch
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <limits.h>		/* for UINT_MAX */
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "instpatch.h"
#include "ippriv.h"
#include "i18n.h"


/* default max sample buffer waste (unref'd samples) in Kbytes */
#define SAMPLE_BUFFER_DEFAULT_MAX_WASTE  16 * 1024


/* define error messages that are used multiple times */

#define INSTP_ERRMSG_WRONG_METHOD_0 \
    "Wrong sample storage method for this function"
#define INSTP_ERRMSG_METH_INSTP_BAD_SFPTR_0 \
    "Sound font pointer not set in IPSAMPLE_METHOD_SFONT IPSampleStore"
#define INSTP_ERRMSG_METH_ALREADY_ALLOC_0 \
    "Allocate function called for already allocated IPSampleStore"
#define INSTP_ERRMSG_METH_NOT_ALLOC_0 \
    "Function expects allocated IPSampleStore"
#define INSTP_ERRMSG_METH_BUFFER_SEEK_2 \
    "Failed to seek to offset %d in sample buffer: %s"
#define INSTP_ERRMSG_METH_BUFFER_TEMP_FILE_1 \
    "Failed to create temporary file for sample buffer: %s"
#define INSTP_ERRMSG_STORE_NOT_IN_SAMPLEDATA_0 \
    "IPSampleStore not owned by IPSampleData object"

/* static function prototypes */
static void method_SFONT_done_func (IPSampleData *sampledata,
				    IPSampleStore *store);
static int method_SFONT_read_func (IPSampleData *sampledata,
				   IPSampleStore *store,
				   int offset, int size, void *buf);
static int method_BUFFER_init_func (IPSampleData *sampledata,
				    IPSampleStore *store);
static int method_BUFFER_alloc_func (IPSampleData *sampledata,
				     IPSampleStore *store);
static void method_BUFFER_free_func (IPSampleData *sampledata,
				     IPSampleStore *store);
static int method_BUFFER_read_func (IPSampleData *sampledata,
				    IPSampleStore *store,
				    int offset, int size, void *buf);
static int method_BUFFER_write_func (IPSampleData *sampledata,
				     IPSampleStore *store,
				     int offset, int size, const void *buf);
static int sample_buffer_init (void);
static void sample_buffer_destroy (void);
static int method_RAM_alloc_func (IPSampleData *sampledata,
				  IPSampleStore *store);
static void method_RAM_free_func (IPSampleData *sampledata,
				  IPSampleStore *store);
static int method_RAM_read_func (IPSampleData *sampledata,
				 IPSampleStore *store,
				 int offset, int size, void *buf);
static int method_RAM_write_func (IPSampleData *sampledata,
				  IPSampleStore *store,
				  int offset, int size, const void *buf);

/* IPSAMPLE_METHOD_SFONT vars */
typedef struct _MethodSFontVars
{
  IPFileInfo *file;		/* info for file containing the sample data */
  guint start;		/* start offset, in samples, of data in sample chunk */
} MethodSFontVars;

/* IPSAMPLE_METHOD_BUFFER vars */
typedef struct _MethodBufferVars
{
  guint start;			/* location, in bytes, within disk buffer */
} MethodBufferVars;

/* IPSAMPLE_METHOD_RAM vars */
typedef struct _MethodRAMVars
{
  void *dataptr;		/* pointer to data in RAM memory */
} MethodRAMVars;

/* IPSAMPLE_METHOD_ROM vars */
typedef struct _MethodROMVars
{
  guint start;			/* start address in ROM (in samples) */
} MethodROMVars;

/*
  Pre-defined sample storage methods.
  { IPSampleMethodType, init_func, done_func, alloc_func, free_func,
  read_func, write_func }
*/
static IPSampleMethod sample_methods[IPSAMPLE_METHOD_COUNT] = {
  { IPSAMPLE_METHOD_NONE, 0, 0, 0,
    NULL, NULL, NULL, NULL, NULL, NULL },
  { IPSAMPLE_METHOD_SFONT, sizeof (MethodSFontVars), 3, 0,
    NULL,			/* init */
    method_SFONT_done_func,
    NULL, NULL,			/* allocate and free */
    method_SFONT_read_func,
    NULL },			/* no write function */
  { IPSAMPLE_METHOD_BUFFER, sizeof (MethodBufferVars), 4, 0,
    method_BUFFER_init_func,
    NULL,			/* no done function */
    method_BUFFER_alloc_func,
    method_BUFFER_free_func,
    method_BUFFER_read_func,
    method_BUFFER_write_func },
  { IPSAMPLE_METHOD_RAM, sizeof (MethodRAMVars), 9, 0,
    NULL, NULL,			/* no init or done function */
    method_RAM_alloc_func,
    method_RAM_free_func,
    method_RAM_read_func,
    method_RAM_write_func },
  { IPSAMPLE_METHOD_ROM, sizeof (MethodROMVars), 2, 0,
    NULL, NULL, NULL, NULL, NULL, NULL }
};

static int sample_next_user_method_id = IPSAMPLE_METHOD_USER;
static IPSampleMethod **sample_user_methods = NULL;

/** List of all managed IPSampleData structures */
IPSampleData *instp_sample_data_list = NULL;

/* Sample BUFFER Method variables */
static FILE *sample_buffer_fd = NULL;
static guint sample_buffer_length; /* sample buffer length in bytes */
static guint sample_buffer_waste; /* current amount of waste (unused data) */

/* maximum sample waste in bytes before cleanup performed */
static guint sample_buffer_max_waste = SAMPLE_BUFFER_DEFAULT_MAX_WASTE;


/* --- Sample Method Functions --- */

/**
 * Allocate a IPSampleMethod structure
 * Returns: Pointer to the new IPSampleMethod structure or NULL on error
 */
IPSampleMethod *
instp_sample_method_new (void)
{
  IPSampleMethod *meth;
  meth = g_malloc0 (sizeof (IPSampleMethod));
  return (meth);
}

/**
 * Register a user defined sample storage method
 * @method New IPSampleMethod to register
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Registers a user defined sample storage method. Pointer to method should
 * not be free'd for the duration of libsoundfont's use.
 */
int
instp_sample_method_register (IPSampleMethod *method)
{
  IPSampleMethod **temp;
  int size, index;

  g_return_val_if_fail (method != NULL, INSTP_FAIL);

  index = sample_next_user_method_id - IPSAMPLE_METHOD_USER;
  size = (index + 1) * sizeof (IPSampleMethod *);

  temp = realloc (sample_user_methods, size);

  if (!temp)
    {
      g_critical (_("Failed to reallocate %d bytes"), size);
      return (INSTP_FAIL);
    }

  sample_user_methods[index] = method;
  method->type = sample_next_user_method_id++;

  return (INSTP_OK);
}

/**
 * Get pointer to an IPSampleMethod type
 * @method_type Method type ID (#IPSampleMethodType or user defined type)
 * Returns: Pointer to the IPSampleMethod or NULL if invalid type
 *
 * Get pointer to an IPSampleMethod type. Function pointers can be modified to
 * change behaviour of the given type.
 */
IPSampleMethod *
instp_sample_method_get_by_type (int method_type)
{
  if (method_type >= 0 && method_type < IPSAMPLE_METHOD_COUNT)
    return (&sample_methods[method_type]);

  if (method_type >= IPSAMPLE_METHOD_USER
      && method_type < sample_next_user_method_id)
    return (sample_user_methods[method_type - IPSAMPLE_METHOD_USER]);

  return (NULL);
}


/* --- Sample Store Functions --- */

/**
 * Create a new sample store
 * @sampledata Sample data object to create store in.
 * @method_type Sample storage method type of store to create.
 * Returns: The new store added to the sample data object or NULL on error.
 */
IPSampleStore *
instp_sample_store_new (IPSampleData *sampledata, int method_type)
{
  IPSampleMethod *method;
  IPSampleStore *store;

  g_return_val_if_fail (sampledata != NULL, NULL);

  method = instp_sample_method_get_by_type (method_type);
  g_return_val_if_fail (method != NULL, NULL);

  /* allocate the sample store structure (including method var area) */
  store = g_malloc0 (sizeof (IPSampleStore) + method->var_size);
  store->method = method;

  if (method->init)
    if ((*method->init)(sampledata, store) != INSTP_OK)
      {
	g_free (store);
	return (NULL);
      }

  /* prepend the store on the sampledata->storage list */
  store->next = sampledata->storage;
  sampledata->storage = store;

  return (store);
}

/**
 * Get the next sample store in a sample data storage list
 * @sampledata Sample data object containing the storage list.
 * @store Sample store to get next store of, if NULL then first store
 *   will be returned.
 * Returns: The next (or first) sample store or NULL if no more stores.
 */
IPSampleStore *
instp_sample_store_next (const IPSampleStore *store)
{
  g_return_val_if_fail (store != NULL, NULL);
  return (store->next);
}

/**
 * Function to destroy a sample store
 * @sampledata Sample data object containing the store.
 * @store Sample store to destroy.
 */
void
instp_sample_store_destroy (IPSampleData *sampledata, IPSampleStore *store)
{
  IPSampleStore *prev = NULL;

  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (store->method != NULL);

  /* find previous store in sampledata's storage list */
  if (sampledata->storage != store)
    {
      prev = sampledata->storage;

      while (prev && prev->next != store)
	prev = instp_sample_store_next (prev);

      if (!prev)		/* store not found in sampledata? */
	{
	  g_warning (INSTP_ERRMSG_STORE_NOT_IN_SAMPLEDATA_0);
	  return;
	}
    }

  /* call the store's "done" function */
  if (store->method->done)
    (*store->method->done)(sampledata, store);

  /* unlink the store */
  if (prev) prev->next = store->next;
  else sampledata->storage = store->next;

  g_free (store);		/* free the store structure */
}

/**
 * Allocate space for sample data in a sample store
 * @sampledata IPSampleData to allocate storage space for
 * @store Sample store to allocate space for
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Allocates space for sample data by calling the "alloc" function of a
 * sample store's assigned method.
 */
int
instp_sample_store_alloc (IPSampleData *sampledata, IPSampleStore *store)
{
  g_return_val_if_fail (sampledata != NULL, INSTP_FAIL);
  g_return_val_if_fail (store != NULL, INSTP_FAIL);
  g_return_val_if_fail (store->method != NULL, INSTP_FAIL);

  if (store->method->alloc)
    return ((*store->method->alloc)(sampledata, store));
  return (INSTP_OK);
}

/**
 * Free storage space allocated for a sample store
 * @sampledata Sample data object owning store
 * @store Store to free sample data of.
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Frees resources allocated for sample data by calling the "free" function
 * of a sample store's assigned method.
 */
void
instp_sample_store_free (IPSampleData *sampledata, IPSampleStore *store)
{
  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (store->method != NULL);

  if (store->method->free)
    (*store->method->free)(sampledata, store);
}

/**
 * Read sample data from a sample store
 * @sampledata Sample data object owning store
 * @store Sample data store to read from.
 * @offset Offset (in samples) to start read
 * @size Number of samples to read
 * @Buffer to place sample data in, should be (size * 2 bytes)
 *   for 16 bit samples.
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sample_store_read (IPSampleData *sampledata, IPSampleStore *store,
			 int offset, int size, void *buf)
{
  g_return_val_if_fail (sampledata != NULL, INSTP_FAIL);
  g_return_val_if_fail (store != NULL, INSTP_FAIL);
  g_return_val_if_fail (store->method != NULL, INSTP_FAIL);
  g_return_val_if_fail (store->method->read != NULL, INSTP_FAIL);

  return ((*store->method->read)(sampledata, store, offset, size, buf));
}

/**
 * Write sample data to a sample store
 * @sampledata Sample data object owning store
 * @store Sample data store to write to (should be writable).
 * @offset Offset (in samples) to start write
 * @size Number of samples to write
 * @buf Buffer of sample data to write (should be size * 2 bytes) for
 *   16 bit samples.
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sample_store_write (IPSampleData *sampledata, IPSampleStore *store,
			  int offset, int size, const void *buf)
{
  g_return_val_if_fail (sampledata != NULL, INSTP_FAIL);
  g_return_val_if_fail (store != NULL, INSTP_FAIL);
  g_return_val_if_fail (store->method != NULL, INSTP_FAIL);
  g_return_val_if_fail (store->method->write != NULL, INSTP_FAIL);

  return ((*store->method->write)(sampledata, store, offset, size, buf));
}

/**
 * Copy data from one sample store to another
 * @dest_sampledata Destination sample data object
 * @dest_store Destination sample store
 * @src_sampledata Source sample data object
 * @src_store Source sample store
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Sample data objects must be the same size.
 */
int
instp_sample_store_copy (IPSampleData *dest_sampledata,
			 IPSampleStore *dest_store,
			 IPSampleData *src_sampledata,
			 IPSampleStore *src_store)
{
  void *buf;
  guint samsize, size, ofs;

  g_return_val_if_fail (dest_sampledata != NULL, INSTP_FAIL);
  g_return_val_if_fail (dest_store != NULL, INSTP_FAIL);
  g_return_val_if_fail (src_sampledata != NULL, INSTP_FAIL);
  g_return_val_if_fail (src_store != NULL, INSTP_FAIL);
  g_return_val_if_fail (dest_store != src_store, INSTP_FAIL);
  g_return_val_if_fail (IPSAMPLE_STORE_IS_READABLE (src_store), INSTP_FAIL);
  g_return_val_if_fail (IPSAMPLE_STORE_IS_WRITABLE (dest_store), INSTP_FAIL);
  g_return_val_if_fail (dest_sampledata->size == src_sampledata->size,
			INSTP_FAIL);

  buf = g_malloc (INSTP_SAMPLE_COPY_BUFFER_SIZE * 2);

  samsize = dest_sampledata->size;
  size = INSTP_SAMPLE_COPY_BUFFER_SIZE;
  ofs = 0;
  while (ofs < samsize)
    {
      if (samsize - ofs < size) /* check for last partial fragment */
	size = samsize - ofs;

      if (instp_sample_store_read (src_sampledata, src_store,
				   ofs, size, buf) != INSTP_OK)
	{
	  g_free (buf);
	  return (INSTP_FAIL);
	}

      if (instp_sample_store_write (dest_sampledata, dest_store,
				    ofs, size, buf) != INSTP_OK)
	{
	  g_free (buf);
	  return (INSTP_FAIL);
	}

      ofs += size;
    }

  g_free (buf);

  return (INSTP_OK);
}

/**
 * Duplicate a sample store within the same sample data object
 * @sampledata Sample data object
 * @store Sample store to duplicate
 * @method_type Sample storage method type to use for new store
 * Returns: The new store with duplicated sample data or NULL on error
 */
IPSampleStore *
instp_sample_store_duplicate (IPSampleData *sampledata, IPSampleStore *store,
			      int method_type)
{
  IPSampleStore *newstore;

  g_return_val_if_fail (sampledata != NULL, NULL);
  g_return_val_if_fail (store != NULL, NULL);

  /* create the new sample store of type RAM */
  newstore = instp_sample_store_new (sampledata, method_type);
  if (!newstore) return (NULL);

  /* allocate sample data storage for new store */
  if (instp_sample_store_alloc (sampledata, newstore) != INSTP_OK)
    {
      instp_sample_store_destroy (sampledata, newstore);
      return (NULL);
    }

  /* copy the data from the store to the new store */
  if (instp_sample_store_copy (sampledata, newstore,
			       sampledata, store) != INSTP_OK)
    {
      instp_sample_store_destroy (sampledata, newstore);
      return (NULL);
    }

  return (newstore);
}

/**
 * Get a pointer to the variable area of a sample store.
 * @store Sample store to get pointer to var area of.
 * Returns: Pointer to variable area in sample store.
 *
 * Only useful to sample storage method functions.
 */
void *
instp_sample_store_get_varptr (IPSampleStore *store)
{
  g_return_val_if_fail (store != NULL, NULL);

  return ((void *)(store + 1));
}


/* --- sample method specific functions --- */

/**
 * Set the parameters of an #IPSAMPLE_METHOD_SFONT sample store
 * @sampledata Sample data object owning store
 * @store An #IPSAMPLE_METHOD_SFONT sample store
 * @file Info on file which contains the sample data
 * @start Offset (in samples) from the beginning of the sample chunk
 *   in the sound font file to the sample data.
 */
void
instp_sample_method_SFONT_set_params (IPSampleData *sampledata,
				      IPSampleStore *store,
				      IPFileInfo *file, guint start)
{
  MethodSFontVars *vars;

  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (file != NULL);
  g_return_if_fail (store->method != NULL);
  g_return_if_fail (store->method->type == IPSAMPLE_METHOD_SFONT);

  vars = instp_sample_store_get_varptr (store);

  instp_file_info_ref (file);
  if (vars->file) instp_file_info_unref (vars->file);

  vars->file = file;
  vars->start = start;
}

/**
 * Get the parameters of an #IPSAMPLE_METHOD_SFONT sample store
 * @sampledata Sample data object owning store
 * @store An #IPSAMPLE_METHOD_SFONT sample store
 * @file OUTPUT: Pointer to info on the file which contains the
 *   sample data.
 * @start OUTPUT: Pointer to store offset (in samples) from
 *   beginning of sample chunk in sound font file to sample data, can be NULL.
 */
void
instp_sample_method_SFONT_get_params (IPSampleData *sampledata,
				      IPSampleStore *store,
				      IPFileInfo **file, guint *start)
{
  MethodSFontVars *vars;

  if (file) *file = NULL;
  if (start) *start = 0;

  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (store->method != NULL);
  g_return_if_fail (store->method->type == IPSAMPLE_METHOD_SFONT);

  vars = instp_sample_store_get_varptr (store);

  if (file) *file = vars->file;
  if (start) *start = vars->start;
}

/* IPSampleDoneFunc for IPSAMPLE_METHOD_SFONT */
static void
method_SFONT_done_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodSFontVars *vars;

  vars = instp_sample_store_get_varptr (store);

  if (vars->file)
    {
      instp_file_info_unref (vars->file);
      vars->file = NULL;
    }
}

/* IPSampleReadFunc for IPSAMPLE_METHOD_SFONT */
static int
method_SFONT_read_func (IPSampleData *sampledata, IPSampleStore *store,
			int offset, int size, void *buf)
{
  MethodSFontVars *method_vars;
  off_t pos;
  int loaded;

  method_vars = instp_sample_store_get_varptr (store);
  g_return_val_if_fail (method_vars->file != NULL, INSTP_FAIL);

  pos = method_vars->file->sample_pos + method_vars->start * 2 + offset * 2;

  if (lseek (method_vars->file->fd, pos, SEEK_SET) == -1)
    {
      g_critical (_("Failed to seek to offset %ld in sound font file: %s"),
		  (long)pos, g_strerror (errno));
      return (INSTP_FAIL);
    }

  loaded = read (method_vars->file->fd, buf, size * 2);

  if (loaded == -1 || loaded != size * 2)
    {
      g_critical (_("Failed to read %d bytes from sound font file: %s"),
		  size * 2, g_strerror (errno));
      return (INSTP_FAIL);
    }
  return (INSTP_OK);
}


/* --- Sample BUFFER method --- */

/* IPSampleInitFunc for IPSAMPLE_METHOD_BUFFER */
static int
method_BUFFER_init_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodBufferVars *vars;

  vars = instp_sample_store_get_varptr (store);
  vars->start = UINT_MAX;	/* UINT_MAX used to indicate no allocation */

  return (INSTP_OK);
}

/* IPSampleAllocFunc for IPSAMPLE_METHOD_BUFFER */
static int
method_BUFFER_alloc_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodBufferVars *vars;

  vars = instp_sample_store_get_varptr (store);

  if (!sample_buffer_fd)
    if (sample_buffer_init () != INSTP_OK)
      return (INSTP_FAIL);

  /* start is set to UINT_MAX when data hasn't been allocated yet */
  if (vars->start != UINT_MAX)
    {
      g_warning (_(INSTP_ERRMSG_METH_ALREADY_ALLOC_0));
      return (INSTP_OK);
    }

  vars->start = sample_buffer_length;
  sample_buffer_length += sampledata->size * 2;

  return (INSTP_OK);
}

/* IPSampleFreeFunc for IPSAMPLE_METHOD_BUFFER */
static void
method_BUFFER_free_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodBufferVars *vars;

  vars = instp_sample_store_get_varptr (store);

  /* if data is actually allocated, add it to waste counter and set
     start to UINT_MAX to indicate no allocation. Sample data will be
     actually de-allocated when sample buffer is cleaned. */
  if (vars->start != UINT_MAX)
    {
      sample_buffer_waste += sampledata->size * 2;
      vars->start = UINT_MAX;
    }
}

/* IPSampleReadFunc for IPSAMPLE_METHOD_BUFFER */
static int
method_BUFFER_read_func (IPSampleData *sampledata, IPSampleStore *store,
			 int offset, int size, void *buf)
{
  MethodBufferVars *vars;
  guint pos, loaded;

  vars = instp_sample_store_get_varptr (store);

  /* start == UINT_MAX indicates that data hasn't been allocated */
  if (vars->start == UINT_MAX)
    {
      g_critical (INSTP_ERRMSG_METH_NOT_ALLOC_0);
      return (INSTP_FAIL);
    }

  pos = vars->start + offset * 2;

  if (fseek (sample_buffer_fd, pos, SEEK_SET) == -1)
    {
      g_critical (_(INSTP_ERRMSG_METH_BUFFER_SEEK_2), pos, g_strerror (errno));
      return (INSTP_FAIL);
    }

  loaded = fread (buf, 1, size * 2, sample_buffer_fd);

  if (loaded != size * 2)
    {
      g_critical (_("Failed to read %d bytes from sample buffer: %s"),
		  size * 2, g_strerror (errno));
      return (INSTP_FAIL);
    }

  return (INSTP_OK);
}

/* IPSampleWriteFunc for IPSAMPLE_METHOD_BUFFER */
static int
method_BUFFER_write_func (IPSampleData *sampledata, IPSampleStore *store,
			  int offset, int size, const void *buf)
{
  MethodBufferVars *vars;
  guint pos, written;

  vars = instp_sample_store_get_varptr (store);

  /* start == UINT_MAX indicates that data hasn't been allocated */
  if (vars->start == UINT_MAX)
    {
      g_critical (_(INSTP_ERRMSG_METH_NOT_ALLOC_0));
      return (INSTP_FAIL);
    }

  pos = vars->start + offset * 2;

  if (fseek (sample_buffer_fd, pos, SEEK_SET) == -1)
    {
      g_critical (_(INSTP_ERRMSG_METH_BUFFER_SEEK_2), pos, g_strerror (errno));
      return (INSTP_FAIL);
    }

  written = fwrite (buf, 1, size * 2, sample_buffer_fd);

  if (written != size * 2)
    {
      g_critical (_("Failed to write %d bytes to sample buffer: %s"),
		  size * 2, g_strerror (errno));
      return (INSTP_FAIL);
    }

  return (INSTP_OK);
}

/* initialize the sample buffer */
static int
sample_buffer_init (void)
{
  if (sample_buffer_fd)
    return (INSTP_OK);

  if (!(sample_buffer_fd = tmpfile ()))
    {
      g_critical (_(INSTP_ERRMSG_METH_BUFFER_TEMP_FILE_1), g_strerror (errno));
      return (INSTP_FAIL);
    }

  sample_buffer_length = 0;
  sample_buffer_waste = 0;

  return (INSTP_OK);
}

/* destroy sample buffer */
static void
sample_buffer_destroy (void)
{
  if (!sample_buffer_fd) return;

  fclose (sample_buffer_fd);

  sample_buffer_fd = NULL;
}

/**
 * Rebuild disk sample buffer
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Call this function to force rebuild of sample disk buffer.  Only
 * useful if in manual cleanup mode.  All used sample data is written
 * to a new temporary file and then the old sample buffer file is
 * deleted. Any unused sample data will therefore be
 * "de-allocated". Does not actually check how much (if any) sample
 * data is wasted, in order to be useful the caller should check this.
 */
int
sample_buffer_clean (void)
{

  /* FIXME */
#if 0
  struct st_saminfo  /* structure stores old sample info */
  {
    IPSampleData *samdata;
    guint start;
  };

  int temp_fd;			/* temp file for new sample buffer */
  char *temp_fname;		/* temp file name */
  int newpos = 0;		/* current position in new sample buffer */
  GArray *backup_saminfo;  /* array of old samdata info (in case of error) */
  IPSamDataInfo *datainfo;
  struct st_saminfo bakinfo;	/* structure stores old sample info */
  struct st_saminfo *reinfo;

  if (sample_buffer_fd == -1) return (INSTP_OK);

  temp_fname = g_strdup ("libsf-XXXXXX");

  if ((temp_fd = mkstemp (temp_fname)) == -1)
    {
      g_free (temp_fname);
      g_critical (_(INSTP_ERRMSG_METH_BUFFER_TEMP_FILE_1), g_strerror (errno));
    }

  backup_saminfo = g_array_new (FALSE, FALSE, sizeof (bakinfo));

  /* copy used samples from old sample buffer to new */
  p = sam_datainfo_list;
  while (p)  /* loop over sample data info */
    {
      datainfo = (IPSamDataInfo *) (p->data);
      p = g_slist_next (p);
      if (!datainfo->samfile)
	continue;		/* only samples in sam buffer */

      /* save sample info (in case of error) */
      bakinfo.datainfo = datainfo;
      bakinfo.start = datainfo->start;
      g_array_append_val (backup_saminfo, bakinfo);

      size = datainfo->size * 2;

      /* copy sample to new sample buffer */
      if (!(buf = safe_malloc (size)))
	error = TRUE;

      /* seek to start of sample buffer */
      if (!error && !safe_fseek (sambuf_fd, datainfo->start * 2, SEEK_SET))
	error = TRUE;
      
      if (!error && !safe_fread (buf, size, sambuf_fd))
	error = TRUE;
      if (!error && !safe_fwrite (buf, size, temp_fd))
	error = TRUE;

      g_free (buf);
      
      datainfo->start = newpos;
      newpos += datainfo->size;

      if (error)
	{		/* if error occured, then restore sample info */
	  i = backup_saminfo->len - 1;
	  while (i >= 0)
	    {
	      reinfo = &g_array_index (backup_saminfo, struct st_saminfo, i);
	      datainfo = reinfo->datainfo;
	      datainfo->start = reinfo->start;
	      i--;
	    }
	  g_array_free (backup_saminfo, TRUE);
	  fclose (temp_fd);
	  return;
	}
    }

  g_array_free (backup_saminfo, TRUE);
  fclose (sambuf_fd);		/* close defunct sample buffer */
  sambuf_fd = temp_fd;		/* set temp buffer to new one */
  sambuf_length = newpos;	/* set length of buffer to new length */

#endif

  return (INSTP_OK);
}


/* --- RAM Sample Method functions --- */


/**
 * Set the sample data pointer of an #IPSAMPLE_METHOD_RAM sample store
 * @sampledata Sample data object owning store.
 * @store Sample store of type #IPSAMPLE_METHOD_RAM to set pointer of.
 * @data Pointer to sample data in RAM.
 *
 * Data pointer \b CAN be set to existing sample data without calling
 * the "alloc" method function.
 */
void
instp_sample_method_RAM_set_pointer (IPSampleData *sampledata,
				     IPSampleStore *store, void *data)
{
  MethodRAMVars *vars;

  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (store->method != NULL);
  g_return_if_fail (store->method->type == IPSAMPLE_METHOD_RAM);

  vars = instp_sample_store_get_varptr (store);
  vars->dataptr = data;
}

/**
 * Get the sample data pointer of an #IPSAMPLE_METHOD_RAM sample store
 * @sampledata Sample data object owning store.
 * @store Sample store of type #IPSAMPLE_METHOD_RAM to get pointer from.
 * Returns: Pointer to entire sample data in memory or NULL if not set.
 */
void *
instp_sample_method_RAM_get_pointer (IPSampleData *sampledata,
				     IPSampleStore *store)
{
  MethodRAMVars *vars;

  g_return_val_if_fail (sampledata != NULL, NULL);
  g_return_val_if_fail (store != NULL, NULL);
  g_return_val_if_fail (store->method != NULL, NULL);
  g_return_val_if_fail (store->method->type == IPSAMPLE_METHOD_RAM, NULL);

  vars = instp_sample_store_get_varptr (store);
  return (vars->dataptr);
}

/* IPSampleAllocFunc for IPSAMPLE_METHOD_RAM */
static int
method_RAM_alloc_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodRAMVars *vars;

  vars = instp_sample_store_get_varptr (store);

  vars->dataptr = malloc (sampledata->size * 2);
  if (!vars->dataptr)
    {
      g_critical (_("Failed to allocate %d bytes"), sampledata->size * 2);
      return (INSTP_FAIL);
    }

  return (INSTP_OK);
}

/* IPSampleFreeFunc for IPSAMPLE_METHOD_RAM */
static void
method_RAM_free_func (IPSampleData *sampledata, IPSampleStore *store)
{
  MethodRAMVars *vars;

  vars = instp_sample_store_get_varptr (store);

  if (vars->dataptr)
    {
      g_free (vars->dataptr);
      vars->dataptr = NULL;
    }
}

/* IPSampleReadFunc for IPSAMPLE_METHOD_RAM */
static int
method_RAM_read_func (IPSampleData *sampledata, IPSampleStore *store,
		      int offset, int size, void *buf)
{
  MethodRAMVars *vars;

  vars = instp_sample_store_get_varptr (store);

  g_return_val_if_fail (vars->dataptr != NULL, INSTP_FAIL);
  memcpy (buf, (char *)(vars->dataptr) + offset * 2, size * 2);

  return (INSTP_OK);
}

/* IPSampleWriteFunc for IPSAMPLE_METHOD_RAM */
static int
method_RAM_write_func (IPSampleData *sampledata, IPSampleStore *store,
		       int offset, int size, const void *buf)
{
  MethodRAMVars *vars;

  vars = instp_sample_store_get_varptr (store);

  g_return_val_if_fail (vars->dataptr != NULL, INSTP_FAIL);
  memcpy ((char *)(vars->dataptr) + offset * 2, buf, size * 2);

  return (INSTP_OK);
}

/**
 * Set the sample start location of a #IPSAMPLE_METHOD_ROM sample store
 * @sampledata Sample data object owning store.
 * @store Sample store of type #IPSAMPLE_METHOD_ROM to set start of.
 * @start Start location within sample ROM (in samples) of data.
 */
void
instp_sample_method_ROM_set_start (IPSampleData *sampledata,
				   IPSampleStore *store, guint start)
{
  MethodROMVars *vars;

  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (store != NULL);
  g_return_if_fail (store->method != NULL);
  g_return_if_fail (store->method->type == IPSAMPLE_METHOD_ROM);

  vars = instp_sample_store_get_varptr (store);
  vars->start = start;
}

/**
 * Get the sample start location of a #IPSAMPLE_METHOD_ROM sample store
 * @sampledata Sample data object owning store.
 * @store Sample store of type #IPSAMPLE_METHOD_ROM to get start from.
 * Returns: Start location within sample ROM (in samples) of data.
 */
guint
instp_sample_method_ROM_get_start (IPSampleData *sampledata,
				   IPSampleStore *store)
{
  MethodROMVars *vars;

  g_return_val_if_fail (sampledata != NULL, 0);
  g_return_val_if_fail (store != NULL, 0);
  g_return_val_if_fail (store->method != NULL, 0);
  g_return_val_if_fail (store->method->type == IPSAMPLE_METHOD_ROM, 0);

  vars = instp_sample_store_get_varptr (store);
  return (vars->start);
}
