/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "WcStatusLvi.h"
#include "ColorId.h"
#include "StatusId.h"
#include "DragDropMimeTypes.h"
#include "sublib/ColorStorage.h"
#include "svn/WcEntry.h"



static QString getStatusCode( svn::WcStatusKind status )
{
  switch( status )
  {
  case svn::WcStatus_None:        return _q(StatusCodes[StatusNone]._code);
  case svn::WcStatus_Unversioned: return _q(StatusCodes[StatusUnversioned]._code);
  case svn::WcStatus_Normal:      return _q(StatusCodes[StatusNormal]._code);
  case svn::WcStatus_Added:       return _q(StatusCodes[StatusAdded]._code);
  case svn::WcStatus_Missing:     return _q(StatusCodes[StatusMissing]._code);
  case svn::WcStatus_Deleted:     return _q(StatusCodes[StatusDeleted]._code);
  case svn::WcStatus_Replaced:    return _q(StatusCodes[StatusReplaced]._code);
  case svn::WcStatus_Modified:    return _q(StatusCodes[StatusModified]._code);
  case svn::WcStatus_Merged:      return _q(StatusCodes[StatusMerged]._code);
  case svn::WcStatus_Conflicted:  return _q(StatusCodes[StatusConflicted]._code);
  case svn::WcStatus_Ignored:     return _q(StatusCodes[StatusIgnored]._code);
  case svn::WcStatus_Obstructed:  return _q(StatusCodes[StatusObstructed]._code);
  case svn::WcStatus_External:    return _q(StatusCodes[StatusExternal]._code);
  case svn::WcStatus_Incomplete:  return _q(StatusCodes[StatusIncomplete]._code);
  default:                        return _q(StatusCodes[StatusFallback]._code);
  }
}

static QColor getStatusColor( svn::WcStatusKind status )
{
  switch( status )
  {
  case svn::WcStatus_None:        return ColorStorage::getColor(ColorNone);
  case svn::WcStatus_Unversioned: return ColorStorage::getColor(ColorUnversioned);
  case svn::WcStatus_Normal:      return ColorStorage::getColor(ColorNormal);
  case svn::WcStatus_Added:       return ColorStorage::getColor(ColorAdded);
  case svn::WcStatus_Missing:     return ColorStorage::getColor(ColorMissing);
  case svn::WcStatus_Deleted:     return ColorStorage::getColor(ColorDeleted);
  case svn::WcStatus_Replaced:    return ColorStorage::getColor(ColorReplaced);
  case svn::WcStatus_Modified:    return ColorStorage::getColor(ColorModified);
  case svn::WcStatus_Merged:      return ColorStorage::getColor(ColorMerged);
  case svn::WcStatus_Conflicted:  return ColorStorage::getColor(ColorConflicted);
  case svn::WcStatus_Ignored:     return ColorStorage::getColor(ColorIgnored);
  case svn::WcStatus_Obstructed:  return ColorStorage::getColor(ColorObstructed);
  case svn::WcStatus_External:    return ColorStorage::getColor(ColorExternal);
  case svn::WcStatus_Incomplete:  return ColorStorage::getColor(ColorIncomplete);
  default:                        return ColorStorage::getColor(ColorFallback);
  }
}

static QString getLockStatusCode( svn::WcStatusLockKind status )
{
  switch( status )
  {
  case svn::WcStatusLock_None:   return _q(StatusCodes[StatusLockNone]._code);
  case svn::WcStatusLock_Locked: return _q(StatusCodes[StatusLockLocked]._code);
  case svn::WcStatusLock_Stolen: return _q(StatusCodes[StatusLockStolen]._code);
  case svn::WcStatusLock_Other:  return _q(StatusCodes[StatusLockOther]._code);
  case svn::WcStatusLock_Broken: return _q(StatusCodes[StatusLockBroken]._code);
  default:                       return _q(StatusCodes[StatusFallback]._code);
  }
}

static QString getStatusCode( StatusId id )
{
  return _q(StatusCodes[id]._code);
}


WcStatusLvi::WcStatusLvi( QListView* parent, const svn::WcStatusPtr status )
: super(parent), _status(status), _rename(false)
{
  init();
}

WcStatusLvi::WcStatusLvi( QListViewItem* parent, const svn::WcStatusPtr status )
: super(parent), _status(status), _rename(false)
{
  init();
}

WcStatusLvi::~WcStatusLvi()
{
}

void WcStatusLvi::init()
{
  QString path = QString::fromUtf8( _status->getName().getStr() );
  path.remove( 0, _parentPath.length()+1 );

  setDropEnabled(_status->isDir());

  if( path.length() == 0 )
  {
    // ie. "."
    return;
  }

  setRenameEnabled( StatusColumnName, _status->isVersioned());
  setDragEnabled(_status->isVersioned());
}

void WcStatusLvi::startRename( int col )
{
  super::startRename(col);
}

void WcStatusLvi::okRename( int col )
{
  // tell the text method to return the new text and not the one
  // from the status.
  _rename = true;
  super::okRename(col);
  _rename = false;
}

void WcStatusLvi::cancelRename( int col )
{
  super::cancelRename(col);
}

#if 0
void WcStatusLvi::paintFocus( QPainter* p, const QColorGroup& cg, const QRect& r )
{
  super::paintFocus(p,cg,r);
}
#endif

void WcStatusLvi::paintCell( QPainter* p, const QColorGroup& cg, int column, int width, int alignment )
{
  QColorGroup g(cg);

  if( _status->getTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getTextStatus()) );
  }
  else if(
    _status->getPropStatus() != svn::WcStatus_Normal &&
    _status->getPropStatus() != svn::WcStatus_None
    )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getPropStatus()) );
  }
#if 0
  else if( _status->getReposTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getReposTextStatus()) );
  }
  else if( _status->getReposPropStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, getStatusColor(_status->getReposPropStatus()) );
  }
#endif
  //g.setColor( QColorGroup::Background, QColor(255,0,0) );
  //g.setColor( QColorGroup::Foreground, QColor(255,0,0) );
  //g.setColor( QColorGroup::Text, QColor(255,0,0) );
  //g.setColor( QColorGroup::Button, QColor(255,0,0) );
  //g.setColor( QColorGroup::ButtonText, QColor(255,0,0) );
  //g.setColor( QColorGroup::Light, QColor(255,0,0) );
  //g.setColor( QColorGroup::Midlight, QColor(255,0,0) );
  //g.setColor( QColorGroup::Dark, QColor(255,0,0) );
  //g.setColor( QColorGroup::Mid, QColor(255,0,0) );
  //g.setColor( QColorGroup::Shadow, QColor(255,0,0) );
  //g.setColor( QColorGroup::Highlight, QColor(255,0,0) );
  //g.setColor( QColorGroup::HighlightedText, QColor(255,0,0) );
  //g.setColor( QColorGroup::BrightText, QColor(255,0,0) );

  super::paintCell(p,g,column,width,alignment);
}

#if 0
void WcStatusLvi::paintBranches( QPainter* p, const QColorGroup& cg, int w, int y, int h, GUIStyle s )
{
  QColorGroup g(cg);
  if( _status->getTextStatus() != svn::WcStatus_Normal )
  {
    g.setColor( QColorGroup::Base, QColor(230,230,230) );
  }
  super::paintBranches(p,g,w,y,h,s);
}
#endif

QString WcStatusLvi::text( int column ) const
{
  switch( column )
  {
  case StatusColumnName:
    {
      if( _rename )
      {
        return super::text(StatusColumnName);
      }

      QString path = QString::fromUtf8( _status->getName().getStr() );
      path.remove( 0, _parentPath.length()+1 );

      if( path.length() == 0 )
      {
        return ".";
      }
      return path;
    }
  case StatusColumnTextStatus:
    {
      return getStatusCode( _status->getTextStatus() );
    }
  case StatusColumnPropStatus:
    {
      return getStatusCode( _status->getPropStatus() );
    }
  case StatusColumnWcLocked:
    {
      return _status->isLocked() ?
        getStatusCode(StatusWcLockLocked) : getStatusCode(StatusWcLockNone);
    }
  case StatusColumnCopied:
    {
      return _status->isCopied() ?
        getStatusCode(StatusHistory) : getStatusCode(StatusHistoryNone);
    }
  case StatusColumnSwitched:
    {
      return _status->isSwitched() ?
        getStatusCode(StatusSwitched) : getStatusCode(StatusSwitchedNo);
    }
  case StatusColumnRepLock:
    {
      return getLockStatusCode( _status->getLockStatus() );
    }
  case StatusColumnUpToDate:
    {
      bool t = _status->getReposTextStatus() != svn::WcStatus_None;
      bool p = _status->getReposTextStatus() != svn::WcStatus_None;

      return ( t || p ) ?
        getStatusCode(StatusUpToDateNo) : getStatusCode(StatusUpToDate);
    }
  case StatusColumnWcRev:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getRevnumber() < 0 )
      {
        return "?";
      }
      else if( _status->isCopied() )
      {
        return "-";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getRevnumber() );
      }
    }
  case StatusColumnCmtRev:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getCmtRevnumber() < 0 )
      {
        return "?";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getCmtRevnumber() );
      }
    }
  case StatusColumnCmtAuthor:
    {
      const svn::WcEntry* e = _status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( ! e->getCmtAuthor().getStr() )
      {
        return "?";
      }
      else
      {
        return QString::fromUtf8( e->getCmtAuthor().getStr() );
      }
    }
  case StatusColumnEmpty:
    {
      return "";
    }
  default:
    {
      return "invalid column";
    }
  }
}

QString WcStatusLvi::key( int column, bool ascending ) const
{
  switch( column )
  {
  case StatusColumnName:
  case StatusColumnTextStatus:
  case StatusColumnPropStatus:
  case StatusColumnWcLocked:
  case StatusColumnCopied:
  case StatusColumnSwitched:
  case StatusColumnRepLock:
  case StatusColumnUpToDate:
  case StatusColumnWcRev:
  case StatusColumnCmtRev:
  case StatusColumnCmtAuthor:
  default:
    {
      return text( column );
    }
  }
}

int WcStatusLvi::compare( QListViewItem * i, int col, bool ascending ) const
{
  if( col != StatusColumnName )
  {
    return super::compare(i,col,ascending);
  }

  WcStatusLvi*     lvi    = dynamic_cast<WcStatusLvi*>(i);
  svn::WcStatusPtr status = lvi->getStatus();

  QString myself = text(StatusColumnName);
  QString other  = lvi->text(StatusColumnName);

  bool otherIsDir  = (status->getWcEntry() && status->getWcEntry()->isDir())
    || (lvi->text(StatusColumnName).contains( '/' ) > 0);

  bool myselfIsDir = (_status->getWcEntry() && _status->getWcEntry()->isDir())
    || (text(StatusColumnName).contains( '/' ) > 0);

  //printf( "compare:\n" );
  //printf( "%s %s\n", myselfIsDir ? "d" : "f", (const char*)myself );
  //printf( "%s %s\n", otherIsDir ?  "d" : "f", (const char*)other );

  if( myselfIsDir && ! otherIsDir )
  {
    return -1;
  }
  else if( ! myselfIsDir && otherIsDir )
  {
    return 1;
  }
  else
  {
    if( myself < other )
    {
      return -1;
    }
    else if( myself > other )
    {
      return 1;
    }
    else
    {
      return 0;
    }
  }
}

void WcStatusLvi::setParentPath( const QString& parentPath )
{
  _parentPath = parentPath;
}

const QString& WcStatusLvi::getParentPath() const
{
  return _parentPath;
}

svn::WcStatusPtr WcStatusLvi::getStatus() const
{
  return _status;
}

bool WcStatusLvi::acceptDrop( const QMimeSource* mime ) const
{
  return dropEnabled()
    && mime->provides( ScMimeTypeWcFilesItem );
}


#if 0
  QColorGroup g(cg);
  g.setColor( QColorGroup::Background, QColor(0,0,0) );
  g.setColor( QColorGroup::Foreground, QColor(0,0,0) );
  g.setColor( QColorGroup::Base, QColor(0,0,0) );
  g.setColor( QColorGroup::Text, QColor(0,0,0) );
  g.setColor( QColorGroup::Button, QColor(0,0,0) );
  g.setColor( QColorGroup::ButtonText, QColor(0,0,0) );
  g.setColor( QColorGroup::Light, QColor(0,0,0) );
  g.setColor( QColorGroup::Midlight, QColor(0,0,0) );
  g.setColor( QColorGroup::Dark, QColor(0,0,0) );
  g.setColor( QColorGroup::Mid, QColor(0,0,0) );
  g.setColor( QColorGroup::Shadow, QColor(0,0,0) );
  g.setColor( QColorGroup::Highlight, QColor(0,0,0) );
  g.setColor( QColorGroup::HighlightedText, QColor(0,0,0) );
  g.setColor( QColorGroup::BrightText, QColor(0,0,0) );
#endif

