/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

//sc
#include "config.h"
#include "MainWindow.h"
#include "ListWidget.h"
#include "ScModel.h"
#include "DialogHandler.h"
#include "Cancel.h"
#include "LogLvi.h"
#include "LogWidget.h"
#include "Project.h"
#include "ProjectPropertiesDialog.h"
#include "ExternProviderImpl.h"
#include "ActionStorage.h"
#include "PostProgressCallback.h"
#include "ConfigManager.h"
#include "ProjectFoldersWidget.h"
#include "settings/ProjectSettingsWidget.h"
#include "settings/ProjectSettingsInfo.h"
#include "settings/ToolsSettingsWidget.h"
#include "settings/ToolsSettingsInfo.h"
#include "settings/FontSettingsWidget.h"
#include "settings/FontSettingsInfo.h"
#include "settings/SubversionSettingsWidget.h"
#include "settings/SubversionSettingsInfo.h"
#include "events/LoggingEvent.h"
#include "events/DialogEvent.h"
#include "events/LviOnItemEvent.h"
#include "events/ExceptionEvent.h"
#include "events/CancelStartEvent.h"
#include "events/CancelStopEvent.h"
#include "events/CustomEvents.h"
#include "sublib/AboutDialog.h"
#include "sublib/ErrorDialog.h"
#include "sublib/SplitLayout.h"
#include "sublib/SettingsInfo.h"
#include "sublib/SettingsDialog.h"
#include "sublib/DebugSettingsInfo.h"
#include "sublib/DebugSettingsWidget.h"
#include "sublib/MessageBox.h"
#include "sublib/Utility.h"
#include "sublib/settings/ColorSettingsWidget.h"
#include "sublib/settings/ColorSettingsInfo.h"
#include "util/Error.h"
#include "util/Condition.h"

// qt
#include <qapplication.h>
#include <qmenubar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qstatusbar.h>
#include <qfiledialog.h>
#include <qprogressbar.h>
#include <qfontmetrics.h>
#include <qtimer.h>
#include <qlistview.h>
#include <qheader.h>
#include <qaction.h>
#include <qtooltip.h>
#include <qobjectlist.h>
#ifdef ENABLE_MANUAL
#include <qassistantclient.h>
#endif // ENABLE_MANUAL

// sys
#include <string>
#include <fstream>
#include <math.h>



//\todo replace with QActions
enum MenuIds
{
  pmFile,
  pmTools,
  pmHelp
};

enum Actions
{
  ActionOptionStatusUpdates,
  ActionOptionStatusIgnored,
  ActionOptionStatusAll,
  ActionOptionStatusRecurse,

  ActionOptionCommandForce,
  ActionOptionCommandRecurse,

  ActionCommandReload
};

const char SplitKey[] = "MainWindow.split";

MainWindow::MainWindow( ScModel* model, ConfigManager* cfg, QWidget *parent,
  const char *wname )
: super(parent, wname, Qt::WStyle_Customize | Qt::WType_TopLevel |
  Qt::WStyle_NormalBorder | Qt::WStyle_Title | Qt::WStyle_SysMenu |
  Qt::WStyle_MinMax | Qt::WResizeNoErase ), TargetId(this), _config(cfg),
  _model(model), _assistant(NULL)
{
  setName( "MainWindow" );
  setCaption( _q("subcommander") );
   
  setToolBarsMovable(false);
  setDockMenuEnabled(true);
  
  // we have to set this before creating the tool bar. Otherwise restoring
  // the split handle position will not work.

  // todo write to ini 
#ifdef _MACOSX
  setUsesBigPixmaps(true);
  setUsesTextLabel(true);
#else  
  setUsesBigPixmaps(false);
  setUsesTextLabel(false);  
#endif // _MACOSX
  
  QToolBar* tbOptStatus  = new QToolBar(this);
  addToolBar(tbOptStatus);
  QToolBar* tbOptCommands = new QToolBar(this);
  addToolBar(tbOptCommands);
  QToolBar* tbCommands = new QToolBar(this);
  addToolBar(tbCommands);
  {
#ifdef _MACOSX
    QFont f = font();
    f.setPixelSizeFloat(10.0);

    tbOptStatus->setFont(f);
    tbOptCommands->setFont(f);
    tbCommands->setFont(f);
#endif // _MACOSX
  }
  
  // widget setup
  QWidget* sw = new QWidget(this);
  super::setCentralWidget(sw);

  VSplitLayout* sl = new VSplitLayout(sw,SplitLayout::pTwo);
  _split = sl;
  {
    _list = new ListWidget(model,sw);
    _list->installChildEventFilter(this);
    sl->addWidgetOne( _list, false, 5 );

    QHBox* h = new QHBox(sw);

    _output = new LogWidget(h);

    connect( _output, SIGNAL(contextMenuRequested( QListViewItem*, const QPoint&, int )),
      this, SLOT(outputMenuRequested( QListViewItem*, const QPoint&, int )) );

    sl->addWidgetTwo( h, false, 2 );
  }
  connect( _list->getProjectFoldersWidget(), SIGNAL(prjSettings(Project*)), SLOT(prjSettings(Project*)) );  

  {
    _actions = new ActionStorage();
    QAction* action;

    QIconSet isetUpdates;
    isetUpdates.setPixmap( getIconDir() + "WcUpdates-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetUpdates.setPixmap( getIconDir() + "WcUpdates-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("updates"), _q("U"), this );
    action->setIconSet( isetUpdates );
    action->setToggleAction(true);
    action->setStatusTip( _q("working copy option: show items with updates in the repository") );
    action->setOn(_model->getOptionStatusUpdates());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionStatusUpdates()) );
    _actions->addAction( ActionOptionStatusUpdates, action );
    action->addTo(tbOptStatus);

    QIconSet isetIgnored;
    isetIgnored.setPixmap( getIconDir() + "WcIgnored-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetIgnored.setPixmap( getIconDir() + "WcIgnored-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("ignored"), _q("I"), this );
    action->setIconSet( isetIgnored );
    action->setToggleAction(true);
    action->setStatusTip( _q("working copy option: show ignored items") );
    action->setOn(_model->getOptionStatusIgnored());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionStatusIgnored()) );
    _actions->addAction( ActionOptionStatusIgnored, action );
    action->addTo(tbOptStatus);

    QIconSet isetAll;
    isetAll.setPixmap( getIconDir() + "WcAll-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetAll.setPixmap( getIconDir() + "WcAll-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("all"), _q("A"), this );
    action->setIconSet( isetAll );
    action->setToggleAction(true);
    action->setStatusTip( _q("working copy option: show all items") );
    action->setOn(_model->getOptionStatusAll());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionStatusAll()) );
    _actions->addAction( ActionOptionStatusAll, action );
    action->addTo(tbOptStatus);

    QIconSet isetRecurse;
    isetRecurse.setPixmap( getIconDir() + "Recursive-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetRecurse.setPixmap( getIconDir() + "Recursive-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("recursive"), _q("E"), this );
    action->setIconSet( isetRecurse );
    action->setStatusTip( _q("working copy option: display recursive (flat)") );
    action->setToggleAction(true);
    action->setEnabled(true);
    action->setOn(_model->getOptionStatusRecurse());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionStatusRecurse()) );
    _actions->addAction( ActionOptionStatusRecurse, action );
    action->addTo(tbOptStatus);

    QIconSet isetForce;
    isetForce.setPixmap( getIconDir() + "Force-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetForce.setPixmap( getIconDir() + "Force-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("force"), _q("F"), this );
    action->setIconSet( isetForce );
    action->setStatusTip( _q("command option: run commands with force option") );
    action->setToggleAction(true);
    action->setEnabled(true);
    action->setOn(_model->getOptionCommandForce());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionCommandForce()) );
    _actions->addAction( ActionOptionCommandForce, action );
    action->addTo(tbOptCommands);

    QIconSet isetRecurse2;
    isetRecurse.setPixmap( getIconDir() + "Recursive-NormalOff.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::Off );
    isetRecurse.setPixmap( getIconDir() + "Recursive-NormalOn.png", QIconSet::Automatic, QIconSet::Normal, QIconSet::On );
    action = new QAction( _q("recursive"), _q("C"), this );
    action->setIconSet( isetRecurse );
    action->setStatusTip( _q("command option: run recursive") );
    action->setToggleAction(true);
    action->setEnabled(true);
    action->setOn(_model->getOptionCommandRecurse());
    connect( action, SIGNAL(activated()), SLOT(toggledOptionCommandRecurse()) );
    _actions->addAction( ActionOptionCommandRecurse, action );
    action->addTo(tbOptCommands);

    QIconSet isetReload;
    isetReload.setPixmap( getIconDir() + "Reload-Normal.png", QIconSet::Automatic, QIconSet::Normal );
    isetReload.setPixmap( getIconDir() + "Reload-Active.png", QIconSet::Automatic, QIconSet::Active );
    isetReload.setPixmap( getIconDir() + "Reload-Disabled.png", QIconSet::Automatic, QIconSet::Disabled );
    action = new QAction( _q("reload"), _q("R"), this );
    action->setIconSet( isetReload );
    action->setStatusTip( _q("command: reload the selected repository or working copy") );
    action->setEnabled(true); // \todo enable/disable based on lvi type
    connect( action, SIGNAL(activated()), _list, SLOT(reload()) );
    _actions->addAction( ActionCommandReload, action );
    action->addTo(tbCommands);

    tbOptStatus->addSeparator();
    tbOptCommands->addSeparator();
    //tbCommands->addSeparator();
  }

  QMenuBar* mb = menuBar();
  {
    QPopupMenu* m1  = new QPopupMenu( mb );
    mb->insertItem( _q("&File"), m1, pmFile );
    _fileMenu = m1;
    {
      m1->insertItem( _q("&Quit"),  qApp, SLOT(quit()), Qt::CTRL+Qt::Key_Q );
    }


    QAction* action;
    QPopupMenu* reload = new QPopupMenu(mb);
    mb->insertItem( _q("&View"), reload );
    
    action = _actions->getAction( ActionCommandReload );
    action->addTo(reload);
    
    
    mb->insertItem( _q("Pr&oject"),        _list->getProjectMenu() );
    mb->insertItem( _q("Re&pos Bookmark"), _list->getRepositoryFolderMenu() );
    mb->insertItem( _q("W&c Bookmark"),    _list->getWorkingCopyFolderMenu() );
    mb->insertItem( _q("&Repos"),          _list->getRepositoryFilesMenu() );
    mb->insertItem( _q("&Wc"),             _list->getWorkingCopyFilesMenu() );

    
    QPopupMenu* opt = new QPopupMenu(mb);
    //mb->insertItem( _q("Opt&ions"), opt );
    
    action = _actions->getAction( ActionOptionStatusUpdates );
    action->addTo(opt);
    opt->changeItem( opt->idAt(0), QIconSet(), action->menuText() );
    opt->setAccel( action->accel(), opt->idAt(0) );

    action = _actions->getAction( ActionOptionStatusIgnored );
    action->addTo(opt);
    opt->changeItem( opt->idAt(2), QIconSet(), action->menuText() );
    opt->setAccel( action->accel(), opt->idAt(2) );

    action = _actions->getAction( ActionOptionStatusAll );
    action->addTo(opt);
    opt->changeItem( opt->idAt(3), QIconSet(), action->menuText() );
    opt->setAccel( action->accel(), opt->idAt(3) );

    action = _actions->getAction( ActionOptionStatusRecurse );
    action->addTo(opt);
    opt->changeItem( opt->idAt(4), QIconSet(), action->menuText() );
    opt->setAccel( action->accel(), opt->idAt(4) );

    action = _actions->getAction( ActionOptionCommandForce );
    action->addTo(opt);
    opt->changeItem( opt->idAt(1), QIconSet(), action->menuText() );
    opt->setAccel( action->accel(), opt->idAt(1) );
   

    QPopupMenu* m1b = new QPopupMenu(mb);
    mb->insertItem( _q("&Tools"), m1b, pmTools );
    {
      m1b->insertItem( _q("&Settings"),  this, SLOT(settings()), Qt::CTRL+Qt::Key_S );
    }
    
    QPopupMenu* m2 = new QPopupMenu(mb);
    mb->insertItem( _q("&Help"), m2 );
    {
      m2->insertItem( _q("A&bout"),  this, SLOT(about()), Qt::CTRL+Qt::Key_B );
#ifdef ENABLE_MANUAL
      m2->insertItem( _q("&Manual"), this, SLOT(manual()), Qt::Key_F1 ); 
#endif // ENABLE_MANUAL
    }
  }
    
  // statusbar
  QStatusBar* sb = statusBar();
  {
    sb->setSizeGripEnabled(true);
    sb->message( _q("Ready"), 5000 );

    _model->setProgressCallback( new PostProgressCallback(this) );
  }

  // restore layout settings
  setGeometry( _config->getWindowRect( name(), QRect(100,100,800,500) ) );
  _split->setHandlePos( _config->getPosition(SplitKey) );
}

MainWindow::~MainWindow()
{
  _config->setWindowRect( name(), geometry() );
  _config->setPosition( SplitKey, _split->getHandlePos() );
  
  delete _actions;
}

bool MainWindow::showDockMenu( const QPoint& globalPos )
{
  // strange, we can't add anything to the default dock menu

  //QPopupMenu* dockMenu = createDockWindowMenu();

  QPopupMenu* menu = new QPopupMenu(this,_q("subcommander"));
  menu->setCheckable(true);
  //menu->insertItem( "tool bars", dockMenu );

  menu->insertItem( _q("show label"),  10 );
  menu->setItemChecked(10,usesTextLabel());

  menu->insertItem( _q("large icons"), 20 );
  menu->setItemChecked(20,usesBigPixmaps());

  int result = menu->exec(globalPos);

  switch( result )
  {
  case 10: {
      setUsesTextLabel( ! menu->isItemChecked(10) );
      break;
    }
  case 20: {
      setUsesBigPixmaps( ! menu->isItemChecked(20) );
      break;
    }
  }

  return true;
}

void MainWindow::about()
{
  AboutDialog* ab = new AboutDialog( this );

  ab->exec();

  removeChild(ab);
  delete ab;
}

void MainWindow::manual()
{
#ifdef ENABLE_MANUAL
  if( _assistant == NULL )
  {
    _assistant = new QAssistantClient( "/usr/bin", this );
    connect( _assistant, SIGNAL(error(const QString&)), SLOT(manualError(const QString&)));

    QStringList args;
    args << "-profile";
    args << QDir::convertSeparators("/usr/share/doc/subcommander/html/subcommander.adp");
    _assistant->setArguments(args);
  }

  _assistant->openAssistant();
#endif // ENABLE_MANUAL
}

void MainWindow::manualError(const QString& msg)
{
  msgCritical( _q("subcommander:manual"), msg, _q("&Ok") );
} 

void MainWindow::settings()
{
  settings("");
}

void MainWindow::settings( const QString& selected )
{
  SettingsDialog* sd = new SettingsDialog( _q("subcommander:settings"), this );
  SettingsInfo*   si;

  // projects root item
  si = new ProjectSettingsInfo( _q("Projects"), "toplevel", 0 );
  si->setSelected(selected);
  sd->addSettingsInfo( si );
  
  // single project settings
  ExternProviderImpl externProvider( _model );
  sd->addSettingsWidget( _q("Project Settings"), new ProjectSettingsWidget(&externProvider) );

  const Projects& prjs = _model->getProjects();
  for( Projects::const_iterator it = prjs.begin(); it != prjs.end(); it++ )
  {
    Project* prj = *it;

    si = new ProjectSettingsInfo( QString::fromUtf8(prj->getName()), _q("Project Settings"), prj );
    si->setSelected(selected);
    sd->addSettingsInfo( si, _q("Projects") );
  }

  int sortIndex = 0;

  // tools settings
  sd->addSettingsWidget( _q("Diff/Merge Settings"), new ToolsSettingsWidget(&externProvider) );
  sd->addSettingsInfo( new ToolsSettingsInfo( _q("Diff/Merge"), _q("Diff/Merge Settings"), _config, ++sortIndex ) );

  // we currently have only windows specific subversion settings
#ifdef _WIN32
  // subversion settings
  sd->addSettingsWidget( _q("Subversion Settings"), new SubversionSettingsWidget() );
  sd->addSettingsInfo( new SubversionSettingsInfo( _q("Subversion"), _q("Subversion Settings"), _config, ++sortIndex ) );
#endif // _WIN32

  // font settings
  sd->addSettingsWidget( _q("Font Settings"), new FontSettingsWidget() );
  sd->addSettingsInfo( new FontSettingsInfo( _q("Fonts"), _q("Font Settings"), _config, ++sortIndex ) );

  // color settings
  sd->addSettingsWidget( _q("Color Settings"), new ColorSettingsWidget() );
  sd->addSettingsInfo( new ColorSettingsInfo( _q("Colors"), _q("Color Settings"), _config, ++sortIndex ) );

  // debug settings
  sd->addSettingsWidget( _q("Debug Settings"), new DebugSettingsWidget() );
  sd->addSettingsInfo( new DebugSettingsInfo( _q("Debug"), _q("Debug Settings"), _config, ++sortIndex ) );

  sd->exec();

  removeChild(sd);
  delete sd;

  _model->saveSettings();

  // update project display if necessary...
  _list->refreshProjects();
}

void MainWindow::prjSettings( Project* prj )
{
  settings( QString::fromUtf8(prj->getName()) );
  _model->saveProject(prj);
}

void MainWindow::toggledOptionStatusAll()
{
  _model->setOptionStatusAll( _actions->getAction(ActionOptionStatusAll)->isOn() );
}

void MainWindow::toggledOptionStatusUpdates()
{
  _model->setOptionStatusUpdates( _actions->getAction(ActionOptionStatusUpdates)->isOn() );
}

void MainWindow::toggledOptionStatusIgnored()
{
  _model->setOptionStatusIgnored( _actions->getAction(ActionOptionStatusIgnored)->isOn() );
}

void MainWindow::toggledOptionStatusRecurse()
{
  _model->setOptionStatusRecurse( _actions->getAction(ActionOptionStatusRecurse)->isOn() );
}

void MainWindow::toggledOptionCommandForce()
{
  _model->setOptionCommandForce( _actions->getAction(ActionOptionCommandForce)->isOn() );
}

void MainWindow::toggledOptionCommandRecurse()
{
  _model->setOptionCommandRecurse( _actions->getAction(ActionOptionCommandRecurse)->isOn() );
}

void MainWindow::customEvent( QCustomEvent* ce )
{
  switch( ce->type() )
  {
  case ScDialogEvent:
    {
      DialogEvent*    de = (DialogEvent*)ce;
      DialogHandler*  h  = de->getHandler();

      h->run( this );
      break;
    }
  case ScExceptionEvent:
    {
      ExceptionEvent* ee = (ExceptionEvent*)ce;

      ErrorDialog* edlg = new ErrorDialog( ee->getError(), ee->getStack() );
      edlg->exec();
      delete edlg;

      break;
    }
  case ScCancelStartEvent:
    {
      log(dynamic_cast<CancelStartEvent*>(ce));
      break;
    }
  case ScCancelStopEvent:
    {
      log(dynamic_cast<CancelStopEvent*>(ce));
      break;
    }
  case ScLoggingEvent:
    {
      log( (LoggingEvent*)ce );
      break;
    }
  default:
    {
      printf( "MainWindow: unknown custom event type %d!\n", ce->type() );
    }
  }
}

bool MainWindow::eventFilter( QObject* t, QEvent* e )
{
  switch( e->type() )
  {
  case ScLoggingEvent:
    {
      log( (LoggingEvent*)e );
      return true;
    }
  case ScLviOnItemEvent:
    {
      LviOnItemEvent* oie = (LviOnItemEvent*)e;
      statusBar()->message( oie->getInfo(), 5000 );
      return true;
    }
  default:
    {
      return super::eventFilter(t,e);
    }
  }
}

void MainWindow::log( LoggingEvent* e )
{
  if( e->isStart() )
  {
    LogLvi* lvi = new LogLvi( _output, e->getData() );
    lvi->start();

    _output->ensureItemVisible( lvi );
  }
  else if( e->isRunning() )
  {
    LogLvi* lvi = _output->findItem( e->getId() );
    if( lvi )
    {
      lvi->add( e->getData() );
    }
  }
  else if( e->isStop() )
  {
    LogLvi* lvi = _output->findItem( e->getId() );
    if( lvi )
    {
      lvi->add( e->getData() );
      lvi->stop();
      lvi->repaint();
    }
  }
  else if( e->isTime() )
  {
    LogLvi* lvi = _output->findItem( e->getId() );
    if( lvi )
    {
      lvi->add( e->getData() );
    }
  }
}

void MainWindow::log( CancelStartEvent* e )
{
  LogLvi* lvi = _output->findItem(e->getId());
  if( lvi )
  {
    lvi->setCancelable(e->getCancel());
    lvi->setCancelNormal();
    _output->ensureItemVisible(lvi);
  }
}

void MainWindow::log( CancelStopEvent* e )
{
  LogLvi* lvi = _output->findItem(e->getId());
  if( lvi )
  {
    lvi->setCancelDisabled();
  }
}

void MainWindow::outputMenuRequested( QListViewItem*, const QPoint& pos, int )
{
  const long clearId = 1;

  QPopupMenu* m = new QPopupMenu(this);
  m->insertItem( _q("clear"), clearId );

  int result = m->exec( pos );

  if( result == clearId )
  {
    _output->clear();
  }

  delete m;
}

ScModel* MainWindow::getModel()
{
  return _model;
}

void MainWindow::runWcAutoUpdate()
{
  _list->getProjectFoldersWidget()->runWcAutoUpdate();
}
