#include <pcre.h>
#include <stdio.h>
#include <string.h>
#include <FL/Fl_Image.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Group.H>
#include "interface.h"
#include "load.h"
#include "html.h"
#include "http.h"
#include "url.h"

extern Fl_Preferences *CurComicCacheGroup;

inline void crop(char *src) {
	if (src[0] == '\'' || src[0] == '\"') {
		memmove(src, (src + 1), strlen(src) + 1);		// Chop off the ' or " at the start
	}
	if (src[strlen(src) - 1] == '\'' || src[strlen(src) - 1] == '\"') {
		src[strlen(src) - 1] = 0;	// Chop off the end
	}
}

void FindImages(const char *File, const char *Pattern, sURL *Source) {
	FILE *fptr;
	char tmp, buf[400], ImageFileName[400], *URL = NULL;
	int tagpos = 0, numimage = 0, length, lastupdate = 0;
	pcre *compile, *crap;
	pcre_extra *extra;
	const char *error;
	int erroffset;
	bool intag = false;
	int result = 0, ovector[30];

	Out("Opening \"%s\" to find images\n", File);

	if (!File) {
		DbgOut("NULL pointer passed to FindImages()... unable to retrieve HTML?\n");
		return;
	}

	DbgOut("Search Pattern: %s\n", Pattern);

	if (!(fptr = fopen(File, "r"))) {
		MOut("Unable to open %s!\n", File);
		return;
	}

	fseek(fptr, 0, SEEK_END);

	length = ftell(fptr);

	fseek(fptr, 0, SEEK_SET);

	compile = pcre_compile(Pattern, PCRE_CASELESS || PCRE_DOTALL, &error, &erroffset, NULL);

	if (!compile) {
		MOut("Error compiling pattern: %s\nError at offset: %d\nPattern: %s\n", error, erroffset, Pattern);
		fclose(fptr);
		return;
	}

	// Keenspot has the previous and next images as 1x1 images to make archive browsing faster, but we don't want them

	crap = pcre_compile("<img.*?WIDTH=\"1\".*?>", PCRE_CASELESS, &error, &erroffset, NULL);		

	extra = pcre_study(compile, 0, &error);

	Interface.BarColor(FL_YELLOW);
	Interface.BarRange(0.0, (float)length);
	Interface.BarValue(0.0);
	Interface.BarLabel("Image searching...");

	do {
		do {
			while (1) {
				fread(&tmp, 1, 1, fptr);
				if (feof(fptr)) {
					break;
				}
				if (tmp == '<') {
					intag = true;
					tagpos = 1;
					buf[0] = '<';
				} else {
					if (intag) {
						if (tagpos == 400) {
							intag = false;
							continue;
						}
						if ((tmp != '\n') && (tmp != '\r')) {		// Ignore newlines, they confuse the parser
							buf[tagpos++] = tmp;
						}
						if (tmp == '>') {
							intag = false;
							buf[tagpos] = '\0';
							break;
						}
					}
				}
			}
			if (feof(fptr)) {
				result = -20;
				Out("Done parsing\n");
				break;
			}
			if ((ftell(fptr) - 2048) > lastupdate) {
				Interface.BarValue((float)ftell(fptr));
				lastupdate = ftell(fptr);
			}
			result = pcre_exec(compile, extra, buf, strlen(buf), 0, 0, ovector, 30);
		} while (result < 0);
		if (result != -20) {
			if (pcre_exec(crap, NULL, buf, strlen(buf), 0, 0, 0, 0) < 0) {		// Filter out the images we don't want
				sURL *NewURL;
				if (result) { 
					pcre_copy_substring(buf, ovector, result, 1, ImageFileName, 399);
					result = 0;
				}
				free(URL);
				URL = NULL;
				if (strncmp(ImageFileName, "http://", 7)) {
					clonestr(MakeNewURL(ImageFileName, Source), &URL);
				} else {
					clonestr(ImageFileName, &URL);
				}
				Out("Found Image: %s\n", URL);
				NewURL = URLCreate(URL);
				CurComicCacheGroup->set(Fl_Preferences::Name("Image%d", numimage++), NewURL->File);
				if (!GrabHTTPLink(URL, URLUnbreak(Source))) {
					MOut("Error downloading %s\n", ImageFileName);		// We can ignore the return unless it fails, in which case warn the user
				}
				URLDelete(NewURL);
				Interface.BarColor(FL_YELLOW);
				Interface.BarRange(0.0, (float)length);
				Interface.BarValue((float)ftell(fptr));
				Interface.BarLabel("Image searching...");
			}
		} else {
			break;
		}
	} while (1);

	fclose(fptr);

	Interface.BarLabel("");
	Interface.BarValue(0.0);

	pcre_free(compile);
	pcre_free(crap);
	pcre_free(extra);

	free(URL);

	if(!numimage) {
		MOut("No images found! Incorrect URL, or incorrect search pattern?\n");
	}
	CurComicCacheGroup->set("numimage", numimage);
}

void FindLinks(const char *File, sURL *Source) {
	FILE *fptr;
	char *PrevPattern, *NextPattern;
	pcre *PrevCompile, *NextCompile, *ExtractLink;
	pcre_extra *PrevExtra, *NextExtra;
	const char *error;
	char tmp, buf[400], linkbuf[400] = "";
	int erroffset, tagpos = 0, length, lastupdate = 0;
	bool intag = false, FoundPrev = false, FoundNext = false;
	int result = 0, ovector[30];
	int linktype;

	if (!File) {
		DbgOut("NULL pointer passed to FindLinks(), incorrect URL?\n");
		return;
	}

	if (!CurComicG->entryExists("prevpattern") || !CurComicG->entryExists("nextpattern")) {
		Out("One or both link patterns are missing from the prefs file.\nNo link searching possible. Re-importing .comic may fix this.\nYou may also try the defaults if it's a Keenspot/Keenspace comic.\n");
		return;
	}

	{ char *Temp;
		CurComicG->get("linktype", Temp, "tag");
		if (!strcmp("tag", Temp)) {
			linktype = 0;
		} else if (!strcmp("text", Temp)) {
			linktype = 1;
		} else if (!strcmp("map", Temp)) {
			linktype = 2;
		} else {
			MOut("Unknown link type specified: %s\n", Temp);
			free(Temp);
			return;
		}
		free(Temp);
	}

	Out("Opening \"%s\" to find links\n", File);

	if (!(fptr = fopen(File, "r"))) {
		MOut("Error opening \"%s\"!\n", File);
		return;
	}

	fseek(fptr, 0, SEEK_END);

	length = ftell(fptr);

	fseek(fptr, 0, SEEK_SET);

	Interface.BarColor(FL_YELLOW);
	Interface.BarRange(0.0, (float)length);
	Interface.BarValue(0.0);
	Interface.BarLabel("Link searching...");

	CurComicG->get("prevpattern", PrevPattern, "");
	CurComicG->get("nextpattern", NextPattern, "");

	DbgOut("PrevPattern: %s\n", PrevPattern);
	DbgOut("NextPattern: %s\n", NextPattern);

	PrevCompile = pcre_compile(PrevPattern, PCRE_CASELESS || PCRE_DOTALL, &error, &erroffset, NULL);

	if (!PrevCompile) {
		MOut("Error compiling pattern: %s\nError at offset: %d\nPattern: %s\n", error, erroffset, PrevPattern);
		free(PrevPattern);
		free(NextPattern);
		return;
	}

	NextCompile = pcre_compile(NextPattern, PCRE_CASELESS || PCRE_DOTALL, &error, &erroffset, NULL);

	if (!NextCompile) {
		MOut("Error compiling pattern: %s\nError at offset: %d\nPattern: %s\n", error, erroffset, NextPattern);
		pcre_free(PrevCompile);
		free(PrevPattern);
		free(NextPattern);
		return;
	}

	ExtractLink = pcre_compile("<a .*href=(.*?)( .*>|>)", PCRE_CASELESS || PCRE_DOTALL, &error, &erroffset, NULL);		// That is such an ugly looking regexp

	if (!ExtractLink) {			// Shouldn't ever happen, but I might tweak this at some point, who knows...
		MOut("Error compiling Link pattern: %s\nError at offset: %d\n", error, erroffset);
		pcre_free(NextCompile);
		pcre_free(PrevCompile);
		free(PrevPattern);
		free(NextPattern);
		return;
	}

	PrevExtra = pcre_study(PrevCompile, 0, &error);
	NextExtra = pcre_study(NextCompile, 0, &error);

	if (linktype == 0) {		// Tag based link search
		while(!feof(fptr) && !(FoundPrev && FoundNext)) {
			while(!feof(fptr)) {		// Search for html tags
				fread(&tmp, 1, 1, fptr);
				if (tmp == '<') {
					intag = true;
				}
				if (intag) {
					if (tmp != '\n' && tmp != '\r') {
						buf[tagpos++] = tmp;
					}
					if (tmp == '>') {
						buf[tagpos] = '\0';
						tagpos = 0;
						intag = false;
						break;
					}
					if (tagpos == 399) {
						tagpos = 0;
						intag = false;
						continue;
					}
				}
			}
			if (Debug) {
				Interface.BarLabel("Link Search: %ld", ftell(fptr));
			}
			if ((ftell(fptr) - 2048) > lastupdate) {
				Interface.BarValue((float)ftell(fptr));
				lastupdate = ftell(fptr);
			}
			if (!feof(fptr)) {
				if (Debug) {
					DbgOut("Checking Tag against regexps:\n%s\n", buf);
				}
				if ((result = pcre_exec(ExtractLink, 0, buf, strlen(buf), 0, 0, ovector, 30)) > 0) {
					pcre_copy_substring(buf, ovector, result, 1, linkbuf, 399);
					crop(linkbuf);
				}
				result = -1;	// Reset result
				if(!FoundPrev) {	// "Prev" Link hasn't been found yet
					result = pcre_exec(PrevCompile, PrevExtra, buf, strlen(buf), 0, 0, ovector, 30);
					if (result >= 0) {
						DbgOut("Found Prev Tag Marker\n");
						DbgOut("Using: %s\n", linkbuf);
						FoundPrev = true;
						if (strlen(linkbuf)) {
							if (strncmp(linkbuf, "http://", 7)) {
								CurComicCacheGroup->set("prevlink", MakeNewURL(linkbuf, Source));
							} else {
								CurComicCacheGroup->set("prevlink", linkbuf);
							}
						}
						strcpy(linkbuf, ""); // Make sure we don't reuse the link
					}
				}
				if((result < 0) && !FoundNext) {	// "Prev" Link didn't match, or was already found, AND "Next" Link hasn't been found yet
					result = pcre_exec(NextCompile, NextExtra, buf, strlen(buf), 0, 0, ovector, 30);
					if (result >= 0) {
						DbgOut("Found Next Tag Marker\n");
						DbgOut("Using: %s\n", linkbuf);
						FoundNext = true;
						if (strlen(linkbuf)) {
							if (strncmp(linkbuf, "http://", 7)) {
								CurComicCacheGroup->set("nextlink", MakeNewURL(linkbuf, Source));
							} else {
								CurComicCacheGroup->set("nextlink", linkbuf);
							}
						}
						strcpy(linkbuf, ""); // Make sure we don't reuse the link
					}
				}
			} // if (!feof...
		} // while (!feof...
	} else if (linktype == 1) {		// Text-based link search
		while (!feof(fptr) && !(FoundPrev && FoundNext)) {
			while(!feof(fptr)) {		// Search for html tags
				fread(&tmp, 1, 1, fptr);
				if (tmp == '<') {
					intag = true;
				}
				if (intag) {
					buf[tagpos++] = tmp;
					if (tmp == '>') {
						buf[tagpos] = '\0';
						tagpos = 0;
						intag = false;
						break;
					}
					if (tagpos == 399) {
						tagpos = 0;
						intag = false;
						continue;
					}
				}
			} // while (!feof...
			if (!feof(fptr)) {
				if (Debug) {
					Interface.BarLabel("Link Search: %ld", ftell(fptr));
				}
				if ((ftell(fptr) - 2048) > lastupdate) {
					Interface.BarValue((float)ftell(fptr));
					lastupdate = ftell(fptr);
				}
				if ((result = pcre_exec(ExtractLink, 0, buf, strlen(buf), 0, 0, ovector, 30)) > 0) {
					bool endlink = false;
					pcre_copy_substring(buf, ovector, result, 1, linkbuf, 399);
					crop(linkbuf);
					do {
						fread(&tmp, 1, 1, fptr);
						buf[tagpos++] = tmp;
						if (tagpos == 399) {		// Don't overflow
							tagpos = 0;
							buf[0] = 0;
							endlink = true;
						}
						if (tmp == '>') {	// Maybe found the end?
							if (!strncmp((buf + tagpos - 4), "</a>", 4) || !strncmp((buf + tagpos - 4), "</A>", 4)) {	// Found the end
								buf[tagpos - 4] = 0;		// Cut off the end tag, only interested in what's between them
								tagpos = 0;
								endlink = true;
							}
						}						
					} while (!endlink);
					endlink = false;
					result = -1;	// Reset result
					if(Debug) {
						DbgOut("Checking against regexps: %s\n", buf);
					}
					if(!FoundPrev) {	// "Prev" Link hasn't been found yet
						result = pcre_exec(PrevCompile, PrevExtra, buf, strlen(buf), 0, 0, ovector, 30);
						if (result >= 0) {
							DbgOut("Found Prev Tag Marker\n");
							DbgOut("Using: %s\n", linkbuf);
							FoundPrev = true;
							if (strlen(linkbuf)) {
								if (strncmp(linkbuf, "http://", 7)) {
									CurComicCacheGroup->set("prevlink", MakeNewURL(linkbuf, Source));
								} else {
									CurComicCacheGroup->set("prevlink", linkbuf);
								}
							}
							strcpy(linkbuf, ""); // Make sure we don't reuse the link
						}
					}
					if((result < 0) && !FoundNext) {	// "Prev" Link didn't match, or was already found, AND "Next" Link hasn't been found yet
						result = pcre_exec(NextCompile, NextExtra, buf, strlen(buf), 0, 0, ovector, 30);
						if (result >= 0) {
							DbgOut("Found Next Tag Marker\n");
							DbgOut("Using: %s\n", linkbuf);
							FoundNext = true;
							if (strlen(linkbuf)) {
								if (strncmp(linkbuf, "http://", 7)) {
									CurComicCacheGroup->set("nextlink", MakeNewURL(linkbuf, Source));
								} else {
									CurComicCacheGroup->set("nextlink", linkbuf);
								}
							}
							strcpy(linkbuf, ""); // Make sure we don't reuse the link
						}
					}
				}	// if
			}
		} // while (!feof...
	} else if (linktype == 2) {	// Map-based link search
		while (!feof(fptr) && !(FoundPrev && FoundNext)) {
			while(!feof(fptr)) {		// Search for html tags
				fread(&tmp, 1, 1, fptr);
				if (tmp == '<') {
					intag = true;
				}
				if (intag) {
					buf[tagpos++] = tmp;
					if (tmp == '>') {
						buf[tagpos] = '\0';
						tagpos = 0;
						intag = false;
						break;
					}
					if (tagpos == 399) {
						tagpos = 0;
						intag = false;
						continue;
					}
				}
			} // while (!feof...
			if (!feof(fptr)) {
				result = 0;
				if (Debug) {
					Interface.BarLabel("Link Search: %ld", ftell(fptr));
				}
				if ((ftell(fptr) - 2048) > lastupdate) {
					Interface.BarValue((float)ftell(fptr));
					lastupdate = ftell(fptr);
				}
				if (!FoundPrev && ((result = pcre_exec(PrevCompile, PrevExtra, buf, strlen(buf), 0, 0, ovector, 30)) > 0)) {	// Match tag, and pull out link if applicable
					pcre_copy_substring(buf, ovector, result, 1, linkbuf, 399);
					DbgOut("Found Prev Tag Marker\n");
					DbgOut("Using: %s\n", linkbuf);
					FoundPrev = true;
					if (strlen(linkbuf)) {
						if (strncmp(linkbuf, "http://", 7)) {
							CurComicCacheGroup->set("prevlink", MakeNewURL(linkbuf, Source));
						} else {
							CurComicCacheGroup->set("prevlink", linkbuf);
						}
					}
					strcpy(linkbuf, ""); // Make sure we don't reuse the link
				}
				if (!FoundNext && ((result = pcre_exec(NextCompile, NextExtra, buf, strlen(buf), 0, 0, ovector, 30)) > 0)) {	// Match tag, and pull out link if applicable
					pcre_copy_substring(buf, ovector, result, 1, linkbuf, 399);
					DbgOut("Found Next Tag Marker\n");
					DbgOut("Using: %s\n", linkbuf);
					FoundNext = true;
					if (strlen(linkbuf)) {
						if (strncmp(linkbuf, "http://", 7)) {
							CurComicCacheGroup->set("nextlink", MakeNewURL(linkbuf, Source));
						} else {
							CurComicCacheGroup->set("nextlink", linkbuf);
						}
					}
					strcpy(linkbuf, ""); // Make sure we don't reuse the link
				}
			}
		} // while (!feof...
	}

	pcre_free(PrevCompile);
	pcre_free(NextCompile);
	pcre_free(PrevExtra);
	pcre_free(NextExtra);
	pcre_free(ExtractLink);
	free(PrevPattern);
	free(NextPattern);

	fclose(fptr);

	Out("Done Link Searching\n");
}
