// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// viewer.h - This class defines a generic graphical interface on which
//            stars can be plotted.  It is purely virtual and its methods
//            must be applied by descendant classes, which will act as
//            wrappers about classes from graphics libraries such as
//            KDE, GTK, etc.  The purpose of this setup is to keep the core
//            StarPlot classes (Star, StarArray) library-independent
//            while allowing them the ability to display themselves on a
//            display area.

#ifndef _VIEWER_H_
#define _VIEWER_H_

class StarViewer {

 protected:
  StarViewer() { } // should not be called directly!

 public:
  virtual ~StarViewer() { }

  // width(), height(): should return width / height of display area in pixels.
  virtual unsigned int width() const = 0;
  virtual unsigned int height() const = 0;

  // setcolor(): should set the drawing color to `color' in 0xRRGGBB format.
  virtual void setcolor(unsigned long color) = 0;

  // setfill(): if fillmode is false, should cause drawbox(), drawcircle(),
  //            and drawellipse() to produce outline shapes.  if fillmode
  //            is true, these functions should produce solid shapes.
  virtual void setfill(bool fillmode) = 0;

  // fill(): should fill the drawing area with the given color.
  virtual void fill(unsigned long color) = 0;

  // drawline(): should draw a line from pixel (x1,y1) to pixel (x2,y2).
  virtual void drawline(unsigned int x1, unsigned int y1,
			unsigned int x2, unsigned int y2) = 0;

  // drawbox(): should draw a box whose upper left corner is at (xul,yul)
  //            with width `width' and height `height'.
  virtual void drawbox(unsigned int xul, unsigned int yul,
		       unsigned int width, unsigned int height) = 0;

  // drawcircle(): should draw a circle centered at (xc,yc) with 
  //               radius `radius'.
  virtual void drawcircle(unsigned int xc, unsigned int yc, int radius) = 0;

  // drawstar(): should draw a filled circle centered at (xc,yc) with
  // 		 radius `radius'.  Also implements primitive anti-aliasing,
  // 		 so this is a separate function from drawcircle().
  virtual void drawstar(unsigned int xc, unsigned int yc, int radius) = 0;
  
  // drawellipse(): should draw an ellipse centered at (xc,yc) with
  //                axes parallel to the x and y axes, with semivertical axis
  //                `semiheight' and semihorizontal axis `semiwidth'.
  virtual void drawellipse(unsigned int xc, unsigned int yc,
			   int semiwidth, int semiheight) = 0;

  // drawarc(): should draw part of the outline of the ellipse which would
  //            be produced by drawellipse(xc,yc,semiwidth,semiheight).
  //            arc should go from `startangle' to `endangle' in a counter-
  //            clockwise direction, with zero in the positive x direction.
  //            both angles should be in radians.
  virtual void drawarc(unsigned int xc, unsigned int yc, // arc center
		       int semiwidth, int semiheight,    // angles in radians:
		       double startangle, double endangle) = 0;

  // drawtext(): should draw the text `text' starting at position (x,y), with
  //             font height in pixels `pointsize'.
  virtual void drawtext(const char *text, unsigned int x, unsigned int y,
			unsigned int pointsize) = 0;
  virtual void drawtext(char text, unsigned int x, unsigned int y,
			unsigned int pointsize) = 0;
} ;

#endif // _VIEWER_H_
