// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// mathops.cc
//
// Vector and SolidAngle class definitions, and a couple random functions

#include "mathops.h"

// sigdigits(): Converts to a given number of significant figures.
double sigdigits(double number, int numdigits)
{
  if (numdigits <= 0) return number; // no rounding
  if (number < 0.0) return -sigdigits(-number, numdigits);
  else if (number == 0.0) return 0.0;

  int exponent;
  double mantissa;

  if (number >= 1.0)
    exponent = (int)log10(number);
  else
    exponent = (int)log10(number) - 1;
  mantissa = pow(10.0, log10(number) - exponent);
  mantissa = (1.0 / pow(10.0, numdigits - 1)
	      * (int)(0.5 + mantissa * pow(10.0, numdigits - 1)));
  return mantissa * pow(10.0, exponent);
}


// For the Vector class ----------------------------------------

// Vector math ops: +-*/, magnitude, unit vector, dot and cross products.
//  Should be self-explanatory.
Vector &Vector::operator+=(Vector v)
{ x += v.x; y += v.y; z += v.z; return *this; }

Vector &Vector::operator-=(Vector v)
{ x -= v.x; y -= v.y; z -= v.z; return *this; }

Vector Vector::operator+(Vector v) const
{ return Vector(x+v.x, y+v.y, z+v.z); }

Vector Vector::operator-(Vector v) const
{ return Vector(x-v.x, y-v.y, z-v.z); }

Vector Vector::operator*(double c) const
{ return Vector(x*c, y*c, z*c); }

Vector operator*(double c, Vector v)
{ return v*c; }

Vector Vector::operator/(double c) const
{ return Vector(x/c, y/c, z/c); }

double Vector::magnitude() const
{ return sqrt( x*x + y*y + z*z ); }

// If object calling this is the zero vector, return another zero vector.
//  Else produce the expected result: a unit vector in the same direction.
Vector Vector::unitvector() const
{
  double radius = magnitude();
  if (radius > 0.0)
    return (*this)/radius;
  else
    return Vector(0,0,0);
}

double Vector::dot(Vector v) const
{ return x*v.x + y*v.y + x*v.z; }

Vector Vector::cross(Vector v) const
{ return Vector(y*v.z - z*v.y, z*v.x - x*v.z, x*v.y - y*v.x); }


// toSpherical(): returns the solid angle {theta,phi} defining the direction
//  of the current vector.  If object calling this is the zero vector,
//  return some random solid angle for definiteness (specifically, theta = 0,
//  phi = 0).  Else return the angular coordinates of the direction the
//  vector is pointing.  (If the vector is polar, return longitude phi = 0
//  for definiteness.)

SolidAngle Vector::toSpherical() const
{
  double theta = 0.0, phi = 0.0;
  Vector unit = unitvector();

  if (magnitude() == 0.0)
    return SolidAngle(0.0, 0.0); // for neatness' sake

  if (unit.z >= 1.0)
    theta = M_PI_2;
  else if (unit.z <= -1.0)
    theta = -M_PI_2;
  else {
    theta = asin(unit.z);
    if (!(unit.x == 0.0 && unit.y == 0.0))
      phi = atan2(unit.y, unit.x);
    if (phi < 0.0) // put phi in the range 0 to 2pi
      phi += (2 * M_PI);
  }
  return SolidAngle(phi, theta);
}


// toGalactic(), toCelestial(): return the conversions of the current vector
//  from celestial to galactic coordinates and vice versa, respectively.
//  We assume that the basis vectors are oriented as follows:
//
// Basis vector:     Celestial coords:      Galactic coords:
// -------------     -----------------      ----------------
//      +x             RA 0h, Dec 0        Lat 0, Lon 0 (Sgr A)
//      +y             RA 6h, Dec 0         Lat 0, Lon 90 deg
//      +z          North Celestial Pole   North Galactic Pole
//
// Note there is no safeguard to prevent applying (e.g.) toCelestial() to
// a vector which is already in celestial coordinates.  Preventing this is
// the responsibility of the main program.

Vector Vector::toGalactic() const
{
  Vector galactic;
  galactic.x =
    x * (COS_A * SIN_D * SIN_T - SIN_A * COS_T)
    + y * (SIN_A * SIN_D * SIN_T + COS_A * COS_T)
    - z * (COS_D * SIN_T);
  galactic.y =
    -x * (COS_A * SIN_D * COS_T + SIN_A * SIN_T)
    + y * (-SIN_A * SIN_D * COS_T + COS_A * SIN_T)
    + z * (COS_D * COS_T);
  galactic.z = 
    x * COS_A * COS_D + y * SIN_A * COS_D + z * SIN_D;
  return galactic;
}

Vector Vector::toCelestial() const
{
  Vector celestial;
  celestial.x =
    x * (COS_A * SIN_D * SIN_T - SIN_A * COS_T)
    - y * (COS_A * SIN_D * COS_T + SIN_A * SIN_T)
    + z * (COS_A * COS_D);
  celestial.y =
    x * (SIN_A * SIN_D * SIN_T + COS_A * COS_T)
    + y * (-SIN_A * SIN_D * COS_T + COS_A * SIN_T)
    + z * (SIN_A * COS_D);
  celestial.z = 
    -x * COS_D * SIN_T + y * COS_D * COS_T + z * SIN_D;
  return celestial;
}


// For the SolidAngle class ------------------------------------

// toCartesian(): returns a vector pointing in the same direction as the
//  current solid angle, with a magnitude specified by radius.

Vector SolidAngle::toCartesian(double radius) const
{
  double x, y, z;

  x = cos(theta) * cos(phi);
  y = cos(theta) * sin(phi);
  z = sin(theta);

  return radius * Vector(x, y, z);
}


// toGalactic(), toCelestial(): return conversions of the current solid angle
//  from celestial to galactic coordinates and vice versa, respectively.
//  These just call the appropriate vector conversion functions in Cartesian
//  coords, where such a transformation is much easier.  See warning
//  in comments above the vector conversion functions.

SolidAngle SolidAngle::toGalactic() const
{ return toCartesian(1.0).toGalactic().toSpherical(); }

SolidAngle SolidAngle::toCelestial() const
{ return toCartesian(1.0).toCelestial().toSpherical(); }

