/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-dav-server.h: DAV server support.
 *
 * Authors:
 *      Alex Graveley (alex@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <ctype.h>
#include <glib.h>
#include <string.h>
#include <tree.h>
#include <parser.h>
#include <xmlmemory.h>

#include "soup-dav.h"
#include "soup-dav-server.h"
#include "soup-message.h"

static GHashTable *dav_handlers;
static guint next_tag = 0; 

typedef struct {
	SoupDavServerHandlers handlers;
	gpointer              user_data;
	gchar                *path;
	guint                 table_tag;
	GHashTable           *null_locks;
	GHashTable           *known_locks;
} InternalHandlers;

#define __uri_exists(_ctx, _ih, _path) \
	(*(_ih)->handlers.uri_exists) (_ctx, _path, (_ih)->user_data)

#define __is_collection(_ctx, _ih, _path) \
	(*(_ih)->handlers.is_collection) (_ctx, _path, (_ih)->user_data)

#define __opt_describe_locks(_ctx, _ih, _path) \
	(*(_ih)->handlers.opt_describe_locks) (_ctx, _path, (_ih)->user_data)

#define __opt_lock(_ctx, _ih, _path, _lock) \
	(*(_ih)->handlers.opt_lock) (_ctx, _path, _lock, (_ih)->user_data)

#define __opt_unlock(_ctx, _ih, _path, _lock) \
	(*(_ih)->handlers.opt_unlock) (_ctx, _path, _lock, (_ih)->user_data)

#define __create_collection(_ctx, _ih, _path) \
	(*(_ih)->handlers.create_collection) (_ctx, _path, (_ih)->user_data)

#define __create_doc(_ctx, _ih, _path, _buf) \
	(*(_ih)->handlers.create_doc) (_ctx, _path, _buf, (_ih)->user_data)

#define __delete_collection(_ctx, _ih, _path) \
	(*(_ih)->handlers.delete_collection) (_ctx, _path, (_ih)->user_data)

#define __delete_doc(_ctx, _ih, _path) \
	(*(_ih)->handlers.delete_doc) (_ctx, _path, (_ih)->user_data)

#define __can_delete(_ctx, _ih, _path) \
	(*(_ih)->handlers.can_delete) (_ctx, _path, (_ih)->user_data)

#define __list_contents(_ctx, _ih, _path) \
	(*(_ih)->handlers.list_contents) (_ctx, _path, (_ih)->user_data)

#define __get_content(_ctx, _ih, _path, _buf) \
	(*(_ih)->handlers.get_content) (_ctx, _path, _buf, (_ih)->user_data)

#define __set_content(_ctx, _ih, _path, _buf) \
	(*(_ih)->handlers.set_content) (_ctx, _path, _buf, (_ih)->user_data)

#define __get_dav_prop(_ctx, _ih, _path, _prop) \
	(*(_ih)->handlers.get_dav_prop) (_ctx, _path, _prop, (_ih)->user_data)

#define __set_dav_prop(_ctx, _ih, _path, _prop) \
	(*(_ih)->handlers.set_dav_prop) (_ctx, _path, _prop, (_ih)->user_data)

#define __list_custom_props(_ctx, _ih, _path) \
	(*(_ih)->handlers.list_custom_props) (_ctx, _path, (_ih)->user_data)

#define __get_custom_prop(_ctx, _ih, _path, _prop) \
	(*(_ih)->handlers.get_custom_prop) (_ctx,_path,_prop,(_ih)->user_data)

#define __set_custom_prop(_ctx, _ih, _path, _prop) \
	(*(_ih)->handlers.set_custom_prop) (_ctx, _path, _prop,(_ih)->user_data)

#define __delete_custom_prop(_ctx, _ih, _path, _prop) \
	(*(_ih)->handlers.delete_custom_prop)(_ctx,_path,_prop,(_ih)->user_data)

#define __opt_move(_ctx, _ih, _path, _dest, _ow) \
	(*(_ih)->handlers.opt_move) (_ctx, _path, _dest, _ow, (_ih)->user_data)

#define __opt_copy(_ctx, _ih, _path, _dest, _ow) \
	(*(_ih)->handlers.opt_move) (_ctx, _path, _dest, _ow, (_ih)->user_data)

#define __opt_search(_ctx, _ih, _cnt, _dest) \
	(*(_ih)->handlers.opt_search) (_ctx, _path, _uri, (_ih)->user_data)

static gchar *
get_handler_for_path (const gchar *path, InternalHandlers **ih)
{
	gchar *dir, *mypath;

	if (!path) goto ERROR;

	mypath = g_strdup (path);
	dir = mypath;

	do {
		*ih = g_hash_table_lookup (dav_handlers, mypath);
		if (*ih) return mypath;

		dir = strrchr (mypath, '/');
		if (dir) *dir = '\0';
	} while (dir);

 ERROR:
	*ih = NULL;
	return NULL;
}

static inline gboolean
get_overwrite (const gchar *val)
{
	gboolean ret = TRUE;
	gchar *val_cpy;

	if (!val) return TRUE;

	val_cpy = g_strdup (val);
	g_strstrip (val_cpy);

	if (toupper (*val_cpy) == 'F') ret = FALSE;

	g_free (val_cpy);

	return ret;
}

static inline gint
get_depth (const gchar *val)
{
	if (!val) return -1;
	else if (*val == '0') return 0;
	else if (*val == '1') return 1;
	else if (!g_strcasecmp (val, "infinity")) return -1;

	return 0;
}

static inline void
set_response_error (SoupMessage *msg, gint errorcode, const gchar *reason)
{
	msg->response_code = errorcode;
	msg->response_phrase = g_strdup (reason);
}

#define make_dav_prop(_name) soup_dav_prop_new (_name, NULL, NULL, NULL)

static GSList *
list_dav_props (gboolean is_col)
{
	GSList *ret = NULL;
	SoupDavProp *prop;
	
	prop = make_dav_prop ("creationdate");
	ret = g_slist_prepend (ret, prop);

	prop = make_dav_prop ("displayname");
	ret = g_slist_prepend (ret, prop);

	prop = make_dav_prop ("resourcetype");
	ret = g_slist_prepend (ret, prop);

	prop = make_dav_prop ("supportedlock");
	ret = g_slist_prepend (ret, prop);

	if (!is_col) {
		prop = make_dav_prop ("getcontentlength");
		ret = g_slist_prepend (ret, prop);

		prop = make_dav_prop ("getlastmodified");
		ret = g_slist_prepend (ret, prop);

		prop = make_dav_prop ("getetag");
		ret = g_slist_prepend (ret, prop);

		prop = make_dav_prop ("getcontenttype");
		ret = g_slist_prepend (ret, prop);
	}

	return ret;
}

static gboolean
is_writable_dav_prop (const gchar *name)
{
	return TRUE;
}

static gboolean
munge_dav_prop (SoupDavContext   *ctx, 
		InternalHandlers *ih, 
		const gchar      *path,
		SoupDavProp      *prop) 
{
	/* FIXME: Implement for resource_type, supportedlock, 
	          getcontentlength. */
	return FALSE;
}

static gboolean
i_delete_ok (SoupDavContext   *ctx,
	     InternalHandlers *ih, 
	     const gchar      *path)
{
	GSList *subitems, *iter;
	gboolean fail = FALSE;

	if (__is_collection (ctx, ih, path)) {
		subitems = __list_contents (ctx, ih, path);

		for (iter = subitems; iter && !fail; iter = iter->next) {
			gchar *sub_path, *child = iter->data;

			sub_path = g_strconcat (path, "/", child);

			if (!i_delete_ok (ctx, ih, sub_path)) {
				for (; iter; iter = iter->next)
					g_free (iter->data);

				g_free (sub_path);
				g_slist_free (subitems);

				return FALSE;
			}				

			g_free (sub_path);
			g_free (child);
		}

		g_slist_free (subitems);
	}

	return __can_delete (ctx, ih, path);
}

static void 
i_delete (SoupDavContext   *ctx,
	  InternalHandlers *ih, 
	  const gchar      *path)
{
	GSList *subitems, *iter;

	if (!i_delete_ok (ctx, ih, path)) return;

	if (__is_collection (ctx, ih, path)) {
		subitems = __list_contents (ctx, ih, path);

		for (iter = subitems; iter; iter = iter->next) {
			gchar *sub_path, *child = iter->data;

			sub_path = g_strconcat (path, "/", child);

			i_delete (ctx, ih, sub_path);

			g_free (sub_path);
			g_free (child);
			iter = iter->next;
		}

		g_slist_free (subitems);

		__delete_collection (ctx, ih, path);
	} else
		__delete_doc (ctx, ih, path);
}

static void 
i_copy (SoupDavContext   *ctx, 
	InternalHandlers *src_ih, 
	InternalHandlers *dest_ih,
	const gchar      *src,
	const gchar      *dest,
	gint              depth)
{
	SoupDataBuffer buf;
	gboolean       is_col;
	GSList        *props, *iter, *subitems;

	/* FIXME: Support propertybehavior (keepalive | omit) */

	is_col = __is_collection (ctx, src_ih, src);

	if (is_col && depth != 0) {
		subitems = __list_contents (ctx, src_ih, src);

		for (iter = subitems; iter; iter = iter->next) {
			gchar *schild, *dchild, *child = iter->data;

			schild = g_strconcat (src, "/", child);
			dchild = g_strconcat (dest, "/", child);

			i_copy (ctx, src_ih, dest_ih, schild, dchild, depth);

			g_free (schild);
			g_free (dchild);
			g_free (child);
			iter = iter->next;
		}

		g_slist_free (subitems);
	} else {
		if (is_col) {
			if (!__create_collection (ctx, dest_ih, dest)) {
				// can't create error
				return;
			}
		} else {
			if (!__get_content (ctx, src_ih, src, &buf)) {
				// error can't get content
				return;
			}

			if (!__create_doc (ctx, dest_ih, dest, &buf)) {
				// can't create error
				return;
			}
		}

		props = list_dav_props (is_col);

		for (props = iter; iter; iter = iter->next) {
			SoupDavProp *prop = iter->data;

			if (!__get_dav_prop (ctx, src_ih, src, prop) &&
			    !munge_dav_prop (ctx, src_ih, src, prop)) {
				soup_dav_prop_free (prop);
				continue;
			}

			__set_dav_prop (ctx, dest_ih, dest, prop);

			soup_dav_prop_free (prop);
		}

		g_slist_free (props);

		props = __list_custom_props (ctx, src_ih, src);

		for (props = iter; iter; iter = iter->next) {
			SoupDavProp *prop = iter->data;

			__get_custom_prop (ctx, src_ih, src, prop);
			__set_custom_prop (ctx, dest_ih, dest, prop);

			soup_dav_prop_free (prop);
		}

		g_slist_free (props);
	}
}

static void
move_copy (SoupDavContext *ctx, InternalHandlers *src_ih, gboolean delete_src)
{
	gboolean overwrite, dest_exists;
	gint depth;
	const gchar *src, *dest, *owh;
	GSList *s_content, *d_content;
	InternalHandlers *dest_ih;

	src = ctx->path;

	depth = get_depth (soup_message_get_request_header (ctx->msg, "Depth"));
	owh = soup_message_get_request_header (ctx->msg, "Overwrite");
	overwrite = get_overwrite (owh);

	dest = soup_message_get_request_header (ctx->msg, "Destination");
	dest = get_handler_for_path (dest, &dest_ih);

	if (!dest_ih) {
		ctx->msg->response_code = 409;
		ctx->msg->response_phrase = g_strdup ("Conflict");
		return;
	}

	dest_exists = __uri_exists (ctx, dest_ih, dest);

	if (dest_exists) {
		if (!overwrite || !__can_delete (ctx, dest_ih, dest)) {
			ctx->msg->response_code = 412;
			ctx->msg->response_phrase = 
				g_strdup ("Precondition Failed");
			return;
		}

		i_delete (ctx, dest_ih, dest);
	}

	i_copy (ctx, src_ih, dest_ih, src, dest, depth);

	if (delete_src) 
		i_delete (ctx, src_ih, src);
}

static void 
dav_move (SoupDavContext *ctx, InternalHandlers *ih)
{
	if (ih->handlers.opt_move) {
		const gchar *dest, *owh;
		gboolean ow;

		dest = soup_message_get_request_header (ctx->msg,
							"Destination");
		owh = soup_message_get_request_header (ctx->msg, "Overwrite");
		ow = get_overwrite (owh);

		__opt_move (ctx, ih, ctx->path, dest, ow);

		return;
	}

	move_copy (ctx, ih, TRUE);
}

static void 
dav_copy (SoupDavContext *ctx, InternalHandlers *ih)
{
	if (ih->handlers.opt_copy) {
		const gchar *dest, *owh;
		gboolean ow;

		dest = soup_message_get_request_header (ctx->msg,
							"Destination");
		owh = soup_message_get_request_header (ctx->msg, "Overwrite");
		ow = get_overwrite (owh);

		__opt_copy (ctx, ih, ctx->path, dest, ow);

		return;
	}

	move_copy (ctx, ih, FALSE);
}

static void 
dav_delete (SoupDavContext *ctx, InternalHandlers *ih)
{
	i_delete (ctx, ih, ctx->path);
}

static gboolean
parse_propfind (SoupDavContext  *ctx,
		SoupDataBuffer  *buf, 
		GSList         **list, 
		gboolean        *show_content) 
{
        xmlDocPtr xml_doc;
        xmlNodePtr cur;
	GSList *propnames;

	LIBXML_TEST_VERSION;
        xmlKeepBlanksDefault (0);

        xml_doc = xmlParseMemory (buf->body, buf->length);
        if (xml_doc == NULL) {
		set_response_error (ctx->msg, 400, "Bad Request");
                return FALSE;
        }

	cur = xmlDocGetRootElement (xml_doc);
        if (cur == NULL) {
		set_response_error (ctx->msg, 400, "Bad Request");
                xmlFreeDoc (xml_doc);
                return FALSE;
        }

	if (g_strcasecmp (cur->name, "PROPFIND") != 0) {
		set_response_error (ctx->msg, 422, "Unprocessable Entity");
                xmlFreeDoc (xml_doc);
                return FALSE;
	}

	cur = cur->xmlChildrenNode;
	if (!cur) goto PARSE_ERROR;

	*list = NULL;

	if (!g_strcasecmp (cur->name, "PROP")) {
		cur = cur->xmlChildrenNode;
		if (!cur) goto PARSE_ERROR;
		
		for (; cur; cur = cur->next) {
			SoupDavProp *prop;
			gboolean is_dav;

			is_dav = g_strcasecmp (cur->ns->href, "DAV:") == 0;

			prop = 
				soup_dav_prop_new (
					cur->name, 
					is_dav ? NULL : cur->ns->prefix,
					is_dav ? NULL : cur->ns->href,
					NULL);

			*list = g_slist_append (*list, prop);
		}

		*show_content = TRUE;
	}
	else if (!g_strcasecmp (cur->name, "ALLPROP")) {
		*show_content = TRUE;
	}
	else if (!g_strcasecmp (cur->name, "PROPNAME")) {
		*show_content = FALSE;
	}
	else goto PARSE_ERROR;

	xmlFreeDoc (xml_doc);
	return TRUE;

 PARSE_ERROR:
	set_response_error (ctx->msg, 422, "Unprocessable Entity");
	xmlFreeDoc (xml_doc);
	return FALSE;
}

static void
i_propfind (SoupDavContext     *ctx, 
	    InternalHandlers   *ih, 
	    const gchar        *path, 
	    gint                depth,
	    GSList             *find_list,
	    gboolean            show_content,
	    SoupDavMultiStatus *ms)
{
	GSList *find_copy;
	SoupDavResponse *resp;
	gboolean is_col;
	gchar *href;

	// on error, use error for all following props

	is_col = __is_collection (ctx, ih, path);

	if (find_list)
		find_copy = soup_dav_prop_list_copy (find_list);
	else {
		find_copy = list_dav_props (is_col);
		find_copy = g_slist_concat (find_copy,
					    __list_custom_props (ctx, 
								 ih, 
								 path));
	}

	if (show_content) {
		GSList *iter;

		for (iter = find_copy; iter; iter = iter->next) {
			SoupDavProp *prop = iter->data;

			if (!prop->ns_uri)
				__get_dav_prop (ctx, ih, path, prop);
			else
				__get_custom_prop (ctx, ih, path, prop);
		}
	}

	href = soup_dav_context_make_href (ctx, path);
	resp = soup_dav_propstat_new (href, find_copy, NULL);
	g_free (href);

	soup_dav_mstat_add_response (ms, resp);

	if (depth == 0) return;

	if (is_col) {
		GSList *contents, *iter;

		contents = __list_contents (ctx, ih, path);

		for (iter = contents; iter; iter = iter->next) {
			gchar *path = iter->data;

			i_propfind (ctx, 
				    ih, 
				    path, 
				    depth == 1 ? 0 : depth, 
				    find_list,
				    show_content,
				    ms);

			g_free (path);
		}

		g_slist_free (contents);
	}
}

static void 
dav_propfind (SoupDavContext *ctx, InternalHandlers *ih)
{
	gint depth;
	SoupDavMultiStatus *ms;
	GSList *find_list;
	gboolean show_content;

	if (!__uri_exists (ctx, ih, ctx->path)) {
		// error
		return;
	}

	if (!parse_propfind (ctx, 
			     &ctx->msg->request, 
			     &find_list, 
			     &show_content)) {
		// error
		return;
	}

	depth = get_depth (soup_message_get_request_header (ctx->msg, "Depth"));

	ms = soup_dav_mstat_new (ctx->msg);

	i_propfind (ctx, ih, ctx->path, depth, find_list, show_content, ms);

	if (find_list)
		soup_dav_prop_list_free (find_list);

	soup_dav_mstat_serialize (ms, &ctx->msg->response);
	soup_dav_mstat_free (ms);
}

static gboolean
parse_proppatch (SoupDavContext  *ctx,
		 SoupDataBuffer  *buf, 
		 GSList        **list) 
{
        xmlDocPtr   xml_doc;
        xmlNodePtr  cur;
	GSList     *propnames;

	LIBXML_TEST_VERSION;
        xmlKeepBlanksDefault (0);

        xml_doc = xmlParseMemory (buf->body, buf->length);
        if (xml_doc == NULL) {
                //g_warning ("XML parse failed");
                return FALSE;
        }

	cur = xmlDocGetRootElement (xml_doc);
        if (cur == NULL) {
                //g_warning ("Couldn't get root element");
                xmlFreeDoc (xml_doc);
                return FALSE;
        }

	if (g_strcasecmp (cur->name, "PROPERTYUPDATE") != 0) {
                //g_warning ("Root element not propfind");
                xmlFreeDoc (xml_doc);
                return FALSE;
	}

	cur = cur->xmlChildrenNode;
	if (!cur) goto PARSE_ERROR;

	*list = NULL;

	for (; cur; cur = cur->next) {
		gboolean is_set;
		xmlNodePtr props;

		if (!g_strcasecmp (cur->name, "SET"))
			is_set = TRUE;
		else if (!g_strcasecmp (cur->name, "REMOVE"))
			is_set = FALSE;
		else goto PARSE_ERROR;

		props = cur->xmlChildrenNode;
		if (!props || g_strcasecmp (props->name, "PROP") != 0) {
			// error
			goto PARSE_ERROR;
		}

		props = props->xmlChildrenNode;
		if (!props) {
			// error
			goto PARSE_ERROR;
		}

		for (; props; props = props->next) {
			SoupDavProp *prop;
			gboolean     is_dav;

			is_dav = g_strcasecmp (props->ns->href, "DAV:") == 0;

			prop = soup_dav_prop_new (
					props->name, 
					is_dav ? NULL : props->ns->prefix,
					is_dav ? NULL : props->ns->href,
					NULL);

			if (is_set)
				prop->content = 
					xmlNodeListGetString (xml_doc, 
							      props, 
							      FALSE);

			*list = g_slist_append (*list, prop);
		}
	}

	xmlFreeDoc (xml_doc);
	return TRUE;

 PARSE_ERROR:
	set_response_error (ctx->msg, 422, "Unprocessable Entity");
	xmlFreeDoc (xml_doc);
	return FALSE;
}

static void 
dav_proppatch (SoupDavContext *ctx, InternalHandlers *ih)
{
	SoupDavMultiStatus *ms;
	GSList *list, *rollback_list, *iter;

	if (!__uri_exists (ctx, ih, ctx->path)) {
		// error
		return;
	}

	if (!parse_proppatch (ctx, &ctx->msg->request, &list)) {
		// error
		return;
	}

	ms = soup_dav_mstat_new (ctx->msg);

	for (iter = list; iter; iter = iter->next) {
		SoupDavProp *prop = iter->data, *backup;

		backup = soup_dav_prop_copy (prop);

		if (!prop->ns_uri)
			__get_dav_prop (ctx, ih, ctx->path, backup);
		else
			__get_custom_prop (ctx, ih, ctx->path, backup);

		if (prop->content) {
			if (!prop->ns_uri) {
				if (!__set_dav_prop (ctx, 
						     ih, 
						     ctx->path, 
						     prop)) { 
					// error
					goto ROLLBACK;
				}
			} else {
				if (!__set_custom_prop (ctx, 
							ih, 
							ctx->path, 
							prop)) { 
					// error
					goto ROLLBACK;
				}				
			}

			
		} else {
			if (!prop->ns_uri) {
				// error
				goto ROLLBACK;
			} else {
				if (!__delete_custom_prop (ctx, 
							   ih, 
							   ctx->path, 
							   prop)) {
					// error
					goto ROLLBACK;
				}
			}
		}

		rollback_list = g_slist_prepend (rollback_list, backup);

	ROLLBACK:
		soup_dav_prop_free (backup);

		for (iter = rollback_list; iter; iter = iter->next) {
			backup = iter->data;

			if (!backup->ns_uri)
				__set_dav_prop (ctx, ih, ctx->path, backup);
			else
				__set_custom_prop (ctx, ih, ctx->path, backup);
		}

		soup_dav_prop_list_free (rollback_list);

		break;
	}

	if (list) soup_dav_prop_list_free (list);

	soup_dav_mstat_serialize (ms, &ctx->msg->response);
	soup_dav_mstat_free (ms);

	// foreach prop
	//   switch op
	//   modify:
	//     if dav prop
	//       set dav prop*
	//   delete:
	//     if dav prop 
	//       error
	//     else
	//       if delete() == error
	//         error
}

static void 
dav_lock (SoupDavContext *ctx, InternalHandlers *ih)
{
	/* FIXME: Implement */

	if (ih->handlers.opt_lock)
		__opt_lock (ctx, ih, ctx->path, NULL);

	// if locked
	//   error
	// else 
	//   lock	
}

static void 
dav_unlock (SoupDavContext *ctx, InternalHandlers *ih)
{
	/* FIXME: Implement */

	if (ih->handlers.opt_unlock)
		__opt_unlock (ctx, ih, ctx->path, NULL);

	// if locked
	//   if auth match
	//     unlock
	//   else
	//     error
	// else
	//   error
}

static void 
dav_search (SoupDavContext *ctx, InternalHandlers *ih)
{
	/* FIXME: Implement */

	// parse sql tree
}

static void
dav_get (SoupDavContext *ctx, InternalHandlers *ih)
{
	if (!__is_collection (ctx, ih, ctx->path)) {
		if (!__get_content (ctx, ih, ctx->path, &ctx->msg->response)) {
			// error
			return;
		}

		ctx->msg->response_code = 200;
		ctx->msg->response_phrase = g_strdup ("OK");
	} else {
		// error
		return;
	}
}

static void
dav_put (SoupDavContext *ctx, InternalHandlers *ih)
{
	if (!__is_collection (ctx, ih, ctx->path)) {
		if (!__set_content (ctx, ih, ctx->path, &ctx->msg->request)) {
			// error
			return;
		}

		ctx->msg->response_code = 200;
		ctx->msg->response_phrase = g_strdup ("OK");
	} else {
		// error
		return;
	}
}

static void
dav_mkcol (SoupDavContext *ctx, InternalHandlers *ih)
{
	if (__uri_exists (ctx, ih, ctx->path)) {
		// error
		return;
	}

	__create_collection (ctx, ih, ctx->path);

	ctx->msg->response_code = 201;
	ctx->msg->response_phrase = g_strdup ("Created");
}

static gchar *
get_supported_methods (SoupDavContext   *ctx, 
		       InternalHandlers *ih, 
		       const gchar      *path)
{
	// FIXME: Implement

	return g_strdup ("LOCK, UNLOCK, GET, PUT, COPY, MOVE, MKCOL, DELETE, "
			 "PROPFIND, PROPPATCH, SEARCH");
}

static void
dav_options (SoupDavContext *ctx, InternalHandlers *ih)
{
	gchar *methods;

	if (!__uri_exists (ctx, ih, ctx->path)) {
		// error
		return;
	}

	methods = get_supported_methods (ctx, ih, ctx->path);

	soup_message_set_response_header (ctx->msg, "DAV", "1,2");
	soup_message_set_response_header (ctx->msg, "Allow", methods);

	g_free (methods);
}

static void
dav_request_handler (SoupMessage *msg, gpointer user_data)
{
	SoupDavContext    ctx;
	InternalHandlers *ih = user_data;
	const SoupUri    *uri;
	gchar            *path, *methods;

	uri = soup_context_get_uri (msg->context);
	path = get_handler_for_path (uri->path, &ih);

	if (!ih) {
		msg->response_code = 404;
		msg->response_phrase = g_strdup ("Not Found");
		return;
	}

	ctx.msg        = msg;
	ctx.path       = path;
	ctx.handler_id = ih->table_tag;
	ctx.auth       = NULL; // FIXME: Get auth

	switch (toupper (msg->method [0])) {
	case 'G':
		if (!g_strcasecmp (msg->method, "GET")) {
			ctx.action = SOUP_DAV_GET;
			dav_get (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;		
	case 'C':
		if (!g_strcasecmp (msg->method, "COPY")) {
			ctx.action = SOUP_DAV_COPY;
			dav_copy (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'M':
		if (!g_strcasecmp (msg->method, "MOVE")) {
			ctx.action = SOUP_DAV_MOVE;
			dav_move (&ctx, ih);
		} else if (!g_strcasecmp (msg->method, "MKCOL")) {
			ctx.action = SOUP_DAV_MKCOL;
			dav_mkcol (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'D':
		if (!g_strcasecmp (msg->method, "DELETE")) {
			ctx.action = SOUP_DAV_DELETE;
			dav_delete (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'P':
		if (!g_strcasecmp (msg->method, "PROPFIND")) {
			ctx.action = SOUP_DAV_PROPFIND;
			dav_propfind (&ctx, ih);
		} else if (!g_strcasecmp (msg->method, "PROPPATCH")) {
			ctx.action = SOUP_DAV_PROPPATCH;
			dav_proppatch (&ctx, ih);
		} else if (!g_strcasecmp (msg->method, "PUT")) {
			ctx.action = SOUP_DAV_PUT;
			dav_put (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'L':
		if (!g_strcasecmp (msg->method, "LOCK")) {
			ctx.action = SOUP_DAV_LOCK;
			dav_lock (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'U':
		if (!g_strcasecmp (msg->method, "UNLOCK")) {
			ctx.action = SOUP_DAV_UNLOCK;
			dav_unlock (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'S':
		if (!g_strcasecmp (msg->method, "SEARCH")) {
			ctx.action = SOUP_DAV_SEARCH;
			dav_search (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	case 'O':
		if (!g_strcasecmp (msg->method, "OPTIONS")) {
			ctx.action = SOUP_DAV_OPTIONS;
			dav_options (&ctx, ih);
		} else goto UNKNOWN_METHOD;
		break;
	default:
		goto UNKNOWN_METHOD;
		break;
	}

	g_free (path);
	return;

 UNKNOWN_METHOD:
	msg->response_code = 304;
	msg->response_phrase = g_strdup ("Method Not Implemented");

	methods = get_supported_methods (&ctx, ih, ctx.path);
	soup_message_set_response_header (msg, "Allow", methods);
	g_free (methods);

	g_free (path);
}

guint 
soup_dav_server_register (const gchar                 *path,
			  const SoupDavServerHandlers *handlers,
			  gpointer                     user_data)
{
	InternalHandlers *ih;

	g_return_val_if_fail (path != NULL, 0);
	g_return_val_if_fail (handlers != NULL, 0);

	ih = g_new0 (InternalHandlers, 1);
	ih->user_data = user_data;
	ih->table_tag = ++next_tag;
	ih->path = g_strdup (path);

	memcpy (&ih->handlers, handlers, sizeof (SoupDavServerHandlers));

	soup_server_register (path, dav_request_handler, ih);

	if (!dav_handlers) 
		dav_handlers = g_hash_table_new (g_direct_hash, g_direct_equal);

	g_hash_table_insert (dav_handlers, GINT_TO_POINTER (ih->table_tag), ih);

	return ih->table_tag;
}

const SoupDavServerHandlers *
soup_dav_server_get_table (guint table_tag)
{
	return dav_handlers ? 
		g_hash_table_lookup (dav_handlers, 
				     GINT_TO_POINTER (table_tag)) : NULL;
}

void 
soup_dav_server_unregister (guint table_tag)
{
	InternalHandlers *ih;

	if (dav_handlers) {
		ih = g_hash_table_lookup (dav_handlers, 
					  GINT_TO_POINTER (table_tag));
		g_hash_table_remove (dav_handlers, 
				     GINT_TO_POINTER (table_tag));

		g_free (ih->path);
		g_free (ih);

		if (g_hash_table_size (dav_handlers) == 0) {
			g_hash_table_destroy (dav_handlers);
			dav_handlers = NULL;
		}
	}
}

void 
soup_dav_server_clear_table (SoupDavServerHandlers *handlers)
{
	g_return_if_fail (handlers != NULL);

	memset (handlers, 0, sizeof (SoupDavServerHandlers));
}

gchar *
soup_dav_context_make_href (SoupDavContext *ctx,
			    const gchar    *path)
{
	SoupUri *suri;
	gchar *path_old, *uri;

	suri = (SoupUri *) soup_context_get_uri (ctx->msg->context);
	path_old = suri->path;

	suri->path = (gchar *) path;
	uri = soup_uri_to_string (suri, FALSE);
	suri->path = path_old;

	return uri;
}
