<?php
/**
 * Forwards_Driver_sql:: implements the Forwards_Driver API for SQL servers.
 *
 * $Horde: forwards/lib/Driver/sql.php,v 1.29 2006/02/17 16:51:01 jan Exp $
 *
 * Copyright 2001-2006 Ilya Krel <mail@krel.org>
 * Copyright 2001-2006 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file LICENSE for license information (ASL). If you
 * did not receive this file, see http://www.horde.org/licenses/asl.php.
 *
 * @author  Ilya Krel <mail@krel.org>
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_sql extends Forwards_Driver {

    /**
     * SQL connection object.
     *
     * @var DB
     */
    var $_db;

    /**
     * State of SQL connection.
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * Constructs a new Forwards_Driver_sql object.
     *
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver_sql($params = null)
    {
        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', 'sql');
        }
        parent::Forwards_Driver($params);
    }

    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean  True on success, false otherwise.
     */
    function _disconnect()
    {
        if ($this->_connected) {
            $this->_connected = false;
            return $this->_db->disconnect();
        }

        return true;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm, $password, $target, $keeplocal)
    {
        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();
        $local_target = $user . '@' . $realm;

        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params['table'] .
                 ' SET ' . $this->_params['forward'] . ' = ?' .
                 ' WHERE ' . $this->_params['user_col'] . ' = ?' .
                 ' AND ' . $this->_params['pass_col'] . ' = ?';
        $values = array('y', $myuser, md5($password));

        $query1 = 'UPDATE ' . $this->_params['table'];
        $values1 = array();
        if ($keeplocal === 'on') {
            if (!strstr($target, $local_target)) {
                $query1 .= ' SET ' . $this->_params['altemail'] . ' = ?';
                $values1[] = $target . ',' . $local_target;
            }
        } else {
            $query1 .= ' SET ' . $this->_params['altemail'] . ' = ?';
            $values1[] = $target;
        }
        $query1 .= ' WHERE ' . $this->_params['user_col'] . ' = ?' .
                   ' AND ' . $this->_params['pass_col'] . ' = ?';
        array_push($values1, $myuser, md5($password));

        // Execute the query.
        $result = $this->_db->query($query, $values);
        $result1 = $this->_db->query($query1, $values1);

        if (!is_a($result, 'PEAR_Error') && !is_a($result1, 'PEAR_Error')) {
            if ($result === DB_OK && $result1 === DB_OK) {
                $this->_disconnect();
                return true;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm, $password)
    {
        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();

        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params['table'] .
                 ' SET ' . $this->_params['forward'] . ' = ?,' .
                 ' ' . $this->_params['altemail'] . ' = ?' .
                 ' WHERE ' . $this->_params['user_col'] . ' = ?' .
                 ' AND ' . $this->_params['pass_col'] . ' = ?';
        $values = array('n', '', $myuser, md5($password));

        // Execute the query.
        $result = $this->_db->query($query, $values);

        if (!is_a($result, 'PEAR_Error')) {
            if ($result === DB_OK) {
                $this->_disconnect();
                return true;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // get current details
        $current_details = $this->_getUserDetails($myuser, $realm, $password);
        if ($current_details === false) {
            return false;
        }

        // check forwarding flag
        if ($current_details[$this->_params['forward']] === 'y' ||
             $current_details[$this->_params['forward']] === 'Y') {
            return 'Y';
        } else {
            return false;
        }
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm, $password)
    {
        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // Get current details.
        $current_details = $this->_getUserDetails($myuser, $realm, $password);
        if ($current_details === false) {
            return false;
        }

        // Check retain copy flag.
        if (substr_count($current_details[$this->_params['altemail']], $user . '@' . $realm)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false on error.
     */
    function currentTarget($user, $realm, $password)
    {
        $searchString = ",$user@$realm";
        $current_details = $this->_getUserDetails($user, $realm, $password);

        // Check current forwarding mail address.
        $targets = $current_details[$this->_params['altemail']];
        return str_replace($searchString, "", $current_details[$this->_params['altemail']]);
    }

    /**
     * Retrieves user details from the backend.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return array|boolean  Hash with user details, or false.
     */
    function _getUserDetails($user, $realm, $password)
    {
        static $row;

        // If we already have the details, return now.
        if (isset($row)) {
            return $row;
        }

        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();

        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // Build the SQL query.
        $query = 'SELECT * FROM ' . $this->_params['table'] .
                 ' WHERE ' . $this->_params['user_col'] . ' = ?' .
                 ' AND ' . $this->_params['pass_col'] . ' = ?';
        $values = array($myuser, md5($password));

        // Execute the query.
        $result = $this->_db->query($query, $values);

        if (!is_a($result, 'PEAR_Error')) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);

            if (is_array($row)) {
                $this->_disconnect();
                return $row;
            } else {
                $this->_disconnect();
                $result->free();
                return false;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @param string $user   Username.
     * @param string $realm  Realm name.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername($user, $realm)
    {
        if ($realm === 'default' ||
            $realm === '') {
            return $user;
        } else {
            return $user . '@' . $realm;
        }
    }

    /**
     * Connects to SQL server and logs in as user with privilege to change
     * password.
     *
     * @return boolean  True on success, false otherwise.
     */
    function _connect()
    {
        if (!$this->_connected) {
            Horde::assertDriverConfig($this->_params, 'server',
                array('phptype', 'table'),
                'Forwards SQL');

            if (!isset($this->_params['database'])) {
                $this->_params['database'] = '';
            }
            if (!isset($this->_params['username'])) {
                $this->_params['username'] = '';
            }
            if (!isset($this->_params['hostspec'])) {
                $this->_params['hostspec'] = '';
            }

            // Connect to SQL server using supplied parameters.
            include_once 'DB.php';
            $this->_db = &DB::connect($this->_params,
                                      array('persistent' => !empty($this->_params['persistent'])));
            if (is_a($this->_db, 'PEAR_Error')) {
                Horde::fatal(PEAR::raiseError(_("Unable to connect to SQL server.")), __FILE__, __LINE__);
            }

            // Set DB portability options.
            switch ($this->_db->phptype) {
            case 'mssql':
                $this->_db->setOption('portability', DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS | DB_PORTABILITY_RTRIM);
                break;
            default:
                $this->_db->setOption('portability', DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS);
            }

            $this->_connected = true;
        }

        return true;
    }

}
