<?php
/**
 * Forwards_Driver_sql:: implements the Forwards_Driver API for SQL servers.
 *
 * $Horde: forwards/lib/Driver/customsql.php,v 1.13 2006/02/17 16:51:01 jan Exp $
 *
 * Copyright 2004-2006 Kendrick Vargas <ken@hudat.com>
 *
 * See the enclosed file LICENSE for license information (ASL). If you
 * did not receive this file, see http://www.horde.org/licenses/asl.php.
 *
 * @author  Kendrick Vargas <ken@hudat.com>
 * @since   Forwards 3.0
 * @package Forwards
 */
class Forwards_Driver_customsql extends Forwards_Driver {

    /**
     * SQL connection object.
     *
     * @var DB
     */
    var $_db;

    /**
     * State of SQL connection.
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * Constructs a new Forwards_Driver_customsql object.
     *
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver_customsql($params = null)
    {
        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', 'sql');
        }
        parent::Forwards_Driver($params);
    }

    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean  True on success, false otherwise.
     */
    function _disconnect()
    {
        if ($this->_connected) {
            $this->_connected = false;
            return $this->_db->disconnect();
        }

        return true;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm, $password, $target, $keeplocal)
    {
        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();
        $keeplocal = ($keeplocal == 'on') ? 'Yes' : 'No';

        // Build the SQL query.
        $query = $this->_params['query_set'];
        $query = str_replace("\U", $this->_db->quote(Auth::getAuth()), $query);
        $query = str_replace("\T", $this->_db->quote($target), $query);
        $query = str_replace("\L", $this->_db->quote($keeplocal), $query);
        $query = str_replace("\P", $this->_db->quote($password), $query);

        // Execute the query.
        $result = $this->_db->query($query);

        if (!is_a($result, 'PEAR_Error')) {
            if ($result === DB_OK) {
                $this->_disconnect();
                return true;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm, $password)
    {
        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();

        // Build the SQL query.
        $query = $this->_params['query_disable'];
        $query = str_replace("\U", $this->_db->quote(Auth::getAuth()), $query);
        $query = str_replace("\P", $this->_db->quote($password), $query);

        // Execute the query.
        $result = $this->_db->query($query);

        if (!is_a($result, 'PEAR_Error')) {
            if ($result === DB_OK) {
                $this->_disconnect();
                return true;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // get current details
        $current_details = $this->_getUserDetails($myuser, $realm, $password);
        if ($current_details === false) {
            return false;
        } else {
            return 'Y';
        }

    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm, $password)
    {
        // Build username.
        $myuser = $this->_buildUsername($user, $realm);

        // Get current details.
        $current_details = $this->_getUserDetails($myuser, $realm, $password);
        if ($current_details === false) {
            return false;
        }

        // Check retain copy flag.
        if ($current_details[$this->_params['column_keeplocal']] == 'Y') {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false on error.
     */
    function currentTarget($user, $realm, $password)
    {
        $current_details = $this->_getUserDetails($user, $realm, $password);

        // Check current forwarding mail address.
        $target = $current_details[$this->_params['column_target']];
        return $target;
    }

    /**
     * Retrieves user details from the backend.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  Hash with user details, or false.
     */
    function _getUserDetails($user, $realm, $password)
    {
        // _connect() will die with Horde::fatal() upon failure.
        $this->_connect();

        // Build the SQL query.
        $query = $this->_params['query_select'];
        $query = str_replace("\U", $this->_db->quote(Auth::getAuth()), $query);
        $query = str_replace("\P", $this->_db->quote($password), $query);

        // Execute the query.
        $result = $this->_db->query($query);

        if (!is_a($result, 'PEAR_Error')) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);

            $this->_disconnect();
            if (is_array($row)) {
                return $row;
            } else {
                $result->free();
                return false;
            }
        }

        $this->_disconnect();
        return false;
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @param string $user   Username.
     * @param string $realm  Realm name.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername($user, $realm)
    {
        if ($realm === 'default' ||
            $realm === '') {
            return $user;
        } else {
            return $user . '@' . $realm;
        }
    }

    /**
     * Connects to SQL server and logs in as user with privilege to change
     * password.
     *
     * @return boolean  True on success, false otherwise.
     */
    function _connect()
    {
        if (!$this->_connected) {
            Horde::assertDriverConfig($this->_params, 'server',
                array('phptype', 'query_select', 'query_set', 'query_disable', 'column_target', 'column_keeplocal'),
                'Forwards SQL');

            if (!isset($this->_params['database'])) {
                $this->_params['database'] = '';
            }
            if (!isset($this->_params['username'])) {
                $this->_params['username'] = '';
            }
            if (!isset($this->_params['hostspec'])) {
                $this->_params['hostspec'] = '';
            }

            // Connect to SQL server using supplied parameters.
            include_once 'DB.php';
            $this->_db = &DB::connect($this->_params,
                                      array('persistent' => !empty($this->_params['persistent'])));
            if (is_a($this->_db, 'PEAR_Error')) {
                Horde::fatal(PEAR::raiseError(_("Unable to connect to SQL server.")), __FILE__, __LINE__);
            }

            // Set DB portability options.
            switch ($this->_db->phptype) {
            case 'mssql':
                $this->_db->setOption('portability', DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS | DB_PORTABILITY_RTRIM);
                break;
            default:
                $this->_db->setOption('portability', DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS);
            }

            $this->_connected = true;
        }

        return true;
    }

}
