/* Somaplayer - Copyright (C)>003 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_AO
#include "../../player.h"
#include "../../audio.h"
#include "../../other.h"

void
ao_quit (audio * output)
{

  if (output && output->data)
    ao_close ((ao_device *) output->data);
  else
    return;

  output->data = NULL;
}

void
ao_show (audio * output)
{
  ao_info *info;
  ao_device *tmp;

  if (!output || !output->data)
    return;

  tmp = (ao_device *) output->data;

  info = ao_driver_info (tmp->driver_id);

  msg (_("Audio Device:\n\tType: %s\n\tAuthor: %s\n\tComments: %s\n"),
       info->name, info->author, info->comment);
}

int
ao_init (audio * output, long rate, int channels, int bitrate)
{
  ao_sample_format format;
  int driver_id;
  ao_option *ao_options;
  ao_device *device = NULL;

  format.bits = bitrate;
  format.rate = rate;
  format.channels = channels;

  format.byte_format = AO_FMT_NATIVE;

  if (output->audio_type)
    {
      switch (output->audio_type)
	{
	case USE_NULL:
	  driver_id = ao_driver_id ("null");

	  if ((device = ao_open_live (driver_id, &format, NULL)) == NULL)
	    {
	      msg_error (_("Error opening NULL driver."));
	      return 1;
	    }

	  break;

	case USE_STDOUT:
	  ao_options = NULL;
	  driver_id = ao_driver_id ("raw");

	  ao_append_option (&ao_options, "byteorder", "little");

	  if ((device =
	       ao_open_file (driver_id, "-", 1, &format, ao_options)) == NULL)
	    {
	      msg_error (_("Error opening stdout."));
	      return 1;
	    }

	  break;
	}
    }

  if (device)
    {
      output->data = (void *) device;
      return 0;
    }

  if (!output->audio_type || output->audio_type == USE_AUDIO)
    {
      ao_options = NULL;
      driver_id = ao_default_driver_id ();

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_AUDIO)
	{
	  msg_error (_("Error opening OSS device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_AUDIO;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_ALSA)
    {
      char str[SIZE_BUFFER];
      char *c;
      ao_options = NULL;
      driver_id = ao_driver_id ("alsa");

      if (output->audio_dev)
	{
	  strncpy(str, output->audio_dev, SIZE_BUFFER);

	  if ((c = strchr (str, ':')) == NULL
	      || strlen (c + 1) < 1)
	    {
	      msg_error (_("Error format ALSA card: device specifiction."));
	      return 1;
	    }

	  *(c++) = '\0';

	  ao_append_option (&ao_options, "card", str);
	  ao_append_option (&ao_options, "dev", c);
	}

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_ALSA)
	{
	  msg_error (_("Error opening ALSA device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_ALSA;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_ALSA09)
    {
      ao_options = NULL;
      driver_id = ao_driver_id ("alsa09");

      if (output->audio_dev)
	ao_append_option (&ao_options, "dev", output->audio_dev);

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_ALSA09)
	{
	  msg_error (_("Error opening ALSA09 device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_ALSA09;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_OSS)
    {
      ao_options = NULL;
      driver_id = ao_driver_id ("oss");

      if (output->audio_dev)
	ao_append_option (&ao_options, "dsp", output->audio_dev);

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_OSS)
	{
	  msg_error (_("Error opening OSS device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_OSS;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_SUN)
    {
      ao_options = NULL;
      driver_id = ao_driver_id ("sun");

      if (output->audio_dev)
	ao_append_option (&ao_options, "dev", output->audio_dev);

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_SUN)
	{
	  msg_error (_("Error opening SUN device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_SUN;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_ESD)
    {
      ao_options = NULL;
      driver_id = ao_driver_id ("esd");

      if (output->audio_dev)
	ao_append_option (&ao_options, "host", output->audio_dev);

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_ESD)
	{
	  msg_error (_("Error opening ESD device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_ESD;
	  output->data = (void *) device;
	  return 0;
	}
    }

  if (!output->audio_type || output->audio_type == USE_ARTS)
    {
      ao_options = NULL;
      driver_id = ao_driver_id ("arts");

      device = ao_open_live (driver_id, &format, ao_options);

      if (!device && output->audio_type == USE_ARTS)
	{
	  msg_error (_("Error opening ARTS device."));
	  return 1;
	}

      if (device)
	{
	  output->audio_type = USE_ARTS;
	  output->data = (void *) device;
	  return 0;
	}
    }

  msg_error (_("Error opening audio device."));
  return 1;
}

int
ao_check (audio * output)
{
  int driver_id = 0;
  ao_info *info;

  if (!output->data)
    return 0;

  switch (output->audio_type)
    {
    case USE_NULL:
      driver_id = ao_driver_id ("null");
      break;

    case USE_STDOUT:
      driver_id = ao_driver_id ("raw");
      break;

    case USE_ESD:
      driver_id = ao_driver_id ("esd");
      break;

    case USE_ARTS:
      driver_id = ao_driver_id ("arts");
      break;

    case USE_ALSA:
      driver_id = ao_driver_id ("alsa");
      break;

    case USE_AUDIO:
      driver_id = ao_default_driver_id ();
      break;

    case USE_OSS:
      driver_id = ao_driver_id ("oss");
      break;

    case USE_SUN:
      driver_id = ao_driver_id ("sun");
      break;

    case USE_ALSA09:
      driver_id = ao_driver_id ("alsa09");
      break;
    }

  if (!(info = ao_driver_info (driver_id)))
    return 0;

  return (info->type == AO_TYPE_LIVE);
}

int
ao_write (audio * output, int channels, int bitrate, void *data, size_t length)
{
  ao_device *tmp = (ao_device *) output->data;

  if (!tmp)
    return 0;

  ao_play (tmp, data, length);
  return 0;
}
#endif
