/***************************************************************************
                          smb4kprint  -  description
                             -------------------
    begin                : Tue Mar 30 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qtimer.h>
#include <qfile.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kfileitem.h>

// system includes
#include <stdlib.h>

// application specific includes
#include "smb4kprint.h"
#include "smb4kdefs.h"
#include "smb4kglobal.h"

using namespace Smb4KGlobal;

bool retry = false;


Smb4KPrint::Smb4KPrint( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "PrintProcess" );
  m_proc->setUseShell( true );

  m_info = NULL;

  m_password_handler = new Smb4KPasswordHandler( this, "PrintPasswordHandler" );

  m_working = false;

  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,   SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,   SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ),
           this,   SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KPrint::~Smb4KPrint()
{
  abort();
}


/****************************************************************************
   Initializes the printing of a file. This function looks for the mime
   type and takes care of the further processing.
****************************************************************************/


bool Smb4KPrint::init( Smb4KPrintInfo *info )
{
  m_working = true;

  config()->setGroup( "Programs" );
  QString enscript = config()->readPathEntry( "enscript" );
  QString dvips = config()->readPathEntry( "dvips" );

  bool ok = true;

  if ( QFile::exists( info->path() ) )
  {
    // We want to determine the mimetype of the incoming
    // file. Let's use KFileItem for this, because KFileMetaInfo
    // does not work very reliable.
    KURL u;
    u.setPath( info->path() );

    KFileItem item = KFileItem( KFileItem::Unknown, KFileItem::Unknown, u, false );

    if ( QString::compare( item.mimetype(), "application/postscript" ) == 0 ||
         QString::compare( item.mimetype(), "application/pdf" ) == 0 ||
         item.mimetype().startsWith( "image" ) )
    {
      m_info = info;
      startPrinting();
    }
    else if ( QString::compare( item.mimetype(), "application/x-dvi" ) == 0 && !dvips.isEmpty() )
    {
      m_info = info;
      convertDVIToPS();
    }
    else if ( ( item.mimetype().startsWith( "text" ) || item.mimetype().startsWith( "message" ) ) && !enscript.isEmpty() )
    {
      m_info = info;
      convertTextToPS();
    }
    else
    {
      showCoreError( ERROR_MIMETYPE_NOT_SUPPORTED, item.mimetype() );
    }
  }
  else
  {
    showCoreError( ERROR_FILE_NOT_FOUND, info->path() );
    ok = false;
  }

  if ( !m_info )
  {
    delete info;
  }

  return ok;
}


/****************************************************************************
   Aborts the current process.
****************************************************************************/

void Smb4KPrint::abort()
{
  if ( m_proc->isRunning() )
  {
    m_proc->kill();
  }
}


/****************************************************************************
   Starts the actual printing.
****************************************************************************/

void Smb4KPrint::startPrinting()
{
  Smb4KAuthInfo *auth = m_password_handler->readAuth( m_info->workgroup(), m_info->host(), m_info->printer() );

  QString uri;

  // It seems that we must not quote the entries for the DEVICE_URI
  // environment variable. Printing will fail if you do it.

  if ( !m_info->workgroup().isEmpty() )
  {
    if ( !auth->user().isEmpty() )
    {
      uri = QString( "smb://%1:%2@%3/%4/%5" ).arg( auth->user(), auth->password() ).arg( m_info->workgroup(), m_info->host(), m_info->printer() );
    }
    else
    {
      uri = QString( "smb://%1/%2/%3" ).arg( m_info->workgroup(), m_info->host(), m_info->printer() );
    }
  }
  else
  {
    if ( !auth->user().isEmpty() )
    {
      uri = QString( "smb://%1:%2@%3/%4" ).arg( auth->user(), auth->password() ).arg( m_info->host(), m_info->printer() );
    }
    else
    {
      uri = QString( "smb://%1/%2" ).arg( m_info->host(), m_info->printer() );
    }
  }

  m_proc->setEnvironment( "DEVICE_URI", uri );

  // FIXME: If smbspool starts to use argv[1] and argv[5]
  // we have to implement them!
  QString command = QString( "smbspool 111 %1 \"Smb4K print job\" %2 \"\" %3" ).arg( getenv( "USER" ) ).arg( m_info->copies() ).arg( KProcess::quote( m_info->path() ) );

  delete auth;

  *m_proc << command;

  startProcess( Print );
}


/****************************************************************************
   Converts DVI files to PS using dvips.
****************************************************************************/

void Smb4KPrint::convertDVIToPS()
{
  QString path = m_info->path().section( "/", 0, -2 );
  QString file = m_info->path().section( "/", -1, -1 );

  *m_proc << "cd "+KProcess::quote( path )+" && dvips -P pdf -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( file );
  startProcess( Convert );
}


/****************************************************************************
   Converts text files to PS using enscript.
****************************************************************************/

void Smb4KPrint::convertTextToPS()
{
  *m_proc << "enscript -1 -B --ps-level=2 -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( m_info->path() );
  startProcess( Convert );
}


/****************************************************************************
   Starts any process of this class.
****************************************************************************/

void Smb4KPrint::startProcess( int internal_state )
{
  m_internal_state = internal_state;
  m_buffer = QString::null;

  if ( m_internal_state == Print )
  {
    emit state( PRINT_SEND_FILE );
  }
  else
  {
    emit state( PRINT_CONVERT_FILE );
  }

  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   If the KProcess ended, this function takes care of the further
   processing.
****************************************************************************/

void Smb4KPrint::endProcess()
{
  switch ( m_internal_state )
  {
    case Print:
      endPrintProcess();
      break;
    case Convert:
      endConversionProcess();
      break;
    default:
      break;
  }

  m_internal_state = Idle;
  m_proc->clearArguments();

  if ( !retry )
  {
    delete m_info;
    m_info = NULL;
  }

  m_working = false;
  emit state( PRINT_STOP );
}


/****************************************************************************
   Post-processing of any print job.
****************************************************************************/

void Smb4KPrint::endPrintProcess()
{
  if ( m_buffer.contains( "NT_STATUS", true ) != 0 )
  {
    if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED" ) != 0 || m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0 )
    {
      int state = Smb4KPasswordHandler::None;

      if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED" ) != 0 )
      {
        state = Smb4KPasswordHandler::AccessDenied;
      }
      else if (m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0  )
      {
        state = Smb4KPasswordHandler::LogonFailure;
      }

      if ( m_password_handler->askpass( m_info->workgroup(), m_info->host(), m_info->printer(), state ) )
      {
        retry = true;
        QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
      }
    }
    else
    {
      showCoreError( ERROR_PRINTING, m_buffer );

      // Clean up:
      QFile::remove( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
    }
  }
  else
  {
    // Clean up:
    QFile::remove( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
  }
}


/****************************************************************************
   Ends the conversion process.
****************************************************************************/

void Smb4KPrint::endConversionProcess()
{
  if ( m_buffer.contains( "command not found", true ) != 0 )
  {
    QString missing = m_buffer.section( ":", -2, -2 ).section( ":", -1, -1 ).stripWhiteSpace();

    showCoreError( ERROR_COMMAND_NOT_FOUND, missing );
  }
  else
  {
    // Set the URL to the newly created PS file:
    m_info->setPath( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );

    retry = true;
    QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KPrint::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KPrint::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );

  if ( m_buffer.contains( "NT_STATUS" ) != 0 )
  {
    abort();
  }
}


void Smb4KPrint::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KPrint::slotRetry()
{
  retry = false;
  startPrinting();
}

#include "smb4kprint.moc"
