/***************************************************************************
    smb4kbookmarkhandler  -  This class handles the bookmarks.
                             -------------------
    begin                : Fr Jan 9 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qfile.h>

// KDE includes
#include <kstandarddirs.h>
#include <kdebug.h>

// application specific includes
#include "smb4kbookmarkhandler.h"
#include "smb4kdefs.h"
#include "smb4kglobal.h"
using namespace Smb4KGlobal;


Smb4KBookmarkHandler::Smb4KBookmarkHandler( QObject *parent, const char *name ) : QObject( parent, name )
{
  // First we need the directory.
  KStandardDirs *stddir = new KStandardDirs();
  QString dir = locateLocal( "data", "smb4k", KGlobal::instance() );

  if ( !stddir->exists( dir ) )
  {
    stddir->makeDir( dir );
  }

  delete stddir;

  loadBookmarks();
}


Smb4KBookmarkHandler::~Smb4KBookmarkHandler()
{
  for ( QValueList<Smb4KBookmark *>::Iterator it = m_bookmarks.begin(); it != m_bookmarks.end(); ++it )
  {
    delete *it;
  }

  m_bookmarks.clear();
}


/****************************************************************************
   Writes the bookmark to the bookmark file, too.
****************************************************************************/

void Smb4KBookmarkHandler::addBookmark( Smb4KBookmark *bookmark )
{
  if ( QString::compare( bookmark->type(), "Printer" ) == 0 )
  {
    showCoreError( ERROR_WRONG_BOOKMARK_TYPE );
    return;
  }

  if ( QString::compare( bookmark->share(), "homes" ) == 0 )
  {
    QString share = specifyUser( bookmark->host() );

    bookmark->setShareName( share );
  }

  for ( QValueList<Smb4KBookmark *>::Iterator it = m_bookmarks.begin(); it != m_bookmarks.end(); ++it )
  {
    if ( QString::compare( (*it)->bookmark().lower(), bookmark->bookmark().lower() ) == 0 &&
         QString::compare( (*it)->workgroup().lower(), bookmark->workgroup().lower() ) == 0 &&
         QString::compare( (*it)->ip(), bookmark->ip() ) == 0 )
    {
      // already exists
      return;
    }
    else if ( QString::compare( (*it)->bookmark().lower(), bookmark->bookmark().lower() ) == 0 &&
              (QString::compare( (*it)->workgroup().lower(), bookmark->workgroup().lower() ) != 0 ||
               QString::compare( (*it)->ip(), bookmark->ip() ) != 0) )
    {
      // remove obsolete bookmark entry
      delete *it;
      *it = NULL;
    }
  }

  m_bookmarks.remove( NULL );

  m_bookmarks.append( bookmark );
  writeBookmarkList( m_bookmarks );
}


/****************************************************************************
   Writes a new list of bookmarks to the bookmark file.
****************************************************************************/

void Smb4KBookmarkHandler::writeBookmarkList( const QValueList<Smb4KBookmark *> &list )
{
  if ( list != m_bookmarks )
  {
    for ( QValueListIterator<Smb4KBookmark *> it = m_bookmarks.begin(); it != m_bookmarks.end(); ++it )
    {
      delete *it;
    }

    m_bookmarks.clear();

    m_bookmarks = list;
  }

  QFile file( locateLocal( "data", "smb4k/bookmarks", KGlobal::instance() ) );

  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    for ( QValueListConstIterator<Smb4KBookmark *> it = m_bookmarks.begin(); it != m_bookmarks.end(); ++it )
    {
      ts << (*it)->host() << "," << (*it)->share() << "," << (*it)->workgroup() << "," << (*it)->ip() << endl;
    }

    file.close();
  }
  else
  {
    showCoreError( ERROR_WRITING_FILE, QDir::currentDirPath()+"/"+file.name() );
    return;
  }

  emit bookmarksUpdated();
}


/****************************************************************************
   Load the bookmarks.
****************************************************************************/

void Smb4KBookmarkHandler::loadBookmarks()
{
  QFile file( locateLocal( "data", "smb4k/bookmarks", KGlobal::instance() ) );

  QStringList contents;

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    contents = QStringList::split( '\n', ts.read(), false );

    file.close();

    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
    {
      if ( (*it).startsWith( "#" ) || (*it).startsWith( "[" ) ||
         QString::compare( (*it).stripWhiteSpace(), QString::null ) == 0 )
      {
        continue;
      }
      else
      {
        // Load old bookmark entries (prior to version 0.7.0)
        if ( ((*it).stripWhiteSpace())[0].isDigit() )
        {
          QString bookmark = (*it).section( "=", 1, -1 ).stripWhiteSpace();
          m_bookmarks.append( new Smb4KBookmark( bookmark.section( "/", 2, 2 ).stripWhiteSpace(), bookmark.section( "/", 3, 3 ).stripWhiteSpace(), QString::null, QString::null, "Disk" ) );
        }
        else
        {
          QString host = (*it).section( ",", 0, 0 ).stripWhiteSpace();
          QString share = (*it).section( ",", 1, 1 ).stripWhiteSpace();
          QString workgroup = (*it).section( ",", 2, 2 ).stripWhiteSpace();
          QString ip = (*it).section( ",", 3, 3 ).stripWhiteSpace();

          m_bookmarks.append( new Smb4KBookmark( host, share, workgroup, ip, "Disk" ) );
        }
      }
    }

    emit bookmarksUpdated();
  }
}


/****************************************************************************
   Return a bookmark object from the list
****************************************************************************/

Smb4KBookmark *Smb4KBookmarkHandler::getBookmark( const QString &bookmark )
{
  QValueListConstIterator<Smb4KBookmark *> it;

  for ( it = m_bookmarks.begin(); it != m_bookmarks.end(); ++it )
  {
    if ( QString::compare( (*it)->bookmark().lower(), bookmark.lower() ) == 0 )
    {
      break;
    }
  }

  return it != m_bookmarks.end() ? *it : NULL;
}

#include "smb4kbookmarkhandler.moc"
