
/*
 * slmon
 *
 * Copyright (C) 2000 Krzysztof Luks <m00se@iq.pl>
 *
 * This program is based on WMMon.app.
 *
 * WMMon.app is copyright (c) 1997, 1998 by Martijn Pieterse and
 * Antoine Nulle.
 *
 * This program is distributed under the GPL license. For details see
 * COPYING text.
 *
 * Author: Krzysztof Luks <m00se@iq.pl>
 *
 */

#include "defines.h"
#include "stat.h"
#include "draw.h"
#include "signal_handling.h"

#ifdef HAVE_SLANG
int keyboard(void);
void change_mode(void);
void change_mem(void);
void change_cpu(void);
void increase(int what);
void decrease(int what);
#endif

void print_version(void);
#ifdef HAVE_POPT
void set_value(char **name, char *color, char **values);
int count_elements(char **array);
#elif defined HAVE_GETOPT
void getopt_help(void);
#endif

int main(int argc, char **argv)
{
    int j, arg;
    extern int cpus, cur_cpu;
#ifdef HAVE_POPT
    char *conffile, *opt_arg;
    poptContext opt_con;
#elif defined HAVE_GETOPT
    int c;
#endif

#ifdef HAVE_GTOP
    struct last l[GLIBTOP_NCPU];
#else
    struct last l[4];		/* FIXME: Temporary hack :( */
#endif

    /* FIXME ARGH! */
    static char *capt[] =
	{ "Total CPU usage", "CPU0 usage", "CPU1 usage", "CPU2 usage",
	"CPU3 usage", "CPU4 usage", "CPU5 usage", "CPU6 usage",
	"CPU7 usage"
    };

    /* Variables initialization */
    cur_cpu = 0;
    mem = 2;
    gauge_len = 30;

#ifdef HAVE_SLANG
    pos = 0;
    rspace = 3;
    cspace = 7;
    redraw = 1;
    mode = MODE_G;
    do_bold = 0;		/* Disable bold by default */
    height = 1000;		/* FIXME is there a better way? */

    col_norm = "green";
    col_valu = "yellow";
    col_stat = "yellow";
    col_stba = "blue";
    col_back = "default";
    col_gra1 = "blue";
    col_gra2 = "brightblue";
    col_gra3 = "white";
#endif

#ifdef HAVE_POPT
    conffile =
	calloc(strlen(getenv("HOME")) + strlen("/.slmonrc") + 1,
	       sizeof(char));
    strcpy(conffile, getenv("HOME"));
    strcat(conffile, "/.slmonrc");

    opt_con =
	poptGetContext("slmon", argc, (const char **) argv, optionsTable,
		       0);
    poptReadConfigFile(opt_con, conffile);

    while ((arg = poptGetNextOpt(opt_con)) > 0) {
	opt_arg = (char *) poptGetOptArg(opt_con);

	switch (arg) {
	case 'b':
	    mem = 0;
	    break;
	case 'k':
	    mem = 1;
	    break;
	case 'm':
	    mem = 2;
	    break;
#ifdef HAVE_SLANG
	case 'g':
	    mode = MODE_G;
	    break;
	case 'h':
	    mode = MODE_H;
	    break;
	case 'B':
	    do_bold = 1;
	    break;
	case COL_STAT:
	    col_stat = opt_arg;
	    break;
	case COL_NORM:
	    set_value(&col_norm, opt_arg, slangColors);
	    break;
	case COL_VALU:
	    set_value(&col_valu, opt_arg, slangColors);
	    break;
	case COL_BACK:
	    set_value(&col_back, opt_arg, slangColors);
	    break;
	case COL_STBA:
	    set_value(&col_stba, opt_arg, slangColors);
	    break;
	case COL_GRA1:
	    set_value(&col_gra1, opt_arg, slangColors);
	    break;
	case COL_GRA2:
	    set_value(&col_gra2, opt_arg, slangColors);
	    break;
	case COL_GRA3:
	    set_value(&col_gra3, opt_arg, slangColors);
	    break;
#endif
	default:
	    fprintf(stderr, "Error in options parsing!\n");
	    cleanup(-1);
	}
    }
#elif defined HAVE_GETOPT
    while ((c = getopt(argc, argv, "?hvbkmBGHE:")) != -1)
	switch (c) {
	case 'h':
		getopt_help();
		cleanup(0);
	    break;
	case '?':
	    getopt_help();
	    cleanup(0);
	case 'v':
	    version = 1;
	    break;
	case 'b':
	    mem = 0;
	    break;
	case 'k':
	    mem = 1;
	    break;
	case 'm':
	    mem = 2;
	    break;
	case 'B':
	    do_bold = 1;
	    break;
	case 'G':
	    mode = MODE_G;
	    break;
	case 'H':
	    mode = MODE_H;
	    break;
	case 'E':
	    height = atoi(optarg);
	    break;
	default:
	    fprintf(stderr, "Wrong option!\n");
	    cleanup(-1);
	}
#endif
    if (version) {
	print_version();
	cleanup(0);
    }
#ifdef HAVE_SLANG
    init_slang();		/* S-Lang initialisation, blablabla */
    init_signals();
    SLsmg_set_color(6);
    SLsmg_cls();
#endif

#ifdef HAVE_GTOP
    glibtop_init();		/* libgtop initialisation */
    cpus = glibtop_global_server->ncpu;
#elif defined HAVE_LINUXPROC
    cpus = count_cpus();
#endif

    /* Main loop */
#ifdef HAVE_SLANG
    do {
	draw_status();
	if (mode == MODE_H) {
	    draw_histogram_value(cpu_calc(cur_cpu, &l[cur_cpu]), pos);
	    if (redraw == 1) {
		draw_histogram(capt[cur_cpu]);
		redraw = 0;
	    }
	    if (++pos >= SLtt_Screen_Cols - (2 * cspace) - 1) {
		clear_histogram();
		pos = 0;
	    }
	} else if (mode == MODE_G) {
	    SLsmg_gotorc(2, 0);
	    SLsmg_set_color(7);
	    SLsmg_forward(5);	/* 5 == "xxx% " */
	    SLsmg_printf("0%%");
	    SLsmg_forward(gauge_len - 4);	/* 4 == "100%%" */
	    SLsmg_printf("100%%");
	    SLsmg_set_color(6);
	    SLsmg_printf(" Scale");
	    for (j = 0; j <= cpus; j++) {
		draw_gauge(3 + j, 0, cpu_calc(j, &l[j]), capt[j]);
	    }
	    print_mem(5 + j, 0);
	    print_ldavg(10 + j, 0);
	}
	print_keys(mode);
	SLsmg_refresh();
	usleep(490000);		/* This should be: 0.5s - SLang_input_pending time */
    }
    while (!keyboard());
#else
    while (1) {
	for (j = 0; j <= cpus; j++) {
	    draw_gauge(0, 0, cpu_calc(j, &l[j]), capt[cur_cpu]);
	}
	print_mem(0, 0);
	print_ldavg(0, 0);
	printf("\n\n");
	usleep(2500000);
    }
#endif

    cleanup(0);
}

#ifdef HAVE_SLANG
int keyboard(void)
{
    int ch;

    if (SLang_input_pending(-1)) {
	if ((ch = SLang_getkey()) == 'q') {
	    return 1;
	} else if (ch == 'm') {
	    change_mode();
	} else if (ch == 'u') {
	    change_mem();
	} else if (ch == 'h' && mode == MODE_H) {
	    change_cpu();
	} else if (ch == '=') {
	    increase(0);
	} else if (ch == '-') {
	    decrease(0);
	} else if (ch == '+') {
	    increase(1);
	} else if (ch == '_') {
	    decrease(1);
	}
	SLang_flush_input();
    }
    return 0;
}

void change_mode(void)
{
    mode++;
    mode %= 2;
    pos = 0;
    SLsmg_cls();
    redraw = 1;
}

/* FIXME: need to set some sane limits */

void increase(int what)
{
    if (mode == MODE_G)
	gauge_len++;
    else if (mode == MODE_H && what == 0)
	rspace++;
    else if (mode == MODE_H && what == 1)
	cspace++;
    pos = 0;
    redraw = 1;
    SLsmg_cls();
}

void decrease(int what)
{
    if (mode == MODE_G) {
	gauge_len--;
	if (gauge_len < 1)
	    gauge_len = 1;
    } else if (mode == MODE_H && what == 0) {
	rspace--;
	if (rspace < 2)
	    rspace = 2;
    } else if (mode == MODE_H && what == 1) {
	cspace--;
	if (cspace < 1)
	    cspace = 1;
    }
    pos = 0;
    redraw = 1;
    SLsmg_cls();
}

void change_mem(void)
{
    mem++;
    mem %= 3;
}

void change_cpu(void)
{
    cur_cpu++;
    cur_cpu %= cpus + 1;
    clear_histogram();
    pos = 0;
    redraw = 1;
}
#endif

void print_version(void)
{
    printf("%s %s\n", PACKAGE, VERSION);
    printf("Copyright (C) 2000 - Krzysztof Luks <m00se@iq.pl>\n\n");
    printf
	("   This program is free software; you can redistribute it and/or modify\n");
    printf
	("   it under the terms of the GNU General Public License as published by\n");
    printf
	("   the Free Software Foundation; either version 2 of the License, or\n");
    printf("   (at your option) any later version.\n\n");
    printf
	("   This program is distributed in the hope that it will be useful,\n");
    printf
	("   but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
    printf
	("   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    printf("   GNU General Public License for more details.\n\n");
    printf
	("   You should have received a copy of the GNU General Public License\n");
    printf
	("   along with this program; if not, write to the Free Software\n");
    printf
	("   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n");

}

#ifdef HAVE_POPT
void set_value(char **name, char *color, char **values)
{
    int i;
    for (i = 0; i < count_elements(values); i++) {
	if (strcmp(color, values[i]) == 0) {
	    *name = color;
	    return;
	}
    }
}

int count_elements(char **array)
{
    int i;
    for (i = 0; array[i]; i++);
    return --i;
}
#endif

#ifdef HAVE_GETOPT
void getopt_help(void)
{
	printf("Usage: slmon [OPTION...]\n");
	printf("  -v         Display version and copying information\n");
	printf("  -h,?       This help message\n");
	printf("  -b         Display memory in bytes\n");
	printf("  -k         Display memory in kilobytes\n");
	printf("  -m         Display memory in megabytes\n");
	printf("  -B         Turn bold on monochrome terminals on\n");
	printf("  -G         Start in gauge mode (default)\n");
	printf("  -H         Start in histogram mode\n");
	printf("  -E <num>   Set display height to <num> lines\n\n");
}
#endif
