/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slidentd.h"
#include "slid_log.h"
#include "slid_datestamp.h"
#include <unistd.h>
#include <byte.h>
#include <sys/types.h>
#include <sys/socket.h>

static int slid_open_log();

static char log_buffer_space[1024];
static buffer log_buffer;
static int buffer_done = 0;

buffer *
slid_get_log_buffer()
{
	if (buffer_done == 0) {
		int log_fd = slid_open_log();
		buffer_init(&log_buffer, write, log_fd, log_buffer_space,
			    sizeof log_buffer_space);
		buffer_done++;
	}

	return &log_buffer;
}

/* log a message */
void
slid_log(stralloc * msg)
{
	buffer *log_buffer = slid_get_log_buffer();
	slid_put_datestamp(log_buffer);
	buffer_puts(log_buffer, " ");
	buffer_putsflush(log_buffer, msg->s);
}

/* open a file descriptor to an appropriate log device */
static int
slid_open_log()
{
	int log_fd = -1;
#if SLIDENTD_USE_SYSLOG != 0
	struct sockaddr syslog_addr;

	errno = 0;
	byte_zero(&syslog_addr, sizeof (syslog_addr));
	syslog_addr.sa_family = AF_UNIX;
	memccpy(syslog_addr.sa_data,
		SLIDENTD_SYSLOG_DEV_PATH,
		0, (sizeof (syslog_addr.sa_data) - 1));
	if ((log_fd = socket(AF_UNIX, SOCK_DGRAM, 0)) == -1) {
		buffer_puts(buffer_2, "Unable to connect to log device ");
		buffer_puts(buffer_2, SLIDENTD_SYSLOG_DEV_PATH);
		buffer_puts(buffer_2, ": ");
		buffer_puts(buffer_2, strerror(errno));
		buffer_putsflush(buffer_2, "\n");
		slid_dies();

	}
	if (connect(log_fd, &syslog_addr, sizeof (syslog_addr)) == -1) {
		int saved_errno = errno;
		close(log_fd);
		log_fd = -1;

		if ((log_fd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
			buffer_puts(buffer_2,
				    "Unable to connect to log device ");
			buffer_puts(buffer_2, SLIDENTD_SYSLOG_DEV_PATH);
			buffer_puts(buffer_2, " using SOCK_STREAM: ");
			buffer_puts(buffer_2, strerror(errno));
			buffer_putsflush(buffer_2, "\n");
			slid_dies();

		}
		if (connect(log_fd, &syslog_addr, sizeof (syslog_addr)) == -1) {
			buffer_puts(buffer_2,
				    "Unable to connect to log device ");
			buffer_puts(buffer_2, SLIDENTD_SYSLOG_DEV_PATH);
			buffer_puts(buffer_2,
				    " tried SOCK_DGRAM and SOCK_STREAM: ");
			buffer_puts(buffer_2, strerror(saved_errno));
			buffer_putsflush(buffer_2, "\n");
			slid_dies();
		}
	}
#else
	if (slid_running_tcpserver) {
		log_fd = STDERR_FILENO;
	} else {
		errno = 0;
		log_fd =
		    open(SLIDENTD_LOG_PATH, O_WRONLY | O_CREAT | O_APPEND,
			 S_IRWXU);
		if (log_fd == -1 || errno) {
			buffer_puts(buffer_2, "Unable to open log file ");
			buffer_puts(buffer_2, SLIDENTD_LOG_PATH);
			buffer_puts(buffer_2, " for writing: ");
			buffer_puts(buffer_2, strerror(errno));
			buffer_putsflush(buffer_2, "\n");
			slid_dies();
		}
	}
#endif
	return log_fd;
}
