(* This section is all for argument processing *)

(* Mode *)

let version = ref false

let auth = ref false

(* file names *)

let output_file = ref "-"

let policy_file = ref ""

let mls_file = ref ""

let anon_var = ref 0

let anon_fun str =
  match !anon_var with
    0 ->
      policy_file := str;
      anon_var := 1
  | 1 ->
      mls_file := str;
      anon_var := 2
  | _ ->
      anon_var := 3

(* set up Arg.parse *)

let arg_spec =
  [("-a", Arg.Set(auth),
    "     - generate an authorization transition relation");
   ("-f", Arg.Clear(auth),
    "     - generate an information flow transition relation (default)");
   ("-o", Arg.String(fun s -> output_file := s),
    "name - send output to this file (default stdout)" );
   ("-v", Arg.Set(version),
    "     - display version information and exit");
   ("--", Arg.Rest(anon_fun),
    "     - treat remaining args as file names, where - means stdin")]

let usage_msg = "Usage: slat [options] policy [mls]";;

Arg.parse arg_spec anon_fun usage_msg;

if !version then
  begin
    Print_version.print_version();
    exit 0
  end;

if !anon_var < 1 || !anon_var > 2 then
  begin
    prerr_string "bad arg count";
    prerr_newline();
    prerr_string usage_msg;
    prerr_newline();
    exit 1
  end;

if !anon_var = 2 && !auth then
  begin
    prerr_string "no mls allowed with -a option";
    prerr_newline();
    prerr_string usage_msg;
    prerr_newline();
    exit 1
  end;

let policy_file_name = !policy_file in
let mls_file_name = !mls_file in
let output_file_name = !output_file in
let auth_flag = !auth in

try

(* read input *)

  Read_policy_mls.read_policy policy_file_name;
  if mls_file_name <> "" then
    Read_policy_mls.read_mls mls_file_name;

(* make lts *)

  let lts = Identifier.slat auth_flag in

(* open output file *)

  let out =
    if output_file_name = "-" then
      stdout
    else
      open_out output_file_name in

  Format.set_formatter_out_channel out;

(* produce output *)

  Formulas.print_lts lts;

  exit 0

with Failure s ->
  prerr_string s;
  prerr_newline();
  exit 1
