/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportXhb.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportxhb.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgimportexportmanager.h"
#include "skgpayeeobject.h"

#include <klocale.h>
#include <kfilterdev.h>

#include <QDomDocument>

SKGError SKGImportXhb::importFile ( SKGImportExportManager* iImporter, SKGDocumentBank* iDocument )
{
    if (!iImporter || !iDocument) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportXhb::importFile", err );
    QString iFileName=iImporter->getFileName();
    SKGTRACEL ( 10 ) << "Input filename=" << iFileName << endl;

    //Initialisation
    //Open file
    QIODevice* file=KFilterDev::deviceForFile ( iFileName, "application/x-gzip" );
    if ( !file->open ( QIODevice::ReadOnly ) )
    {
        err.setReturnCode ( ERR_INVALIDARG );
        err.setMessage ( i18nc ( "Error message",  "Open file '%1' failed",iFileName ) );
    }
    else
    {
        QDomDocument doc;

        //Set the file without uncompression
        QString errorMsg;
        int errorLine=0;
        int errorCol=0;
        bool contentOK=doc.setContent ( file->readAll(), &errorMsg, &errorLine, &errorCol );
        file->close();

        //Get root
        QDomElement docElem = doc.documentElement();

        if ( !contentOK )
        {
            err.setReturnCode ( ERR_ABORT );
            err.setMessage ( i18nc ( "Error message",  "%1-%2: '%3'",errorLine, errorCol, errorMsg ) );
        }


        if (!contentOK)
        {
            err.addError ( ERR_INVALIDARG, i18nc ( "Error message",  "Invalid XML content in file '%1'",iFileName ) );
        }
        else
        {
            err=iDocument->beginTransaction ( "#INTERNAL#", 4 );

            QMap<QString, SKGAccountObject> mapIdAccount;
            QMap<QString, SKGCategoryObject> mapIdCategory;
            QMap<QString, SKGPayeeObject> mapIdPayee;

            SKGUnitObject unit;
            if ( err.isSucceeded() ) err=iImporter->getDefaultUnit(unit);

            //Step 1-Create accounts
            if ( err.isSucceeded() )
            {
                QDomNodeList accountList=docElem.elementsByTagName ( "account" );
                int nb=accountList.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    //Get account object
                    QDomElement account=accountList.at ( i ).toElement();

                    //Create the bank
                    SKGBankObject bank ( iDocument );
                    QString bname=getAttribute ( account,  "bankname" );
                    if (bname.isEmpty()) bname="HOMEBANK";
                    if ( err.isSucceeded() ) err=bank.setName ( bname );
                    if ( err.isSucceeded() ) err=bank.save();

                    //Creation of the account
                    SKGAccountObject accountObj;
                    if ( err.isSucceeded() ) err=bank.addAccount ( accountObj );
                    if ( err.isSucceeded() ) err=accountObj.setName ( getAttribute ( account,  "name" ) );
                    if ( err.isSucceeded() ) err=accountObj.setNumber ( getAttribute ( account,  "number" ) );
                    if ( err.isSucceeded() )
                    {
                        QString flags=getAttribute( account,  "flags" );
                        err=accountObj.setClosed ( flags=="2" ||  flags=="3" );
                    }
                    if ( err.isSucceeded() ) err=accountObj.save();
                    if ( err.isSucceeded() )
                    {
                        accountObj.setClosed(false); //To be sure that the modification is possible. NOT SAVED
                        err=accountObj.setInitialBalance(SKGServices::stringToDouble(getAttribute ( account,  "initial")), unit);
                    }

                    mapIdAccount[getAttribute ( account,  "key" )]=accountObj;

                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
            if ( err.isSucceeded() ) err=iDocument->stepForward ( 1 );

            //Step 2-Get payees
            if ( err.isSucceeded() )
            {
                QDomNodeList partyList=docElem.elementsByTagName ( "pay" );
                int nb=partyList.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    //Get payee object
                    QDomElement party=partyList.at ( i ).toElement();
                    SKGPayeeObject payeeObject;
                    err=SKGPayeeObject::createPayee(iDocument, getAttribute ( party,  "name" ), payeeObject);
                    mapIdPayee[getAttribute ( party,  "key" )]=payeeObject;

                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
            if ( err.isSucceeded() ) err=iDocument->stepForward ( 2 );

            //Step 3-Create categories
            if ( err.isSucceeded() )
            {
                QDomNodeList categoryList=docElem.elementsByTagName ( "cat" );
                int nb=categoryList.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", 2*nb );
                for ( int j=0; err.isSucceeded() && j<2; ++j )
                {
                    for ( int i=0; err.isSucceeded() && i<nb; ++i )
                    {
                        //Get account object
                        QDomElement category=categoryList.at ( i ).toElement();

                        //Creation of the category
                        QString parent=getAttribute ( category,  "parent" );
                        if (parent=="0" && j==0)
                        {
                            SKGCategoryObject catObj(iDocument);
                            if ( err.isSucceeded() ) err=catObj.setName ( getAttribute ( category,  "name" ) );
                            if ( err.isSucceeded() ) err=catObj.save();

                            mapIdCategory[getAttribute ( category,  "key" )]=catObj;

                        }
                        else if (parent!="0" && j==1)
                        {
                            SKGCategoryObject catParentObj=mapIdCategory[parent];
                            SKGCategoryObject catObj(iDocument);
                            if ( err.isSucceeded() ) err=catObj.setName ( getAttribute ( category,  "name" ) );
                            if ( err.isSucceeded() ) err=catObj.setParentCategory ( catParentObj );
                            if ( err.isSucceeded() ) err=catObj.save();

                            mapIdCategory[getAttribute ( category,  "key" )]=catObj;
                        }

                        if ( err.isSucceeded() ) err=iDocument->stepForward ( nb*j+i+1 );
                    }
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
            if ( err.isSucceeded() ) err=iDocument->stepForward ( 3 );

            //Step 4-Create transaction
            if ( err.isSucceeded() )
            {
                QDomNodeList transactionList=docElem.elementsByTagName ( "ope" );
                int nb=transactionList.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    //Get account object
                    QDomElement transaction=transactionList.at ( i ).toElement();

                    //Creation of the operation
                    SKGAccountObject account=mapIdAccount[getAttribute ( transaction,  "account" )];
                    SKGOperationObject opObj;
                    if ( err.isSucceeded() ) err=account.addOperation ( opObj );
                    if ( err.isSucceeded() ) err=opObj.setDate ( QDate(1,1,1).addDays ( 1+SKGServices::stringToInt(getAttribute ( transaction,  "date" ))) );
                    if ( err.isSucceeded() ) err=opObj.setUnit (unit);
                    if ( err.isSucceeded() ) err=opObj.setPayee (mapIdPayee[ getAttribute ( transaction,  "payee" ) ] );
                    if ( err.isSucceeded() )
                    {
                        QString mode=getAttribute ( transaction,  "paymode" );
                        if (mode=="1") mode=i18nc("Noun: type of payement", "Credit card");
                        else if (mode=="2") mode=i18nc("Noun: type of payement", "Check");
                        else if (mode=="3") mode=i18nc("Noun: type of payement", "Cash");
                        else if (mode=="4" || mode=="5") mode=i18nc("Noun: type of payement", "Transfer");
                        else mode=i18nc("Noun: type of payement", "Other");
                        err=opObj.setMode (mode);
                    }
                    QString comment=(getAttribute ( transaction,  "wording" )+' '+getAttribute ( transaction,  "info" )).trimmed();
                    if ( err.isSucceeded() ) err=opObj.setComment(comment);
                    if ( err.isSucceeded() ) err=opObj.setImported ( true );
                    if ( err.isSucceeded() ) err=opObj.setImportID ( "HXB-"+SKGServices::intToString(i) );
                    if ( err.isSucceeded() )
                    {
                        QString flags=getAttribute( transaction,  "flags" );
                        err=opObj.setStatus ( flags=="1" ||  flags=="3" ? SKGOperationObject::CHECKED : SKGOperationObject::NONE );
                    }
                    if ( err.isSucceeded() ) err=opObj.save();

                    SKGSubOperationObject subObj;
                    if ( err.isSucceeded() ) err=opObj.addSubOperation ( subObj );
                    if ( err.isSucceeded() ) err=subObj.setCategory ( mapIdCategory[getAttribute ( transaction,  "category" )] );
                    if ( err.isSucceeded() ) err=subObj.setComment (comment);
                    if ( err.isSucceeded() ) err=subObj.setQuantity ( SKGServices::stringToDouble(getAttribute ( transaction,  "amount" )) );
                    if ( err.isSucceeded() ) err=subObj.save();

                    if ( err.isSucceeded() && i%100==0 ) err = SKGServices::executeSqliteOrder ( iDocument, "ANALYZE" );
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
            if ( err.isSucceeded() ) err=iDocument->stepForward ( 4 );

            if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
            else  iDocument->endTransaction ( false );

            if ( err.isSucceeded() ) err = SKGServices::executeSqliteOrder ( iDocument, "ANALYZE" );
        }
    }

    delete file;
    return err;
}


QString SKGImportXhb::getAttribute ( const QDomElement& iElement, const QString& iAttribute )
{
    QString val=iElement.attribute ( iAttribute );
    if (val=="(null)") val="";
    return val;
}

#include "skgimportxhb.moc"
