/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportSkg.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportskg.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgimportexportmanager.h"

#include <kio/netaccess.h>

#include <QFile>
#include <QSqlDatabase>
#include <QSqlError>

#include <klocale.h>

SKGError SKGImportSkg::importFile(SKGImportExportManager* iImporter, SKGDocumentBank* iDocument)
{
    if (!iImporter || !iDocument) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportSkg::importFile", err );
    QString iFileName=iImporter->getFileName();
    SKGTRACEL ( 10 ) << "Input filename=" << iFileName << endl;

    SKGDocumentBank docOrigin;
    err=docOrigin.load ( iFileName );
    if ( err.isSucceeded() )
    {
        QMap<QString, SKGObjectBase> mapOriginNew;
        err=iDocument->beginTransaction ( "#INTERNAL#", 10 );

        //Step 1 - units
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listUnits;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_unit", "1=1 order by t_type", listUnits );
            if ( err.isSucceeded() )
            {
                int nb=listUnits.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGUnitObject unitOrigin=listUnits.at ( i );
                    SKGUnitObject unit=unitOrigin.cloneInto ( iDocument );
                    if ( unit.load().isFailed() )
                    {
                        //This unit does not exist yet
                        if ( err.isSucceeded() )
                        {
                            SKGUnitObject::UnitType unitType=unitOrigin.getType();
                            if ( unitType==SKGUnitObject::PRIMARY || unitType==SKGUnitObject::SECONDARY ) unitType=SKGUnitObject::CURRENCY;
                            err=unit.setType ( unitType );
                        }
                        if ( err.isSucceeded() )
                        {
                            SKGUnitObject parentUnitOrigin;
                            unitOrigin.getUnit ( parentUnitOrigin );

                            SKGUnitObject parentUnit=mapOriginNew[parentUnitOrigin.getUniqueID() ];
                            if ( parentUnit!=unit ) err=unit.setUnit ( parentUnit );
                        }
                        if ( err.isSucceeded() ) err=unit.save ( false ); //Save only
                    }
                    mapOriginNew[unitOrigin.getUniqueID() ]=unit;

                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( unitOrigin, unit );

                    //Unit values
                    SKGObjectBase::SKGListSKGObjectBase listUnitsValues;
                    if ( err.isSucceeded() ) err=unitOrigin.getUnitValues ( listUnitsValues );
                    int nb2=listUnitsValues.count();
                    for ( int j=0; err.isSucceeded() && j<nb2; ++j )
                    {
                        SKGUnitValueObject unitValueOrigin=listUnitsValues.at ( j );

                        SKGUnitValueObject  unitval;
                        err=unit.addUnitValue ( unitval );
                        if ( err.isSucceeded() ) err=unitval.setDate ( unitValueOrigin.getDate() );
                        if ( err.isSucceeded() ) err=unitval.setQuantity ( unitValueOrigin.getQuantity() );
                        if ( err.isSucceeded() ) err=unitval.save ( true ); //Save or update

                        //Duplicate properties
                        if ( err.isSucceeded() ) err=copyParameters ( unitValueOrigin, unitval );
                    }
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 1 );

        //Step 2 - bank and accounts
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listBanks;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_bank", "", listBanks );
            if ( err.isSucceeded() )
            {
                int nb=listBanks.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGBankObject bankOrigin=listBanks.at ( i );
                    SKGBankObject bank=bankOrigin.cloneInto ( iDocument );
                    if ( bank.load().isFailed() )
                    {
                        //This bank does not exist yet
                        if ( err.isSucceeded() ) err=bank.save ( false ); //Save only
                    }

                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( bankOrigin, bank );

                    //Accounts
                    SKGObjectBase::SKGListSKGObjectBase listAccounts;
                    if ( err.isSucceeded() ) err=bankOrigin.getAccounts ( listAccounts );
                    int nb2=listAccounts.count();
                    for ( int j=0; err.isSucceeded() && j<nb2; ++j )
                    {
                        SKGAccountObject accountOrigin=listAccounts.at ( j );
                        SKGAccountObject account=accountOrigin.cloneInto ( iDocument );
                        if ( account.load().isFailed() )
                        {
                            //This account does not exist yet
                            if ( err.isSucceeded() ) err=account.setBank ( bank );
                            //Initial balance will be set on operation creation
                            if ( err.isSucceeded() ) err=account.save ( false ); //Save only
                        }

                        //Duplicate properties
                        if ( err.isSucceeded() ) err=copyParameters ( accountOrigin, account );

                        mapOriginNew[accountOrigin.getUniqueID() ]=account;
                    }
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 2 );

        //Step 3 - categories
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listCategories;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_category", "", listCategories );
            if ( err.isSucceeded() )
            {
                int nb=listCategories.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGCategoryObject catOrigin=listCategories.at ( i );

                    SKGCategoryObject cat;
                    err=SKGCategoryObject::createPathCategory ( iDocument, catOrigin.getFullName(), cat );
                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( catOrigin, cat );
                    mapOriginNew[catOrigin.getUniqueID() ]=cat;
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 3 );

        //Step 4 - trackers
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listRefund;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_refund", "", listRefund );
            if ( err.isSucceeded() )
            {
                int nb=listRefund.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGTrackerObject tracOrigin=listRefund.at ( i );
                    SKGTrackerObject trac=tracOrigin.cloneInto ( iDocument );
                    err=trac.save ( );
                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( tracOrigin, trac );
                    mapOriginNew[tracOrigin.getUniqueID() ]=trac;
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 4 );

        //Step 5 - rules
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listRules;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_rule", "", listRules );
            if ( err.isSucceeded() )
            {
                int nb=listRules.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGRuleObject rulOrigin=listRules.at ( i );
                    SKGRuleObject rul=rulOrigin.cloneInto ( iDocument );
                    err=rul.save ( false ); //Save only
                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( rulOrigin, rul );
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 5 );

        //Step 6 - payee
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listPayee;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_payee", "", listPayee );
            if ( err.isSucceeded() )
            {
                int nb=listPayee.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGPayeeObject paylOrigin=listPayee.at ( i );

                    SKGPayeeObject pay;
                    err=SKGPayeeObject::createPayee ( iDocument, paylOrigin.getName(), pay );
                    if ( err.isSucceeded() ) err=pay.setAddress(paylOrigin.getAddress());
                    if ( err.isSucceeded() ) err=pay.save ();
                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( paylOrigin, pay );
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 6 );

        //Step 7 - operations and suboperation
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listOperations;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_operation", "", listOperations );
            if ( err.isSucceeded() )
            {
                int nb=listOperations.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGOperationObject operationOrigin=listOperations.at ( i );
                    SKGOperationObject operation=operationOrigin.cloneInto ( iDocument );
                    if ( err.isSucceeded() ) err=operation.setAttribute ( "r_recurrentoperation_id", "" );
                    if ( err.isSucceeded() ) err=operation.setImported ( true );
                    if ( err.isSucceeded() )
                    {
                        QString importID=operationOrigin.getImportID();
                        if ( importID.isEmpty() ) importID="SKG-"+SKGServices::intToString ( operationOrigin.getID() );
                        err=operation.setImportID ( importID );
                    }
                    if ( err.isSucceeded() )
                    {
                        SKGAccountObject actOrig;
                        err=operationOrigin.getParentAccount ( actOrig );
                        if ( err.isSucceeded() )
                        {
                            SKGAccountObject act=mapOriginNew[actOrig.getUniqueID() ];
                            act.setClosed(false); //To be sure that the modification is possible. NOT SAVED
                            if ( err.isSucceeded() ) err=operation.setParentAccount ( act );
                        }
                    }
                    if ( err.isSucceeded() )
                    {
                        SKGUnitObject unitOrig;
                        err=operationOrigin.getUnit ( unitOrig );
                        if ( err.isSucceeded() ) err=operation.setUnit ( mapOriginNew[unitOrig.getUniqueID() ] );
                    }
                    if ( err.isSucceeded() )
                    {
                        SKGOperationObject groupOrig;
                        operationOrigin.getGroupOperation ( groupOrig );
                        err=operation.setGroupOperation ( mapOriginNew[groupOrig.getUniqueID() ] );
                    }
                    if ( err.isSucceeded() ) err=operation.save ( false ); //Save only

                    mapOriginNew[operationOrigin.getUniqueID() ]=operation;

                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( operationOrigin, operation );

                    //Sub operation
                    SKGObjectBase::SKGListSKGObjectBase listSuboperations;
                    if ( err.isSucceeded() ) err=operationOrigin.getSubOperations ( listSuboperations );
                    int nb2=listSuboperations.count();
                    for ( int j=0; err.isSucceeded() && j<nb2; ++j )
                    {
                        SKGSubOperationObject subopOrigin=listSuboperations.at ( j );

                        SKGSubOperationObject subop=subopOrigin.cloneInto ( iDocument );
                        err=subop.setParentOperation ( operation );
                        if ( err.isSucceeded() )
                        {
                            SKGCategoryObject catOrig;
                            subopOrigin.getCategory ( catOrig ); //Error not managed
                            err=subop.setCategory ( mapOriginNew[catOrig.getUniqueID() ] );
                        }
                        if ( err.isSucceeded() )
                        {

                            SKGTrackerObject tracOrig;
                            subopOrigin.getTracker ( tracOrig ); //Error not managed
                            SKGTrackerObject trac=mapOriginNew[tracOrig.getUniqueID() ];
                            trac.setClosed(false); //To be sure that the modification is possible. NOT SAVED
                            if ( err.isSucceeded() ) err=subop.setTracker ( trac );
                        }
                        if ( err.isSucceeded() ) err=subop.save ( false ); //Save only

                        //Duplicate properties
                        if ( err.isSucceeded() ) err=copyParameters ( subopOrigin, subop );
                    }
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );

                    if ( err.isSucceeded() && i%100==0 ) err = SKGServices::executeSqliteOrder ( iDocument, "ANALYZE" );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 7 );
        /*	                                    SKGObjectBase opWithThisHash;
                                            if ( SKGObjectBase::getObject ( iDocument, "operation", "t_imported IN ('Y','P') AND t_import_id='"+QString ( hash.toHex() ) +'\'', opWithThisHash ).isSucceeded() )*/

        //Step 8 - recurrent
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase ListRecurrentOperations;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_recurrentoperation", "", ListRecurrentOperations );
            if ( err.isSucceeded() )
            {
                int nb=ListRecurrentOperations.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGRecurrentOperationObject recuOrigin=ListRecurrentOperations.at ( i );
                    SKGRecurrentOperationObject recu=recuOrigin.cloneInto ( iDocument );

                    SKGOperationObject opOrig;
                    err=recuOrigin.getParentOperation ( opOrig );
                    if ( err.isSucceeded() ) err=recu.setParentOperation ( mapOriginNew[opOrig.getUniqueID() ] );
                    if ( err.isSucceeded() ) err=recu.save ( false ); //Save only

                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( recuOrigin, recu );

                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 8 );

        //Step 9 - nodes
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listNodes;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_node", "", listNodes );
            if ( err.isSucceeded() )
            {
                int nb=listNodes.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGNodeObject nodeOrigin=listNodes.at ( i );

                    SKGNodeObject node;
                    err=SKGNodeObject::createPathNode ( iDocument, i18n("Imported bookmarks")+OBJECTSEPARATOR+nodeOrigin.getFullName(), node );
                    if (err.isSucceeded()) err=node.setData(nodeOrigin.getData());
                    if (err.isSucceeded()) err=node.setOrder(nodeOrigin.getOrder());
                    if (err.isSucceeded()) err=node.setAutoStart(nodeOrigin.isAutoStart());
                    if (err.isSucceeded()) err=node.save( true, false );
                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 9 );

        //Step 10 - interest
        if ( err.isSucceeded() )
        {
            SKGObjectBase::SKGListSKGObjectBase listInterests;
            err=SKGObjectBase::getObjects ( &docOrigin, "v_interest", "", listInterests );
            if ( err.isSucceeded() )
            {
                int nb=listInterests.count();
                err=iDocument->beginTransaction ( "#INTERNAL#", nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i )
                {
                    SKGInterestObject interestOrigin=listInterests.at ( i );
                    SKGInterestObject interest=interestOrigin.cloneInto ( iDocument );
                    if ( err.isSucceeded() )
                    {
                        SKGAccountObject actOrig;
                        err=interestOrigin.getAccount ( actOrig );
                        if ( err.isSucceeded() ) err=interest.setAccount ( mapOriginNew[actOrig.getUniqueID() ] );
                    }
                    if ( err.isSucceeded() ) err=interest.save ();

                    //Duplicate properties
                    if ( err.isSucceeded() ) err=copyParameters ( interestOrigin, interest );

                    if ( err.isSucceeded() ) err=iDocument->stepForward ( i+1 );
                }

                if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
                else  iDocument->endTransaction ( false );
            }
        }
        if ( err.isSucceeded() ) err=iDocument->stepForward ( 10 );

        if ( err.isSucceeded() ) err=iDocument->endTransaction ( true );
        else  iDocument->endTransaction ( false );

        if ( err.isSucceeded() ) err = SKGServices::executeSqliteOrder ( iDocument, "ANALYZE" );
    }
    return err;
}


SKGError SKGImportSkg::exportFile(SKGImportExportManager* iImporter, SKGDocumentBank* iDocument, bool iSqliteFile)
{
    if (!iImporter || !iDocument) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC ( 2, "SKGImportSkg::exportFile", err );
    QString iFileName=iImporter->getFileName();
    SKGTRACEL ( 10 ) << "Input filename=" << iFileName << endl;

    QString tempFile=iDocument->getCurrentTemporaryFile();
    if (iDocument->getCurrentFileName().isEmpty())
    {
        //The database is only in memory
        KIO::NetAccess::del(KUrl(tempFile), NULL);
        QSqlDatabase* fileDb=new QSqlDatabase(QSqlDatabase::addDatabase("QSQLITE", tempFile));
        fileDb->setDatabaseName(tempFile);
        if (!fileDb->open()) {
            //Set error message
            QSqlError sqlErr=fileDb->lastError();
            err = SKGError(SQLLITEERROR + sqlErr.number(), sqlErr.text());
        }
        else
        {
            iDocument->getDatabase()->commit();
            if (err.isSucceeded()) err=SKGServices::copySqliteDatabase(fileDb, iDocument->getDatabase(), false);
            iDocument->getDatabase()->transaction();
        }

        fileDb->close();
        delete fileDb;
        QSqlDatabase::removeDatabase(tempFile);
    }

    //Copy file to file
    QFile::remove(iFileName);
    if (iSqliteFile)
    {
        if (!KIO::NetAccess::file_copy(tempFile, iFileName))
        {
            //Set error message
            err.setReturnCode(ERR_FAIL);
            err.setMessage(i18nc("An error message", "Creation file '%1' failed",iFileName));
        }
    }
    else
    {
        err=SKGServices::cryptFile(tempFile, iFileName, "", true, iDocument->getDocumentHeader());
        if (err.isSucceeded())
        {
            SKGDocumentBank doc;
            err=doc.load(iFileName);
            if (err.isSucceeded()) err=doc.removeAllTransactions();
            if (err.isSucceeded()) err=doc.save();
        }
    }

    if (iDocument->getCurrentFileName().isEmpty())  KIO::NetAccess::del(KUrl(tempFile), NULL);

    return err;
}

SKGError SKGImportSkg::copyParameters ( const SKGObjectBase& iFrom, const SKGObjectBase& iTo )
{
    SKGError err;

    SKGObjectBase::SKGListSKGObjectBase params;
    err = SKGObjectBase::getObjects ( iFrom.getDocument(), "parameters", "t_uuid_parent='"+SKGServices::stringToSqlString ( iFrom.getUniqueID() ) +'\'', params );
    if ( err.isSucceeded() )
    {
        int nb=params.count();
        SKGDocument* documentTarget=(SKGDocument*) iTo.getDocument();
        for ( int i=0; err.isSucceeded() && i<nb; ++i )
        {
            SKGObjectBase orig=params.at ( i );
            SKGObjectBase param=orig.cloneInto ( documentTarget );
            err=param.setAttribute ( "t_uuid_parent", iTo.getUniqueID() );
            if ( err.isSucceeded() ) err=param.save ( true, false );
        }
    }
    return err;
}

#include "skgimportskg.moc"
