#include "sjog.h"
#include "sjog_config.h"
#include "sjog_list.h"
#include "sjog_brightness.h"
#include "sjog_volume.h"
#include "sjog_scroll.h"
#include "sjog_spic.h"
#include "sjog_sonypi.h"

/* -------------------------------------------------------------------------- */

gboolean reverseBrightness = TRUE;
gboolean useSpic = FALSE;
gboolean mousewheelFakeKeys = FALSE;
gboolean debug = FALSE;
gboolean reverseSpinDirection = FALSE;
gboolean windowsAreTransient = FALSE;
gboolean startMenu = TRUE;
gboolean resetMenu = FALSE;
gchar *sonypiDevice = NULL;
gchar *mixerDevice = NULL;
sjog_gravity listWindowGravity = SJOG_GRAVITY_CENTER;
gboolean eMenu = FALSE;

/* -------------------------------------------------------------------------- */

GtkWidget *listWindow;
GtkWidget *cmdListUI;
guint listTimer;
guint selectedIndex;
GList *list = NULL;
GNode *currentTreeNode = NULL;

/* -------------------------------------------------------------------------- */

/*
 * Displays the usage information.
 * @param programname the name of the executable
 */

void
sjog_show_usage(gchar * programname)
{
   printf("S-Jog -- A tool to make Sony's Jog Dial less useless.\n"
          "Usage: %s [OPTIONS]\n" "\n"
          
          "  -rb,  --reverse-brightness      "
          "reverse brightness setting behaviour\n"

          "  -fk,  --fake-keys               "
          "send page up/down instead of buttons 4/5\n"

          "  -s ,  --spic                    "
          "use the legacy spic driver\n"

          "  -S ,  --sonypi-device           "
          "specify the sonypi device location\n"

          "  -m ,  --mixer-device            "
          "specify the mixer device location\n"

          "  -g ,  --gravity <gravity>       "
          "set the main window's position\n"
          "        gravities: "
          "tleft|top|tright|left|center|right|bleft|bottom|bright|none\n"

          "  -r ,  --reset-menu              "
          "reset launcher list item position on click\n"

          "  -rd,  --reverse-direction       "
          "reverse the spin direction (ex: up is down)\n"

          "  -t ,  --transient               "
          "make the windows transient like a dialog box\n"

          "  -ns,  --no-startmenu            "
          "do not display the menu on startup\n"

          "  -e ,  --enlightenment-menu      "
          "use user enlightenment menu config instead of jogrc\n"

          "  -h ,  --help                    " "display this message\n" "\n",
          programname);
}

/* -------------------------------------------------------------------------- */

/* 
 * Sets the current mode.
 * Mode can be one of list, brightness, volume or scroll.
 * See sjog_mode enum in sjog.h
 *
 * @param mode the new current mode.
 */
void
sjog_set_mode(guint mode)
{
   currentMode = mode;
}

/* -------------------------------------------------------------------------- */

/* 
 * Returns the current mode.
 */
guint
sjog_get_mode()
{
   return currentMode;
}

/* -------------------------------------------------------------------------- */

/* 
 * Gets the current jog position and call a function according to the
 * current mode.
 */
void
sjog_jog_position_changed(gint jog)
{

   D("MODE=%d\n", sjog_get_mode());

   switch (sjog_get_mode())
   {
     case SJOG_MODE_VOLUME:
        sjog_volume_check_jog_position(jog);
        break;
     case SJOG_MODE_BRIGHTNESS:
        sjog_brightness_check_jog_position(jog);
        break;
     case SJOG_MODE_LIST:
        sjog_list_check_jog_position(jog);
        break;
     case SJOG_MODE_SCROLL:
        sjog_scroll_check_jog_position(jog);
   }

}

/* -------------------------------------------------------------------------- */

/*
 * Parse the command line and set the options.
 */
static void
sjog_check_options(gint argc, gchar * argv[])
{
   int i;

   for (i = 1; i < argc; i++)
   {
      /* some vaio models (i.e. 505) reverse the brightness value
       * so this option allows to reverse the brightness
       * setting behavior.
       */
      if (!strcmp(argv[i], "-rb") || !strcmp(argv[i], "--reverse-brightness"))
      {
         reverseBrightness = FALSE;

      }

      /* TT - Provide a way to make the directions on the jogdial reversable */
      else if (!strcmp(argv[i], "-rd")
               || !strcmp(argv[i], "--reverse-direction"))
      {
         reverseSpinDirection = TRUE;

      }

      /* keep compatibility with legacy spicdriver */
      else if (!strcmp(argv[i], "-s") || !strcmp(argv[i], "--spic"))
      {
         useSpic = TRUE;
      }

      /* TT - Allow the user to choose the spic device, if not specified,
       * search for the standard /dev/sonypi device, and if that fails,
       * search for the newer devfs /dev/misc/sonypi device. The actual
       * searching is done in the volume window callbacks. This option
       * supercedes the --spic option, and will set useSpic to FALSE if
       * specified.
       */
      else if (!strcmp(argv[i], "-S") || !strcmp(argv[i], "--sonypi-device"))
      {
         if ((argc > i + 1) && (argv[i + 1][0] != '-'))
         {
            sonypiDevice = strdup(argv[++i]);
            useSpic = FALSE;
         }
         else
         {
            sjog_show_usage(argv[0]);
            exit(0);
         }

      }

      /* TT - Allow the user to choose the mixer device, following the same
       * rules as mentioned above if not. (/dev/mixer, /dev/sound/mixer)
       */
      else if (!strcmp(argv[i], "-m") || !strcmp(argv[i], "--mixer-device"))
      {
         if ((argc > i + 1) && (argv[i + 1][0] != '-'))
         {
            mixerDevice = strdup(argv[++i]);
         }
         else
         {
            sjog_show_usage(argv[0]);
            exit(0);
         }

      }
      else if (!strcmp(argv[i], "-fk") || !strcmp(argv[i], "--fake-keys"))
      {
         mousewheelFakeKeys = TRUE;
      }

      /* SG - keep compatibility by setting the default gravity to none */
      else if (!strcmp(argv[i], "-nc") || !strcmp(argv[i], "--no-center")) {
         listWindowGravity = SJOG_GRAVITY_NONE;
      }

      /* TT - Allow the user to specify the gravity of the main window */
      else if (!strcmp(argv[i], "-g") || !strcmp(argv[i], "--gravity"))
      {
         if ((argc > i + 1) && (argv[i + 1][0] != '-'))
         {
            i++;
            if (!strcmp(argv[i], "tleft"))
            {
               listWindowGravity = SJOG_GRAVITY_TOP_LEFT;
            }
            else if (!strcmp(argv[i], "top"))
            {
               listWindowGravity = SJOG_GRAVITY_TOP;
            }
            else if (!strcmp(argv[i], "tright"))
            {
               listWindowGravity = SJOG_GRAVITY_TOP_RIGHT;
            }
            else if (!strcmp(argv[i], "left"))
            {
               listWindowGravity = SJOG_GRAVITY_LEFT;
            }
            else if (!strcmp(argv[i], "center"))
            {
               listWindowGravity = SJOG_GRAVITY_CENTER;
            }
            else if (!strcmp(argv[i], "right"))
            {
               listWindowGravity = SJOG_GRAVITY_RIGHT;
            }
            else if (!strcmp(argv[i], "bleft"))
            {
               listWindowGravity = SJOG_GRAVITY_BOTTOM_LEFT;
            }
            else if (!strcmp(argv[i], "bottom"))
            {
               listWindowGravity = SJOG_GRAVITY_BOTTOM;
            }
            else if (!strcmp(argv[i], "bright"))
            {
               listWindowGravity = SJOG_GRAVITY_BOTTOM_RIGHT;
            }
            else if (!strcmp(argv[i], "none"))
            {
               listWindowGravity = SJOG_GRAVITY_NONE;
            }
            else
            {
               sjog_show_usage(argv[0]);
               exit(0);
            }
         }
         else
         {
            sjog_show_usage(argv[0]);
            exit(0);
         }

      }

      /* TT - Allow setting the main window to a GTK_WINDOW_POPUP window.
       * This allows most users to not have to mess with the placement
       * of the window. (such as in Golem or TWM)
       */
      else if (!strcmp(argv[i], "-t") || !strcmp(argv[i], "--transient"))
      {
         windowsAreTransient = TRUE;

      }

      else if (!strcmp(argv[i], "-ns") || !strcmp(argv[i], "--no-startmenu"))
      {
         startMenu = FALSE;
      }

      else if (!strcmp(argv[i], "-r") || !strcmp(argv[i], "--reset-menu"))
      {
         resetMenu = TRUE;
      }

#ifdef DEBUG
      else if (!strcmp(argv[i], "-d") || !strcmp(argv[i], "--debug"))
      {
         debug = TRUE;
         D("Debug is on\n");
      }
#endif

      else if (!strcmp(argv[i], "-e") || !strcmp(argv[i], "--enlightenment-menu"))
      {
      	eMenu = TRUE;
      }

      /* usage information */
      else
      {
         sjog_show_usage(argv[0]);
         exit(0);
      }

   }

}

/* -------------------------------------------------------------------------- */

gint
main(gint argc, gchar * argv[])
{

   /* read command line options */
   sjog_check_options(argc, argv);

   /* gtk initialization */
   gtk_init(&argc, &argv);

   /* rgb_initialzation for pixmaps */
   gdk_rgb_init();

   /* choose between legacy spic driver polling or new sonypi
    * driver IRQ handling depending on command line option.
    * Default is to use the new sonypi driver.
    */
   if (useSpic == TRUE)
   {
      /* prepare to read the device */
      sjog_spic_init();
      /* the legacy driver requires us to poll the device instead */
      gtk_timeout_add(100, (GtkFunction) sjog_spic_read, NULL);
   }
   else
   {
      /* prepare to read the device */
      sjog_sonypi_init();
   }

   /* show initial window */
   if(startMenu)
      sjog_list_show();
   else
      sjog_set_mode(SJOG_MODE_SCROLL);

   /* main event loop */
   gtk_main();

   return 0;

}
