/*
 * Copyright (c) 1998 - 2001 Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * The parse tree transformation module for SIP.
 */


#include <stddef.h>
#include <string.h>
#include <stdlib.h>

#include "sip.h"


#define	pyToBasicInt(t)	((t) == short_type || (t) == ushort_type || \
			 (t) == enum_type || (t) == bool_type || \
	    		 (t) == int_type || (t) == uint_type || \
	    		 (t) == long_type || (t) == ulong_type)

#define	pyToInt(t)	(pyToBasicInt(t) || (t) == cint_type)
#define	pyToDouble(t)	(pyToBasicInt(t) || (t) == float_type || (t) == double_type)


/* A set of types specific to a version. */

typedef struct _versTypes {
	versionDef	version;		/* The version. */
	argDef		types[MAX_NR_ARGS + 1];	/* The types. */
} versTypes;


static int supportedSimpleType(argDef *);
static int supportedComplexType(argDef *);
static void addAutoOverloads(classVersDef *,char *,char *);
static void ifaceFileIsUsed(sipSpec *,moduleDef *,classVersDef *,argDef *,int);
static void scopeDefaultValue(sipSpec *,classVersDef *,argDef *);
static void setHierachy(sipSpec *,classVersDef *,classVersDef *);
static void transformCtors(sipSpec *,classVersDef *);
static void transformOverloads(sipSpec *,classVersDef *,overDef *);
static void transformVariableList(sipSpec *);
static void getVisibleMembers(classVersDef *);
static void getVirtuals(classVersDef *);
static void getVirtualsVersions(virtOverDef **,classVersList *,versionDef *);
static void addSignalSignatures(sipSpec *,classVersDef *);
static void addSignalArgs(sipSpec *,funcArgs *,versionDef *);
static void resolveCtorTypes(sipSpec *,classVersDef *,ctorDef *,argDef *,
			     versionDef *);
static void resolveFuncTypes(sipSpec *,classVersDef *,overDef *,argDef *,
			     argDef *,versionDef *);
static void resolveVariableType(sipSpec *,varDef *,argDef *,versionDef *);
static void fatalNoDefinedType(sipSpec *,scopedNameDef *,versionDef *);
static void getBaseType(sipSpec *,classVersDef *,argDef *,versionDef *);
static void searchScope(sipSpec *,classVersDef *,scopedNameDef *,versionDef *,argDef *);
static void searchMappedTypes(sipSpec *,scopedNameDef *,versionDef *,argDef *);
static void searchTypedefs(sipSpec *,scopedNameDef *,versionDef *,argDef *);
static void searchEnums(sipSpec *,scopedNameDef *,versionDef *,argDef *);
static void searchClasses(sipSpec *,scopedNameDef *,versionDef *,argDef *);
static void appendToClassVersList(classVersList **,classVersDef *);
static int versTypesArray(sipSpec *,versTypes **);
static int typesAreDifferent(int,int,versTypes *);


/*
 * Transform the parse tree.
 */

void transform(sipSpec *pt)
{
	int classnr;
	classDef *cd;
	classVersDef *cvd;

	if (pt -> module -> name == NULL)
		fatal("No %%Module has been specified for the module\n");

	if (pt -> cppmname == NULL)
		pt -> cppmname = pt -> module -> name -> text;

	/* The simple class checks. */

	classnr = 0;

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		/* Allocate a class number if it is in the main module. */

		if (cd -> iff -> module == pt -> module)
		{
			cd -> classnr = classnr++;

			/*
			 * If we find a class defined in this module called
			 * QObject, assume it's Qt.
			 */

			if (strcmp(cd -> iff -> name -> text,"QObject") == 0)
				pt -> qobjclass = cd -> classnr;
		}
	}

	/* Set the super-class hierachy for each class. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		setHierachy(pt,cvd,cvd);

	/* Add any automatically generated methods. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		if (!isNamespace(cvd))
		{
			addAutoOverloads(cvd,"QObject","tr");
			addAutoOverloads(cvd,"QObject","className");
		}

	/* Transform variables and global functions. */
 
	transformVariableList(pt);
	transformOverloads(pt,NULL,pt -> overs);
 
	/* Transform class variables and functions. */
 
	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		transformCtors(pt,cvd);
		transformOverloads(pt,cvd,cvd -> overs);
	}

	/* Generate the different class views. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		if (isNamespace(cvd))
			continue;

		/* Get the list of visible member functions. */

		getVisibleMembers(cvd);

		if (cvd -> common -> iff -> module == pt -> module)
			addSignalSignatures(pt,cvd);

		/* Get the virtual members. */

		getVirtuals(cvd);
	}
}


/*
 * Add an overload that is automatically generated (typically by Qt's moc).  It
 * must already have been defined in the base class.
 */

static void addAutoOverloads(classVersDef *cvd,char *bcname,char *oname)
{
	classVersList *cvl;
	memberDef *md = NULL;
	overDef *od;

	/* See if this class is derived from the base class. */

	for (cvl = cvd -> hierachy -> next; cvl != NULL; cvl = cvl -> next)
		if (strcmp(classVersBaseName(cvl -> cv),bcname) == 0)
			break;

	if (cvl == NULL)
		return;

	/* Copy all overloads from the base class. */

	for (od = cvl -> cv -> overs; od != NULL; od = od -> next)
		if (strcmp(od -> common -> pyname -> text,oname) == 0)
		{
			overDef *newod;

			if (md == NULL)
			{
				md = sipMalloc(sizeof (memberDef));

				*md = *od -> common;
				md -> module = cvd -> common -> iff -> module;
				md -> next = cvd -> members;
				cvd -> members = md;
			}

			newod = sipMalloc(sizeof (overDef));

			*newod = *od;
			newod -> common = md;
			newod -> next = cvd -> overs;
			cvd -> overs = newod;
		}
}


/*
 * Set the complete hierachy for a class version.
 */

static void setHierachy(sipSpec *pt,classVersDef *base,classVersDef *cvd)
{
	classList *cl;

	/* See if it is a namespace or has already been done. */

	if (isNamespace(cvd) || cvd -> hierachy != NULL)
		return;

	/*
	 * If the base class is in the main module then we need the interface
	 * files for all classes in its hierachy.
	 */

	if (base -> common -> iff -> module == pt -> module)
		setIsUsed(cvd -> common -> iff);

	/* The first thing is itself. */

	appendToClassVersList(&cvd -> hierachy,cvd);

	/* Now do it's superclasses. */

	for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
	{
		classVersDef *scvd;
		classVersList *scvl;

		/*
		 * Find the particular version of this super-class.  There
		 * should be exactly one.
		 */

		for (scvd = pt -> classversions; scvd != NULL; scvd = scvd -> next)
			if (scvd -> common == cl -> c && versionIsSubset(&cvd -> version,&scvd -> version))
				break;

		if (scvd == NULL)
			fatalNoDefinedType(pt,cl -> c -> iff -> fqname,&cvd -> version);

		/* Make sure the super-class's hierachy has been done. */

		setHierachy(pt,base,scvd);

		/* Append the super-classes hierachy. */

		for (scvl = scvd -> hierachy; scvl != NULL; scvl = scvl -> next)
		{
			appendToClassVersList(&cvd -> hierachy,scvl -> cv);

			/* Ripple through the complex flag. */

			if (isComplex(scvl -> cv))
				setIsComplex(cvd);
		}
	}
}


/*
 * Transform the data types for a list of ctors.
 */

static void transformCtors(sipSpec *pt,classVersDef *cvd)
{
	ctorDef *ct, *copyct;
	int privateCopyCt;
	classVersList *cvl;

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		int nrsecs;
		versTypes *vta;

		/*
		 * See if this is already versioned or there are no
		 * secondary versions at all.
		 */

		if (ct -> version.secondary != NULL || (nrsecs = versTypesArray(pt,&vta)) == 0)
			resolveCtorTypes(pt,cvd,ct,ct -> args.args,&ct -> version);
		else
		{
			versionQual *vq;
			versTypes *vt = vta;

			/* Populate the array. */

			for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
			{
				int a;

				vt -> version.secondary = vq;

				for (a = 0; a < ct -> args.nrArgs; ++a)
					vt -> types[a] = ct -> args.args[a];

				resolveCtorTypes(pt,cvd,ct,vt -> types,&vt -> version);

				++vt;
			}

			/* See if different versions have different types. */

			if (typesAreDifferent(nrsecs,ct -> args.nrArgs,vta))
			{
				int i;
				ctorDef oldctor = *ct, *prev;

				for (i = 0; i < nrsecs; ++i)
				{
					int a;

					if (i > 0)
					{
						ct = sipMalloc(sizeof (ctorDef));
						*ct = oldctor;

						prev -> next = ct;
					}

					ct -> version = vta[i].version;

					for (a = 0; a < ct -> args.nrArgs; ++a)
						ct -> args.args[a] = vta[i].types[a];

					prev = ct;
				}
			}
			else
			{
				int a;

				for (a = 0; a < ct -> args.nrArgs; ++a)
					ct -> args.args[a] = vta[0].types[a];
			}

			free(vta);
		}
	}

	/* See if there is a private copy ctor in the hierachy. */
 
	copyct = NULL;
	privateCopyCt = FALSE;
 
	for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		for (ct = cvl -> cv -> ctors; ct != NULL; ct = ct -> next)
		{
			argDef *ad = &ct -> args.args[0];
 
			/* See if is a copy ctor. */
 
			if (ct -> args.nrArgs != 1 || ad -> nrderefs != 0 || !isReference(ad))
				continue;

			/*
			 * We might not have transformed this class yet so
			 * deal with classes and defined types.
			 */
 
			if (ad -> atype == class_type)
			{
				if (ad -> u.cvd != cvl -> cv)
					continue;
			}
			else if (ad -> atype == defined_type)
                        {
				if (strcmp(ad -> u.snd -> name,classVersBaseName(cvl -> cv)) != 0 || ad -> u.snd -> next != NULL)
					continue;
			}
			else
				continue;
 
			/*
			 * Remember if it's in the class we are dealing with.
			 */
 
			if (cvl == cvd -> hierachy)
				copyct = ct;
 
			/* Remember if it is private. */
 
			if (isPrivateCtor(ct))
				privateCopyCt = TRUE;
 
			break;
		}
 
		/* No need to look any further if we've found a private one. */
 
		if (privateCopyCt)
			break;
	}
 
	if (!privateCopyCt && !isOpaque(cvd) && copyct == NULL)
	{
		ctorDef **tailp;
 
		/* Create a default public copy ctor. */
 
		copyct = sipMalloc(sizeof (ctorDef));
 
		copyct -> args.nrArgs = 1;
		copyct -> args.args[0].atype = class_type;
		copyct -> args.args[0].u.cvd = cvd;
		copyct -> args.args[0].argflags = (SECT_IS_PUBLIC | ARG_IS_REF | ARG_IS_CONST);
		copyct -> args.args[0].nrderefs = 0;
		copyct -> args.args[0].defval = NULL;
 
		copyct -> cppcode = NULL;
		copyct -> version = cvd -> version;
		copyct -> next = NULL;
 
		/* Append it to the list. */
 
		for (tailp = &cvd -> ctors; *tailp != NULL; tailp = &(*tailp) -> next)
			;
 
		*tailp = copyct;
	}
}


/*
 * Transform the data types for a list of overloads.
 */

static void transformOverloads(sipSpec *pt,classVersDef *scope,overDef *overs)
{
	overDef *od;

	for (od = overs; od != NULL; od = od -> next)
	{
		int nrsecs;
		versTypes *vta;

		/*
		 * See if this is already versioned or there are no
		 * secondary versions at all.
		 */

		if (od -> version.secondary != NULL || (nrsecs = versTypesArray(pt,&vta)) == 0)
			resolveFuncTypes(pt,scope,od,od -> result,od -> args.args,&od -> version);
		else
		{
			int nrtypes;
			versionQual *vq;
			versTypes *vt = vta;

			nrtypes = od -> args.nrArgs;

			if (od -> result != NULL)
				++nrtypes;

			/* Populate the array. */

			for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
			{
				int a;
				argDef *res;

				vt -> version.secondary = vq;

				for (a = 0; a < od -> args.nrArgs; ++a)
					vt -> types[a] = od -> args.args[a];

				if (od -> result != NULL)
				{
					res = &vt -> types[nrtypes - 1];
					*res = *od -> result;
				}
				else
					res = NULL;

				resolveFuncTypes(pt,scope,od,res,vt -> types,&vt -> version);

				++vt;
			}

			/* See if different versions have different types. */

			if (typesAreDifferent(nrsecs,nrtypes,vta))
			{
				int i;
				overDef oldover = *od, *prev;

				for (i = 0; i < nrsecs; ++i)
				{
					int a;

					if (i > 0)
					{
						od = sipMalloc(sizeof (overDef));
						*od = oldover;

						if (od -> result != NULL)
							od -> result = sipMalloc(sizeof (argDef));

						prev -> next = od;
					}

					od -> version = vta[i].version;

					if (od -> result != NULL)
						*od -> result = vta[i].types[nrtypes - 1];

					for (a = 0; a < od -> args.nrArgs; ++a)
						od -> args.args[a] = vta[i].types[a];

					prev = od;
				}
			}
			else
			{
				int a;

				if (od -> result != NULL)
					*od -> result = vta[0].types[nrtypes - 1];

				for (a = 0; a < od -> args.nrArgs; ++a)
					od -> args.args[a] = vta[0].types[a];
			}

			free(vta);
		}
	}
}


/*
 * Transform the data types for the variables.
 */

static void transformVariableList(sipSpec *pt)
{
	varDef *vd;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		int nrsecs;
		versTypes *vta;

		/*
		 * See if this is already versioned or there are no
		 * secondary versions at all.
		 */

		if (vd -> version.secondary != NULL || (nrsecs = versTypesArray(pt,&vta)) == 0)
			resolveVariableType(pt,vd,&vd -> type,&vd -> version);
		else
		{
			versionQual *vq;
			versTypes *vt = vta;

			/* Populate the array. */

			for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
			{
				vt -> version.secondary = vq;
				vt -> types[0] = vd -> type;

				resolveVariableType(pt,vd,&vt -> types[0],&vt -> version);

				++vt;
			}

			/* See if different versions have different types. */

			if (typesAreDifferent(nrsecs,1,vta))
			{
				int i;
				varDef oldvar = *vd, *prev;

				for (i = 0; i < nrsecs; ++i)
				{
					if (i > 0)
					{
						vd = sipMalloc(sizeof (varDef));
						*vd = oldvar;

						prev -> next = vd;
					}

					vd -> version = vta[i].version;
					vd -> type = vta[i].types[0];

					prev = vd;
				}
			}
			else
				vd -> type = vta[0].types[0];

			free(vta);
		}
	}
}


/*
 * Append a class version to a list of them.
 */

static void appendToClassVersList(classVersList **headp,classVersDef *cvd)
{
	classVersList *new;

	/* Find the end of the list. */

	while (*headp != NULL)
		headp = &(*headp) -> next;

	new = sipMalloc(sizeof (classVersList));

	new -> cv = cvd;
	new -> next = NULL;

	*headp = new;
}


/*
 * Set the list of visible member functions for a class version.
 */

static void getVisibleMembers(classVersDef *cvd)
{
	classVersList *cvl;

	cvd -> visible = NULL;

	for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		memberDef *md;

		for (md = cvl -> cv -> members; md != NULL; md = md -> next)
		{
			visibleList *vl;

			/*
			 * See if it is already in the list.  This has the
			 * desired side effect of eliminating any functions
			 * that have a version closer to this class in the
			 * hierachy.  This is the only reason to define private
			 * functions.
			 */

			for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
				if (vl -> m -> pyname == md -> pyname)
					break;

			/* See if it is a new member function. */

			if (vl == NULL)
			{
				vl = sipMalloc(sizeof (visibleList));

				vl -> m = md;
				vl -> cv = cvl -> cv;
				vl -> next = cvd -> visible;

				cvd -> visible = vl;
			}
		}
	}
}


/*
 * Get all the virtuals for a particular class version.
 */

static void getVirtuals(classVersDef *cvd)
{
	virtOverDef *head;

	head = NULL;

	getVirtualsVersions(&head,cvd -> hierachy,&cvd -> version);

	/* See if there were any for this version. */

	if (head != NULL)
	{
		virtVersDef *vvd;

		vvd = sipMalloc(sizeof (virtVersDef));
 
		vvd -> vo = head;
		vvd -> next = cvd -> vmembers;
 
		cvd -> vmembers = vvd;
	}
}


/*
 * Get the list of visible virtual functions for a class.
 */
 
static void getVirtualsVersions(virtOverDef **headp,classVersList *cvl,versionDef *vd)
{
	classVersDef *cvd = cvl -> cv;
	overDef *od;

	/* Got through the next in the super-class hierachy. */

	for (od = cvd -> overs; od != NULL; od = od -> next)
	{
		virtOverDef *vod;

		if (!isVirtual(od) || !versionIsSubset(&od -> version,vd))
			continue;

		/*
		 * See if a virtual of this name and signature is already in
		 * the list.
		 */
 
		for (vod = *headp; vod != NULL; vod = vod -> next)
			if (strcmp(vod -> o -> cppname,od -> cppname) == 0 && sameOverload(vod -> o,od,TRUE))
				break;
 
		if (vod == NULL)
		{
			vod = sipMalloc(sizeof (virtOverDef));
 
			vod -> o = od;
			vod -> next = *headp;
 
			*headp = vod;
		}
 
		/*
		 * Set the far pointer, possibly overwriting one that wasn't so
		 * far.
		 */

		vod -> farc = cvd;
	}
 
	/* Now go through the next in the hierachy. */
 
	if (cvl -> next != NULL)
		getVirtualsVersions(headp,cvl -> next,vd);
 
	/*
	 * Look for any members in this class that match any in the virtual
	 * list and set the "near" class.
	 */
 
	for (od = cvd -> overs; od != NULL; od = od -> next)
	{
		virtOverDef *vod;

		if (!versionIsSubset(&od -> version,vd))
			continue;

		for (vod = *headp; vod != NULL; vod = vod -> next)
			if (strcmp(vod -> o -> cppname,od -> cppname) == 0 && sameOverload(vod -> o,od,TRUE))
				break;
 
		if (vod != NULL)
		{
			/*
			 * Use this version of the overload to pick up its
			 * protected/private flags.
			 */

			vod -> o = od;
			vod -> nearc = cvd;
		}
	}
}


/*
 * Resolve the types of a function.
 */

static void resolveCtorTypes(sipSpec *pt,classVersDef *scope,ctorDef *ct,
			     argDef *args,versionDef *version)
{
	int a;

	for (a = 0; a < ct -> args.nrArgs; ++a)
	{
		argDef *ad = &args[a];

		getBaseType(pt,scope,ad,version);

		if (ct -> cppcode == NULL && !supportedComplexType(ad))
		{
			fatalScopedName(scope -> common -> iff -> fqname);
			fatalVersion(pt,version);
			fatal(" unsupported ctor argument type - provide explicit code\n");
		}

		ifaceFileIsUsed(pt,scope -> common -> iff -> module,scope,ad,FALSE);
		scopeDefaultValue(pt,scope,ad);
	}
}


/*
 * Resolve the types of a function.
 */

static void resolveFuncTypes(sipSpec *pt,classVersDef *scope,overDef *od,
			     argDef *res,argDef *args,versionDef *version)
{
	int a;
 
	if (res != NULL)
	{
		getBaseType(pt,scope,res,version);

		/* Results must be simple. */

		if (od -> cppcode == NULL && !isAbstract(od))
		{
			if (isSignal(od))
			{
				if (scope != NULL)
				{
					fatalScopedName(scope -> common -> iff -> fqname);
					fatal("::");
				}

				fatal("%s() - signals must return void\n",od -> cppname);
			}

			if (!supportedSimpleType(res))
			{
				if (scope != NULL)
				{
					fatalScopedName(scope -> common -> iff -> fqname);
					fatal("::");
				}

				fatal("%s() - unsupported member function return type - provide %%MemberCode\n",od -> cppname);
			}
		}

		ifaceFileIsUsed(pt,od -> common -> module,scope,res,FALSE);
	}

	for (a = 0; a < od -> args.nrArgs; ++a)
	{
		argDef *ad = &args[a];

		getBaseType(pt,scope,ad,version);

		if (isVirtual(od))
		{
			/* Arguments to virtual functions must be simple. */

			if (!supportedSimpleType(ad))
			{
				if (od -> cppcode == NULL && !isAbstract(od))
				{
					if (scope != NULL)
					{
						fatalScopedName(scope -> common -> iff -> fqname);
						fatal("::");
					}

					fatal("%s() - unsupported argument type for virtual functions - provide %%MemberCode\n",od -> cppname);
				}

				if (od -> virtcode == NULL)
				{
					if (scope != NULL)
					{
						fatalScopedName(scope -> common -> iff -> fqname);
						fatal("::");
					}

					fatal("%s() - unsupported argument type for virtual functions - provide %%VirtualCode\n",od -> cppname);
				}
			}
		}
		else if (od -> cppcode == NULL && !isAbstract(od))
		{
			if (isSignal(od) || isSlot(od))
			{
				/* Signal arguments must be simple. */

				if (!supportedSimpleType(ad))
				{
					if (scope != NULL)
					{
						fatalScopedName(scope -> common -> iff -> fqname);
						fatal("::");
					}

					fatal("%s() - unsupported argument type for signals and slots\n",od -> cppname);
				}
			}
			else if (!supportedComplexType(ad))
			{
				if (scope != NULL)
				{
					fatalScopedName(scope -> common -> iff -> fqname);
					fatal("::");
				}

				fatal("%s() - unsupported member function argument type - provide %%MemberCode\n",od -> cppname);
			}
		}

		ifaceFileIsUsed(pt,od -> common -> module,scope,ad,isSignal(od));
 
		if (scope != NULL)
			scopeDefaultValue(pt,scope,ad);
	}
}


/*
 * Resolve the type of a variable.
 */

static void resolveVariableType(sipSpec *pt,varDef *vd,argDef *vtype,
				versionDef *version)
{
	int bad = TRUE;

	getBaseType(pt,vd -> ecvd,vtype,version);

	switch (vtype -> atype)
	{
	case mapped_type:
	case class_type:
		/* Class, Class & and Class * are supported. */

		if (vtype -> nrderefs <= 1)
			bad = FALSE;
		break;

	case ustring_type:
	case string_type:
		/* (unsigned) char, (unsigned) char * are supported. */

		if (!isReference(vtype) && vtype -> nrderefs <= 1)
			bad = FALSE;
		break;

	case float_type:
	case double_type:
	case enum_type:
	case bool_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case cint_type:
	case int_type:
	case ulong_type:
	case long_type:
		/* These are supported without pointers or references. */

		if (!isReference(vtype) && vtype -> nrderefs == 0)
			bad = FALSE;
		break;

	case struct_type:
	case voidptr_type:
		/* A simple pointer is supported. */

		if (!isReference(vtype) && vtype -> nrderefs == 1)
			bad = FALSE;
		break;
	}

	if (bad)
	{
		fatalScopedName(vd -> fqname);
		fatalVersion(pt,version);
		fatal(" has an unsupported type\n");
	}
 
	if (vtype -> atype != class_type && vd -> accessfunc != NULL)
	{
		fatalScopedName(vd -> fqname);
		fatalVersion(pt,version);
		fatal(" has %%VariableCode but isn't a class instance\n");
	}

	ifaceFileIsUsed(pt,vd -> module,vd -> ecvd,vtype,FALSE);
}



/*
 * Add any signal signatures defined by a class to the list that the module's
 * proxy needs to deal with.
 */

static void addSignalSignatures(sipSpec *pt,classVersDef *cvd)
{
	visibleList *vl;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			int a;

			if (od -> common != vl -> m)
				continue;

			if (isSignal(od))
				addSignalArgs(pt,&od -> args,&od -> version);

			for (a = 0; a < od -> args.nrArgs; ++a)
				if (od -> args.args[a].atype == slotcon_type)
					addSignalArgs(pt,od -> args.args[a].u.sa,&od -> version);
		}
	}
}


/*
 * Add a set of arguments to the list of signals that might be emitted by this
 * module.
 */

static void addSignalArgs(sipSpec *pt,funcArgs *fa,versionDef *vd)
{
	funcArgsList *fl;

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		/*
		 * Note that there is a bug here.  If a module has a signal
		 * with the signature (char *,int) and another
		 * (SIP_ARRAY,SIP_ARRAY_LEN) then SIP thinks they are different
		 * but it will produce two signal handlers with the same C++
		 * signature.  There isn't such a case in PyQt or PyKDE (yet)
		 * so we are OK for now.
		 */

		if (sameFuncArgs(fa,fl -> fa,TRUE))
			break;
	}
 
	/* Add if it is a new signature. */

	if (fl == NULL)
	{
		fl = sipMalloc(sizeof (funcArgsList));
 
		fl -> fa = fa;
		fl -> vol = NULL;
		fl -> next = pt -> sigargs;
 
		pt -> sigargs = fl;
	}

	orVersionQual(pt,&fl -> vol,vd);
}


/*
 * Add a version qualifier to an or'ed list of them.
 */

void orVersionQual(sipSpec *pt,versionOrList **headp,versionDef *vd)
{
	versionOrList *vol;
	versionQual *vq;

	/* Check if the expression is already true. */

	if (*headp != NULL && (*headp) -> version.secondary == NULL)
		return;

	/* See if the secondary is already included. */

	for (vol = *headp; vol != NULL; vol = vol -> next)
		if (vol -> version.secondary == vd -> secondary)
			return;

	/* Create the new part of the expression. */

	vol = sipMalloc(sizeof (versionOrList));

	vol -> version = *vd;
	vol -> next = *headp;

	*headp = vol;

	/* If the new part was true, then the whole expression is true. */

	if (vol -> version.secondary == NULL)
	{
		/* Truncate the list, keeping the new (true) part. */

		vol -> next = NULL;
		return;
	}

	/*
	 * Make sure that at least one of the defined secondaries is missing
	 * from the expression.  If not then the expression is unconditionally
	 * true.
	 */

	for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
	{
		for (vol = *headp; vol != NULL; vol = vol -> next)
			if (vol -> version.secondary == vq)
				break;

		if (vol == NULL)
			return;
	}

	(*headp) -> version.secondary = NULL;
	(*headp) -> next = NULL;
}


/*
 * See if a simple type is supported by the generated code.  Simple types are
 * typically those that we can deal with when getting them from C++ code.
 */

static int supportedSimpleType(argDef *ad)
{
	/*
	 * In addition to complex types, we can't support references to
	 * anything other than classes.
	 */

	return (supportedComplexType(ad) &&
		(ad -> atype == class_type || ad -> atype == mapped_type || !isReference(ad)));
}


/*
 * See if a complex type is supported by the generated code.  Complex types are
 * typically those that we can deal with when passing them to C++ code.
 */

static int supportedComplexType(argDef *ad)
{
	switch (ad -> atype)
	{
	case signal_type:
	case rxcon_type:
	case rxdis_type:
	case slotcon_type:
	case slotdis_type:
	case array_type:
	case uarray_type:
	case arraysize_type:
	case arrayusize_type:
		/* These can only appear in argument lists without * or &. */

		return TRUE;

	case ustring_type:
	case string_type:
		/* (unsigned) char and (unsigned) char * are supported. */

		if (!isReference(ad) && ad -> nrderefs <= 1)
			return TRUE;
		break;

	case float_type:
	case double_type:
	case enum_type:
	case bool_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case cint_type:
	case int_type:
	case ulong_type:
	case long_type:
		/* These are supported without pointers or references. */

		if (!isReference(ad) && ad -> nrderefs == 0)
			return TRUE;
		break;

	case mapped_type:
	case class_type:
		/* Class, Class & and Class * are supported. */

		if ((isReference(ad) && ad -> nrderefs == 0) || (!isReference(ad) && ad -> nrderefs <= 1))
			return TRUE;
		break;

	case struct_type:
	case voidptr_type:
		/*
		 * A simple pointer is supported (although we could easily
		 * handle greater indirections.
		 */

		if (!isReference(ad) && ad -> nrderefs == 1)
			return TRUE;
		break;
	}

	/* Not supported. */

	return FALSE;
}


/*
 * Put a scoped name to stderr.
 */

void fatalScopedName(scopedNameDef *snd)
{
	while (snd != NULL)
	{
		fatal("%s",snd -> name);

		snd = snd -> next;

		if (snd != NULL)
			fatal("::");
	}
}


/*
 * Compare two overloads and return TRUE if they are the same as far as C++ is
 * concerned.
 */

int sameOverload(overDef *od1,overDef *od2,int strict)
{
	/* They must both be const, or both not. */

	if (isConst(od1) != isConst(od2))
		return FALSE;

	/* The return type must be the same. */

	if (od1 -> result == NULL && od2 -> result != NULL)
		return FALSE;

	if (od1 -> result != NULL && od2 -> result == NULL)
		return FALSE;

	if (od1 -> result != NULL && od2 -> result != NULL && !sameArgType(od1 -> result,od2 -> result,strict))
		return FALSE;

	return sameFuncArgs(&od1 -> args,&od2 -> args,strict);
}


/*
 * Compare two sets of function arguments are return TRUE if they are the same.
 */

int sameFuncArgs(funcArgs *fa1,funcArgs *fa2,int strict)
{
	int a;

	if (strict)
	{
		/* The number of arguments must be the same. */

		if (fa1 -> nrArgs != fa2 -> nrArgs)
			return FALSE;
	}
	else
	{
		int na1, na2;

		/* We only count the compulsory arguments. */

		na1 = 0;

		for (a = 0; a < fa1 -> nrArgs; ++a)
		{
			if (fa1 -> args[a].defval != NULL)
				break;

			++na1;
		}

		na2 = 0;

		for (a = 0; a < fa2 -> nrArgs; ++a)
		{
			if (fa2 -> args[a].defval != NULL)
				break;

			++na2;
		}

		if (na1 != na2)
			return FALSE;
	}

	/* The arguments must be the same. */

	for (a = 0; a < fa1 -> nrArgs; ++a)
	{
		if (!strict && fa1 -> args[a].defval != NULL)
			break;

		if (!sameArgType(&fa1 -> args[a],&fa2 -> args[a],strict))
			return FALSE;
	}

	/* Must be the same if we've got this far. */

	return TRUE;
}


/*
 * Compare two argument types and return TRUE if they are the same.
 */

int sameArgType(argDef *a1,argDef *a2,int strict)
{
	/* The indirection and the references must be the same. */

	if (isReference(a1) != isReference(a2) || a1 -> nrderefs != a2 -> nrderefs)
		return FALSE;

	/*
	 * In non-strict, just check if they are subject to Python conversions.
	 */

	if (!strict)
		if ((pyToInt(a1 -> atype) && pyToInt(a2 -> atype)) ||
		    (pyToDouble(a1 -> atype) && pyToDouble(a2 -> atype)))
		return TRUE;

	return sameBaseType(a1,a2,strict);
}


/*
 * Compare two basic types and return TRUE if they are the same.
 */

int sameBaseType(argDef *a1,argDef *a2,int strict)
{
	/* The types must be the same. */

	if (a1 -> atype != a2 ->atype)
		return FALSE;

	switch (a1 -> atype)
	{
	case class_type:
		if (a1 -> u.cvd != a2 -> u.cvd)
			return FALSE;

		break;

	case enum_type:
		if (strict && a1 -> u.ed != a2 -> u.ed)
			return FALSE;

		break;

	case slotcon_type:
	case slotdis_type:
		if (!sameFuncArgs(a1 -> u.sa,a2 -> u.sa,strict))
			return FALSE;

		break;

	case template_type:
		{
			int a;
			templateDef *td1, *td2;

			td1 = a1 -> u.td;
			td2 = a2 -> u.td;

			if (!sameScopedName(td1 -> fqname,td2 -> fqname) != 0 ||
			    td1 -> types.nrArgs != td2 -> types.nrArgs)
				return FALSE;

			for (a = 0; a < td1 -> types.nrArgs; ++a)
				if (!sameBaseType(&td1 -> types.args[a],&td2 -> types.args[a],strict))
					return FALSE;

			break;
		}

	case struct_type:
		if (!sameScopedName(a1 -> u.sname,a2 -> u.sname) != 0)
			return FALSE;

		break;

	case defined_type:
		if (!sameScopedName(a1 -> u.snd,a2 -> u.snd))
			return FALSE;

		break;

	case mapped_type:
		if (a1 -> u.mtd != a2 -> u.mtd)
			return FALSE;

		break;
	}

	/* Must be the same if we've got this far. */

	return TRUE;
}


/*
 * Return TRUE if two scoped names are the same.
 */

int sameScopedName(scopedNameDef *snd1,scopedNameDef *snd2)
{
	while (snd1 != NULL && snd2 != NULL && strcmp(snd1 -> name,snd2 -> name) == 0)
	{
		snd1 = snd1 -> next;
		snd2 = snd2 -> next;
	}

	return (snd1 == NULL && snd2 == NULL);
}


/*
 * Print an optional version to stderr.
 */

void fatalVersion(sipSpec *pt,versionDef *vd)
{
	if (pt -> module -> required != NULL || vd -> secondary != NULL)
	{
		fatal("(");

		if (pt -> module -> required != NULL)
		{
			fatal("%s",pt -> module -> required -> name);

			if (vd -> secondary != NULL)
				fatal(" && ");
		}

		if (vd -> secondary != NULL)
			fatal("%s",vd -> secondary -> name);

		fatal(")");
	}
}


/*
 * Add an explicit scope to the default value of an argument if possible.
 */

static void scopeDefaultValue(sipSpec *pt,classVersDef *cvd,argDef *ad)
{
	valueDef *vd, **tailp;

	/*
	 * We do a quick check to see if we need to do anything.  This means
	 * we can limit the times we need to copy the default value.  It needs
	 * to be copied because it will be shared by class versions that have
	 * been created on the fly and it may need to be scoped differently for
	 * each of those versions.
	 */

	for (vd = ad -> defval; vd != NULL; vd = vd -> next)
		if (vd -> vtype == scoped_value && vd -> u.vscp -> next == NULL)
			break;

	if (vd == NULL)
		return;

	/*
	 * It's not certain that we will do anything, but we assume we will and
	 * start copying.
	 */

	tailp = &ad -> defval;

	for (vd = ad -> defval; vd != NULL; vd = vd -> next)
	{
		classVersList *cvl;
		scopedNameDef *origname;

		/* Make the copy. */

		*tailp = sipMalloc(sizeof (valueDef));

		**tailp = *vd;
		vd = *tailp;
		tailp = &vd -> next;

		/*
		 * Skip this part of the expression if it isn't a named value
		 * or it already has a scope.
		 */

		if (vd -> vtype != scoped_value || vd -> u.vscp -> next != NULL)
			continue;

		/*
		 * Search the class hierachy for an enum value with the same
		 * name.  If we don't find one, leave it as it is (the compiler
		 * will find out if this is a problem).
		 */

		origname = vd -> u.vscp;

		for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
		{
			enumDef *ed;

			for (ed = pt -> enums; ed != NULL; ed = ed -> next)
			{
				enumValueDef *evd;

				if (ed -> ecvd != cvl -> cv)
					continue;

				for (evd = ed -> values; evd != NULL; evd = evd -> next)
					if (strcmp(evd -> name -> text,origname -> name))
					{
						scopedNameDef *snd;

						/*
						 * Take the scope from the
						 * class that the enum was
						 * defined in.
						 */

						snd = copyScopedName(cvl -> cv -> common -> iff -> fqname);
						appendScopedName(&snd,origname);

						vd -> u.vscp = snd;

						/* Nothing more to do. */

						break;
					}

				if (evd != NULL)
					break;
			}

			if (ed != NULL)
				break;
		}
	}
}


/*
 * Make sure the version of a type is a base type.
 */

static void getBaseType(sipSpec *pt,classVersDef *defscope,argDef *type,versionDef *vd)
{
	/* Loop until we've got to a base type. */

	while (type -> atype == defined_type)
	{
		scopedNameDef *snd = type -> u.snd;

		type -> atype = no_type;

		if (defscope != NULL)
			searchScope(pt,defscope,snd,vd,type);

		if (type -> atype == no_type)
			searchMappedTypes(pt,snd,vd,type);

		if (type -> atype == no_type)
			searchTypedefs(pt,snd,vd,type);

		if (type -> atype == no_type)
			searchEnums(pt,snd,vd,type);

		if (type -> atype == no_type)
			searchClasses(pt,snd,vd,type);

		if (type -> atype == no_type)
			fatalNoDefinedType(pt,snd,vd);
	}

	/* Get the base of type of any slot arguments. */

	if (type -> atype == slotcon_type || type -> atype == slotdis_type)
	{
		int sa;

		for (sa = 0; sa < type -> u.sa -> nrArgs; ++sa)
			getBaseType(pt,defscope,&type -> u.sa -> args[sa],vd);
	}

	/* Replace the base type if it has been mapped. */

	if (type -> atype == struct_type || type -> atype == template_type)
		searchMappedTypes(pt,NULL,vd,type);
}


/*
 * Search for a versioned name in a scope and return the corresponding type.
 */

static void searchScope(sipSpec *pt,classVersDef *scope,scopedNameDef *snd,
			versionDef *vd,argDef *ad)
{
	scopedNameDef *tmpsnd = NULL;
	classVersList *cvl;

	for (cvl = scope -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		/* Append the name to the scope and see if it exists. */

		tmpsnd = copyScopedName(classVersFQName(cvl -> cv));
		appendScopedName(&tmpsnd,copyScopedName(snd));

		searchMappedTypes(pt,tmpsnd,vd,ad);

		if (ad -> atype != no_type)
			break;

		searchTypedefs(pt,tmpsnd,vd,ad);

		if (ad -> atype != no_type)
			break;

		searchEnums(pt,tmpsnd,vd,ad);

		if (ad -> atype != no_type)
			break;

		searchClasses(pt,tmpsnd,vd,ad);

		if (ad -> atype != no_type)
			break;

		freeScopedName(tmpsnd);
		tmpsnd = NULL;
	}

	if (tmpsnd != NULL)
		freeScopedName(tmpsnd);
}


/*
 * Search the mapped types for a versioned name and return the type.
 */

static void searchMappedTypes(sipSpec *pt,scopedNameDef *snd,versionDef *vd,argDef *ad)
{
	mappedTypeDef *mtd;
	scopedNameDef *oname;

	/* Patch back to defined types so we can use sameBaseType(). */

	if (snd != NULL)
	{
		oname = ad -> u.snd;
		ad -> u.snd = snd;
		ad -> atype = defined_type;
	}

	for (mtd = pt -> mappedtypes; mtd != NULL; mtd = mtd -> next)
	{
		if (!sameBaseType(ad,&mtd -> type,TRUE))
			continue;

		if (!versionIsSubset(vd,&mtd -> version))
			continue;

		/* Copy the type. */

		ad -> atype = mapped_type;
		ad -> u.mtd = mtd;

		return;
	}

	/* Restore because we didn't find anything. */

	if (snd != NULL)
	{
		ad -> u.snd = oname;
		ad -> atype = no_type;
	}
}


/*
 * Search the typedefs for a versioned name and return the type.
 */

static void searchTypedefs(sipSpec *pt,scopedNameDef *snd,versionDef *vd,argDef *ad)
{
	typedefDef *td;

	for (td = pt -> typedefs; td != NULL; td = td -> next)
	{
		if (!sameScopedName(td -> fqname,snd))
			continue;

		if (!versionIsSubset(vd,&td -> version))
			continue;

		/* Copy the type. */

		ad -> atype = td -> type.atype;
		ad -> argflags |= td -> type.argflags;
		ad -> nrderefs += td -> type.nrderefs;
		ad -> u = td -> type.u;

		break;
	}
}


/*
 * Search the enums for a versioned name and return the type.
 */

static void searchEnums(sipSpec *pt,scopedNameDef *snd,versionDef *vd,argDef *ad)
{
	enumDef *ed;

	for (ed = pt -> enums; ed != NULL; ed = ed -> next)
	{
		if (ed -> fqname == NULL)
			continue;

		if (!sameScopedName(ed -> fqname,snd))
			continue;

		if (!versionIsSubset(vd,&ed -> version))
			continue;

		ad -> atype = enum_type;
		ad -> u.ed = ed;

		break;
	}
}


/*
 * Search the classes for a versioned name and return the type.
 */

static void searchClasses(sipSpec *pt,scopedNameDef *snd,versionDef *vd,argDef *ad)
{
	classVersDef *cvd;

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		if (!sameScopedName(classVersFQName(cvd),snd))
			continue;

		if (!versionIsSubset(vd,&cvd -> version))
			continue;

		ad -> atype = class_type;
		ad -> u.cvd = cvd;

		break;
	}
}


/*
 * Print an error message describing an undefined defined type to stderr and
 * terminate.
 */

static void fatalNoDefinedType(sipSpec *pt,scopedNameDef *snd,versionDef *vd)
{
	fatalScopedName(snd);
	fatalVersion(pt,vd);
	fatal(" is undefined\n");
}


/*
 * If a type has an interface file then add it to the appropriate list of used
 * interface files so that the header file is #included in the generated code.
 */

static void ifaceFileIsUsed(sipSpec *pt,moduleDef *mod,classVersDef *cvd,argDef *ad,int issig)
{
	ifaceFileDef *iff;

	switch (ad -> atype)
	{
	case class_type:
		iff = ad -> u.cvd -> common -> iff;
		break;

	case mapped_type:
		iff = ad -> u.mtd -> iff;
		break;

	default:
		return;
	}

	if (cvd != NULL && cvd -> common -> iff != iff)
		addToUsedList(&cvd -> used,iff);

	/*
	 * It's used by the main module if it is outside a class or it is a
	 * signal.
	 */

	if (mod == pt -> module && (cvd == NULL || issig))
		addToUsedList(&pt -> used,iff);
}


/*
 * Return an array (one element for each possible version) of a set of
 * versioned types.  Return NULL if there are no versions.
 */

static int versTypesArray(sipSpec *pt,versTypes **vtap)
{
	int nvers;
	versionQual *vq;

	/* Count the number of versions. */

	nvers = 0;

	for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
		++nvers;

	*vtap = (nvers > 0 ? sipMalloc(sizeof (versTypes) * nvers) : NULL);

	return nvers;
}


/*
 * Return TRUE if an array of versioned types has at least two different rows.
 */

static int typesAreDifferent(int nrvers,int nrtypes,versTypes *vta)
{
	int v1;

	for (v1 = 0; v1 < nrvers; ++v1)
	{
		int v2;

		for (v2 = v1 + 1; v2 < nrvers; ++v2)
		{
			int i;

			for (i = 0; i < nrtypes; ++i)
				if (!sameArgType(&vta[v1].types[i],&vta[v2].types[i],TRUE))
					return TRUE;
		}
	}

	/* Must be the same if we have got this far. */

	return FALSE;
}
